%{
/*	$Id: parse.y,v 1.12 2006/02/17 09:35:14 mbalmer Exp $	*/

/*
 * Copyright (c) 2004, 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#if defined(__OpenBSD__)
#include <sys/limits.h>
#endif

#include <err.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>

#include "smtp-vilter.h"
#include "pathnames.h"

#if defined(__FreeBSD__) || defined(__linux__)
#include "strtonum.h"
#endif

extern FILE	*spamdin;
extern int	 spamdlineno;
extern char	*spamdtext;

extern int	 spamd_port;
extern char	*spamd_host;
extern char	*bind_addr;
extern int	 spamd_tries;
extern int	 spamd_timeout;
extern off_t	 spamd_maxsize;

static int	 spamderrcnt;
static char	*spamdcfgfile;

int	spamderror(const char *, ...);
int	spamdparse(void);
int	spamdlex(void);
%}

%union {
	long number;
	char *string;
}

%token	HOST PORT TRIES TIMEOUT MAXSIZE BIND
%token	<string>	NUMBER
%token	<string>	TEXT

%%
statement	: /* empty */
		| statement '\n'
		| statement host '\n'
		| statement port '\n'
		| statement bind '\n'
		| statement tries '\n'
		| statement timeout '\n'
		| statement maxsize '\n'
		;

host		: HOST '=' TEXT				{
			spamd_host = $3;
		}
		;

port		: PORT '=' NUMBER			{
			const char *errstr;

			spamd_port = strtonum($3, 1, 65535, &errstr);
			if (errstr)
				errx(1, "spamd: port number is %s:%s", errstr, $3);
			free($3);
		}
		;

bind		: BIND '=' TEXT				{
			bind_addr = $3;
		}
		;

tries		: TRIES '=' NUMBER			{
			const char *errstr;

			spamd_tries = strtonum($3, 1, 1024, &errstr);
			if (errstr)
				errx(1, "spamd: number of tries is %s:%s", errstr, $3);
			free($3);
		}
		;
timeout		: TIMEOUT '=' NUMBER			{
			const char *errstr;

			spamd_timeout = strtonum($3, 0, 65535, &errstr);
			if (errstr)
				errx(1, "spamd: timeout is %s:%s", errstr, $3);
			free($3);
		}
		;

maxsize		: MAXSIZE '=' NUMBER			{
			const char *errstr;

			spamd_maxsize = strtonum($3, 0, LLONG_MAX, &errstr);
			if (errstr)
				errx(1, "spamd: maxsize is %s:%s", errstr, $3);
			free($3);
		}
		;
%%

int
vilter_init(char *cfgfile)
{
	if (verbose)
		warnx("spamd: vilter_init()");

	/* Default values */

	spamd_port = 783;
	if ((spamd_host = strdup("localhost")) == NULL)
		err(1, "spamd: out of memory");
	spamd_tries = 1;
	spamd_timeout = TIMEOUT;
	spamd_maxsize = 0;
	bind_addr = NULL;

	if (cfgfile == NULL)
		cfgfile = SPAMD_CONF;

	spamdlineno = 1;
	spamderrcnt = 0;
	spamdcfgfile = cfgfile;
	if ((spamdin = fopen(cfgfile, "r")) != NULL) {
		while (!feof(spamdin)) {
			spamdparse();
		}
		fclose(spamdin);
		if (spamderrcnt)
			errx(1, "configuration file contains errors, terminating");
	} else if (verbose)
		warnx("spamd: configuration file %s for spamd backend not found, using default values", cfgfile);

	return 0;
}

int
spamderror(const char *fmt, ...)
{
	va_list		 ap;
	char		*nfmt;

	++spamderrcnt;
	va_start(ap, fmt);
	if (asprintf(&nfmt, "%s, line %d: %s near '%s'",
	     spamdcfgfile, spamdlineno, fmt, spamdtext) == -1)
		errx(1, "spamd: asprintf failed");
	va_end(ap);
	fprintf(stderr, "%s\n", nfmt);
	free(nfmt);
	return (0);
}
