/*	$Id: vilter-icap.c,v 1.32 2006/02/17 07:33:14 mbalmer Exp $	*/

/*
 * Copyright (c) 2004, 2005, 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include "pathnames.h"
#include "smtp-vilter.h"

#ifdef __linux__
#include "strlfunc.h"
#endif

#define TIMEOUT 	60
#define MAXBUF		8160
#define MAXSTR		64
#define MAXTRIES	256

int	 icap_port;
char	*icap_host;
char	*bind_addr;
char	*icap_uri;
char	*icap_query;
int	 icap_tries;
int	 icap_timeout;

int
vilter_scan(char *fn, size_t fnlen, char *chroot, char *virus, size_t namelen)
{
	int fd;
	int try;
	int nread, nsent;
	FILE *fp;
	struct sockaddr_in server_sockaddr;
	struct sockaddr_in bind_sockaddr;

	struct timeval tv, *t;
	struct hostent *hostent;
	char req_hdr[MAXBUF];
	char res_hdr[MAXBUF];
	char buf[MAXBUF];
	struct stat statbuf;
	SOCK *sp;
	int result;
	char *p;
	int n;

	if (stat(fn, &statbuf)) {
		syslog(LOG_ERR, "icap: can't stat file to scan");
		return SCAN_ERROR;
	}

	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog(LOG_ERR, "icap: unable to obtain network");
		return SCAN_ERROR;
	}

	if (bind_addr != NULL) {
		bzero((char *)&server_sockaddr, sizeof(server_sockaddr));
		bind_sockaddr.sin_family = AF_INET;

		if (inet_aton(bind_addr, &bind_sockaddr.sin_addr) == 0) {
			hostent = gethostbyname(bind_addr);
			if (!hostent) {
				syslog(LOG_ERR, "icap: unknown bind address: %s", bind_addr);
				close(fd);
				return SCAN_ERROR;
			}
			bind_sockaddr.sin_family = hostent->h_addrtype;
			memcpy(&bind_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
		}
		if (bind(fd, (struct sockaddr *)&bind_sockaddr, sizeof(bind_sockaddr))) {
			syslog(LOG_ERR, "icap: can't bind to address %s", bind_addr);
			close(fd);
			return SCAN_ERROR;
		}
	}
	bzero((char *)&server_sockaddr, sizeof(server_sockaddr));

	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(icap_port);

	if (inet_aton(icap_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(icap_host);
		if (!hostent) {
			syslog(LOG_ERR, "icap: unknown host: %s", icap_host);
			close(fd);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}
	try = 0;

	while (connect(fd, (struct sockaddr *)&server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;

		if (try == icap_tries) {
			syslog(LOG_ERR, "icap: unable to connect socket");
			close(fd);
			return SCAN_ERROR;
		} else
			sleep(1);
	}
	if (icap_timeout > 0) {
		tv.tv_sec = icap_timeout;
		tv.tv_usec = 0;
		t = &tv;
	} else
		t = NULL;

	sp = fdsock(fd);

	snprintf(req_hdr, sizeof(req_hdr),
	    "GET http://www.msys.ch%s%s HTTP/1.1\r\n\r\n",
	    *fn == '/' ? "" : "/", fn);

	snprintf(res_hdr, sizeof(res_hdr),
	    "HTTP/1.1 200 OK\r\n"
	    "Transfer-Encoding: chunked\r\n\r\n");

	bzero(buf, sizeof(buf));

	snprintf(buf, sizeof(buf), "RESPMOD %s%s%s ICAP/1.0\r\n"
	    "Host: %s:%d\r\n"
	    "User-Agent: smtp-vilter-%s/icap\r\n"
	    "Allow: 204\r\n"
	    "Encapsulated: req-hdr=0, res-hdr=%ld, res-body=%ld\r\n\r\n",
	    icap_uri, icap_query != NULL ? "?" : "", icap_query != NULL ? icap_query : "",
	    icap_host, icap_port, VERSION, (long)strlen(req_hdr), (long)(strlen(req_hdr) + strlen(res_hdr)));

	if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf)) {
		syslog(LOG_ERR, "icap: send error");
		close(fd);
		return SCAN_ERROR;
	}
	if (to_send(fd, req_hdr, strlen(req_hdr), 0, t) != strlen(req_hdr)) {
		syslog(LOG_ERR, "icap: send error");
		close(fd);
		return SCAN_ERROR;
	}
	if (to_send(fd, res_hdr, strlen(res_hdr), 0, t) != strlen(res_hdr)) {
		syslog(LOG_ERR, "icap: send error");
		close(fd);
		return SCAN_ERROR;
	}
	fp = fopen(fn, "r");

	if (fp == NULL) {
		syslog(LOG_ERR, "icap: can't open file");
		close(fd);
		return SCAN_ERROR;
	}

	bzero(buf, sizeof(buf));

	while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0) {
		bzero(req_hdr, sizeof(req_hdr));
		snprintf(req_hdr, sizeof(req_hdr), "%x\r\n", nread);

		if (to_send(fd, req_hdr, strlen(req_hdr), 0, t) != strlen(req_hdr)) {
			syslog(LOG_ERR, "icap: chunksize send error");
			goto response;
		}

		if ((nsent = to_send(fd, buf, nread, 0, t)) != nread) {
			syslog(LOG_ERR, "icap: error sending chunk data of %d bytes (%d)", nread, nsent);
			goto response;
		}
		bzero(buf, sizeof(buf));

		snprintf(req_hdr, sizeof(req_hdr), "\r\n");

		if (to_send(fd, req_hdr, strlen(req_hdr), 0, t) != strlen(req_hdr)) {
			syslog(LOG_ERR, "icap: crlf send error %ld bytes", (long)strlen(req_hdr));
			goto response;
		}
	}
	fclose(fp);	
	fflush(stdout);

	snprintf(buf, sizeof(buf), "\r\n0\r\n\r\n");

	if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf))
		syslog(LOG_ERR, "icap: send error");

response:	
	result = SCAN_ERROR;

	while (to_readln(buf, sizeof(buf), sp, t) > 0) {
		if (verbose) {
			buf[strlen(buf) - 1] = '\0';
			warnx("<-- %s", buf);
		}
		if (!strncmp(buf, "ICAP/1.0 204", strlen("ICAP/1.0 204")))
			result = SCAN_OK;
		else if (!strncmp(buf, "ICAP/1.0 403", strlen("ICAP/1.0 403")))
			result = SCAN_VIRUS;
		else if (!strncmp(buf, "X-Violations-Found", strlen("X-Violations-Found")))
			result = SCAN_VIRUS;
		else if (!strncmp(buf, "X-Infection-Found", strlen("X-Infection-Found"))) {
			if ((p = strstr(buf, "Threat=")) != NULL) {
				p += strlen("Threat=");

				n = 0;
				while (n < namelen && *p && *p != ';')
					virus[n++] = *p++;
				virus[n] = '\0';
			}
		}
	}
	sclose(sp);

	if (verbose) {
		switch (result) {
		case SCAN_OK:
			warnx("icap: file contains no virus");
			break;
		case SCAN_VIRUS:
			warnx("icap: found a virus: %s", virus);
			break;
		case SCAN_ERROR:
			warnx("icap: error during virus scan");
			break;
		}
	}

	if (result == SCAN_ERROR)
		syslog(LOG_ERR, "icap: error during virus scan");

	return result;
}

char *
vilter_name(void)
{
	return "ICAP 1.0 Vilter (icap)";
}

int
vilter_type(void)
{
	return BE_SCAN_VIRUS;
}

void
vilter_exit(void)
{
	if (verbose)
		warnx("icap: vilter_exit()");

	free(icap_host);
	free(icap_uri);
	free(icap_query);
}
