/*	$Id: engine.c,v 1.37 2006/02/08 14:47:28 mbalmer Exp $	*/

/*
 * Copyright (c) 2003, 2004, 2005, 2006 Marc Balmer <marc@msys.ch>.
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <dlfcn.h>
#include <err.h>
#include <errno.h>
#include <grp.h>
#ifdef ENABLE_LDAP
#include <ldap.h>
#endif
#include <math.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "libmilter/mfapi.h"

#include "pathnames.h"
#include "smtp-vilter.h"

#include "imsg.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#include "strtonum.h"
#else
#include <sys/syslimits.h>
#endif

#ifdef __FreeBSD__
#include "strtonum.h"
#endif

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256

#define LOG_UNSET	-1

extern char	*port;
extern int	 virus_strategy;
extern int	 spam_strategy;
extern int	 unwanted_strategy;
extern int	 error_strategy;

extern int	 virus_reaction;
extern int	 spam_reaction;
extern int	 unwanted_reaction;
extern int	 clean_reaction;

/* there is only one reaction: add to table */

extern char	*virus_table;
extern long	 virus_duration;
extern char	*spam_table;
extern long	 spam_duration;
extern char	*unwanted_table;
extern long	 unwanted_duration;
extern char	*clean_table;
extern long	 clean_duration;

extern int	 markall;
extern int	 verbose;
extern char	*pidfile;
extern char	*cfgfile;
extern char	*tmpdir;
extern int	 tmpfiles_gr;
extern int	 tmpfiles_setgrp;
extern char	*backend;
extern char	*recipient_notification;
extern char	*notify_only;
extern int	 log_facility;
extern char	*logfile;
extern FILE	*logfp;
extern char	*user;
extern char	*group;
extern char	*chrootdir;
extern rlim_t	 rlimit_nofile;
extern rlim_t	 rlimit_nproc;
extern int	 nbe;
extern char	*spamprefix;
extern int	 logvirus, logspam, logunwanted;
extern int	 keep_tmpfiles;
extern int	 ex_timeout;

struct imsgbuf	*ibuf_main;

/* For ceonnection reference counting */

static int		conn_cnt = 0;
static pthread_mutex_t	conn_cnt_mutex = PTHREAD_MUTEX_INITIALIZER;

extern char *__progname;

#ifdef ENABLE_LDAP
extern LDAP	*ld;
#endif
extern char	*searchbase;

struct recipient {
	char		*recipient;
	char		*mailLocalAddress;	/* canonicalized recipient */
#ifdef ENABLE_LDAP
	int		 msgid;
#endif
	int		 virus_strategy;
	int		 spam_strategy;
	int		 unwanted_strategy;
	char		*spamprefix;
	struct domain	*domain;
	SLIST_ENTRY(recipient) entries;
};

struct domain {
	char	*mailLocalDomain;
#ifdef ENABLE_LDAP
	int	 msgid;
#endif
	int	 virus_strategy;
	int	 spam_strategy;
	int	 unwanted_strategy;
	char	*spamprefix;
	SLIST_ENTRY(domain) entries;
};

extern struct backend *be;

struct connection {
	char		*hostname;
	_SOCK_ADDR	*hostaddr;

	char		*clientaddr;
	char		*heloname;

	int		 blocked;

	/*
	 * The following variables are for statistics,
	 * but not used at the moment
	 */

	int		 num_msgs;
	int		 num_virus;
	int		 num_spam;
	int		 num_unwanted;
};

struct message {
	char			 tfn[MAX_FILE];	/* Temporary file name */
	char			*msgid;		/* Message ID */
	char			*envfrom;	/* Envelope sender */
	SLIST_HEAD(, recipient)	 rhead;		/* List of recipients */
	SLIST_HEAD(, domain)	 dhead;		/* List of recp. domains */
	char			*hdrfrom;	/* From: address in message header */
	char			*hdrto;		/* To: address in message header */
	char			*subject;	/* The message subject */
	int			 notify;	/* if != 0, notify recipient */
	FILE			*fp;
};

struct privdata {
	struct connection	*conn;
	struct message		*msg;
	struct be_data		*backend_data;
};

#ifdef ENABLE_PF
static void pf_table_add(const char *, const char *, long);
static void pf_table_clr(const char *, const char *, long);
static void imsg_commit();
#endif

void
e_sighdlr(int signum)
{
	switch (signum) {
	case SIGHUP:
		break;
	}
}

#ifdef ENABLE_LDAP
/*
 * Decode strategies represented as string values
 */

static int
get_v_strategy(const char *s)
{
	if (!strcmp(s, "discard"))
		return STRATEGY_DISCARD;
	else if (!strcmp(s, "mark"))
		return STRATEGY_MARK;
	else if (!strcmp(s, "notify-recipient"))
		return STRATEGY_NOTIFY_RECIPIENT;
	else if (!strcmp(s, "reject"))
		return STRATEGY_REJECT;
	else
		return STRATEGY_UNSET;
}

static int
get_su_strategy(const char *s)
{
	if (!strcmp(s, "discard"))
		return STRATEGY_DISCARD;
	else if (!strcmp(s, "mark"))
		return STRATEGY_NOTIFY_RECIPIENT;
	else if (!strcmp(s, "reject"))
		return STRATEGY_REJECT;
	else
		return STRATEGY_UNSET;
}
#endif

/* Allocate and initialize per-connection memory */

static sfsistat
connection_setup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct connection	*conn;

	if (verbose)
		warnx("connection setup");

	if ((priv = malloc(sizeof(struct privdata))) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for internal data");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv, sizeof(struct privdata));

	if ((conn = malloc(sizeof(struct connection))) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for connection data");
		free(priv);
		return SMFIS_TEMPFAIL;
	}
	bzero(conn, sizeof(struct connection));

	priv->conn = conn;

	if ((priv->backend_data = malloc(sizeof(struct be_data) * nbe)) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for backend data");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv->backend_data, sizeof(struct be_data) * nbe);

	if (smfi_setpriv(ctx, priv)) {
		syslog(LOG_ERR, "unable to store internal data");
		free(priv->conn);
		free(priv->backend_data);
		free(priv);
		return SMFIS_TEMPFAIL;
	}

	if (pthread_mutex_lock(&conn_cnt_mutex))
		syslog(LOG_ERR, "unable to increase connection count");
	else {
		++conn_cnt;
		if (verbose)
			warnx("connection increased to %d", conn_cnt);
		pthread_mutex_unlock(&conn_cnt_mutex);
	}

	return SMFIS_CONTINUE;
}

/* Allocate and initialize per-message memory */

static sfsistat
message_setup(SMFICTX *ctx)
{
	struct privdata	*priv;
	struct message	*msg;

	if (verbose)
		warnx("message setup");

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		if ((msg = malloc(sizeof(struct message))) != NULL) {
			bzero(msg, sizeof(struct message));
			SLIST_INIT(&msg->rhead);
			SLIST_INIT(&msg->dhead);
			priv->msg = msg;
			return SMFIS_CONTINUE;
		} else {
			syslog(LOG_ERR, "unable to allocate memory for message");
			return SMFIS_TEMPFAIL;
		}
	} else
		syslog(LOG_ERR, "unable to get internal data in message_setup(),");

	return SMFIS_TEMPFAIL;
}

/* Deallocate per-message memory and ressources*/

static void
message_cleanup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct message		*msg;
	struct recipient	*r;
	struct domain		*d;

	if (verbose)
		warnx("message cleanup");

	priv = smfi_getpriv(ctx);
	if (priv != NULL && priv->msg != NULL) {
		msg = priv->msg;
		if (msg->fp != NULL)
			fclose(msg->fp);

		if (!keep_tmpfiles && unlink(msg->tfn))
			syslog(LOG_ERR, "can't delete temp file %s", msg->tfn);

		if (msg->msgid != NULL)
			free(msg->msgid);
		if (msg->envfrom != NULL)
			free(msg->envfrom);
		/*
		 * if (msg->envto != NULL)
	 	 *	free(msg->envto);
		 */
		while (!SLIST_EMPTY(&msg->rhead)) {
			r = SLIST_FIRST(&msg->rhead);

			free(r->recipient);
			free(r->mailLocalAddress);
			if (r->spamprefix != NULL)
				free(r->spamprefix);

			SLIST_REMOVE_HEAD(&msg->rhead, entries);
			free(r);
		}

		while (!SLIST_EMPTY(&msg->dhead)) {
			d = SLIST_FIRST(&msg->dhead);
			free(d->mailLocalDomain);
			if (d->spamprefix != NULL)
				free(d->spamprefix);
			SLIST_REMOVE_HEAD(&msg->dhead, entries);
			free(d);
		}

		if (msg->hdrfrom != NULL)
			free(msg->hdrfrom);
		if (msg->hdrto != NULL)
			free(msg->hdrto);
		if (msg->subject != NULL)
			free(msg->subject);
		free(msg);
		priv->msg = NULL;
	}
}

/* Free per-connection memory */

static void
connection_cleanup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct connection	*conn;
	
	if (verbose)
		warnx("connection cleanup");

	priv = smfi_getpriv(ctx);
	if (priv != NULL) {
		if (priv->conn != NULL) {
			conn = priv->conn;

			if (conn->hostname != NULL)
				free(conn->hostname);
			if (conn->heloname != NULL)
				free(conn->heloname);
			free(conn);
		}
		if (priv->backend_data != NULL)
			free(priv->backend_data);

		free(priv);
		smfi_setpriv(ctx, NULL);
	} else
		syslog(LOG_ERR, "unable to get internal data in connection_cleanup()");

	if (pthread_mutex_lock(&conn_cnt_mutex))
		syslog(LOG_ERR, "can't decrease connection count");
	else {
		--conn_cnt;
		if (verbose)
			warnx("connection count decreased to %d", conn_cnt);
		if (conn_cnt < 0)
			syslog(LOG_ERR, "connection count drops below zero");
		pthread_mutex_unlock(&conn_cnt_mutex);
	}
}

static sfsistat
smtp_vilter_connect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *hostaddr)
{
	struct privdata		*priv;
	struct connection	*conn;
	sfsistat		 retval;

	if (verbose)
		warnx("connect from: %s", hostname);

	if ((retval = connection_setup(ctx)) != SMFIS_CONTINUE)
		return retval;

	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog(LOG_ERR, "unable to get internal data");
		return SMFIS_TEMPFAIL;
	}

	conn = priv->conn;

	if ((conn->hostname = strdup(hostname)) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for hostname %s", hostname);
		return SMFIS_TEMPFAIL;
	}

	conn->clientaddr = smfi_getsymval(ctx, "_");

	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_helo(SMFICTX *ctx, char *heloname)
{
	struct privdata		*priv;
	struct connection	*conn;

	if (verbose)
		warnx("helo from: %s", heloname);

	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog(LOG_ERR, "unable to get internal data");
		return SMFIS_TEMPFAIL;
	}

	conn = priv->conn;

	if ((conn->heloname = strdup(heloname)) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for heloname %s", heloname);
		return SMFIS_TEMPFAIL;
	}
	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envfrom(SMFICTX *ctx, char **argv)
{
	struct privdata		*priv;
	struct message		*msg;
	struct connection	*conn;
	char			 hostname[128];
	int			 fd;
	int			 n;
	sfsistat		 retval;

	if (verbose)
		warnx("envelope sender: %s", argv[0]);

	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog(LOG_ERR, "unable to get internal data");
		return SMFIS_TEMPFAIL;
	}

	if ((retval = message_setup(ctx)) != SMFIS_CONTINUE)
		return retval;

	msg = priv->msg;
	conn = priv->conn;

	if ((msg->envfrom = strdup(argv[0])) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for envelope address %s", argv[0]);
		return SMFIS_TEMPFAIL;
	}
	if (virus_strategy == STRATEGY_NOTIFY_RECIPIENT)
		msg->notify = 1;

	strlcpy(msg->tfn, tmpdir, sizeof(msg->tfn));
	strlcat(msg->tfn, "/vilter.XXXXXXXXXX", sizeof(msg->tfn));

	if ((fd = mkstemp(msg->tfn)) == -1) {
		syslog(LOG_ERR, "unable to create temp file");
		return SMFIS_TEMPFAIL;
	}
	if (tmpfiles_gr && fchmod(fd, S_IRUSR | S_IWUSR | S_IRGRP)) {
		syslog(LOG_ERR, "unable to change access mode of temp file");
		close(fd);	
		unlink(msg->tfn);
		return SMFIS_TEMPFAIL;
	}
	if (tmpfiles_setgrp && fchown(fd, (uid_t)-1, getgid())) {
		syslog(LOG_ERR, "unable to change group ownership of temp file");
		close(fd);	
		unlink(msg->tfn);
		return SMFIS_TEMPFAIL;
	}
	if ((msg->fp = fdopen(fd, "w")) == NULL) {
		syslog(LOG_ERR, "unable to open temp file %s for writing", msg->tfn);
		if (fd != -1) {
			close(fd);
			unlink(msg->tfn);
		}
		return SMFIS_TEMPFAIL;
	}		
	for (n = 0; n < nbe; n++) {
		if (be[n].be_new != NULL) {
			if (be[n].be_new(&priv->backend_data[n], argv[0], msg->fp)) {
				syslog(LOG_ERR, "vilter-engine refuses to work on this message");
				/* vilter_cleanup(ctx); */
				return SMFIS_TEMPFAIL;
			}
		}
	}

	/* Write an artifical Received: from: Header to the message */

	if (gethostname(hostname, sizeof(hostname))) {
		syslog(LOG_ERR, "can't get local hostname");
		strlcpy(hostname, "localhost", sizeof(hostname));
	}
	if (conn->clientaddr == NULL)
		fprintf(msg->fp, "Received: from %s by %s\r\n", conn->hostname, hostname);
	else
		fprintf(msg->fp, "Received: from %s (%s) by %s\r\n", conn->heloname,
		    conn->clientaddr, hostname);

	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envrcpt(SMFICTX *ctx, char **argv)
{
	struct privdata		*priv;
	struct message		*msg;
	struct recipient	*r;
	struct domain		*d;
	char			*p;
#ifdef ENABLE_LDAP
	char			*attrs[] = { "spamStrategy",
			  	    "virusStrategy",
			  	    "spamSubjectPrefix",
			  	    NULL 
				 };
	char			 filter[1024];
#endif

	if (verbose)
		warnx("envelope recipient: %s", argv[0]);

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (notify_only != NULL && strstr(argv[0], notify_only) == NULL)
			msg->notify = 0;

		/*
		 * Keep a list of all recipients and all domains and initiate
		 * an asynchroneous LDAP search for each recipient and each
		 * domain to retrieve per-user and per-domain configuration.
		 */

		if ((r = malloc(sizeof(struct recipient))) == NULL) {
			syslog(LOG_ERR,"out of memory");
			goto bailout;
		}
		bzero(r, sizeof(struct recipient));

		if ((r->recipient = strdup(argv[0])) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		if ((r->mailLocalAddress = strdup(r->recipient + 1)) == NULL) {
			free(r->recipient);
			free(r);
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		if ((p = strrchr(r->mailLocalAddress, '>')) != NULL)
			*p = '\0';

		r->virus_strategy = r->spam_strategy = -1;
		SLIST_INSERT_HEAD(&msg->rhead, r, entries);

#ifdef ENABLE_LDAP		
		/* Search for per-user configuration */
		
		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterUserConf)(mailLocalAddress=%s))", r->mailLocalAddress);

		if ((r->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(r->recipient);
			free(r->mailLocalAddress);
			free(r);
			syslog(LOG_ERR, "ldap_search failed");
			goto bailout;
		}
#endif		
		/* Search for per-domain configuration */

		if ((p = strchr(r->mailLocalAddress, '@')) == NULL) {
			syslog(LOG_ERR, "strange email address %s", r->mailLocalAddress);
			goto bailout;
		}
		p++;

		SLIST_FOREACH(d, &msg->dhead, entries) {
			if (!strcmp(d->mailLocalDomain, p)) { /* Domain already listed */
				r->domain = d;
				goto bailout;
			}
		}
		if ((d = malloc(sizeof(struct domain))) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		bzero(d, sizeof(struct domain));

		if ((d->mailLocalDomain = strdup(p)) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		d->virus_strategy = d->spam_strategy = -1;
		SLIST_INSERT_HEAD(&msg->dhead, d, entries);
		r->domain = d;

#ifdef ENABLE_LDAP
		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterDomainConf)(mailLocalDomain=%s))", d->mailLocalDomain);
		
		if ((d->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(d->mailLocalDomain);
			free(d);
			syslog(LOG_ERR, "ldap_search failed");
			goto bailout;
		} 
#endif
	} else
		syslog(LOG_ERR, "unable to access internal data");

bailout:
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_header(SMFICTX *ctx, char *headerf, char *headerv)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fprintf(msg->fp, "%s: %s\r\n", headerf, headerv) >= 0) {

			if (!strncmp(headerf, "From", 4)) {
				if (msg->hdrfrom != NULL) {
					syslog(LOG_INFO, "duplicate From: header");
					free(msg->hdrfrom);
					msg->hdrfrom = NULL;
				}
				if ((msg->hdrfrom = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for header from: address");
					goto badluck;
				}
			} else if (!strncmp(headerf, "To", 2)) {
				if (msg->hdrto != NULL) {
					syslog(LOG_INFO, "duplicate To: header");
					free(msg->hdrto);
					msg->hdrto = NULL;
				}
				if ((msg->hdrto = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for header to: address");
					goto badluck;
				}
			} else if (!strncmp(headerf, "Message-Id", 10)) {
				if (msg->msgid != NULL) {
					syslog(LOG_INFO, "duplicate message id");
					free(msg->msgid);
					msg->msgid = NULL;
				}
				if ((msg->msgid = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for message id");
					goto badluck;
				}
			} else if (spamprefix != NULL && !strncasecmp(headerf, "Subject", 7)) {
				if (msg->subject != NULL) {
					syslog(LOG_INFO, "duplicate subject");
					free(msg->subject);
					msg->subject = NULL;
				}
				if ((msg->subject = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for subject");
					goto badluck;
				}
			}	
			return SMFIS_CONTINUE;
		} else
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

badluck:		
	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_eoh(SMFICTX *ctx)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fprintf(msg->fp, "\r\n") >= 0)
			return SMFIS_CONTINUE;
		else 
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_body(SMFICTX *ctx, u_char *bodyp, size_t len)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fwrite(bodyp, len, 1, msg->fp) == 1)
			return SMFIS_CONTINUE;
		else
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

	return SMFIS_TEMPFAIL;
}

static void
mark_virus(SMFICTX *ctx, char *status, char *virus, char *backend)
{
	char headerf[MAXSTR];

	smfi_addheader(ctx, "X-SMTP-Vilter-Virus-Backend", backend);

	if (status != NULL) {
		smfi_addheader(ctx, "X-SMTP-Vilter-Status", status);
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}

	if (virus != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Name", backend);

		smfi_addheader(ctx, headerf, virus);
	}
}

static void
mark_spam(SMFICTX *ctx, char *status, char *backend, double score, double threshold)
{
	char	hdr[32];
	char	slvl[] = "************************************";
	int	lvl;

	smfi_addheader(ctx, "X-SMTP-Vilter-Spam-Backend", backend);

	snprintf(hdr, sizeof(hdr), "%.1f", score);
	smfi_addheader(ctx, "X-Spam-Score", hdr);

	lvl = (int) rint(score);

	if (lvl > 0) {
		if (lvl < strlen(slvl))
			slvl[lvl] = '\0';

		smfi_addheader(ctx, "X-Spam-Level", slvl);
	}

	if (threshold > 0.0) {
		snprintf(hdr, sizeof(hdr), "%.1f", threshold);
		smfi_addheader(ctx, "X-Spam-Threshold", hdr);

		snprintf(hdr, sizeof(hdr), "%.1f", score / threshold);
		smfi_addheader(ctx, "X-Spam-Probability", hdr);
	}

	if (status != NULL)
		smfi_addheader(ctx, "X-Spam-Status", status);
}

static void
mark_unwanted(SMFICTX *ctx, char *status, char *reason, char *backend)
{
	char headerf[MAXSTR];

	smfi_addheader(ctx, "X-SMTP-Vilter-Unwanted-Backend", backend);

	if (status != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}
	if (reason != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Reason", backend);
		smfi_addheader(ctx, headerf, reason);
	}
}

static int
vilter_replacebody(SMFICTX *ctx, char *s)
{
	return smfi_replacebody(ctx, (u_char *) s, (int)strlen(s));
}

static int
replace_msgbody(SMFICTX *ctx, char *fn)
{
	FILE	*fp;
	char	 buf[1024];

	if ((fp = fopen(fn, "r")) != NULL) {
		while (fgets(buf, sizeof(buf), fp)) {
			if (vilter_replacebody(ctx, buf) != MI_SUCCESS) {
				fclose(fp);
				goto discard;
			}
		}
		fclose(fp);
		return SMFIS_CONTINUE;				
	} else
		syslog(LOG_ERR, "can not open replacement file %s", fn);

discard:
	syslog(LOG_ERR, "failed to replace message body");		
	return SMFIS_DISCARD;
}

static int
notify_recipient(SMFICTX *ctx, const char *virus)
{
	char msg[256];

	if (recipient_notification != NULL)
		return replace_msgbody(ctx, recipient_notification);

	/*
	 * If there is no recipient-notification file or there was an error
	 * processing it, fall through to the standard message.
	 */

	snprintf(msg, sizeof(msg), "\r\n*** The message body has been "
	    "deleted because it was infected with the %s virus ***\r\n", \
	    virus);

	if (vilter_replacebody(ctx, msg) == MI_SUCCESS)
		return SMFIS_CONTINUE;

	syslog(LOG_ERR, "failed to replace message body");		
	return SMFIS_DISCARD;
}

static void
log_incident(struct privdata *priv, char *type, char *backend, char *value)
{
	time_t			 now;
	struct message		*msg;
	struct connection	*conn;
	struct recipient	*r;

	msg = priv->msg;
	conn = priv->conn;

	if (logfp != NULL) {
		flockfile(logfp);
		time(&now);
		fprintf(logfp, "smtp-vilter incident report\n");
		fprintf(logfp, "timestamp: %s", ctime(&now));
		fprintf(logfp, "type: %s\n", type);
		fprintf(logfp, "peer: %s\n", (conn->hostname != NULL) ? conn->hostname : "n/a");
		fprintf(logfp, "envelope-sender: %s\n", (msg->envfrom != NULL) ? msg->envfrom : "n/a");
		fprintf(logfp, "envelope-recipients: ");
		SLIST_FOREACH(r, &msg->rhead, entries)
			fprintf(logfp, "%s ", r->recipient);
		fprintf(logfp, "\n");
		fprintf(logfp, "message-id: %s\n", (msg->msgid != NULL) ? msg->msgid : "n/a");

		fprintf(logfp, "from: %s\n", (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a");
		fprintf(logfp, "to: %s\n", (msg->hdrto != NULL) ? msg->hdrto : "n/a");
		fprintf(logfp, "subject: %s\n", (msg->subject != NULL) ? msg->subject : "n/a");
		fprintf(logfp, "backend: %s\n", backend);
		if (value != NULL)
			fprintf(logfp, "%s: %s\n", type, value);
		fprintf(logfp, "\n");
		fflush(logfp);
		funlockfile(logfp);
	}
}

static sfsistat
vilter_virus(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain, int *isvirus)
{
	int			 retval = SMFIS_CONTINUE;
	char			 virus[MAXLEN];
	int			 scanresult;
	char			 scanfn[PATH_MAX];
	struct message		*msg;

	msg = priv->msg;

	bzero(virus, sizeof(virus));

	/* Scan for virus */

	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_virus(backend_data, scanfn, chrootdir, virus, sizeof(virus));

	switch (scanresult) {
	case SCAN_VIRUS:
		syslog(LOG_NOTICE, "virus '%s' detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s",
		    virus,
		    (msg->msgid != NULL) ? msg->msgid : "n/a",
		    (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		    (msg->envfrom != NULL) ? msg->envfrom : "n/a",
		    (msg->hdrto != NULL) ? msg->hdrto : "n/a",
		    backend->name);

		if (logvirus)
			log_incident(priv, "virus", backend->name, virus);

		switch (virus_strategy) {
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_virus(ctx, "infected", virus, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			/* XXX include virus name */
			smfi_setreply(ctx, "554", "5.6.1", "Virus detected in the message");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_NOTIFY_RECIPIENT:
			if (msg->notify) {
				syslog(LOG_INFO, "marking message and replacing message body");
				mark_virus(ctx, "cleaned", virus, backend->name);
				if ((retval = notify_recipient(ctx, virus)) == SMFIS_CONTINUE)
					*stop_chain = 1;

				/* Set the message content type to text/plain */

				if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain"))
					syslog(LOG_INFO, "failed to set message content type to text/plain");
				break;
			}
			/* FALLTHROUGH */
		case STRATEGY_DISCARD:
		default:
			syslog(LOG_INFO, "silently discarding message");
			retval = SMFIS_DISCARD;
			break;
		}

		switch (virus_reaction) {
		case REACTION_ADDTBL:
#ifdef ENABLE_PF
			pf_table_add(virus_table, priv->conn->hostname, virus_duration);
#endif
			break;
		case REACTION_DELTBL:
#ifdef ENABLE_PF
			pf_table_clr(virus_table, priv->conn->hostname, virus_duration);
#endif
			break;
		}
		*isvirus = 1;	
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message contains no virus");

		if (markall)
			mark_virus(ctx, "clean", NULL, backend->name);

		if (strcmp(scanfn, msg->tfn)) {
			syslog(LOG_INFO, "replace message body");
			(void)replace_msgbody(ctx, scanfn);
			if (unlink(scanfn))
				syslog(LOG_ERR, "can't delete temp file %s", scanfn);
		}
		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during virus scan of file %s", msg->tfn);

		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");

			mark_virus(ctx, "unchecked", NULL, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
			
	return retval;
}

static sfsistat
vilter_spam(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *spam)
{
	int			 retval = SMFIS_CONTINUE;
	int			 scanresult;
	double			 score, threshold;
	char			*subject;
	unsigned int		 len;
	struct message		*msg;

	msg = priv->msg;
	score = threshold = 0.0;

	/* Scan for spam */

	scanresult = backend->be_scan.be_scan_spam(backend_data, msg->tfn, chrootdir, &score, &threshold);

	switch (scanresult) {
	case SCAN_SPAM:
		syslog(LOG_NOTICE, "message with id '%s' from '%s' (env '%s') to '%s' classified as spam by backend %s",
		    (msg->msgid != NULL) ? msg->msgid : "n/a",
		    (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		    (msg->envfrom != NULL) ? msg->envfrom : "n/a",
		    (msg->hdrto != NULL) ? msg->hdrto : "n/a",
		    backend->name);

		if (logspam)
			log_incident(priv, "spam", backend->name, NULL);

		switch (spam_strategy) {
		case STRATEGY_DISCARD:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			smfi_setreply(ctx, "554", "5.6.1", "Message considered spam");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_spam(ctx, "spam", backend->name, score, threshold);

			if (spamprefix != NULL) {
				if (msg->subject != NULL) {
					len = strlen(spamprefix) + strlen(msg->subject) + 2;
					subject = malloc(len);
					if (subject != NULL) {
						snprintf(subject, len, "%s %s", spamprefix, msg->subject);
						smfi_chgheader(ctx, "Subject", 1, subject);
						free(subject);
					} else
						syslog(LOG_ERR, "memory error");
				} else {
					len = strlen(spamprefix) + 2;
					subject = malloc(len);
					if (subject != NULL) {
						snprintf(subject, len, "%s", spamprefix);
						smfi_addheader(ctx, "Subject", subject);
						free(subject);
					} else
						syslog(LOG_ERR, "memory error");
				}
			}

			retval = SMFIS_CONTINUE;
			break;
		}

		switch (spam_reaction) {
		case REACTION_ADDTBL:
#ifdef ENABLE_PF
			pf_table_add(spam_table, priv->conn->hostname, spam_duration);
#endif
			break;
		case REACTION_DELTBL:
#ifdef ENABLE_PF
			pf_table_clr(spam_table, priv->conn->hostname, spam_duration);
#endif
			break;
		}
		*spam = 1;
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message is not spam");
		mark_spam(ctx, NULL, backend->name, score, threshold);
		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during spam scan of file %s", msg->tfn);
		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_spam(ctx, "unchecked", backend->name, 0.0, 0.0);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
	return retval;
}

static sfsistat
vilter_unwanted(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *unwanted)
{
	int			 retval = SMFIS_CONTINUE;
	char			 reason[MAXLEN];
	int			 scanresult;
	char			 scanfn[PATH_MAX];
	struct message		*msg;

	msg = priv->msg;

	bzero(reason, sizeof(reason));

	/* Scan for unwanted content */

	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_unwanted(backend_data, scanfn, chrootdir, reason, sizeof(reason));

	switch (scanresult) {
	case SCAN_UNWANTED:
		syslog(LOG_NOTICE, "unwanted content  detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s, reason %s",
		(msg->msgid != NULL) ? msg->msgid : "n/a",
		(msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		(msg->envfrom != NULL) ? msg->envfrom : "n/a",
		(msg->hdrto != NULL) ? msg->hdrto : "n/a",
		backend->name, reason);

		if (logunwanted)
			log_incident(priv, "unwanted-content", backend->name, reason);

		switch (unwanted_strategy) {
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_unwanted(ctx, "unwanted content", reason, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			smfi_setreply(ctx, "554", "5.6.1", "Unwanted content detected in the message");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_NOTIFY_RECIPIENT:
			if (msg->notify) {
				mark_unwanted(ctx, "cleaned", reason, backend->name);
				retval = notify_recipient(ctx, reason);

				/* Set the message content type to text/plain */

				if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain"))
					syslog(LOG_INFO, "failed to set message content type to text/plain");
				break;
			}
			/* FALLTHROUGH */
		case STRATEGY_DISCARD:
		default:
			syslog(LOG_INFO, "silently discarding message");
			retval = SMFIS_DISCARD;
			break;
		}

		switch (unwanted_reaction) {
		case REACTION_ADDTBL:
#ifdef ENABLE_PF
			pf_table_add(unwanted_table, priv->conn->hostname, unwanted_duration);
#endif
			break;
		case REACTION_DELTBL:
#ifdef ENABLE_PF
			pf_table_clr(unwanted_table, priv->conn->hostname, unwanted_duration);
#endif
			break;
		}
		*unwanted = 1;
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message contains no unwanted content");
		if (strcmp(scanfn, msg->tfn)) {
			if (markall)
				mark_unwanted(ctx, "cleaned", NULL, backend->name);

			if (logunwanted)
				log_incident(priv, "unwanted-content", backend->name, reason);

			syslog(LOG_INFO, "replace message body");
			(void)replace_msgbody(ctx, scanfn);

			/* Make sure later backends get the new content */

			if (unlink(scanfn))
				syslog(LOG_ERR, "can't delete temp file %s", scanfn);

		} else if (markall)
			mark_unwanted(ctx, "clean", NULL, backend->name);
		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during scan of file %s", msg->tfn);
		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_unwanted(ctx, "unchecked", NULL, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
	return retval;
}

#ifdef ENABLE_LDAP
static void
print_entries_with_values(LDAP *ld, LDAPMessage *result)
{
	LDAPMessage	*e;
	BerElement	*ber;
	char		*dn, *attr;
	char		**vals;
	int		 i;

	for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
		if ((dn = ldap_get_dn(ld, e)) != NULL) {
			warnx("dn: %s", dn);
			ldap_memfree(dn);
		}
		for (attr = ldap_first_attribute(ld, e, &ber); attr != NULL; attr = ldap_next_attribute(ld, e, ber)) {
			if ((vals = ldap_get_values(ld, e, attr)) != NULL) {
				for (i = 0; vals[i] != NULL; i++)
					warnx("%s: %s", attr, vals[i]);
				ldap_value_free(vals);
			}
			ldap_memfree(attr);
		}
		if (ber != NULL)
			ber_free(ber, 0);
	}
}
#endif

static sfsistat
vilter_eom(SMFICTX *ctx)
{
	struct		 privdata *priv;
	struct		 message *msg;
	int		 retval = SMFIS_CONTINUE;
	int		 n;
	struct		 be_data *backend_data;
	int		 stop_chain;
	int		 status, allstat;
#ifdef ENABLE_LDAP
	struct		 recipient *r;
	struct		 domain *d;
	LDAPMessage	*result;
	LDAPMessage	*e;
	char		**vals;
	int		 nentries, rc;	
#endif

	/*
	 * XXX: If there is only one recipient, things are easy:  Take this
	 * recipients strategy settings, if present, else the domain settings,
	 * if these are absent, use the system default.
	 *
	 * If there is more than one recipient, things can get complicated.
	 * Check if all strategies are compatible, if they are, treat the
	 * message accordingly.  If one recipient has a discard strategy
	 * whereas others have a mark strategy, the recipient with discard is
	 * removed from the list of recipients.  The following table gives an
	 * idea how the situation is handled.
	 *
	 * R1		R2			Action
	 * any		the same	use strategy
	 * discard	mark		remove R1, then mark
	 * discard	notify		remove R1, then notify
	 * mark		notify		the only real conflict: notify
	 */

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		fclose(msg->fp);
		msg->fp = NULL;

		smfi_addheader(ctx, "X-SMTP-Vilter-Version", VERSION);

		/* Scan content */

		allstat = 0;
		for (n = stop_chain = 0; n < nbe && !stop_chain && retval == SMFIS_CONTINUE; n++) {
			backend_data = &priv->backend_data[n];
			status = 0;
			if (be[n].type & BE_SCAN_VIRUS) {
				if (verbose)
					warnx("scanning for viruses using backend %s", be[n].name);
				retval = vilter_virus(&be[n], backend_data, ctx, priv, &stop_chain, &status);
			} else if (be[n].type & BE_SCAN_SPAM) {
				if (verbose)
					warnx("scanning for spam using backend %s", be[n].name);

				retval = vilter_spam(&be[n], backend_data, ctx, priv, &status);
			} else if (be[n].type & BE_SCAN_UNWANTED) {
				if (verbose)
					warnx("scanning for unwanted content using backend %s", be[n].name);
				retval = vilter_unwanted(&be[n], backend_data, ctx, priv, &status);
			} else
				return SMFIS_TEMPFAIL;
			allstat += status;
		}

		if (allstat == 0) {	/* message is clean */
			switch (clean_reaction) {
			case REACTION_ADDTBL:
#ifdef ENABLE_PF
				pf_table_add(clean_table, priv->conn->hostname, clean_duration);
#endif
				break;
			case REACTION_DELTBL:
#ifdef ENABLE_PF
				pf_table_clr(clean_table, priv->conn->hostname, clean_duration);
#endif
				break;
			}
		}
#ifdef ENABLE_LDAP
		SLIST_FOREACH(r, &msg->rhead, entries) {
			warnx("recipient: %s (domain %s)", r->mailLocalAddress, r->domain->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						r->spam_strategy = get_su_strategy(vals[0]);
						if (r->spam_strategy == STRATEGY_UNSET)
							warnx("unknown spam strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						r->virus_strategy = get_v_strategy(vals[0]);
						if (r->virus_strategy == STRATEGY_UNSET)
							warnx("unknown virus strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "unwantedStrategy")) != NULL) {
						r->unwanted_strategy = get_su_strategy(vals[0]);
						if (r->unwanted_strategy == STRATEGY_UNSET)
							warnx("unknown unwanted strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						r->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				ldap_msgfree(result);
			}
		}
		SLIST_FOREACH(d, &msg->dhead, entries) {
			warnx("domain: %s", d->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						d->spam_strategy = get_su_strategy(vals[0]);
						if (d->spam_strategy == STRATEGY_UNSET)
							warnx("unknown spam strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						d->virus_strategy = get_v_strategy(vals[0]);
						if (d->virus_strategy == STRATEGY_UNSET)
							warnx("unknown virus strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "unwantedStrategy")) != NULL) {
						d->unwanted_strategy = get_su_strategy(vals[0]);
						if (d->unwanted_strategy == STRATEGY_UNSET)
							warnx("unknown unwanted strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						d->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				ldap_msgfree(result);
			}
		}	
#endif
		message_cleanup(ctx);
		
	} else {
		syslog(LOG_ERR, "unable to access internal data");
		retval = SMFIS_TEMPFAIL;
	}
	return retval;
}


static sfsistat
vilter_abort(SMFICTX *ctx)
{
#ifdef ENABLE_LDAP
	struct privdata *priv;
	struct message *msg;
	struct recipient *r;
	struct domain *d;
	LDAPMessage *result;
	int rc;
#endif

	syslog(LOG_INFO, "message aborted");
#ifdef ENABLE_LDAP
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		SLIST_FOREACH(r, &msg->rhead, entries) {
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}	
		SLIST_FOREACH(d, &msg->dhead, entries) {
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}
	}
#endif
	message_cleanup(ctx);
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_close(SMFICTX *ctx)
{
	struct privdata *priv;
	struct connection *conn;

	if (verbose) {
		if ((priv = smfi_getpriv(ctx)) != NULL) {
			conn = priv->conn;
			if (conn != NULL && conn->hostname != NULL) {
				warnx("closing connection with %s", conn->hostname);
				goto cleanup;
			}
		}
		warnx("closing connection");
	}
cleanup:
	connection_cleanup(ctx);
	return SMFIS_CONTINUE;
}

#ifdef ENABLE_PF
static void
pf_table_add(const char *table, const char *addr, long duration)
{
	pid_t pid;
	struct pftable_msg m;

	if (table == NULL || addr == NULL)
		return;
	
	pid = getpid();

	strlcpy(m.pftable, table, PFTABLE_LEN);
	strlcpy(m.addr, addr, ADR_LEN);
	m.duration = duration;
	m.len = 0;
	
	imsg_compose(ibuf_main, IMSG_PFTABLE_ADD, 0, pid, -1, &m, sizeof(m)); 
	imsg_commit();
}

static void
pf_table_clr(const char *table, const char *addr, long duration)
{
	pid_t pid;
	struct pftable_msg m;

	if (table == NULL || addr == NULL)
		return;
	
	pid = getpid();

	strlcpy(m.pftable, table, PFTABLE_LEN);
	strlcpy(m.addr, addr, ADR_LEN);
	m.duration = duration;
	m.len = 0;
	
	imsg_compose(ibuf_main, IMSG_PFTABLE_DEL, 0, pid, -1, &m, sizeof(m)); 
	imsg_commit();
}

static void
imsg_commit()
{
	struct pollfd	pfd[1];
	int nfds;

	for (;;) {
		bzero(pfd, sizeof(pfd));
		pfd[0].fd = ibuf_main->fd;
		pfd[0].events = POLLOUT;

		if ((nfds = poll(pfd, 1, INFTIM)) == -1)
			if (errno != EINTR)
				errx(1, "engine: poll error");

		if (nfds > 0 && ibuf_main->w.queued) {
			if (msgbuf_write(&ibuf_main->w) < 0)
				errx(1, "pipe write error");
			else
				break; /* XXX All bytes sent? */
		}
	}
}
#endif

int
e_main(uid_t uid, gid_t gid, int pipe_m2e[2])
{
	smfiDesc_str	 desc;
	FILE		*fp;
	struct rlimit	 rl;
	int		 n;
	int		 pid;
	int		 nconn;
	int		 tout;

	switch (pid = fork()) {
	case -1:
		errx(1, "can't fork");
		/* NOTREACHED */
	case 0:
		break;
	default:
		return pid;
	}

#ifndef __linux__	
	setproctitle(NULL);
#endif		
	/* Set ressource limits */

	if (rlimit_nofile > 0) {
		rl.rlim_cur = rl.rlim_max = rlimit_nofile;

		if (setrlimit(RLIMIT_NOFILE, &rl))
			err(1, "can set number of files to %lld", (long long)rlimit_nofile);
	}

	if (rlimit_nproc > 0) {
		rl.rlim_cur = rl.rlim_max = rlimit_nproc;

		if (setrlimit(RLIMIT_NPROC, &rl))
			err(1, "can set number of processes to %lld", (long long)rlimit_nproc);
	}

	if (logfile != NULL && (logfp = fopen(logfile, "a")) == NULL)
		err(1, "can't open logfile %s", logfile);
	else
		logfp = NULL;

	if ((fp = fopen(pidfile, "w")) == NULL) 
		err(1, "can't open pidfile %s", pidfile);
	if (fprintf(fp, "%d\n", getpid()) == -1)
		err(1, "can't write pid to pidfile %s", pidfile);
	fclose(fp);

	/* chroot, if necessary */

	if (chrootdir) {
		if (getuid())
			errx(1, "must be started by root if using chroot");
		if (chroot(chrootdir))
			err(1, "can't chroot to %s", chrootdir);
		if (chdir("/"))
			err(1, "can't chdir to / after chroot to %s", chrootdir);
	}

	/* Change the user and group id */

	setgid(gid);
	setuid(uid);

	if (!getuid() || !getgid())
		errx(1, "must not be run as root");

	/* imsg stuff */

	close(pipe_m2e[0]);
	if ((ibuf_main = malloc(sizeof(struct imsgbuf))) == NULL)
		errx(1, "memory error");
	imsg_init(ibuf_main, pipe_m2e[1]);

	syslog(LOG_INFO, "dropped privileges, running as %d:%d", uid, gid);

	/* Initialize sendmail milter */

	desc.xxfi_name = "smtp-vilter";
	desc.xxfi_version = SMFI_VERSION;
	desc.xxfi_flags = SMFIF_ADDHDRS | SMFIF_CHGHDRS | SMFIF_CHGBODY;

	desc.xxfi_connect = smtp_vilter_connect;	
	desc.xxfi_helo = smtp_vilter_helo;
	desc.xxfi_envfrom = smtp_vilter_envfrom;
	desc.xxfi_envrcpt = smtp_vilter_envrcpt;
#ifdef ENABLE_LDAP
	desc.xxfi_flags |= SMFIF_DELRCPT;
#endif
	desc.xxfi_header = vilter_header;
	desc.xxfi_eoh = vilter_eoh;
	desc.xxfi_body = vilter_body;
	desc.xxfi_eom = vilter_eom;
	desc.xxfi_abort = vilter_abort;
	desc.xxfi_close = vilter_close;

	if (smfi_register(desc) == MI_FAILURE)
		err(1, "failed to register with sendmail");	

	if (!strncasecmp(port, "unix:", 5) || !strncasecmp(port, "local:", 6))
		unlink(strchr(port, ':') + 1);

	smfi_setconn(port);
	if (smfi_main() == MI_FAILURE)
		syslog(LOG_ERR, "failed to establish a connection on port %s", port);

	/* Wait for the connection reference count to drop to zero */

	if (verbose)
		warnx("gracefully terminating");

	if (ex_timeout > 0)
		tout = ex_timeout;

	nconn = 0;
	do {
		if (pthread_mutex_lock(&conn_cnt_mutex))
			syslog(LOG_ERR, "can get connection count");
		else {
			nconn = conn_cnt;
			pthread_mutex_unlock(&conn_cnt_mutex);
		}

		if (nconn > 0) {
			if (verbose)
				warnx("waiting for connection count to reach zero (now %d)", nconn);
			sleep(1);
			if (ex_timeout > 0) {
				if (tout-- == 0)
					nconn = 0;
			}
		}
	} while (nconn > 0);

	if (verbose)
		warnx("connection count reached zero, terminating instance");

	for (n = 0; n < nbe; n++) {
		if (be[n].be_exit != NULL)
			be[n].be_exit();
		free(be[n].name);
		dlclose(be[n].dlhandle);
	}
	free(be);

	syslog(LOG_INFO, "child process exiting");

	closelog();
	if (logfp != NULL)
		fclose(logfp);
	_exit(2);
	/* NOTREACHED */
}
