/*	$Id: smtp-vilter.c,v 1.103 2005/04/24 10:20:17 marc Exp $	*/

/*
 * Copyright (c) 2003-2005 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/time.h>
#include <sys/types.h>
#include <sys/queue.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <dlfcn.h>
#include <err.h>
#include <errno.h>
#include <grp.h>
#ifdef ENABLE_LDAP
#include <ldap.h>
#endif
#include <math.h>
#include <pthread.h>
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "libmilter/mfapi.h"

#include "pathnames.h"
#include "smtp-vilter.h"

#ifdef __linux__
#include <sys/param.h>
#include "strlfunc.h"
#include "strtonum.h"
#else
#include <sys/syslimits.h>
#endif

#ifdef __FreeBSD__
#include "strtonum.h"
#endif

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256

#define LOG_UNSET	-1

char	*port;
char	*vilterd_host;
int	 vilterd_port;
int	 virus_strategy;
int	 spam_strategy;
int	 unwanted_strategy;
int	 error_strategy;
int	 markall;
int	 verbose;
char	*pidfile;
char	*cfgfile;
char	*tmpdir;
int	 tmpfiles_gr;
int	 tmpfiles_setgrp;
char	*backend;
char	*recipient_notification;
char	*notify_only;
int	 log_facility;
char	*logfile;
FILE	*logfp;
char	*user;
char	*group;
char	*chrootdir;
int	 rlimit_nofile;
int	 nbe;
char	*spamprefix;
int	 logvirus, logspam, logunwanted;
int	 keep_tmpfiles;

extern char *__progname;

#define DEFLT_LDAPHOST	"localhost"
#define DEFLT_LDAPPORT	389

#ifdef ENABLE_LDAP
LDAP	*ld;
#endif
char	*ldaphost;
int	 ldapport;
char	*searchbase;
char	*binddn;
char	*bindpasswd;
int	 ldap_use_tls;

struct recipient {
	char		*recipient;
	char		*mailLocalAddress;	/* canonicalized recipient */
#ifdef ENABLE_LDAP
	int		 msgid;
#endif
	int		 virus_strategy;
	int		 spam_strategy;
	char		*spamprefix;
	struct domain	*domain;
	SLIST_ENTRY(recipient) entries;
};

struct domain {
	char	*mailLocalDomain;
#ifdef ENABLE_LDAP
	int	 msgid;
#endif
	int	 virus_strategy;
	int	 spam_strategy;
	char	*spamprefix;
	SLIST_ENTRY(domain) entries;
};

struct backend *be;

struct connection {
	char		*hostname;
	_SOCK_ADDR	*hostaddr;

	int		 blocked;

	/*
	 * The following variables are for statistics,
	 * but not used at the moment
	 */

	int		 num_msgs;
	int		 num_virus;
	int		 num_spam;
	int		 num_unwanted;
};

struct message {
	char			 tfn[MAX_FILE];	/* Temporary file name */
	char			*msgid;		/* Message ID */
	char			*envfrom;	/* Envelope sender */
	SLIST_HEAD(, recipient)	 rhead;		/* List of recipients */
	SLIST_HEAD(, domain)	 dhead;		/* List of recp. domains */
	char			*hdrfrom;	/* From: address in message header */
	char			*hdrto;		/* To: address in message header */
	char			*subject;	/* The message subject */
	int			 notify;	/* if != 0, notify recipient */
	FILE			*fp;
};

struct privdata {
	struct connection	*conn;
	struct message		*msg;
	struct be_data		*backend_data;
};

extern void smtp_vilter_init(void);

static void
sighand(int signum)
{
	exit(0);
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-V] [-m] [-k] [-p port] [-d dir] [-T tmpfile-opt] [-e logile] [-b backend] [-n recipient-notification] [-o notify-only] [-C configfile] [-P pidfile] [-t chroot-dir] [-u user] [-g group] [-f maxfiles] [-a spam-subject-prefix]", __progname);
#ifdef ENABLE_LDAP
	fprintf(stderr, " [-D binddn] [-h ldaphost] [-L ldapport] [-w bindpasswd] [-B searchbase]");
#endif
	fprintf(stderr, "\n");
	exit(1);
}

int
decode_backend(char *s)
{
	char	*p;
	int	 n;
	char	*q, *r;

	if ((q = strdup(s)) == NULL)
		err(1, "memory allocation error");

	nbe = 0;
	r = q;
	do {
		p = strsep(&r, ",");
		if (*p != '\0')
			++nbe;
	} while (r != NULL);

	free(q);

	if ((be = (struct backend *) malloc(nbe * sizeof(struct backend))) == NULL)
		err(1, "error allocating memory for backends");
	bzero(be, nbe * sizeof(struct backend));

	n = 0;
	do {
		p = strsep(&s, ",");
		if (*p != '\0') {
			if ((be[n].name = strdup(p)) == NULL)
				err(1, "memory allocation error");
			++n;
		}
	} while (s != NULL);

	return 0;
}

/* Allocate and initialize per-connection memory */

static sfsistat
connection_setup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct connection	*conn;

	if (verbose)
		warnx("connection setup");

	if ((priv = malloc(sizeof(struct privdata))) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for internal data");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv, sizeof(struct privdata));

	if ((conn = malloc(sizeof(struct connection))) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for connection data");
		free(priv);
		return SMFIS_TEMPFAIL;
	}
	bzero(conn, sizeof(struct connection));

	priv->conn = conn;

	if ((priv->backend_data = malloc(sizeof(struct be_data) * nbe)) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for backend data");
		return SMFIS_TEMPFAIL;
	}
	bzero(priv->backend_data, sizeof(struct be_data) * nbe);

	if (smfi_setpriv(ctx, priv)) {
		syslog(LOG_ERR, "unable to store internal data");
		free(priv->conn);
		free(priv->backend_data);
		free(priv);
		return SMFIS_TEMPFAIL;
	}

	return SMFIS_CONTINUE;
}

/* Allocate and initialize per-message memory */

static sfsistat
message_setup(SMFICTX *ctx)
{
	struct privdata	*priv;
	struct message	*msg;

	if (verbose)
		warnx("message setup");

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		if ((msg = malloc(sizeof(struct message))) != NULL) {
			bzero(msg, sizeof(struct message));
			SLIST_INIT(&msg->rhead);
			SLIST_INIT(&msg->dhead);
			priv->msg = msg;
			return SMFIS_CONTINUE;
		} else {
			syslog(LOG_ERR, "unable to allocate memory for message");
			return SMFIS_TEMPFAIL;
		}
	} else
		syslog(LOG_ERR, "unable to get internal data in message_setup(),");

	return SMFIS_TEMPFAIL;
}

/* Deallocate per-message memory and ressources*/

static void
message_cleanup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct message		*msg;
	struct recipient	*r;
	struct domain		*d;

	if (verbose)
		warnx("message cleanup");

	priv = smfi_getpriv(ctx);
	if (priv != NULL && priv->msg != NULL) {
		msg = priv->msg;
		if (msg->fp != NULL)
			fclose(msg->fp);

		if (!keep_tmpfiles && unlink(msg->tfn))
			syslog(LOG_ERR, "can't delete temp file %s", msg->tfn);

		if (msg->msgid != NULL)
			free(msg->msgid);
		if (msg->envfrom != NULL)
			free(msg->envfrom);
		/*
		 * if (msg->envto != NULL)
	 	 *	free(msg->envto);
		 */
		while (!SLIST_EMPTY(&msg->rhead)) {
			r = SLIST_FIRST(&msg->rhead);

			free(r->recipient);
			free(r->mailLocalAddress);
			if (r->spamprefix != NULL)
				free(r->spamprefix);

			SLIST_REMOVE_HEAD(&msg->rhead, entries);
			free(r);
		}

		while (!SLIST_EMPTY(&msg->dhead)) {
			d = SLIST_FIRST(&msg->dhead);
			free(d->mailLocalDomain);
			if (d->spamprefix != NULL)
				free(d->spamprefix);
			SLIST_REMOVE_HEAD(&msg->dhead, entries);
			free(d);
		}

		if (msg->hdrfrom != NULL)
			free(msg->hdrfrom);
		if (msg->hdrto != NULL)
			free(msg->hdrto);
		if (msg->subject != NULL)
			free(msg->subject);
		free(msg);
		priv->msg = NULL;
	}
}

/* Free per-connection memory */

static void
connection_cleanup(SMFICTX *ctx)
{
	struct privdata		*priv;
	struct connection	*conn;
	
	if (verbose)
		warnx("connection cleanup");

	priv = smfi_getpriv(ctx);
	if (priv != NULL) {
		if (priv->conn != NULL) {
			conn = priv->conn;

			if (conn->hostname != NULL)
				free(conn->hostname);
			free(conn);
		}
		if (priv->backend_data != NULL)
			free(priv->backend_data);

		free(priv);
		smfi_setpriv(ctx, NULL);
	} else
		syslog(LOG_ERR, "unable to get internal data in connection_cleanup()");
}

static sfsistat
smtp_vilter_connect(SMFICTX *ctx, char *hostname, _SOCK_ADDR *hostaddr)
{
	struct privdata		*priv;
	struct connection	*conn;
	sfsistat		 retval;

	if (verbose)
		warnx("connect from: %s", hostname);

	if ((retval = connection_setup(ctx)) != SMFIS_CONTINUE)
		return retval;

	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog(LOG_ERR, "unable to get internal data");
		return SMFIS_TEMPFAIL;
	}

	conn = priv->conn;

	if ((conn->hostname = strdup(hostname)) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for hostname %s", hostname);
		return SMFIS_TEMPFAIL;
	}
	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envfrom(SMFICTX *ctx, char **argv)
{
	struct privdata		*priv;
	struct message		*msg;
	struct connection	*conn;
	char			 hostname[128];
	int			 fd;
	int			 n;
	sfsistat		 retval;

	if (verbose)
		warnx("envelope sender: %s", argv[0]);

	if ((priv = smfi_getpriv(ctx)) == NULL) {
		syslog(LOG_ERR, "unable to get internal data");
		return SMFIS_TEMPFAIL;
	}

	if ((retval = message_setup(ctx)) != SMFIS_CONTINUE)
		return retval;

	msg = priv->msg;
	conn = priv->conn;

	if ((msg->envfrom = strdup(argv[0])) == NULL) {
		syslog(LOG_ERR, "unable to allocate memory for envelope address %s", argv[0]);
		return SMFIS_TEMPFAIL;
	}
	if (virus_strategy == STRATEGY_NOTIFY_RECIPIENT)
		msg->notify = 1;

	strlcpy(msg->tfn, tmpdir, sizeof(msg->tfn));
	strlcat(msg->tfn, "/vilter.XXXXXXXXXX", sizeof(msg->tfn));

	if ((fd = mkstemp(msg->tfn)) == -1) {
		syslog(LOG_ERR, "unable to create temp file");
		return SMFIS_TEMPFAIL;
	}
	if (tmpfiles_gr && fchmod(fd, S_IRUSR | S_IWUSR | S_IRGRP)) {
		syslog(LOG_ERR, "unable to change access mode of temp file");
		close(fd);	
		unlink(msg->tfn);
		return SMFIS_TEMPFAIL;
	}
	if (tmpfiles_setgrp && fchown(fd, -1, getgid())) {
		syslog(LOG_ERR, "unable to change group ownership of temp file");
		close(fd);	
		unlink(msg->tfn);
		return SMFIS_TEMPFAIL;
	}
	if ((msg->fp = fdopen(fd, "w")) == NULL) {
		syslog(LOG_ERR, "unable to open temp file %s for writing", msg->tfn);
		if (fd != -1) {
			close(fd);
			unlink(msg->tfn);
		}
		return SMFIS_TEMPFAIL;
	}		
	for (n = 0; n < nbe; n++) {
		if (be[n].be_new != NULL) {
			if (be[n].be_new(&priv->backend_data[n], argv[0], msg->fp)) {
				syslog(LOG_ERR, "vilter-engine refuses to work on this message");
				/* vilter_cleanup(ctx); */
				return SMFIS_TEMPFAIL;
			}
		}
	}

	/* Write an artifical Received: from: Header to the message */

	if (gethostname(hostname, sizeof(hostname))) {
		syslog(LOG_ERR, "can't get local hostname");
		strlcpy(hostname, "localhost", sizeof(hostname));
	}
	fprintf(msg->fp, "Received: from %s by %s\r\n", conn->hostname, hostname);

	return SMFIS_CONTINUE;
}

static sfsistat
smtp_vilter_envrcpt(SMFICTX *ctx, char **argv)
{
	struct privdata		*priv;
	struct connection	*conn;
	struct message		*msg;
	struct recipient	*r;
	struct domain		*d;
	char			*p;
#ifdef ENABLE_LDAP
	char			*attrs[] = { "spamStrategy",
			  	    "virusStrategy",
			  	    "spamSubjectPrefix",
			  	    NULL 
				 };
	char			 filter[1024];
#endif

	if (verbose)
		warnx("envelope recipient: %s", argv[0]);

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;
		conn = priv->conn;

		if (notify_only != NULL && strstr(argv[0], notify_only) == NULL)
			msg->notify = 0;

		/*
		 * Keep a list of all recipients and all domains and initiate
		 * an asynchroneous LDAP search for each recipient and each
		 * domain to retrieve per-user and per-domain configuration.
		 */

		if ((r = malloc(sizeof(struct recipient))) == NULL) {
			syslog(LOG_ERR,"out of memory");
			goto bailout;
		}
		bzero(r, sizeof(struct recipient));

		if ((r->recipient = strdup(argv[0])) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		if ((r->mailLocalAddress = strdup(r->recipient + 1)) == NULL) {
			free(r->recipient);
			free(r);
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		if ((p = strrchr(r->mailLocalAddress, '>')) != NULL)
			*p = '\0';

		r->virus_strategy = r->spam_strategy = -1;
		SLIST_INSERT_HEAD(&msg->rhead, r, entries);

#ifdef ENABLE_LDAP		
		/* Search for per-user configuration */
		
		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterUserConf)(mailLocalAddress=%s))", r->mailLocalAddress);

		if ((r->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(r->recipient);
			free(r->mailLocalAddress);
			free(r);
			syslog(LOG_ERR, "ldap_search failed");
			goto bailout;
		}
#endif		
		/* Search for per-domain configuration */

		if ((p = strchr(r->mailLocalAddress, '@')) == NULL) {
			syslog(LOG_ERR, "strange email address %s", r->mailLocalAddress);
			goto bailout;
		}
		p++;

		SLIST_FOREACH(d, &msg->dhead, entries) {
			if (!strcmp(d->mailLocalDomain, p)) { /* Domain already listed */
				r->domain = d;
				goto bailout;
			}
		}
		if ((d = malloc(sizeof(struct domain))) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		bzero(d, sizeof(struct domain));

		if ((d->mailLocalDomain = strdup(p)) == NULL) {
			syslog(LOG_ERR, "out of memory");
			goto bailout;
		}
		d->virus_strategy = d->spam_strategy = -1;
		SLIST_INSERT_HEAD(&msg->dhead, d, entries);
		r->domain = d;

#ifdef ENABLE_LDAP
		snprintf(filter, sizeof(filter), "(&(objectClass=SMTPVilterDomainConf)(mailLocalDomain=%s))", d->mailLocalDomain);
		
		if ((d->msgid = ldap_search(ld, searchbase, LDAP_SCOPE_SUBTREE, filter, attrs, 0)) == -1) {
			free(d->mailLocalDomain);
			free(d);
			syslog(LOG_ERR, "ldap_search failed");
			goto bailout;
		} 
#endif
	} else
		syslog(LOG_ERR, "unable to access internal data");

bailout:
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_header(SMFICTX *ctx, char *headerf, char *headerv)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fprintf(msg->fp, "%s: %s\r\n", headerf, headerv) >= 0) {

			if (!strncmp(headerf, "From", 4)) {
				if (msg->hdrfrom != NULL) {
					syslog(LOG_INFO, "duplicate From: header");
					free(msg->hdrfrom);
					msg->hdrfrom = NULL;
				}
				if ((msg->hdrfrom = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for header from: address");
					goto badluck;
				}
			} else if (!strncmp(headerf, "To", 2)) {
				if (msg->hdrto != NULL) {
					syslog(LOG_INFO, "duplicate To: header");
					free(msg->hdrto);
					msg->hdrto = NULL;
				}
				if ((msg->hdrto = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for header to: address");
					goto badluck;
				}
			} else if (!strncmp(headerf, "Message-Id", 10)) {
				if (msg->msgid != NULL) {
					syslog(LOG_INFO, "duplicate message id");
					free(msg->msgid);
					msg->msgid = NULL;
				}
				if ((msg->msgid = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for message id");
					goto badluck;
				}
			} else if (spamprefix != NULL && !strncmp(headerf, "Subject", 7)) {
				if (msg->subject != NULL) {
					syslog(LOG_INFO, "duplicate subject");
					free(msg->subject);
					msg->subject = NULL;
				}
				if ((msg->subject = strdup(headerv)) == NULL) {
					syslog(LOG_ERR, "unable to allocate memory for subject");
					goto badluck;
				}
			}	
			return SMFIS_CONTINUE;
		} else
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

badluck:		
	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_eoh(SMFICTX *ctx)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fprintf(msg->fp, "\r\n") >= 0)
			return SMFIS_CONTINUE;
		else 
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

	return SMFIS_TEMPFAIL;
}

static sfsistat
vilter_body(SMFICTX *ctx, u_char *bodyp, size_t len)
{
	struct privdata	*priv;
	struct message	*msg;

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		if (fwrite(bodyp, len, 1, msg->fp) == 1)
			return SMFIS_CONTINUE;
		else
			syslog(LOG_ERR, "write error");
	} else
		syslog(LOG_ERR, "unable to access internal data");

	return SMFIS_TEMPFAIL;
}

static void
mark_virus(SMFICTX *ctx, char *status, char *virus, char *backend)
{
	char headerf[MAXSTR];

	smfi_addheader(ctx, "X-SMTP-Vilter-Virus-Backend", backend);

	if (status != NULL) {
		smfi_addheader(ctx, "X-SMTP-Vilter-Status", status);
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}

	if (virus != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Virus-Name", backend);

		smfi_addheader(ctx, headerf, virus);
	}
}

static void
mark_spam(SMFICTX *ctx, char *status, char *backend, double score, double threshold)
{
	char	hdr[32];
	char	slvl[] = "************************************";
	int	lvl;

	smfi_addheader(ctx, "X-SMTP-Vilter-Spam-Backend", backend);

	snprintf(hdr, sizeof(hdr), "%.1f", score);
	smfi_addheader(ctx, "X-Spam-Score", hdr);

	lvl = (int) rint(score);

	if (lvl > 0) {
		if (lvl < strlen(slvl))
			slvl[lvl] = '\0';

		smfi_addheader(ctx, "X-Spam-Level", slvl);
	}

	if (threshold > 0.0) {
		snprintf(hdr, sizeof(hdr), "%.1f", threshold);
		smfi_addheader(ctx, "X-Spam-Threshold", hdr);

		snprintf(hdr, sizeof(hdr), "%.1f", score / threshold);
		smfi_addheader(ctx, "X-Spam-Probability", hdr);
	}

	if (status != NULL)
		smfi_addheader(ctx, "X-Spam-Status", status);
}

static void
mark_unwanted(SMFICTX *ctx, char *status, char *reason, char *backend)
{
	char headerf[MAXSTR];

	smfi_addheader(ctx, "X-SMTP-Vilter-Unwanted-Backend", backend);

	if (status != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Status", backend);
		smfi_addheader(ctx, headerf, status);
	}
	if (reason != NULL) {
		snprintf(headerf, sizeof(headerf), "X-SMTP-Vilter-%s-Unwanted-Reason", backend);
		smfi_addheader(ctx, headerf, reason);
	}
}

static int
vilter_replacebody(SMFICTX *ctx, char *s)
{
	return smfi_replacebody(ctx, (u_char *) s, strlen(s));
}

static int
replace_msgbody(SMFICTX *ctx, char *fn)
{
	FILE	*fp;
	char	 buf[1024];

	if ((fp = fopen(fn, "r")) != NULL) {
		while (fgets(buf, sizeof(buf), fp)) {
			if (vilter_replacebody(ctx, buf) != MI_SUCCESS) {
				fclose(fp);
				goto discard;
			}
		}
		fclose(fp);
		return SMFIS_CONTINUE;				
	} else
		syslog(LOG_ERR, "can not open replacement file %s", fn);

discard:
	syslog(LOG_ERR, "failed to replace message body");		
	return SMFIS_DISCARD;
}

static int
notify_recipient(SMFICTX *ctx, char *virus)
{
	if (recipient_notification != NULL)
		return replace_msgbody(ctx, recipient_notification);

	/*
	 * If there is no recipient-notification file or there was an error
	 * processing it, fall through to the standard message.
	 */

	if (vilter_replacebody(ctx, "\r\n*** The message body has been deleted because it was infected with a virus ***\r\n") == MI_SUCCESS) {
		return SMFIS_CONTINUE;
	}

	syslog(LOG_ERR, "failed to replace message body");		
	return SMFIS_DISCARD;
}

static void
log_incident(struct privdata *priv, char *type, char *backend, char *value)
{
	time_t			 now;
	struct message		*msg;
	struct connection	*conn;
	struct recipient	*r;

	msg = priv->msg;
	conn = priv->conn;

	if (logfp != NULL) {
		flockfile(logfp);
		time(&now);
		fprintf(logfp, "smtp-vilter incident report\n");
		fprintf(logfp, "timestamp: %s", ctime(&now));
		fprintf(logfp, "type: %s\n", type);
		fprintf(logfp, "peer: %s\n", (conn->hostname != NULL) ? conn->hostname : "n/a");
		fprintf(logfp, "envelope-sender: %s\n", (msg->envfrom != NULL) ? msg->envfrom : "n/a");
		fprintf(logfp, "envelope-recipients: ");
		SLIST_FOREACH(r, &msg->rhead, entries) {
			fprintf(logfp, "%s ", r->recipient);
		}
		fprintf(logfp, "\n");
		fprintf(logfp, "message-id: %s\n", (msg->msgid != NULL) ? msg->msgid : "n/a");

		fprintf(logfp, "from: %s\n", (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a");
		fprintf(logfp, "to: %s\n", (msg->hdrto != NULL) ? msg->hdrto : "n/a");
		fprintf(logfp, "subject: %s\n", (msg->subject != NULL) ? msg->subject : "n/a");
		fprintf(logfp, "backend: %s\n", backend);
		if (value != NULL)
			fprintf(logfp, "%s: %s\n", type, value);
		fprintf(logfp, "\n");
		fflush(logfp);
		funlockfile(logfp);
	}
}

static sfsistat
vilter_virus(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int			 retval = SMFIS_CONTINUE;
	char			 virus[MAXLEN];
	int			 scanresult;
	char			 scanfn[PATH_MAX];
	struct message		*msg;
	struct connection	*conn;

	msg = priv->msg;
	conn = priv->conn;

	bzero(virus, sizeof(virus));

	/* Scan for virus */

	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_virus(backend_data, scanfn, chrootdir, virus, sizeof(virus));

	switch(scanresult) {
	case SCAN_VIRUS:
		syslog(LOG_NOTICE, "virus '%s' detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s",
		    virus,
		    (msg->msgid != NULL) ? msg->msgid : "n/a",
		    (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		    (msg->envfrom != NULL) ? msg->envfrom : "n/a",
		    (msg->hdrto != NULL) ? msg->hdrto : "n/a",
		    backend->name);

		if (logvirus)
			log_incident(priv, "virus", backend->name, virus);

		switch (virus_strategy) {
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_virus(ctx, "infected", virus, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			smfi_setreply(ctx, "554", "5.6.1", "Virus detected in the message");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_NOTIFY_RECIPIENT:
			if (msg->notify) {
				syslog(LOG_INFO, "marking message and replacing message body");
				mark_virus(ctx, "cleaned", virus, backend->name);
				if ((retval = notify_recipient(ctx, virus)) == SMFIS_CONTINUE)
					*stop_chain = 1;

				/* Set the message content type to text/plain */

				if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain"))
					syslog(LOG_INFO, "failed to set message content type to text/plain");
				break;
			}
			/* FALLTRHOUGH */
		case STRATEGY_DISCARD:
		default:
			syslog(LOG_INFO, "silently discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message contains no virus");

		if (markall)
			mark_virus(ctx, "clean", NULL, backend->name);

		if (strcmp(scanfn, msg->tfn)) {
			syslog(LOG_INFO, "replace message body");
			replace_msgbody(ctx, scanfn);
			if (unlink(scanfn))
				syslog(LOG_ERR, "can't delete temp file %s", scanfn);
		}

		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during virus scan of file %s", msg->tfn);

		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");

			mark_virus(ctx, "unchecked", NULL, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
			
	return retval;
}

static sfsistat
vilter_spam(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int			 retval = SMFIS_CONTINUE;
	int			 scanresult;
	double			 score, threshold;
	char			*subject;
	int			 len;
	struct message		*msg;
	struct connection	*conn;

	msg = priv->msg;
	conn = priv->conn;
	score = threshold = 0.0;

	/* Scan for spam */

	scanresult = backend->be_scan.be_scan_spam(backend_data, msg->tfn, chrootdir, &score, &threshold);

	switch(scanresult) {
	case SCAN_SPAM:
		syslog(LOG_NOTICE, "message with id '%s' from '%s' (env '%s') to '%s' classified as spam by backend %s",
		    (msg->msgid != NULL) ? msg->msgid : "n/a",
		    (msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		    (msg->envfrom != NULL) ? msg->envfrom : "n/a",
		    (msg->hdrto != NULL) ? msg->hdrto : "n/a",
		    backend->name);

		if (logspam)
			log_incident(priv, "spam", backend->name, NULL);

		switch (spam_strategy) {
		case STRATEGY_DISCARD:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			smfi_setreply(ctx, "554", "5.6.1", "Message considered spam");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_spam(ctx, "spam", backend->name, score, threshold);

			/* Idea:  Add a subject with spamprefix even if there is no
			   subject. */

			if (spamprefix != NULL && msg->subject != NULL) {
				len = strlen(spamprefix) + strlen(msg->subject) + 2;
				subject = malloc(len);
				if (subject != NULL) {
					snprintf(subject, len, "%s %s", spamprefix, msg->subject);
					smfi_chgheader(ctx, "Subject", 1, subject);
					free(subject);
				} else
					syslog(LOG_ERR, "no space to construct a new subject line");
			}

			retval = SMFIS_CONTINUE;
			break;
		}
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message is not spam");
		mark_spam(ctx, NULL, backend->name, score, threshold);
		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during spam scan of file %s", msg->tfn);
		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_spam(ctx, "unchecked", backend->name, 0.0, 0.0);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
	return retval;
}

static sfsistat
vilter_unwanted(struct backend *backend, struct be_data *backend_data, SMFICTX *ctx, struct privdata *priv, int *stop_chain)
{
	int			 retval = SMFIS_CONTINUE;
	char			 reason[MAXLEN];
	int			 scanresult;
	char			 scanfn[PATH_MAX];
	struct message		*msg;
	struct connection	*conn;

	msg = priv->msg;
	conn = priv->conn;

	bzero(reason, sizeof(reason));

	/* Scan for unwanted content */

	strlcpy(scanfn, msg->tfn, sizeof(scanfn));
	scanresult = backend->be_scan.be_scan_unwanted(backend_data, scanfn, chrootdir, reason, sizeof(reason));

	switch(scanresult) {
	case SCAN_UNWANTED:
		syslog(LOG_NOTICE, "unwanted content  detected in message with id '%s' from '%s' (env '%s') to '%s' by backend %s, reason %s",
		(msg->msgid != NULL) ? msg->msgid : "n/a",
		(msg->hdrfrom != NULL) ? msg->hdrfrom : "n/a",
		(msg->envfrom != NULL) ? msg->envfrom : "n/a",
		(msg->hdrto != NULL) ? msg->hdrto : "n/a",
		backend->name, reason);

		if (logunwanted)
			log_incident(priv, "unwanted-content", backend->name, reason);

		switch (unwanted_strategy) {
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_unwanted(ctx, "unwanted content", reason, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_REJECT:
			syslog(LOG_INFO, "rejecting message");
			smfi_setreply(ctx, "554", "5.6.1", "Unwanted content detected in the message");
			retval = SMFIS_REJECT;
			break;
		case STRATEGY_NOTIFY_RECIPIENT:
			if (msg->notify) {
				mark_unwanted(ctx, "cleaned", reason, backend->name);
				retval = notify_recipient(ctx, reason);

				/* Set the message content type to text/plain */

				if (smfi_chgheader(ctx, "Content-Type", 1, "text/plain"))
					syslog(LOG_INFO, "failed to set message content type to text/plain");
				break;
			}
			/* FALLTRHOUGH */
		case STRATEGY_DISCARD:
		default:
			syslog(LOG_INFO, "silently discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	case SCAN_OK:
		syslog(LOG_INFO, "message contains no unwanted content");
		if (strcmp(scanfn, msg->tfn)) {
			if (markall)
				mark_unwanted(ctx, "cleaned", NULL, backend->name);

			if (logunwanted)
				log_incident(priv, "unwanted-content", backend->name, reason);

			syslog(LOG_INFO, "replace message body");
			replace_msgbody(ctx, scanfn);

			/* Make sure later backends get the new content */

			if (unlink(scanfn))
				syslog(LOG_ERR, "can't delete temp file %s", scanfn);

		} else if (markall)
			mark_unwanted(ctx, "clean", NULL, backend->name);

		break;
	case SCAN_ERROR:
		syslog(LOG_ERR, "error during scan of file %s", msg->tfn);
		switch (error_strategy) {
		case STRATEGY_IGNORE:
			syslog(LOG_INFO, "ignoring the error");
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_MARK:
			syslog(LOG_INFO, "marking message (but deliver anyway)");
			mark_unwanted(ctx, "unchecked", NULL, backend->name);
			retval = SMFIS_CONTINUE;
			break;
		case STRATEGY_TEMPFAIL:
			syslog(LOG_INFO, "temporarily failing message");
			retval = SMFIS_TEMPFAIL;
			break;
		case STRATEGY_DISCARD:
			/* FALLTHROUGH */
		default:
			syslog(LOG_INFO, "discarding message");
			retval = SMFIS_DISCARD;
			break;
		}
		break;
	}
	return retval;
}

#ifdef ENABLE_LDAP
static void
print_entries_with_values(LDAP *ld, LDAPMessage *result)
{
	LDAPMessage	*e;
	BerElement	*ber;
	char		*dn, *attr;
	char		**vals;
	int		 i;

	for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
		if ((dn = ldap_get_dn(ld, e)) != NULL) {
			warnx("dn: %s", dn);
			ldap_memfree(dn);
		}
		for (attr = ldap_first_attribute(ld, e, &ber); attr != NULL; attr = ldap_next_attribute(ld, e, ber)) {
			if ((vals = ldap_get_values(ld, e, attr)) != NULL) {
				for (i = 0; vals[i] != NULL; i++)
					warnx("%s: %s", attr, vals[i]);
				ldap_value_free(vals);
			}
			ldap_memfree(attr);
		}
		if (ber != NULL)
			ber_free(ber, 0);
	}
}
#endif

static sfsistat
vilter_eom(SMFICTX *ctx)
{
	struct		 privdata *priv;
	struct		 message *msg;
	int		 retval = SMFIS_CONTINUE;
	int		 n;
	struct		 be_data *backend_data;
	int		 stop_chain;
#ifdef ENABLE_LDAP
	struct		 recipient *r;
	struct		 domain *d;
	LDAPMessage	*result;
	LDAPMessage	*e;
	char		**vals;
	int		 nentries, rc;	
#endif

	/*
	 * XXX: If there is only one recipient, things are easy:  Take this
	 * recipients strategy settings, if present, else the domain settings,
	 * if these are absent, use the system default.
	 *
	 * If there is more than one recipient, things can get complicated.
	 * Check if all strategies are compatible, if they are, treat the
	 * message accordingly.  If one recipient has a discard strategy
	 * whereas others have a mark strategy, the recipient with discard is
	 * removed from the list of recipients.  The following table gives an
	 * idea how the situation is handled.
	 *
	 * R1		R2			Action
	 * any		the same	use strategy
	 * discard	mark		remove R1, then mark
	 * discard	notify		remove R1, then notify
	 * mark		notify		the only real conflict: notify
	 */

	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		fclose(msg->fp);
		msg->fp = NULL;

		smfi_addheader(ctx, "X-SMTP-Vilter-Version", VERSION);

		/* Scan content */

		for (n = stop_chain = 0; n < nbe && !stop_chain && retval == SMFIS_CONTINUE; n++) {
			backend_data = &priv->backend_data[n];

			if (be[n].type & BE_SCAN_VIRUS) {
				if (verbose)
					warnx("scanning for viruses using backend %s", be[n].name);

				retval = vilter_virus(&be[n], backend_data, ctx, priv, &stop_chain);
			} else if (be[n].type & BE_SCAN_SPAM) {
				if (verbose)
					warnx("scanning for spam using backend %s", be[n].name);

				retval = vilter_spam(&be[n], backend_data, ctx, priv, &stop_chain);
			} else if (be[n].type & BE_SCAN_UNWANTED) {
				if (verbose)
					warnx("scanning for unwanted content using backend %s", be[n].name);
				retval = vilter_unwanted(&be[n], backend_data, ctx, priv, &stop_chain);
			} else
				return SMFIS_TEMPFAIL;
		}

#ifdef ENABLE_LDAP
		SLIST_FOREACH(r, &msg->rhead, entries) {
			warnx("recipient: %s (domain %s)", r->mailLocalAddress, r->domain->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						r->spam_strategy = get_token(spam_strategy_tab, vals[0], TOK_UNKNOWN);
						if (r->spam_strategy == TOK_UNKNOWN) {
							warnx("unknown spam strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
							r->spam_strategy = -1;
						}
						ldap_value_free(vals);
					}

					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						r->virus_strategy = get_token(virus_strategy_tab, vals[0], TOK_UNKNOWN);
						if (r->virus_strategy == TOK_UNKNOWN) {
							warnx("unknown virus strategy '%s' for recipient '%s'", vals[0], r->mailLocalAddress);
							r->virus_strategy = -1;
						}
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						r->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				ldap_msgfree(result);
			}
		}
		SLIST_FOREACH(d, &msg->dhead, entries) {
			warnx("domain: %s", d->mailLocalDomain);
			nentries = 0;
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY) {
				nentries++;
				if (verbose)
					print_entries_with_values(ld, result);
				for (e = ldap_first_entry(ld, result); e != NULL; e = ldap_next_entry(ld, e)) {
					if ((vals = ldap_get_values(ld, e, "spamStrategy")) != NULL) {
						d->spam_strategy = get_token(spam_strategy_tab, vals[0], TOK_UNKNOWN);
						if (d->spam_strategy == TOK_UNKNOWN) {
							warnx("unknown spam strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
							d->spam_strategy = -1;
						}
						ldap_value_free(vals);
					}

					if ((vals = ldap_get_values(ld, e, "virusStrategy")) != NULL) {
						d->virus_strategy = get_token(virus_strategy_tab, vals[0], TOK_UNKNOWN);
						if (d->virus_strategy == TOK_UNKNOWN) {
							warnx("unknown virus strategy '%s' for recipient domain '%s'", vals[0], d->mailLocalDomain);
							d->virus_strategy = -1;
						}
						ldap_value_free(vals);
					}
					if ((vals = ldap_get_values(ld, e, "spamSubjectPrefix")) != NULL) {
						d->spamprefix = strdup(vals[0]);
						ldap_value_free(vals);
					}
				}
				ldap_msgfree(result);
			}
		}	
#endif
		message_cleanup(ctx);
		
	} else {
		syslog(LOG_ERR, "unable to access internal data");
		retval = SMFIS_TEMPFAIL;
	}
	return retval;
}


static sfsistat
vilter_abort(SMFICTX *ctx)
{
#ifdef ENABLE_LDAP
	struct privdata *priv;
	struct message *msg;
	struct recipient *r;
	struct domain *d;
	LDAPMessage *result;
	int rc;
#endif

	syslog(LOG_INFO, "message aborted");
#ifdef ENABLE_LDAP
	if ((priv = smfi_getpriv(ctx)) != NULL) {
		msg = priv->msg;

		SLIST_FOREACH(r, &msg->rhead, entries) {
			while ((rc = ldap_result(ld, r->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}	
		SLIST_FOREACH(d, &msg->dhead, entries) {
			while ((rc = ldap_result(ld, d->msgid, 0, NULL, &result)) == LDAP_RES_SEARCH_ENTRY)
				ldap_msgfree(result);
		}
	}
#endif
	message_cleanup(ctx);
	return SMFIS_CONTINUE;
}

static sfsistat
vilter_close(SMFICTX *ctx)
{
	struct privdata *priv;
	struct connection *conn;

	if (verbose) {
		if ((priv = smfi_getpriv(ctx)) != NULL) {
			conn = priv->conn;
			if (conn != NULL && conn->hostname != NULL) {
				warnx("closing connection with %s", conn->hostname);
				goto cleanup;
			}
		}
		warnx("closing connection");
	}
cleanup:
	connection_cleanup(ctx);
	return SMFIS_CONTINUE;
}
	
#ifdef ENABLE_LDAP
static int
get_rebind_credentials(LDAP *ld, char **dnp, char **pwp, int *authmethodp, int freeit)
{
	warnx("get_rebind_credentials() called");
	return LDAP_SUCCESS;
}
#endif

int
main(int argc, char *argv[])
{
	smfiDesc_str	 desc;
	int		 ch;
	FILE		*fp;
	struct stat	 statbuf;
	char		 libname[MAX_FILE];
	struct passwd	*passwd;
	struct group	*grp;
	uid_t		 uid;
	gid_t		 gid;
	struct rlimit	 rl;
	int		 n;
	int		 status;
	pid_t		 child_pid;
	time_t		 now, last_start;
#ifdef ENABLE_LDAP
	int		 version;
#endif
	int		 xmode;
	const char	*errstr;
	long long rlimit_max;

	/* Set initial values */

	port = NULL;
	tmpdir = NULL;
	tmpfiles_gr = 0;
	tmpfiles_setgrp = 0;
	pidfile = NULL;
	user = NULL;
	group = NULL;
	chrootdir = NULL;
	cfgfile = _PATH_CFGFILE;
	backend = NULL;
	virus_strategy = -1;
	spam_strategy = -1;
	unwanted_strategy = -1;
	error_strategy = -1;

	log_facility = LOG_UNSET;
	logfile = NULL;
	recipient_notification = NULL;
	notify_only = NULL;
	markall = 0;
	rlimit_nofile = -1;
	spamprefix = NULL;
	logvirus = logspam = logunwanted = 0;
	keep_tmpfiles = 0;
	vilterd_host = NULL;
	vilterd_port = -1;
#ifdef ENABLE_LDAP
	ldaphost = NULL;
	ldapport = -1;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
	ldap_use_tls = -1;
#endif
	xmode = 0;

	/* Process the commandline */

#ifdef ENABLE_LDAP	
	while ((ch = getopt(argc, argv, "vmp:d:T:b:e:n:o:C:P:t:u:g:Vf:ka:xD:h:L:w:B:?")) != -1) {
#else
	while ((ch = getopt(argc, argv, "vmp:d:T:b:e:n:o:C:P:t:u:g:Vf:ka:x?")) != -1) {
#endif
		switch (ch) {
		case 'v':
			++verbose;
			break;
		case 'm':
			markall = 1;
			break;
		case 'p':
			port = optarg;
			break;
		case 'd':
			tmpdir = optarg;
			break;
		case 'T':
			if (!strcmp(optarg, "g+r+"))
				tmpfiles_gr = 1;
			else if (!strcmp(optarg, "setgrp"))
				tmpfiles_setgrp = 1;
			else
				errx(1, "unknown temp file option %s", optarg);
			break;
		case 'b':
			backend = optarg;
			decode_backend(backend);
			break;
		case 'e':
			logfile = optarg;
			break;
		case 'n':
			recipient_notification = optarg;
			break;
		case 'o':
			notify_only = optarg;
			break;
		case 'C':
			cfgfile = optarg;
			break;
		case 'P':
			pidfile = optarg;
			break;
		case 't':
			chrootdir = optarg;
			break;
		case 'u':
			user = optarg;
			break;
		case 'g':
			group = optarg;
			break;
		case 'V':
			printf("%s %s\n", __progname, VERSION);
#ifdef ENABLE_LDAP
			printf("(with LDAP support enabled)\n");
#endif
			exit(0);
		case 'f':
			if (getuid()) {
				getrlimit(RLIMIT_NOFILE, &rl);
				rlimit_max = rl.rlim_max;
			}
				rlimit_max = RLIM_INFINITY;
			rlimit_nofile = strtonum(optarg, 1, rlimit_max, &errstr);
			if (errstr)
				errx(1, "number of files is %s: %s", errstr, optarg);
			break; 
		case 'a':
			spamprefix = optarg;
			break;
		case 'k':
			keep_tmpfiles = 1;
			break;
#ifdef ENABLE_LDAP
		case 'D':
			binddn = optarg;
			break;
		case 'h':
			ldaphost = optarg;
			break;
		case 'L':
			ldapport = strtonum(optarg, 1, 65535, &errstr);
			if (errstr)
				errx(1, "ldap port number is %s: %s", errstr, optarg);
			break;
		case 'w':
			bindpasswd = optarg;
			break;
		case 'B':
			searchbase = optarg;
			break;
#endif
		case 'x':
			xmode = 1;
			break;
		default:
			usage();
		}
	}

	/*
	argc -= optind;
	argv += optind;
	*/

	/* Read config file */

	printf("cfgfile: %s\n", cfgfile);

	smtp_vilter_init();

	if (xmode) {
		printf("backends: ");
		for (n = 0; n < nbe; n++)
			printf("%s", be[n].name);
		printf("\n");
		printf("user: %s\n", user != NULL ? user : "not set");
		printf("group: %s\n", group != NULL ? group : "not set");
		printf("chroot: %s\n", chrootdir != NULL ? chrootdir : "not set");
		printf("recipient-notification: %s\n", recipient_notification != NULL ? recipient_notification : "not set");
		printf("spam-subject-prefix: %s\n", spamprefix != NULL ? spamprefix : "not set");
		printf("port: %s\n", port != NULL ? port : "not set");
		printf("tmpdir: %s\n", tmpdir != NULL ? tmpdir : "not set");
		printf("pidfile: %s\n", pidfile != NULL ? pidfile : "not set");
		printf("logfile: %s\n", logfile != NULL ? logfile : "not set");
		printf("ldaphost: %s\n", ldaphost != NULL ? ldaphost : "not set");
		printf("searchbase: %s\n", searchbase != NULL ? searchbase : "not set");
		printf("binddn: %s\n", binddn != NULL ? binddn : "not set");
		printf("bindpasswd: %s\n", bindpasswd != NULL ? bindpasswd : "not set");
		return 0;
	}
	/* Set default values if some variables are not set */

	if (port == NULL)
		port = _PATH_PORT;
	if (tmpdir == NULL)
		tmpdir = _PATH_TMPDIR;
	if (pidfile == NULL)
		pidfile = _PATH_PIDFILE;
	if (backend == NULL)
		backend = "clamd";
	if (virus_strategy == -1)
		virus_strategy = STRATEGY_DISCARD;
	if (spam_strategy == -1)
		spam_strategy = STRATEGY_MARK;
	if (unwanted_strategy == -1)
		unwanted_strategy = STRATEGY_MARK;
	if (error_strategy == -1)
		error_strategy = STRATEGY_TEMPFAIL;
	if (log_facility == LOG_UNSET)
		log_facility = LOG_MAIL;

#ifdef ENABLE_LDAP
	if (ldaphost == NULL)
		ldaphost = DEFLT_LDAPHOST;
	if (ldapport == -1)
		ldapport = DEFLT_LDAPPORT;
	if (ldap_use_tls == -1)
		ldap_use_tls = 0;
		
	if ((ld = ldap_init(ldaphost, ldapport)) == NULL)
		errx(1, "No directory server at %s, %m", ldaphost);
	else if (verbose)
		warnx("using LDAP server %s:%d", ldaphost, ldapport);

	version = LDAP_VERSION3;
	if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version) != LDAP_OPT_SUCCESS)
		errx(1, "Failed to set LDAP version 3 protocol");

	if (ldap_use_tls)
		if (ldap_start_tls_s(ld, NULL, NULL) != LDAP_SUCCESS)
			errx(1, "Failed to start TLS for LDAP");

	/*
	 * Setup LDAP rebind, eventually continue with system defaults if
	 * the initial bind fails?
 	 */

	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS)
		errx(1, "Failed to bind to directory server as '%s'", binddn);

	ldap_set_rebind_proc(ld, get_rebind_credentials, NULL);

	if (verbose)
		warnx("Directory server at %s server is up, %susing TLS", ldaphost,
			ldap_use_tls ? "" : "not ");
#endif
	openlog(__progname, verbose ? LOG_CONS | LOG_NDELAY | LOG_PID | LOG_PERROR : LOG_CONS | LOG_NDELAY | LOG_PID, log_facility);

	/* Load the backends */

	for (n = 0; n < nbe; n++) {
		snprintf(libname, sizeof(libname), "libvilter-%s.so.%d", be[n].name, MAJOR);

		if (verbose)
			warnx("loading backend %s from file %s", be[n].name, libname);
			
		if ((be[n].dlhandle = dlopen(libname, RTLD_LAZY)) == NULL)
			err(1, "error loading backend %s (%s), %s", be[n].name, libname, dlerror());
		
		be[n].be_init = dlsym(be[n].dlhandle, "vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "vilter_end");	/* Optional */
		
		if ((be[n].be_type = dlsym(be[n].dlhandle, "vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);

		be[n].type = be[n].be_type();

		if (be[n].type & BE_SCAN_VIRUS)	{
			if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
			if (be[n].type & (BE_SCAN_SPAM | BE_SCAN_UNWANTED))
				errx(1, "illegal backend type combination %s", be[n].name);
		} else if (be[n].type & BE_SCAN_SPAM) {
			if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
			if (be[n].type & BE_SCAN_UNWANTED)
				errx(1, "illegal backend type combination %s", be[n].name);
		} else if (be[n].type & BE_SCAN_UNWANTED) {
			if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
				errx(1, "no function vilter_scan in backend %s", be[n].name);
		} else
			errx(1, "backend %s has an unknown vilter type", be[n].name);

		be[n].be_exit = dlsym(be[n].dlhandle, "vilter_exit");
		if (be[n].be_init != NULL) {
			if (be[n].be_init(be[n].config_file))
				errx(1, "error initializing backend %s", be[n].name);
		}
	}

	uid = getuid();
	gid = getgid();

	if (user != NULL) {
		if ((passwd = getpwnam(user)) != NULL) {
			uid = passwd->pw_uid;
			gid = passwd->pw_gid;
		} else
			err(1, "no such user '%s'", user);
		
		if (group != NULL) {
			if ((grp = getgrnam(group)) != NULL)
				gid = grp->gr_gid;
			else
				err(1, "no such group '%s'", group);
		}
	}

	if (!stat(pidfile, &statbuf))
		errx(1, "pid file %s exists, another copy running?", pidfile);

	if (verbose == 0 &&  daemon(0, 0))
		err(1, "can't run as daemon");

	/*
	 * Fork into two processes, one to remove the PID file, one to do the
	 * work.
	 */

	do {
		child_pid = 0;
		if (!verbose)
			child_pid = fork();

		if (child_pid == 0) {
#ifndef __linux__	
			setproctitle(NULL);
#endif		

			/* Set up the signal handlers */

			signal(SIGPIPE, sighand);

			/* Set ressource limits */

			if (rlimit_nofile > 0) {
				rl.rlim_cur = rl.rlim_max = rlimit_nofile;

				if (setrlimit(RLIMIT_NOFILE, &rl))
					err(1, "can set number of files to %d", rlimit_nofile);
			}

			if (logfile != NULL && (logfp = fopen(logfile, "a")) == NULL)
				err(1, "can't open logfile %s", logfile);
			else
				logfp = NULL;

			if (verbose)
				warnx("not writing a pid file when running in verbose mode");
			else {
				if ((fp = fopen(pidfile, "w")) == NULL) 
					err(1, "can't open pidfile %s", pidfile);
				if (fprintf(fp, "%d\n", getpid()) == -1)
					err(1, "can't write pid to pidfile %s", pidfile);
				if (fclose(fp))
					err(1, "error closing pidfile %s", pidfile);
			}

			/* chroot, if necessary */

			if (chrootdir) {
				if (getuid())
					errx(1, "must be started by root if using chroot");
				if (chroot(chrootdir))
					err(1, "can't chroot to %s", chrootdir);
				if (chdir("/"))
					err(1, "can't chdir to / after chroot to %s", chrootdir);
			}

			/* Change the user and group id */

			setgid(gid);
			setuid(uid);

			if (!getuid() || !getgid())
				errx(1, "must not be run as root");

			syslog(LOG_INFO, "dropped privileges, running as %d:%d", uid, gid);

			/* Initialize sendmail milter */

			desc.xxfi_name = "smtp-vilter";
			desc.xxfi_version = SMFI_VERSION;
			desc.xxfi_flags = SMFIF_ADDHDRS | SMFIF_CHGHDRS | SMFIF_CHGBODY;

			desc.xxfi_connect = smtp_vilter_connect;	
			desc.xxfi_helo = NULL;
			desc.xxfi_envfrom = smtp_vilter_envfrom;
			desc.xxfi_envrcpt = smtp_vilter_envrcpt;
#ifdef ENABLE_LDAP
			desc.xxfi_flags |= SMFIF_DELRCPT;
#endif
			desc.xxfi_header = vilter_header;
			desc.xxfi_eoh = vilter_eoh;
			desc.xxfi_body = vilter_body;
			desc.xxfi_eom = vilter_eom;
			desc.xxfi_abort = vilter_abort;
			desc.xxfi_close = vilter_close;

			if (smfi_register(desc) == MI_FAILURE)
				err(1, "failed to register with sendmail");	

			if (!strncasecmp(port, "unix:", 5) || !strncasecmp(port, "local:", 6))
				unlink(strchr(port, ':') + 1);

			smfi_setconn(port);
			if (smfi_main() == MI_FAILURE)
				syslog(LOG_ERR, "failed to establish a connection on port %s", port);

			for (n = 0; n < nbe; n++) {
				if (be[n].be_exit != NULL)
					be[n].be_exit();
				free(be[n].name);
				dlclose(be[n].dlhandle);
			}
			free(be);

			syslog(LOG_INFO, "exiting");

			closelog();
			if (logfp != NULL)
				fclose(logfp);
			_exit(2);
		} else {
			time(&last_start);
			if (!verbose) {
				syslog(LOG_INFO, "started child process with pid %d", child_pid);
#ifndef __linux__
				setproctitle("[priv]");
#endif
				wait(&status);
				time(&now);
				sleep(1);
			} else
				now = last_start;
		}
		if (unlink(pidfile))
			syslog(LOG_ERR, "failed to unlink pidfile %s", pidfile);
	} while (!verbose);

#ifdef ENABLE_LDAP
	ldap_unbind(ld);
#endif	
	if (!verbose)
		closelog();
	return 0;
}
