/*
 * Copyright (c) 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <err.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <err.h>
#include <dlfcn.h>
#include <ctype.h>
#include <syslog.h>
#include <math.h>


#include "smtp-vilter.h"

#ifdef LINUX
#include <sys/param.h>
#include "strlfunc.h"
#else
#include <sys/syslimits.h>	/* for PATH_MAX */
#endif

#include "libmilter/mfapi.h"

#define MAX_FILE	1024
#define MAXLEN		256
#define MAXSTR		256
#define	CFGFILE		"/etc/smtp-vilter/smtp-vilter.conf"

enum strategy_tok {
	STRATEGY_UNSET	= -1,
	STRATEGY_DISCARD,
	STRATEGY_MARK,
	STRATEGY_NOTIFY_RECIPIENT,
	STRATEGY_TEMPFAIL,
	STRATEGY_REJECT
};

#define LOG_UNSET					-1

char *port;
int virus_strategy;
int spam_strategy;
int unwanted_strategy;
int error_strategy;
int markall;
int verbose;
char *pidfile;
char *cfgfile;
char *tmpdir;
int tmpfiles_gr;
int tmpfiles_setgrp;
char *backend;
char *recipient_notification;
char *notify_only;
int log_facility;
int logvirus;
int logspam;
int logunwanted;
char *user;
char *group;
char *chrootdir;
int rlimit_nofile;
int nbe;
char *spamprefix;

extern char *__progname;

struct backend {
	char *name;
	char *config_file;
	void *dlhandle;
	int (*be_init)(char *);	/* One time initialization */
	char *(*be_name)(void);
	int (*be_type)(void);
	int (*be_new)(struct be_data *priv, char *from, FILE *tmpfp);	/* Per thread initializer */
	union {
		int (*be_scan_virus)(struct be_data *priv, char *path, char *chroot, char *virus, int viruslen);
		int (*be_scan_spam)(struct be_data *priv, char *path, char *chroot, double *score, double *threshold);
		int (*be_scan_unwanted)(struct be_data *priv, char *path, char *chroot, char *reason, int rlen);
	} be_scan;
	int (*be_end)(struct be_data *priv);	/* Per thread finalizer */
	void (*be_exit)(void);
};

struct backend *be;

struct privdata {
	char tfn[MAX_FILE];	/* Temporary file name */
	char *msgid;		/* Message ID */
	char *envfrom;		/* Envelope sender */
	char *hdrfrom;		/* From: address in message header */
	char *hdrto;		/* To: address in message header */
	char *subject;		/* The message subject */
	int  notify;		/* if != 0, notify recipient */
	FILE *fp;
	struct syslog_data sdata;
	struct be_data *backend_data;
};

enum cmd_tok {
	CMD_BACKEND = 0,
	CMD_VIRUS_STRATEGY,
	CMD_ERROR_STRATEGY,
	CMD_SPAM_STRATEGY,
	CMD_UNWANTED_STRATEGY,
	CMD_PORT,
	CMD_TMPDIR,
	CMD_PIDFILE,
	CMD_LOG_FACILITY,
	CMD_RECIPIENT_NOTIFICATION,
	CMD_NOTIFY_ONLY,
	CMD_OPTION,
	CMD_USER,
	CMD_GROUP,
	CMD_CHROOT,
	CMD_TMPFILES,
	CMD_MAXFILES,
	CMD_SPAM_SUBJECT_PREFIX,
	CMD_LOGFILE,
	CMD_CFGFILE
};
	
static struct token cmd_tab[] = {
	{ "backend",	CMD_BACKEND },
	{ "virus-strategy",	CMD_VIRUS_STRATEGY },
	{ "error-strategy",	CMD_ERROR_STRATEGY },
	{ "spam-strategy",	CMD_SPAM_STRATEGY },
	{ "unwanted-strategy", CMD_UNWANTED_STRATEGY },
	{ "port",	CMD_PORT },
	{ "tmpdir",	CMD_TMPDIR },
	{ "pidfile",	CMD_PIDFILE },
	{ "log-facility",	CMD_LOG_FACILITY },
	{ "recipient-notification",	CMD_RECIPIENT_NOTIFICATION },
	{ "notify-only",	CMD_NOTIFY_ONLY },
	{ "pidfile",	CMD_PIDFILE },
	{ "option",	CMD_OPTION },
	{ "user",	CMD_USER },
	{ "group",	CMD_GROUP },
	{ "chroot",	CMD_CHROOT },
	{ "tmpfiles",	CMD_TMPFILES },
	{ "maxfiles",	CMD_MAXFILES },
	{ "spam-subject-prefix",	CMD_SPAM_SUBJECT_PREFIX },
	{ "logfile",	CMD_LOGFILE },
	{ "config-file",	CMD_CFGFILE },
	{ NULL, 0 }
};

static struct token lfacilities_tab[] = {
	{ "daemon",	LOG_DAEMON, },
	{ "mail",	LOG_MAIL },
	{ "user",	LOG_USER },
	{ "local0",	LOG_LOCAL0 },
	{ "local1",	LOG_LOCAL1 },
	{ "local2",	LOG_LOCAL2 },
	{ "local3",	LOG_LOCAL3 },
	{ "local4",	LOG_LOCAL4 },
	{ "local5",	LOG_LOCAL5 },
	{ "local6",	LOG_LOCAL6 },
	{ "local7",	LOG_LOCAL7 },
	{ NULL,		0 }
};

static struct token virus_strategy_tab[] = {
	{ "discard",			STRATEGY_DISCARD },
	{ "mark",				STRATEGY_MARK },
	{ "notify-recipient",	STRATEGY_NOTIFY_RECIPIENT },
	{ "reject",				STRATEGY_REJECT },
	{ NULL,	0 }
};

static struct token error_strategy_tab[] = {
	{ "discard",	STRATEGY_DISCARD },
	{ "mark",		STRATEGY_MARK },
	{ "tempfail",	STRATEGY_TEMPFAIL },
	{ NULL,	0 }
};

static struct token spam_strategy_tab[] = {
	{ "discard",	STRATEGY_DISCARD },
	{ "mark",		STRATEGY_MARK },
	{ "reject",		STRATEGY_REJECT },
	{ NULL,	0 }
};

static struct token unwanted_strategy_tab[] = {
	{ "discard",			STRATEGY_DISCARD },
	{ "mark",				STRATEGY_MARK },
	{ "reject",				STRATEGY_REJECT },
	{ NULL,	0 }
};

enum option_tok {
	OPT_MARKALL = 0,
	OPT_LOGVIRUS,
	OPT_LOGSPAM,
	OPT_LOGUNWANTED
};

static struct token option_tab[] = {
	{ "markall",	OPT_MARKALL },
	{ "logvirus",		OPT_LOGVIRUS },
	{ "logspam", 		OPT_LOGSPAM },
	{ "logunwanted",	OPT_LOGUNWANTED },
	{ NULL, 0 }
};

enum tmpfiles_opt {
	TF_GRPRD = 0,
	TF_SETGRP
};

static struct token tmpfiles_tab[] = {
	{ "g+r",	TF_GRPRD },
	{ "setgrp",	TF_SETGRP },
	{ NULL,	0 }
};

int
get_token(struct token *token_tab, char *string, int deflt)
{
	int n;
	
	for (n = 0; token_tab[n].string != NULL; n++)
		if (!strcmp(string, token_tab[n].string))
			return token_tab[n].token;
			
	return deflt;
}

static void
usage(void)
{
	fprintf(stderr, "usage: %s [-v] [-V] [-m] [-p port] [-d dir] [-T tmpfile-opt] [-l log-facility] [-s virus-strategy] [-U spam-strategy] [-c unwanted-strategy] [-S error-strategy] [-b backend] [-n recipient-notification] [-o notify-only] [-C configfile] [-P pidfile] [-t chroot-dir] [-u user] [-g group] [-f maxfiles] [-a spam-subject-prefix]", __progname);
	fprintf(stderr, "\n");
	exit(1);
}

static int
decode_backend(char *s)
{
	char *p;
	int n;
	char *q, *r;
	
	if ((q = strdup(s)) == NULL)
		err(1, "memoy allocation error");
	
	nbe = 0;
	r = q;
	do {
		p = strsep(&r, ",");
		if (*p != '\0')
			++nbe;
	} while (r != NULL);
	
	free(q);
		
	be = (struct backend *) malloc(nbe * sizeof(struct backend));
	if (be == NULL)
		err(1, "error allocating memory for backends");
	bzero(be, nbe * sizeof(struct backend));
	
	n = 0;
	do {
		p = strsep(&s, ",");
		if (*p != '\0') {
			if ((be[n].name = strdup(p)) == NULL)
				err(1, "memory allocation error");
			++n;
		}
	} while (s != NULL);
	
	
	return 0;
}

static int
smtp_vilter_init(void)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];
	char *be_cfgfile;
	int n, be_found;
	
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			switch (get_token(cmd_tab, field, TOK_UNKNOWN)) {
				case CMD_BACKEND:
					if (backend == NULL)
						decode_backend(value);
					break;
				case CMD_VIRUS_STRATEGY:
					if (virus_strategy == STRATEGY_UNSET) {
						virus_strategy = get_token(virus_strategy_tab, value, TOK_UNKNOWN);
						if (virus_strategy == TOK_UNKNOWN)
							errx(1, "unknown virus-strategy %s", value);
					}
					break;
				case CMD_ERROR_STRATEGY:
					if (error_strategy == STRATEGY_UNSET) {
						error_strategy = get_token(error_strategy_tab, value, TOK_UNKNOWN);
						if (error_strategy == TOK_UNKNOWN)
							errx(1, "unknown error-strategy %s", value);
					}
					break;
				case CMD_SPAM_STRATEGY:
					if (spam_strategy == STRATEGY_UNSET) {
						spam_strategy = get_token(spam_strategy_tab, value, TOK_UNKNOWN);
						if (spam_strategy == TOK_UNKNOWN)
							errx(1, "unknown spam-strategy %s", value);
					}
					break;
				case CMD_UNWANTED_STRATEGY:
					if (unwanted_strategy == STRATEGY_UNSET) {
						unwanted_strategy = get_token(unwanted_strategy_tab, value, TOK_UNKNOWN);
						if (unwanted_strategy == TOK_UNKNOWN)
							errx(1, "unknown unwanted-strategy %s", value);
					}
					break;
				case CMD_PORT:
					if (port == NULL)
						port = strdup(value);
					break;
				case CMD_TMPDIR:
					if (tmpdir == NULL)
						tmpdir = strdup(value);
					break;
				case CMD_PIDFILE:
					if (pidfile == NULL)
						pidfile = strdup(value);
					break;
				case CMD_LOG_FACILITY:
					if (log_facility == LOG_UNSET) {
						log_facility = get_token(lfacilities_tab, value, TOK_UNKNOWN);
						if (log_facility == TOK_UNKNOWN) {
							warnx("unknown log facility %s, resetting to LOG_MAIL", value);
							log_facility = LOG_MAIL;
						}
					}
					break;
				case CMD_LOGFILE:
					/* We silently ignore this command */
					break;
				case CMD_CFGFILE:
					be_cfgfile = strchr(value, ':');
					if (be_cfgfile != NULL) {
						*be_cfgfile++ = '\0';
						be_found = 0;
						for (n = 0; n < nbe; n++) {
							if (!strcmp(be[n].name, value)) {
								be[n].config_file = strdup(be_cfgfile);
								be_found = 1;
								break;
							}
						}
						if (!be_found) {
							warnx("config-file: backend %s not found (hint: config-file commands must be listed after the backend command)", value);
						}
						
					} else
						warnx("config-file syntax error");
					break;
				case CMD_RECIPIENT_NOTIFICATION:
					recipient_notification = strdup(value);
					break;
				case CMD_NOTIFY_ONLY:
					notify_only = strdup(value);
					break;
				case CMD_OPTION:
					switch (get_token(option_tab, value, TOK_UNKNOWN)) {
						case OPT_MARKALL:
							markall = 1;
							break;
						case OPT_LOGVIRUS:
							logvirus = 1;
							break;
						case OPT_LOGSPAM:
							logspam = 1;
							break;
						case OPT_LOGUNWANTED:
							logunwanted = 1;
							break;
						default:
							warnx("unknown option %s", value);
					}
					break;
				case CMD_USER:
					if (user == NULL)
						user = strdup(value);
					break;
				case CMD_GROUP:
					if (group == NULL)
						group = strdup(value);
					break;
				case CMD_CHROOT:
					if (chrootdir == NULL)
						chrootdir = strdup(value);
					break;
				case CMD_TMPFILES:
					switch (get_token(tmpfiles_tab, value, TOK_UNKNOWN)) {
						case TF_GRPRD:
							tmpfiles_gr = 1;
							break;
						case TF_SETGRP:
							tmpfiles_setgrp = 1;
							break;
						default:
							warnx("unknown mode for tmpfiles: %s", value);
					}
					break;
				case CMD_MAXFILES:
					if (rlimit_nofile == -1)
						rlimit_nofile = atoi(value);
					break;
				case CMD_SPAM_SUBJECT_PREFIX:
					if (spamprefix == NULL)
						spamprefix = strdup(value);
					break;
			}
		}
		fclose(fp);
	} else
		warn("can't open configuration file %s", cfgfile);

	return 0;
}



static sfsistat
vilter_virus(struct backend *backend, struct be_data *backend_data, char *fn)
{
	int retval = SMFIS_CONTINUE;
	char virus[MAXLEN];
	int scanresult;
	char scanfile[PATH_MAX];
	
	bzero(virus, sizeof(virus));
		
	/* Scan for virus */
	
	strlcpy(scanfile, fn, sizeof(scanfile));
	
	scanresult = backend->be_scan.be_scan_virus(backend_data, scanfile, chrootdir, virus, sizeof(virus));
	
	if (strcmp(scanfile, fn)) {
		printf("backend wants to replace original message with content of file %s\n", scanfile);
	}
	 
	switch(scanresult) {
		case SCAN_VIRUS:
			printf("file '%s' contains a virus (%s)\n", fn, virus);
			
			switch (virus_strategy) {
				case STRATEGY_MARK:
					printf("message would be marked, but delivered anyway\n");
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_REJECT:
					printf("message would be rejected\n");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_NOTIFY_RECIPIENT:
					printf("recipient would be notified\n");
					
					/* FALLTRHOUGH */
				case STRATEGY_DISCARD:
				default:
					printf("message would be silently discarded\n");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
		case SCAN_OK:
			printf("file '%s' is clean\n", fn);
			if (markall)
				printf("message would be marked\n");
			break;
		case SCAN_ERROR:
			warnx("error during virus scan of file %s", fn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					printf("marking message (but deliver anyway)\n");

					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					printf("temporarily failing message\n");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					printf("discarding message\n");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}
			
	return retval;
}

static sfsistat
vilter_spam(struct backend *backend, struct be_data *backend_data, char *fn)
{
	int retval = SMFIS_CONTINUE;
	int scanresult;
	double score, threshold;
	
	score = threshold = 0.0;
	
	/* Scan for spam */
	
	scanresult = backend->be_scan.be_scan_spam(backend_data, fn, chrootdir, &score, &threshold);

	switch(scanresult) {

		case SCAN_SPAM:
			printf("file '%s' is considered spam\n", fn);
			switch (spam_strategy) {
				case STRATEGY_DISCARD:
					printf("discarding message\n");
					retval = SMFIS_DISCARD;
					break;
				case STRATEGY_REJECT:
					printf("rejecting message\n");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_MARK:
					printf("marking message (but deliver anyway)\n");
					retval = SMFIS_CONTINUE;
					break;
			}

			break;
		case SCAN_OK:
			printf("file '%s' contains no spam\n", fn);
			break;
		case SCAN_ERROR:
			warnx("error during spam scan of file %s", fn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					printf("marking message (but deliver anyway)\n");
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					printf("temporarily failing message\n");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					printf("discarding message\n");

					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}

	return retval;
}

static sfsistat
vilter_unwanted(struct backend *backend, struct be_data *backend_data, char *fn)
{
	int retval = SMFIS_CONTINUE;
	char reason[MAXLEN];
	int scanresult;
	char scanfile[PATH_MAX];
	
	bzero(reason, sizeof(reason));
		
	/* Scan for virus */
	
	strcpy(scanfile, fn);
	
	scanresult = backend->be_scan.be_scan_unwanted(backend_data, scanfile, chrootdir, reason, sizeof(reason));
	
	if (strcmp(scanfile, fn)) {
		printf("backend wants to replace original message with content of file %s\n", scanfile);
	}
	 
	switch(scanresult) {
		case SCAN_UNWANTED:
			printf("file '%s' contains unwanted content (%s)\n", fn, reason);
			
			switch (unwanted_strategy) {
				case STRATEGY_MARK:
					printf("message would be marked, but delivered anyway\n");
					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_REJECT:
					printf("message would be rejected\n");
					retval = SMFIS_REJECT;
					break;
				case STRATEGY_NOTIFY_RECIPIENT:
					printf("recipient would be notified\n");
					
					/* FALLTRHOUGH */
				case STRATEGY_DISCARD:
				default:
					printf("message would be silently discarded\n");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
		case SCAN_OK:
			printf("file '%s' is clean\n", fn);
			if (markall)
				printf("message would be marked\n");
			break;
		case SCAN_ERROR:
			warnx("error during virus scan of file %s", fn);

			switch (error_strategy) {
				case STRATEGY_MARK:
					printf("marking message (but deliver anyway)\n");

					retval = SMFIS_CONTINUE;
					break;
				case STRATEGY_TEMPFAIL:
					printf("temporarily failing message\n");
					retval = SMFIS_TEMPFAIL;
					break;
				case STRATEGY_DISCARD:
					/* FALLTHROUGH */
				default:
					printf("discarding message\n");
					retval = SMFIS_DISCARD;
					break;
			}
			break;
	}
			
	return retval;
}

int
vilter_test(struct be_data *priv_backend_data, char *fn)
{
	int retval = SMFIS_CONTINUE;
	int n;
	struct be_data *backend_data;
		 		
	/* Scan content */

	printf("Testing file '%s'\n", fn);
	
	/* Call the new message function of each backend */
	
	for (n = 0; n < nbe; n++) {
		if (be[n].be_new != NULL)
			be[n].be_new(&priv_backend_data[n], "user@dom", NULL);
	}
	
	for (n = 0; n < nbe && retval == SMFIS_CONTINUE; n++) {
		backend_data = &priv_backend_data[n];
		switch (be[n].be_type()) {
			case SCAN_VIRUS:
				printf("Scanning for viruses using backend '%s'\n", be[n].name);
				retval = vilter_virus(&be[n], backend_data, fn);
				break;
			case SCAN_SPAM:
				printf("Scanning for spam using backend '%s'\n", be[n].name);
				retval = vilter_spam(&be[n], backend_data, fn);
				break;
			case SCAN_UNWANTED:
				printf("Scanning for unwanted content using backend '%s'\n", be[n].name);
				retval = vilter_unwanted(&be[n], backend_data, fn);
				break;
			default:
				printf("Unknown scan type in backend '%s'\n", be[n].name);
		}
	}

	/* Call the end message function of each backend */
	
	for (n = 0; n < nbe; n++) {
		if (be[n].be_end != NULL)
			be[n].be_end(&priv_backend_data[n]);
	}
	
	return retval;
}

int
main(int argc, char *argv[])
{
	int ch;
	char libname[MAX_FILE];
	struct passwd *passwd;
	struct group *grp;
	struct syslog_data sdata = SYSLOG_DATA_INIT;
	struct be_data *priv_backend_data;
	uid_t uid;
	gid_t gid;
	int n;
	
	/* Set initial values */

	port = NULL;
	tmpdir = NULL;
	tmpfiles_gr = 0;
	tmpfiles_setgrp = 0;
	pidfile = NULL;
	user = NULL;
	group = NULL;
	chrootdir = NULL;
	cfgfile = CFGFILE;
	backend = NULL;
	virus_strategy = STRATEGY_UNSET;
	spam_strategy = STRATEGY_UNSET;
	unwanted_strategy = STRATEGY_UNSET;
	error_strategy = STRATEGY_UNSET;
	log_facility = LOG_UNSET;
	recipient_notification = NULL;
	notify_only = NULL;
	markall = 0;
	rlimit_nofile = -1;
	spamprefix = NULL;
	logvirus = logspam = logunwanted = 0;
	
	/* Process the commandline */

	while ((ch = getopt(argc, argv, "vmp:d:T:s:S:U:b:l:n:o:C:P:t:u:g:Vf:a:?")) != -1) {
		switch (ch) {
			case 'v':
				++verbose;
				break;
			case 'm':
				markall = 1;
				break;
			case 'p':
				port = optarg;
				break;
			case 'd':
				tmpdir = optarg;
				break;
			case 'T':
				switch (get_token(tmpfiles_tab, optarg, TOK_UNKNOWN)) {
					case TF_GRPRD:
						tmpfiles_gr = 1;
						break;
					case TF_SETGRP:
						tmpfiles_setgrp = 1;
						break;
					default:
						errx(1, "unknown temp file option %s", optarg);
				}
				break;
			case 's':
				virus_strategy = get_token(virus_strategy_tab, optarg, TOK_UNKNOWN);
				if (virus_strategy == TOK_UNKNOWN)
					errx(1, "unknown virus-strategy %s", optarg);
				break;
			case 'S':
				error_strategy = get_token(error_strategy_tab, optarg, TOK_UNKNOWN);
				if (error_strategy == TOK_UNKNOWN)
					errx(1, "unknown error-strategy %s", optarg);
				break;
			case 'U':
				spam_strategy = get_token(spam_strategy_tab, optarg, TOK_UNKNOWN);
				if (error_strategy == TOK_UNKNOWN)
					errx(1, "unknow spam-strategy %s", optarg);
				break;
			case 'c':
				unwanted_strategy = get_token(unwanted_strategy_tab, optarg, TOK_UNKNOWN);
				if (unwanted_strategy == TOK_UNKNOWN)
					errx(1, "unknow unwanted-strategy %s", optarg);
				break;
			case 'b':
				backend = optarg;
				decode_backend(backend);
				break;
			case 'l':
				log_facility = get_token(lfacilities_tab, optarg, TOK_UNKNOWN);
				if (log_facility == TOK_UNKNOWN)
					errx(1, "unknown log facility %s", optarg);
				break;
			case 'n':
				recipient_notification = optarg;
				break;
			case 'o':
				notify_only = optarg;
				break;
			case 'C':
				cfgfile = optarg;
				break;
			case 'P':
				pidfile = optarg;
				break;
			case 't':
				chrootdir = optarg;
				break;
			case 'u':
				user = optarg;
				break;
			case 'g':
				group = optarg;
				break;
			case 'V':
				printf("%s %s\n", __progname, VERSION);
				exit(0);
			case 'f':
				rlimit_nofile = atoi(optarg);
				break; 
			case 'a':
				spamprefix = optarg;
				break;
			default:
				usage();
		}
	}
		
	/* Read config file */
	
	smtp_vilter_init();

	/* Set default values if some variables are not set */

	if (port == NULL)
		port = "unix:/var/run/smtp-vilter.sock";
	if (tmpdir == NULL)
		tmpdir = "/tmp";
	if (backend == NULL)
		backend = "clamd";
	if (virus_strategy == STRATEGY_UNSET)
		virus_strategy = STRATEGY_DISCARD;
	if (spam_strategy == STRATEGY_UNSET)
		spam_strategy = STRATEGY_MARK;
	if (unwanted_strategy == STRATEGY_UNSET)
		unwanted_strategy = STRATEGY_MARK;
	if (error_strategy == STRATEGY_UNSET)
		error_strategy = STRATEGY_TEMPFAIL;
	if (log_facility == LOG_UNSET)
		log_facility = LOG_MAIL;

	/* Load the backends */

	for (n = 0; n < nbe; n++) {
		snprintf(libname, sizeof(libname), "libvilter-%s.so.%d", be[n].name, MAJOR);

		if ((be[n].dlhandle = dlopen(libname, RTLD_LAZY)) == NULL)
			err(1, "error loading backend %s (%s), %s", be[n].name, libname, dlerror());

#ifdef __ELF__
		be[n].be_init = dlsym(be[n].dlhandle, "vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "vilter_end");	/* Optional */

		if ((be[n].be_type = dlsym(be[n].dlhandle, "vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);
		
		switch (be[n].be_type()) {
			case SCAN_VIRUS:	
				if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_SPAM:
				if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_UNWANTED:
				if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			default:
				errx(1, "backend %s has an unknown vilter type", be[n].name);
		}
			

		be[n].be_exit = dlsym(be[n].dlhandle, "vilter_exit");
#else	/* assume a.out format */
		be[n].be_init = dlsym(be[n].dlhandle, "_vilter_init");

		if ((be[n].be_name = dlsym(be[n].dlhandle, "_vilter_name")) == NULL)
			errx(1, "no function vilter_name in backend %s", be[n].name);

		be[n].be_new = dlsym(be[n].dlhandle, "_vilter_new");	/* Optional */
		be[n].be_end = dlsym(be[n].dlhandle, "_vilter_end");	/* Optional */

		if ((be[n].be_type = dlsym(be[n].dlhandle, "_vilter_type")) == NULL)
			errx(1, "no function vilter_type in backend %s", be[n].name);
		
		switch (be[n].be_type()) {
			case SCAN_VIRUS:	
				if ((be[n].be_scan.be_scan_virus = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_SPAM:
				if ((be[n].be_scan.be_scan_spam = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			case SCAN_UNWANTED:
				if ((be[n].be_scan.be_scan_unwanted = dlsym(be[n].dlhandle, "_vilter_scan")) == NULL)
					errx(1, "no function vilter_scan in backend %s", be[n].name);
				break;
			default:
				errx(1, "backend %s has an unknown vilter type", be[n].name);
		}
			

		be[n].be_exit = dlsym(be[n].dlhandle, "_vilter_exit");
#endif
		if (be[n].be_init != NULL) {
			if (be[n].be_init(be[n].config_file))
				errx(1, "error initializing backend %s", be[n].name);
		}
	}

	/* Allocate private memory for the backends */
	
	/* Setup private data for syslog, per backend */
	
	sdata.log_stat |= LOG_PERROR;
	
	
	if ((priv_backend_data = malloc(sizeof(struct be_data) * nbe)) == NULL) {
		errx(1, "unable to allocate memory for backend data, %m");
	}
	
	/* Setup syslog data for each backend */
	
	for (n = 0; n < nbe; n++)
		priv_backend_data[n].sdata = &sdata;

	uid = getuid();
	gid = getgid();
		
	if (user != NULL) {
		if ((passwd = getpwnam(user)) != NULL) {
			uid = passwd->pw_uid;
			gid = passwd->pw_gid;
		} else
			err(1, "no such user '%s'", user);
		
		if (group != NULL) {
			if ((grp = getgrnam(group)) != NULL)
				gid = grp->gr_gid;
			else
				err(1, "no such group '%s'", group);
		}
	}
		
	/* Chroot, if necessary */
	
	if (chrootdir && strlen(chrootdir) > 0) {
		if (getuid())
			errx(1, "must be started by root if using chroot");
		if (chroot(chrootdir))
			err(1, "can't chroot to %s", chrootdir);
		if (chdir("/"))
			err(1, "can't chdir to / after chroot to %s", chrootdir);
	}

	/* Change the user and group id */
	
	setgid(gid);
	setuid(uid);
		
	if (!getuid() || !getgid())
		errx(1, "must not be run as root");
	
	if (optind < argc)
		vilter_test(priv_backend_data, argv[optind]);
	
	for (n = 0; n < nbe; n++) {
		if (be[n].be_exit != NULL)
			be[n].be_exit();
		free(be[n].name);
		dlclose(be[n].dlhandle);
	}
	free(be);

	return 0;
}
