/*
 * Copyright (c) 2003, 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>

#include "smtp-vilter.h"

#ifdef LINUX
#include "strlfunc.h"
#endif

#define	SPAMD_CONF	"/etc/smtp-vilter/spamd.conf"

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256

int spamd_port;
char *spamd_host;
int spamd_tries;
int spamd_timeout;
off_t spamd_maxsize;

int
vilter_init(char *cfgfile)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];

	/* Default values */
			
	spamd_port = 783;
	if ((spamd_host = strdup("localhost")) == NULL)
		err(1, "spamd: out of memory");
	spamd_tries = 1;
	spamd_timeout = TIMEOUT;
	spamd_maxsize = 0;
	
	if (cfgfile == NULL)
		cfgfile = SPAMD_CONF;
		
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("spamd: using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			if (!strcmp(field, "host")) {
				free(spamd_host);
				if ((spamd_host = strdup(value)) == NULL)
					err(1, "spamd: out of memory");
				if (verbose)
					warnx("spamd: setting spamd host to %s", spamd_host);
			} else if (!strcmp(field, "port")) {
				spamd_port = atoi(value);
				if (verbose)
					warnx("spamd: setting spamd port to %d", spamd_port);
			} else if (!strcmp(field, "tries")) {
				spamd_tries = atoi(value);
				if (spamd_tries < 1)
					spamd_tries = 1;
				else if (spamd_tries > MAXTRIES)
					spamd_tries = MAXTRIES;
				if (verbose)
					warnx("spamd: setting nr of tries to %d", spamd_tries);
			} else if (!strcmp(field, "timeout")) {
				spamd_timeout = atoi(value);
				if (spamd_timeout < 0)
					spamd_timeout = 0;
				if (verbose)
					warnx("spamd: setting timeout to %d", spamd_timeout);
			} else if (!strcmp(field, "maxsize")) {
				spamd_maxsize = (off_t) atoll(value);
				if (spamd_maxsize < 0)
					spamd_maxsize = 0;
				if (verbose)
					warnx("spamd: setting maxsize to %lld", (long long) spamd_maxsize);
			} else if (verbose)
				warnx("spamd: unknown command %s=%s", field, value);
		}
		fclose(fp);
	} else if (verbose)
		warnx("spamd: configuration file %s for spamd backend not found, using default values", cfgfile);
	
	return 0;
}

int
vilter_scan(struct be_data *priv, char *fn, char *chroot, double *score, double *threshold)
{
	int fd;
	int try;
	int nread;
	FILE *fp;
	struct sockaddr_in server_sockaddr;
	struct timeval tv, *to;
	struct hostent *hostent;
	SOCK *sp;
	char buf[MAXBUF];
	int buflen;
	struct stat statbuf;
	char status[6];
	double s, t;
		
	if (stat(fn, &statbuf)) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: can't stat file to scan, %m");
		return SCAN_ERROR;
	}

	if ((spamd_maxsize > 0) && (statbuf.st_size > spamd_maxsize)) {
		if (verbose)
			warnx("spamd: skipping large message of %lld bytes", (long long) statbuf.st_size);
		return SCAN_OK;
	}

	
	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: unable to obtain network, %m");
		return SCAN_ERROR;
	}

  
	bzero((char *) &server_sockaddr, sizeof(server_sockaddr));
		
	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(spamd_port);

	if (inet_aton(spamd_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(spamd_host);
		if (!hostent) {
			syslog_r(LOG_ERR, priv->sdata, "spamd: unknown host: %s, %m", spamd_host);
			close(fd);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}

	try = 0;
	
	while (connect(fd, (struct sockaddr *) &server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;
		
		if (try == spamd_tries) {
			syslog_r(LOG_ERR, priv->sdata, "spamd: unable to connect socket, %m");
			close(fd);
			return SCAN_ERROR;
		} else
			sleep(1);
	}

	if (spamd_timeout > 0) {
		tv.tv_sec = spamd_timeout;
		tv.tv_usec = 0;
		to = &tv;
	} else
		to = NULL;
	
	snprintf(buf, sizeof(buf), "CHECK SPAMC/1.2\r\nContent-length: %lld\r\n\r\n",
		(long long) statbuf.st_size);
	buflen = strlen(buf);
		
	if (to_send(fd, buf, buflen, 0, to) != buflen) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: %m");
		close(fd);
		return SCAN_ERROR;
	}

	if ((fp = fopen(fn, "r")) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: error opening file to scan, %m");
		close(fd);
		return SCAN_ERROR;
	}
			
	while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0)
		if (to_send(fd, buf, nread, 0, to) != nread) {
			syslog_r(LOG_ERR, priv->sdata, "spamd: error sending file, %m");
			close(fd);
			fclose(fp);
			return SCAN_ERROR;
		}

	fclose(fp);	

	if (shutdown(fd, SHUT_WR)) {
		syslog_r(LOG_ERR, priv->sdata, "error shutting down writing on socket, %m");
		close(fd);
		return SCAN_ERROR;
	}
	
	if ((sp = fdsock(fd)) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "error buffering socker, %m");
		close(fd);
		return SCAN_ERROR;
	}
	
	if (sgets(buf, sizeof(buf), sp, to) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: lost header - no response to CHECK cmd, %m");
		sclose(sp);
		return SCAN_ERROR;
	}
		
	/* SPAMD/1.1 <result_code> <msg> */
	
	
	if (sgets(buf, sizeof(buf), sp, to) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "spamd: spamd: lost report - no response to CHECK cmd, %m");
		sclose(sp);
		return SCAN_ERROR;
	}
	
	/* Spam: True|False ; score / threshold */
	
	sclose(sp);
	sscanf(buf, "Spam: %5s ; %lf / %lf", status, &s, &t);
	
	if (score != NULL)
		*score = s;
	if (threshold != NULL)
		*threshold = t;
		
	if (!strcmp(status, "True"))
		return SCAN_SPAM;
	else
		return SCAN_OK;
		
	return SCAN_ERROR;
}

char *
vilter_name(void)
{
	return "Spamassassin Daemon (spamd)";
}

int
vilter_type(void)
{
	return SCAN_SPAM;
}

void
vilter_exit(void)
{
	free(spamd_host);
}
