/*
 * Copyright (c) 2003, 2004 Marc Balmer <marc@msys.ch>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <ctype.h>
#include <netdb.h>
#include <err.h>
#include <syslog.h>

#include "smtp-vilter.h"

#ifdef LINUX
#include "strlfunc.h"
#endif

#define	NVCD_CONF	"/etc/smtp-vilter/nvcd.conf"

#define TIMEOUT 	60
#define MAXBUF		1024
#define MAXSTR		64
#define MAXTRIES	256

#define SCANTYPE_FILESYSTEM	1
#define SCANTYPE_SOCKET		2	/* Not supported by nvcd at the moment */

int nvcd_port;
char *nvcd_host;
int nvcd_tries;
int nvcd_timeout;
int scantype;
int chroot_scanrealpath;


int
vilter_init(char *cfgfile)
{
	FILE *fp;
	char field[MAXSTR];
	char value[MAXSTR];

	/* Default values */
			
	nvcd_port = 3310;
	if ((nvcd_host = strdup("localhost")) == NULL)
		err(1, "nvcd: out of memory");
	nvcd_tries = 1;
	nvcd_timeout = TIMEOUT;
	scantype = SCANTYPE_FILESYSTEM;
	chroot_scanrealpath = 0;
	
	if (cfgfile == NULL)
		cfgfile = NVCD_CONF;
		
	if ((fp = fopen(cfgfile, "r")) != NULL) {
		if (verbose)
			warnx("nvcd: using configuration from file %s", cfgfile);
			
		while (!read_conf(fp, field, sizeof(field), value, sizeof(value))) {
			if (!strcmp(field, "host")) {
				free(nvcd_host);
				if ((nvcd_host = strdup(value)) == NULL)
					err(1, "nvcd: out of memory");
				if (verbose)
					warnx("nvcd: setting nvcd host to %s", nvcd_host);
			} else if (!strcmp(field, "port")) {
				nvcd_port = atoi(value);
				if (verbose)
					warnx("nvcd: setting nvcd port to %d", nvcd_port);
			} else if (!strcmp(field, "tries")) {
				nvcd_tries = atoi(value);
				if (nvcd_tries < 1)
					nvcd_tries = 1;
				else if (nvcd_tries > MAXTRIES)
					nvcd_tries = MAXTRIES;
				if (verbose)
					warnx("nvcd: setting nr of tries to %d", nvcd_tries);
			} else if (!strcmp(field, "timeout")) {
				nvcd_timeout = atoi(value);
				if (nvcd_timeout < 0)
					nvcd_timeout = 0;
				if (verbose)
					warnx("nvcd: setting timeout to %d", nvcd_timeout);
			} else if (!strcmp(field, "scantype")) {
				if (!strcmp(value, "filesystem")) {
					scantype = SCANTYPE_FILESYSTEM;
					if (verbose)
						warnx("nvcd: set scantype to filesystem");
				} else if (!strcmp(value, "socket")) {
					errx(1, "nvcd: scantype socket not supported at the moment");
					/*
					scantype = SCANTYPE_SOCKET;
					if (verbose)
						warnx("nvcd: set scantype to socket");
					*/
				} else if (verbose)
					warnx("nvcd: unknown scantype %s, using default", value);
			} else if (!strcmp(field, "option")) {
				if (!strcmp(value, "chroot-scanrealpath")) {
					chroot_scanrealpath = 1;
					if (verbose)
						warnx("nvcd: scanning realpath when chrooted");
				} else
					warnx("nvcd: unknown option %s", value);
			} else if (verbose)
				warnx("nvcd: unknown command %s=%s", field, value);
		}
		fclose(fp);
	} else if (verbose)
		warnx("nvcd: configuration file %s for nvcd backend not found, using default values", cfgfile);
	
	return 0;
}

int
vilter_newmsg(struct be_data *priv, char *env_from, FILE *fp)
{
	/*
		Write "From <sender>" to the temporary file so that nvcd
		recognizes the file as e-mail
	*/
	
	fprintf(fp, "From %s\r\n", env_from);
	
	return 0;
}

int
vilter_scan(struct be_data *priv, char *fn, char *chroot, char *virus, int namelen)
{
	int fd, datfd;
	int try;
	int nread;
	FILE *fp;
	unsigned short port;
	struct sockaddr_in server_sockaddr, data_sockaddr;
	struct timeval tv, *t;
	struct hostent *hostent;
	char buf[MAXBUF];
	char *p, *q;
	
	printf("start scanning\n");
	
	if ((fd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to obtain network, %m");
		return SCAN_ERROR;
	}

  
	bzero((char *) &server_sockaddr, sizeof(server_sockaddr));
		
	server_sockaddr.sin_family = AF_INET;
	server_sockaddr.sin_port = htons(nvcd_port);

	if (inet_aton(nvcd_host, &server_sockaddr.sin_addr) == 0) {
		hostent = gethostbyname(nvcd_host);
		if (!hostent) {
			syslog_r(LOG_ERR, priv->sdata, "nvcd: unknown host: %s, %m", nvcd_host);
			close(fd);
			return SCAN_ERROR;
		}
		server_sockaddr.sin_family = hostent->h_addrtype;
		memcpy(&server_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
	}

	try = 0;
	
	while (connect(fd, (struct sockaddr *) &server_sockaddr, sizeof(server_sockaddr)) == -1) {
		++try;
		
		if (try == nvcd_tries) {
			syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to connect socket, %m");
			close(fd);
			return SCAN_ERROR;
		} else
			sleep(1);
	}

	
	if (nvcd_timeout > 0) {
		tv.tv_sec = nvcd_timeout;
		tv.tv_usec = 0;
		t = &tv;
	} else
		t = NULL;
		
	switch (scantype) {
		case SCANTYPE_FILESYSTEM:
			if (chroot_scanrealpath && chroot != NULL) {
				snprintf(buf, sizeof(buf), "%s/%s\r\n", chroot, *fn == '/' ? fn + 1 : fn);
			} else
				snprintf(buf, sizeof(buf), "%s\r\n", fn);
			
			if (to_send(fd, buf, strlen(buf), 0, t) != strlen(buf)) {
				syslog_r(LOG_ERR, priv->sdata, "nvcd: send error, %m");
				close(fd);
				return SCAN_ERROR;
			}
			break;
		case SCANTYPE_SOCKET:
			errx(1, "nvcd: internal error");
			if (to_send(fd, "STREAM\r\n", 8, 0, t) != 8) {
				syslog_r(LOG_ERR, priv->sdata, "nvcd: %m");
				close(fd);
				return SCAN_ERROR;
			}
			if ((nread = to_recv(fd, buf, sizeof(buf), 0, t)) == -1 || nread == 0) {
				syslog_r(LOG_ERR, priv->sdata, "nvcd: no response to STREAM cmd, %m");
				close(fd);
				return SCAN_ERROR;
			}
			
			if (sscanf(buf, "PORT %hu\n", &port) != 1) {
				syslog_r(LOG_ERR, priv->sdata, "nvcd: did not receive port number");
				close(fd);
				return SCAN_ERROR;
			}

			if (verbose)
				warnx("sending data to port %d", port);
				
			if ((datfd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
				syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to obtain network, %m");
				close(fd);
				return SCAN_ERROR;
			}

			bzero((char *) &data_sockaddr, sizeof(data_sockaddr));

			data_sockaddr.sin_family = AF_INET;
			data_sockaddr.sin_port = htons(port);

			if (inet_aton(nvcd_host, &data_sockaddr.sin_addr) == 0) {
				hostent = gethostbyname(nvcd_host);
				if (!hostent) {
					syslog_r(LOG_ERR, priv->sdata, "nvcd: unknown host: %s, %m", nvcd_host);
					close(datfd);
					close(fd);
					return SCAN_ERROR;
				}
				data_sockaddr.sin_family = hostent->h_addrtype;
				memcpy(&data_sockaddr.sin_addr, hostent->h_addr, hostent->h_length);
			}

			try = 0;

			while (connect(datfd, (struct sockaddr *) &data_sockaddr, sizeof(data_sockaddr)) == -1) {
				++try;

				if (try == nvcd_tries) {
					syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to connect socket, %m");
					close(datfd);
					close(fd);
					return SCAN_ERROR;
				} else
					sleep(1);
			}
			
			fp = fopen(fn, "r");
			
			while ((nread = fread(buf, 1, sizeof(buf), fp)) > 0)
				if (to_send(datfd, buf, nread, 0, t) != nread) {
					syslog_r(LOG_ERR, priv->sdata, "nvcd: error sending file, %m");
					close(datfd);
					close(fd);
					fclose(fp);
					return SCAN_ERROR;
				}
			fclose(fp);	
			close(datfd);
			
			break;
		default:
			syslog_r(LOG_ERR, priv->sdata, "nvcd: internal error, unknown scantype %d", scantype);
			close(fd);
			return SCAN_ERROR;
	}
		
	if ((nread = to_recv(fd, buf, sizeof(buf), 0, t)) == -1 || nread == 0) {
		syslog_r(LOG_ERR, priv->sdata, "nvcd: no response from nvcd, %m");
		close(fd);
		return SCAN_ERROR;
	}
	
	close(fd);
	
	printf("got: %s\n", buf);
	
	if ((p = strrchr(buf, ' ')) == NULL) {
		syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to interpret result '%s'", buf);
		return SCAN_ERROR;
	} else {
		++p;
		if ((p = strrchr(buf, ' ')) == NULL) {
			syslog_r(LOG_ERR, priv->sdata, "nvcd: unable to interpret result '%s'", buf);
			return SCAN_ERROR;
		}
		
		
		if (!strncmp(p, "OK", 2)) {/* File is OK */
			if (verbose)
				warnx("message contains no virus");
			return SCAN_OK;
		} else if (!strncmp(p, "FOUND", 5)) {
			*--p = '\0';
			q = strrchr(buf, ' ') + 1;
			syslog_r(LOG_ERR, priv->sdata, "nvcd: found '%s'", q);
			strlcpy(virus, q, namelen);
			return SCAN_VIRUS;
		} else if (!strncmp(p, "ERROR", 5)) {
			*--p = '\0';
			syslog_r(LOG_ERR, priv->sdata, "nvcd: error '%s'", buf);
			return SCAN_ERROR;
		} else {
			syslog_r(LOG_ERR, priv->sdata, "nvcd: unrecognized response: '%s'", p);
		}
 	}
		
	return SCAN_ERROR;
}

char *
vilter_name(void)
{
	return "Norman Virus Control Daemon (nvcd)";
}

int
vilter_type(void)
{
	return SCAN_VIRUS;
}

void
vilter_exit(void)
{
	free(nvcd_host);
}
