/*	$Id: smsmail.c,v 1.5 2006/06/24 10:08:27 mbalmer Exp $	*/

/*
 * Copyright (c) 2006 Marc Balmer <marc@msys.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/param.h>
#include <sys/stat.h>

#include <ctype.h>
#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <ldap.h>
#include <paths.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <tzfile.h>
#include <unistd.h>

#include "smsmail.h"
#include "pathnames.h"

/*
 * SMSMAIL -- send mail using a pager service
 */

#define	MAXLINE	1024			/* max line from mail header */

#define MAXLEN		256
#define MAXSTR		256

#define LDAPHOST	"localhost"
#define LDAPPORT	389
#define LDAPURL		NULL
#define LDAPQUERY	"(&(objectClass=smsRecipient)(smsUserID=$u))"
#define PAGERATTR	"smsPagerID"


LDAP	*ld;
char	*ldaphost;
int	 ldapport;
char	*ldapurl;
char	*searchbase;
char	*binddn;
char	*bindpasswd;
int	 ldap_use_tls;
char	*query;
char	*cfgfile;
char	*pagerattr;
char	*vacationSent;
int	 log_facility;
int	 verbose;
char	*spamheader;

extern char *__progname;

char from[MAXLINE];
char subj[MAXLINE];
time_t interval;
int ldap_has_interval;
char *dn;

int junkmail(void);
int nsearch(char *, char *);
void readheaders(void);
void sendmessage(char *, char **);
void usage(void);
extern void smsmail_init(void);

/*
 * readheaders --
 *	read mail headers
 */
void
readheaders(void)
{
	char buf[MAXLINE], *p;

	while (fgets(buf, sizeof(buf), stdin) && *buf != '\n')
		switch (*buf) {
		case 'F':		/* "From " */
		case 'f':
			if (!strncasecmp(buf, "From ", 5)) {
				for (p = buf + 5; *p && *p != ' '; ++p)
					;
				*p = '\0';
				(void)strlcpy(from, buf + 5, sizeof(from));
				if ((p = strchr(from, '\n')))
					*p = '\0';
				if (junkmail())
					exit(0);
			}
			break;
		case 'R':		/* "Return-Path:" */
		case 'r':
			if (strncasecmp(buf, "Return-Path:",
			    sizeof("Return-Path:")-1) ||
			    (buf[12] != ' ' && buf[12] != '\t'))
				break;
			for (p = buf + 12; *p && isspace(*p); ++p)
				;
			if (strlcpy(from, p, sizeof(from)) >= sizeof(from)) {
				syslog(LOG_NOTICE,
				    "Return-Path %s exceeds limits", p);
				exit(1);
			}
			if ((p = strchr(from, '\n')))
				*p = '\0';
			if (junkmail())
				exit(0);
			break;
		case 'P':		/* "Precedence:" */
		case 'p':
			if (strncasecmp(buf, "Precedence", 10) ||
			    (buf[10] != ':' && buf[10] != ' ' &&
			    buf[10] != '\t'))
				break;
			if (!(p = strchr(buf, ':')))
				break;
			while (*++p && isspace(*p))
				;
			if (!*p)
				break;
			if (!strncasecmp(p, "junk", 4) ||
			    !strncasecmp(p, "bulk", 4) ||
			    !strncasecmp(p, "list", 4))
				exit(0);
			break;
		case 'S':		/* Subject: */
		case 's':
			if (strncasecmp(buf, "Subject:",
			    sizeof("Subject:")-1) ||
			    (buf[8] != ' ' && buf[8] != '\t'))
				break;
			for (p = buf + 8; *p && isspace(*p); ++p)
				;
			if (strlcpy(subj, p, sizeof(subj)) >= sizeof(subj)) {
				syslog(LOG_NOTICE,
				    "Subject %s exceeds limits", p);
				exit(1);
			}
			if ((p = strchr(subj, '\n')))
				*p = '\0';
			break;
		case 'X':		/* X-Spam-Status: spam" */
		case 'x':
			if (spamheader != NULL && !strncasecmp(buf, spamheader,
			    strlen(spamheader)))
				exit(0);
			break;
		default:
			break;
		}

	if (!*from) {
		syslog(LOG_INFO, "no initial \"From\" or \"Return-Path\"line.");
		exit(0);
	}
}

/*
 * nsearch --
 *	do a nice, slow, search of a string for a substring.
 */
int
nsearch(char *name, char *str)
{
	size_t len;

	for (len = strlen(name); *str; ++str)
		if (!strncasecmp(name, str, len))
			return(1);
	return(0);
}

/*
 * junkmail --
 *	read the header and return if automagic/junk/bulk/list mail
 */
int
junkmail(void)
{
	static struct ignore {
		char	*name;
		size_t	 len;
	} ignore[] = {
		{ "-request", 8 },
		{ "postmaster", 10 },
		{ "uucp", 4 },
		{ "mailer-daemon", 13 },
		{ "mailer", 6 },
		{ "-relay", 6 },
		{ NULL, 0 }
	};
	struct ignore *cur;
	size_t len;
	char *p;

	/*
	 * This is mildly amusing, and I'm not positive it's right; trying
	 * to find the "real" name of the sender, assuming that addresses
	 * will be some variant of:
	 *
	 * From site!site!SENDER%site.domain%site.domain@site.domain
	 */
	if (!(p = strchr(from, '%'))) {
		if (!(p = strchr(from, '@'))) {
			if ((p = strrchr(from, '!')))
				++p;
			else
				p = from;
			for (; *p; ++p)
				;
		}
	}
	len = p - from;
	for (cur = ignore; cur->name; ++cur)
		if (len >= cur->len &&
		    !strncasecmp(cur->name, p - cur->len, cur->len))
			return(1);
	return(0);
}

/*
 * sendmessage --
 *	exec sendmail to send the vacation msg to sender
 */
void
sendmessage(char *uid, char **pagerid)
{
	FILE *sfp;
	char cmd[MAXLINE];
	char buf[MAXLINE];
	int n;
	
	snprintf(cmd, sizeof(cmd), "%s -q", _PATH_SENDPAGE);
	for (n = 0; pagerid[n] != NULL; n++) {
		strlcat(cmd, " -p ", sizeof(cmd));
		strlcat(cmd, pagerid[n], sizeof(cmd));
	}

	if (verbose)
		syslog(LOG_INFO, "send sms msg from %s to %s using "
		    "command '%s'", uid, from, cmd);

	sfp = popen(cmd, "w");
	if (sfp == NULL) {
		syslog(LOG_ERR, "cant open pipe");
		exit(1);
	}
	fprintf(sfp, "%s\n\n", subj);
	while (fgets(buf, sizeof(buf), stdin))
		fprintf(sfp, "%s", buf);
	fprintf(sfp, "\n(%s)\n", from);
	pclose(sfp);
}

__dead void
usage(void)
{
	fprintf(stderr, "usage: %s [-VvZ[Z]] [-b searchbase] [-C configfile] "
	    "[-D binddn] [-d smsUserID] [-f from] [-h ldaphost] [-L ldapport] "
	    "[-P pagerattr] [-q query] [-u ldapurl] [-w bindpasswd] "
	    "[-x spamheader]\n", __progname);
	exit(1);
}

int
main(int argc, char *argv[])
{
	int ch;
	int version;
	char *s;
	char *p;
	char *uid;
	char *attrs[] = {
			NULL,	/* pagerattr will be put here */
			NULL
		};
	LDAPMessage *result, *e;
	size_t size;
	char **vals;
	const char *errstr;

	/* Set initial values */
	verbose = 0;

	log_facility = LOG_MAIL;
	ldaphost = NULL;
	ldapport = 0;
	ldapurl = NULL;
	searchbase = NULL;
	binddn = NULL;
	bindpasswd = NULL;
	query = NULL;
	cfgfile = _PATH_CONFIG;
	ldap_use_tls = -1;
	uid = NULL;
	pagerattr = NULL;
	spamheader = NULL;
	
	/* Process the commandline */
	while ((ch = getopt(argc, argv, "b:C:Dd:f:h:P:p:q:u:Vvw:x:Z?")) != -1) {
		switch (ch) {
		case 'b':
			searchbase = optarg;
			break;
		case 'C':
			cfgfile = optarg;
			break;
		case 'D':
			binddn = optarg;
			break;
		case 'd':
			uid = optarg;
			break;
		case 'f':
			if (strlen(optarg)) {
				strlcpy(from, optarg, sizeof(from));
				if (junkmail())
					exit(0);
			}
			break;
		case 'h':
			ldaphost = optarg;
			break;
		case 'P':
			pagerattr = optarg;
			break;
		case 'p':
			ldapport = (int)strtonum(optarg, 1LL, 65535LL, &errstr);
			if (errstr)
				errx(1, "port number is %s:%s", errstr, optarg);
			break;
		case 'q':
			query = optarg;
			break;
		case 'u':
			ldapurl = optarg;
			break;
		case 'V':
			printf("%s %s\n", __progname, VERSION);
			exit(0);
		case 'v':
			verbose = 1;
			break;
		case 'w':
			bindpasswd = optarg;
			break;
		case 'x':
			if (*optarg == 'x' || *optarg == 'X')
				spamheader = optarg;
			else {
				fprintf(stderr, "Spam header must start with "
				    "'x' or 'X'");
				exit(1);
			}
			break;
		case 'Z':
			switch (ldap_use_tls) {
			case -1:
				ldap_use_tls = 1;
				break;
			case 1:
				ldap_use_tls = 2;
				break;
			default:
				usage();
			}
			break;
		default:
			usage();
		}
	}

	argc -= optind;
	argv += optind;

	/* Read config file */
	smsmail_init();

	/* Set default values if some variables are not set */
	if (ldaphost == NULL)
		ldaphost = LDAPHOST;
	if (ldapport == 0)
		ldapport = LDAPPORT;
	if (ldapurl == NULL)
		ldapurl = LDAPURL;
	if (ldap_use_tls == -1)
		ldap_use_tls = 0;
	if (query == NULL)
		query = strdup(LDAPQUERY);
	if (pagerattr == NULL)
		pagerattr = PAGERATTR;

	openlog(__progname, LOG_CONS | LOG_NDELAY | LOG_PID | verbose ?
	    LOG_PERROR : 0, log_facility);

	if (uid == NULL) {
		syslog(LOG_ERR, "No destination uid set");
		exit(1);
	}

	attrs[0] = pagerattr;

	if (searchbase == NULL) {
		syslog(LOG_ERR, "No LDAP basedn set");
		exit(1);
	}

	if (binddn == NULL) {
		syslog(LOG_ERR, "No bind DN set");
		exit(1);
	}

	if (ldapurl == NULL) {
		if ((ld = ldap_init(ldaphost, ldapport)) == NULL) {
			syslog(LOG_ERR, "No directory server at %s, %m",
			    ldaphost);
			exit(1);
		}
	} else {
		if (ldap_initialize(&ld, ldapurl)) {
			syslog(LOG_ERR, "No directory server at %s, %m",
			    ldapurl);
			exit(1);
		}
	}

	version = LDAP_VERSION3;
	if (ldap_set_option(ld, LDAP_OPT_PROTOCOL_VERSION, &version) !=
	    LDAP_OPT_SUCCESS) {
		syslog(LOG_ERR, "Failed to set LDAP version 3 protocol");
		exit(1);
	}

	if (ldap_use_tls > 0 && ldap_start_tls_s(ld, NULL, NULL) !=
	    LDAP_SUCCESS) {
		syslog(LOG_ERR, "Failed to start TLS for LDAP");
		if (ldap_use_tls > 1)
			exit(1);
	}

	if (ldap_simple_bind_s(ld, binddn, bindpasswd) != LDAP_SUCCESS) {
		syslog(LOG_ERR, "Failed to bind to directory server as '%s'",
		    binddn);
		bzero(bindpasswd, strlen(bindpasswd));
		exit(1);
	}
	bzero(bindpasswd, strlen(bindpasswd));

	if (verbose)
		syslog(LOG_INFO, "bound to LDAP server as %s", binddn);

	if ((s = strstr(query, "$u")) != NULL) {
		size = strlen(query) + strlen(uid) - 1;

		if ((p = malloc(size)) == NULL) {
			syslog(LOG_ERR, "memory allocation error");
			exit(1);
		}

		bzero(p, size);

		*s = 0;
		strlcpy(p, query, size);
		strlcat(p, uid, size);
		strlcat(p, s+2, size);
		query = p; 
	}

	if (verbose)
		syslog(LOG_INFO, "LDAP query: %s", query);

	if (ldap_search_s(ld, searchbase, LDAP_SCOPE_SUBTREE, query, attrs, 0,
	    &result) == LDAP_SUCCESS) {
		if (!ldap_count_entries(ld, result))
			exit(0);	

		dn = NULL;

		if ((e = ldap_first_entry(ld, result)) != NULL)
			dn = ldap_get_dn(ld, e);

		if (dn == NULL) {
			syslog(LOG_ERR, "Can't get dn");
			exit(1);
		}

		if (verbose)
			syslog(LOG_INFO, "DN: %s", dn);

		if ((vals = ldap_get_values(ld, e, pagerattr)) != NULL) {
			readheaders();
			sendmessage(uid, vals);
			ldap_value_free(vals);
		} else {
			syslog(LOG_ERR, "no pager id defined");
			exit(0);
		}

	} else if (verbose)
		syslog(LOG_INFO, "query for sms user failed");

	return 0;
}
