/***********************************************************
node.c: network core for serving the master (master.c).
   Copyright (C) 1999. HaPpY (katami@hotmail.com).
   This source code is protected under the GPL. Read the
   README file for details.
***********************************************************/


/////////////////////////////
#include "node.h"


/////////////////////////////
typedef struct
{
   Terminal self, master;
   fd_set net_fds;
   struct timeval timeout;
} GlbData;


/////////////////////////////
static GlbData global;
extern u_char *message_handler( u_char *, u_int * );
extern int commline_handler( int, u_char ** );
extern int exit_handler( void );
extern void print_handler( const u_char *, ... );

u_short *node_port = &global.self.addr.sin_port;


/////////////////////////////
int init( void );
void sleep_mode( void );
void disconnect( void );
void error( const u_char *str, ... );
void clean_up( int exit_status );


/////////////////////////////
int init( void )
{
   global.timeout.tv_sec = CONNECT_TIMEOUT;
   global.timeout.tv_usec = 0;
   FD_ZERO(&global.net_fds);

   global.master.socket = NULL_SOCKET;
   global.self.socket = socket(AF_INET,SOCK_STREAM,0);
   global.self.addr.sin_family = AF_INET;
   global.self.addr.sin_addr.s_addr = INADDR_ANY;
   if( !global.self.addr.sin_port )
      global.self.addr.sin_port = DEFAULT_PORT;		// no port chosen, use default
   global.self.sa_len = sizeof(global.self.addr);
   
   if( bind( global.self.socket,
      (struct sockaddr *)&global.self.addr,
      global.self.sa_len) )
      return 0;

   return 1;
}

void sleep_mode( void )
{
   listen( global.self.socket, 1 );
   global.master.socket = accept(global.self.socket,
      (struct sockaddr *)&global.master.addr,
      &global.master.sa_len );
   FD_SET(global.master.socket, &global.net_fds);
   strcpy(global.master.ipstr,(u_char *)inet_ntoa(&global.master.addr.sin_addr));
// indicate a first time connect to master
   write( global.master.socket, FIRST_SEND, strlen(FIRST_SEND) );
}

void disconnect( void )
{
   close(global.master.socket);
   if(global.master.socket!=NULL_SOCKET)
      FD_CLR(global.master.socket, &global.net_fds);
   global.master.socket = NULL_SOCKET;
}

void error( const u_char *str, ... )
{
   static u_char tbuf[MAX_STR_LEN];
   static int id;
   va_list parg;
   
   va_start(parg, str);
   vsprintf(tbuf,str,parg);
   va_end(parg);
   print_handler( tbuf );

   clean_up(1);
}

void clean_up( int exit_status )
{
   disconnect();
   exit(exit_status);
}


/////////////////////////////
int main( int argc, u_char **argv )
{
   u_int exit_state = 0;
   global.self.addr.sin_port = 0;

// check for specific port request at command line
   if(!commline_handler(argc,argv))
      clean_up(1);
   if(!init())
      error("could not initialize on port: %d\n", global.self.addr.sin_port );

   while(!exit_state)
   {
   // sleep
      print_handler("listening on: %d\n", global.self.addr.sin_port );
      sleep_mode();
   // awake
      print_handler("master connected at %s.\n",
         global.master.ipstr);
   // work
      while( global.master.socket != NULL_SOCKET )
      {
         static struct timeval tv;
         static u_char netbuf[MAX_PACKET_SIZE];
         tv  = global.timeout;

         if( select(FD_SETSIZE, &global.net_fds, (fd_set *)0, (fd_set *)0, &tv) < 1 )
         {
            print_handler("error: connection timed out.\n");
            disconnect();
         }
         else
         {
            static u_char *pstr;
            static u_int len;

            read(global.master.socket, netbuf, MAX_PACKET_SIZE);
            if( pstr = message_handler(netbuf,&len) )
            {
              	write( global.master.socket, pstr, len );
               print(".\n");
            }
            else
            {
               disconnect();
               exit_state = exit_handler();
            }
         }
         netbuf[0]=0;
      }
   }

   clean_up(exit_state);
}

