/*
** Copyright (c) 1986, 1994, 1996, 2000
**	Jeff Forys (jeff@forys.cranbury.nj.us).  All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that: (1) Redistributions of
** source code must retain the above copyright notice, this list of
** conditions and the following disclaimer, (2) Redistributions in
** binary form must reproduce the above copyright notice, this list
** of conditions and the following disclaimer in the documentation
** and/or other materials provided with the distribution, (3) All
** advertising materials mentioning features or use of this software
** must display the following acknowledgment: ``This product includes
** software developed by Jeff Forys (jeff@forys.cranbury.nj.us).'', (4)
** The name of the author may not be used to endorse or promote products
** derived from this software without specific prior written permission.
**
** THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
** WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
** MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** SunOS 4.0 support by Greg Earle (earle@isolar.Tujunga.CA.US)
*/

#ifndef lint
static char rcsid[] = "$Id: sunos-40.c,v 1.9 2000/05/30 01:41:08 forys Exp $";
#endif

#define	NO_MEXTERN
#include "conf.h"
#undef	NO_MEXTERN

#include <sys/user.h>
#include <sys/proc.h>

#include <stdio.h>

/*
 * Define SigNames, NSig, and TtyDevDir here; they are used by other
 * routines and must be global.  Everyone seems to have their own
 * idea as to what NSIG should be.  Here, `NSig' is the number of
 * signals available, not counting zero.
 */
char *SigMap[] = { "0",
	"HUP", "INT", "QUIT", "ILL", "TRAP", "IOT",		/*  1 -  6 */
	"EMT", "FPE", "KILL", "BUS", "SEGV", "SYS",		/*  7 - 12 */
	"PIPE", "ALRM", "TERM", "URG", "STOP", "TSTP",		/* 13 - 18 */
	"CONT", "CHLD", "TTIN", "TTOU", "IO", "XCPU",		/* 19 - 24 */
	"XFSZ", "VTALRM", "PROF", "WINCH", "LOST", "USR1",	/* 25 - 30 */
	"USR2", "32",						/* 31 - 32 */
};
int NSig = NSIG;

#define	SETCMD(dst,src,maxlen) {			\
	extern char *rindex();				\
	if (maxlen > 0) src[maxlen] = '\0';		\
	dst = (dst = rindex(src, '/')) ? ++dst: src;	\
}

static char *TtyDevDir = "/dev";

int	Skill;			/* set 1 if running `skill', 0 if `snice' */
int	PrioMin, PrioMax;	/* min and max process priorities */
int	SigPri;			/* signal to send or priority to set */
pid_T	MyPid;			/* pid of this process */
uid_T	MyUid;			/* uid of this process */
char	*ProgName;		/* program name */

/*
 * This is the machine-dependent initialization routine.
 *
 *   - The following global variables must be initialized:
 *     MyPid, MyUid, ProgName, Skill, PrioMin, PrioMax, SigPri
 *   - The working directory will be changed to that which contains the
 *     tty devices (`TtyDevDir'); this makes argument parsing go faster.
 *   - If possible, this routine should raise the priority of this process.
 */
void
MdepInit(pname)
	char *pname;
{
	extern char *rindex(), *SysErr();

	MyPid = (pid_T) getpid();
	MyUid = (uid_T) getuid();
	SETCMD(ProgName, pname, 0)

	/*
	 * If we are running as root, raise our priority to better
	 * catch runaway processes.
	 */
	if (MyUid == ROOTUID)
		(void) setpriority(PRIO_PROCESS, MyPid, PRIO_MIN);

	/*
	 * Determine what we are doing to processes we find.  We will
	 * either send them a signal (skill), or renice them (snice).
	 */
	Skill = (strcmp(ProgName, "snice") != 0);

	/*
	 * chdir to `TtyDevDir' to speed up tty argument parsing.
	 */
	if (chdir(TtyDevDir) < 0) {
		fprintf(stderr, "%s: chdir(%s): %s\n", ProgName, TtyDevDir,
		        SysErr());
		exit(EX_SERR);
	}

	/*
	 * Set up minimum and maximum process priorities.
	 * Initialize SigPri to either default signal (`skill') or
	 * default priority (`snice').
	 */
	PrioMin = PRIO_MIN;
	PrioMax = PRIO_MAX;
	SigPri = Skill? SIGTERM: 4;
}

/*
 * Carry out an action on a particular process.  If this is `skill',
 * then send the process a signal, otherwise this is `snice' so change
 * it's priority.
 *
 * If 0 is returned, the operation was successful, otherwise -1 is
 * returned and `errno' set.
 */
int
MdepAction(pid)
	pid_T pid;
{
	if (Skill)
		return(kill((int)pid, SigPri));
	else
		return(setpriority(PRIO_PROCESS, (int)pid, SigPri));
}

/*
 * Now, set up everything we need to write a GetProc() routine.
 */

#include <sys/file.h>

#include <kvm.h>
#include <nlist.h>

static kvm_t *kd = NULL;

static	char	*pidmap[] = { "swapper", "init", "pagedaemon" };
static	int	pidmapsiz = sizeof(pidmap) / sizeof(pidmap[0]);

extern	off_t lseek();

/*
 * GetProc()
 *
 * Fill in and return a `struct ProcInfo' with information about the
 * next process.  If no processes are left, return NULL.
 */
struct ProcInfo *
GetProc()
{
	static char *WarnMsg = "Warning: can't read ";
	static struct ProcInfo procinfo;
	register struct user *auser;
	register struct proc *aproc;

	/*
	 * If this is our first time here, prepare to read procs from kernel.
	 */
	if (kd == NULL) {
		if ((kd = kvm_open((char *)NULL, (char *)NULL, (char *)NULL,
		              O_RDONLY, ProgName)) == NULL)
			exit(EX_SERR);
		if (kvm_setproc(kd) == -1) {
			fprintf(stderr,"%s: kvm_setproc: failed\n", ProgName);
			exit(EX_SERR);
		}
	}

	do {
		if ((aproc = kvm_nextproc(kd)) == NULL)
			return((struct ProcInfo *)NULL);

		if (aproc->p_stat != 0) {
			/*
			 * Before we go through the trouble of reading
			 * in the user struct, let's make sure this isn't
			 * a "zombie" or "exiting" process.  If it is,
			 * we have all the information we need; fill in
			 * procinfo and return.
			 */
			procinfo.pi_flags = 0;
			procinfo.pi_pid = (pid_T) aproc->p_pid;
			procinfo.pi_uid = (uid_T) aproc->p_uid;

			if (aproc->p_stat == SZOMB) {		/* zombie */
				static char *zombie = "<defunct>";
				procinfo.pi_flags |= PI_ZOMBIE;
				procinfo.pi_cmd = zombie;
			} else if (aproc->p_flag & SWEXIT) {	/* exiting */
				static char *exiting = "<exiting>";
				procinfo.pi_flags |= PI_SWEXIT;
				procinfo.pi_cmd = exiting;
			}

			if (procinfo.pi_flags)
				return(&procinfo);
			else if ((auser = kvm_getu(kd, aproc)) == NULL && Wflag)
				printf("%su for pid %d with kvm_getu\n",
				       WarnMsg, aproc->p_pid);
		}

	} while (aproc->p_stat == 0 || auser == NULL);

	/*
	 * We now have a process (`aproc') and a user (`auser').
	 * Fill in the rest of `procinfo'.
	 */
	if (auser->u_ttyp != 0) {	/* has a controlling tty */
		procinfo.pi_flags |= PI_CTLTTY;
		procinfo.pi_tty = (tty_T) auser->u_ttyd;
	}

	if (aproc->p_pid < pidmapsiz) {	/* special */
		procinfo.pi_cmd = pidmap[aproc->p_pid];
		procinfo.pi_flags |= PI_ASKUSR;
	} else				 /* set path-stripped command name */
		SETCMD(procinfo.pi_cmd, auser->u_comm, MAXCOMLEN)

	return(&procinfo);
}
