/*
 * SiSCtrl - SiS Display Control Panel
 * for the SiS X.org/XFree86 driver
 *
 * Configuration page generators
 *
 * (C) 2003-2005 Thomas Winischhofer <thomas@winischhofer.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "sisctrl_def.h"

void 	cf_generate_current(int hints);

extern sc_globals	sc;

/* Routines from MAIN part */

extern int	getxvval(Atom atom);
extern int	newgetxvval(unsigned int cmd, Atom atom);
extern int	newgetrealxvval(unsigned int cmd, Atom atom);
extern int	sd_gettvposscale(unsigned int cmd, Atom atom);
extern int	sd_readcolorcalib(Atom myatom, Bool IsCoarse);

extern Bool	isusb(void);
extern Bool	supportvihuesat(void);
extern Bool	vbhavevb(void);
extern Bool	vbissisbridge(void);
extern Bool	vbischrontel(void);
extern Bool	vbsupportsvga2(void);
extern Bool	vbsupportsscart(void);
extern Bool	vbsupportstv(void);
extern Bool	vbsupportsoverscan(void);
extern Bool	vbsupportstmds(void);
extern Bool	vbsupportsettvtype(void);
extern Bool	vbsupportgamma2(void);
extern Bool	vbsupporttvedgeenhance(void);
extern Bool	vbsupporttvsaturation(void);
extern Bool	vbsupporttvsize(void);

/* Our GUI routines */

extern void gui_cf_clear_text(void);
extern void gui_cf_append_text(char *mytext, int texttag);
extern void gui_cf_insert_option(char *oname, char *ovalue, int texttag);

/* Code start */

static void
conf_ap_three(char *text1, char *text2, char *text3, int texttag)
{
   gui_cf_append_text(text1, texttag);
   gui_cf_append_text(text2, texttag);
   gui_cf_append_text(text3, texttag);
}

static void
conf_ap_four(char *text1, char *text2, char *text3, char *text4, int texttag)
{
   gui_cf_append_text(text1, texttag);
   gui_cf_append_text(text2, texttag);
   gui_cf_append_text(text3, texttag);
   gui_cf_append_text(text4, texttag);
}

static void
conf_fix_comma(char *text)
{
   int i;

   for(i = 0; i < strlen(text); i++) {
      if(text[i] == ',') text[i] = '.';
   }
}

static void
conf_append_header_basic(int hints)
{
   char header[] = "For the current configuration - except mode and desktop size settings - "
		   "to be permanent, you need to edit your X configuration file. This file ";
   char header1[]= "is probably named ";
   char header1a[]="consists of several \"sections\". Most of the configuration SiSCtrl can do "
		   "affects the \"";
   char header1b[]="\" sections. In the following, you see the "
		   "data to be put in these sections to preserve the current setup.\n";

   gui_cf_append_text(header, CU_TT_BLACK);

   if(strstr(XServerVendor(sc.dpy), "XFree86")) {

      gui_cf_append_text(header1, CU_TT_BLACK);
      gui_cf_append_text("/etc/X11/XF86Config", CU_TT_BLACK_BOLD);
      gui_cf_append_text(" or ", CU_TT_BLACK);
      gui_cf_append_text("/etc/X11/XF86Config-4", CU_TT_BLACK_BOLD);
      gui_cf_append_text(" and ", CU_TT_BLACK);

   } else if(strstr(XServerVendor(sc.dpy), "X.Org")) {

      gui_cf_append_text(header1, CU_TT_BLACK);
      gui_cf_append_text("/etc/X11/xorg.conf", CU_TT_BLACK_BOLD);
      gui_cf_append_text(" and ", CU_TT_BLACK);

   }
   gui_cf_append_text(header1a, CU_TT_BLACK);
   gui_cf_append_text("Monitor", CU_TT_BLACK_BOLD);
   gui_cf_append_text("\" and \"", CU_TT_BLACK);
   gui_cf_append_text("Device", CU_TT_BLACK_BOLD);
   gui_cf_append_text(header1b, CU_TT_BLACK);
}

static void
conf_append_header_extended(int hints)
{

   char header2[]= "\n(Note that this fragment does not suffice to fully configure X";
   char header2a[]=", not to mention Dual Head, Xinerama or MergedFB mode";
   char header2b[]=". There are several other sections required, see ";
   char header3[]= ". For detailed SiS driver related documentation, see "
		   "http://www.winischhofer.at/linuxsisvga.shtml)\n";

   gui_cf_append_text(header2, CU_TT_BLACK);
   if(vbhavevb()) {
      gui_cf_append_text(header2a, CU_TT_BLACK);
   }
   gui_cf_append_text(header2b, CU_TT_BLACK);
   gui_cf_append_text("man Xserver", CU_TT_BLACK_BOLD);
   gui_cf_append_text(header3, CU_TT_BLACK);
}

static void
conf_append_dhm_header(int crtnum, int hints)
{
   char header[] = "\nYou are currently running in Dual Head or Xinerama mode. Note that "
		   "this instance of SiSCtrl can - mostly - only configure settings for ";

   char headerdhm[]= "as it was started from this head's session.\n";

   char headerxin[]= "as it connected to this head's screen, either by default or by your choice. "
		     "Use the -screen "
		     "parameter to make SiSCtrl connect to a specific screen. Note that "
		     "\"-display :0.1\" will not work if Xinerama is active! Remember "
		     "that Xinerama fakes ONE big display to clients!\n";
   char *thisone;
   char buffer[8];

   sprintf(buffer, "CRT%d ", crtnum);

   if(sc.xinerama) thisone = headerxin;
   else            thisone = headerdhm;

   conf_ap_three(header, buffer, thisone, CU_TT_BLACK);
}

static void
conf_append_legend(int hints)
{
   gui_cf_append_text("\nblack/bold = keywords; section delimiters\n", CU_TT_BLACK_BOLD);
   gui_cf_append_text("grey", CU_TT_GREY);
   gui_cf_append_text(" = configuration hints\n", CU_TT_BLACK);
   gui_cf_append_text("red", CU_TT_RED);
   gui_cf_append_text(" = settings changed/added by sisctrl\n", CU_TT_BLACK);
   if(hints != CF_HINTS_NO) {
      gui_cf_append_text("blue", CU_TT_BLUE);
      gui_cf_append_text(" = additional configuration hints\n", CU_TT_BLACK);
   }
   gui_cf_append_text("Lines starting with \"#\" are comments and will be ignored\n", CU_TT_BLACK);
}

static void
conf_append_section_monitor(void)
{
   gui_cf_append_text("Section \"Monitor\"\n", CU_TT_BLACK_BOLD);
}

static void
conf_append_section_device(void)
{
   gui_cf_append_text("Section \"Device\"\n", CU_TT_BLACK_BOLD);
}

static void
conf_append_endsection(void)
{
   gui_cf_append_text("EndSection\n", CU_TT_BLACK_BOLD);
}

static void
conf_append_hsvr_hint(int hints)
{
   if(hints == CF_HINTS_GE) {
      if(!(sc.sd2flags & SiS_SD2_NODDCSUPPORT)) {
         gui_cf_append_text("\t# HorizSync and VertRefresh are only required if\n", CU_TT_BLUE);
         gui_cf_append_text("\t# your display device does not support DDC\n", CU_TT_BLUE);
      } else {
         gui_cf_append_text("\t# HorizSync and VertRefresh are required because\n", CU_TT_BLUE);
         gui_cf_append_text("\t# this hardware does not support DDC\n", CU_TT_BLUE);
      }
   }
}

static void
conf_append_driver(Bool ishint)
{
   int col = ishint ? CU_TT_BLUE : CU_TT_GREY;
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_BLACK_BOLD;

   gui_cf_append_text("\n\t# Now for the real thing:\n", col);
   if(!sc.busisusb) {
      gui_cf_append_text("\tDriver \"sis\"\n", colb);
   } else {
      gui_cf_append_text("\tDriver \"sisusb\"\n", colb);
   }
}

static void
conf_append_enablesisctrl_option(Bool ishint)
{
   int col = ishint ? CU_TT_BLUE : CU_TT_GREY;
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_BLACK_BOLD;

   gui_cf_append_text("\n\t# EnableSiSCtrl must be set to use SiSCtrl\n", col);
   gui_cf_append_text("\tOption \"EnableSiSCtrl\" \"yes\"\n", colb);
}

static void
conf_append_usesse(int hints)
{
   if((hints == CF_HINTS_GE) && (sc.sd2flags & SiS_SD2_NEEDUSESSE)) {
      gui_cf_append_text("\n\t# Make use of SSE CPU instructions.\n", CU_TT_BLUE);
      gui_cf_append_text("\t# If your operating system supports using the CPU's\n", CU_TT_BLUE);
      gui_cf_append_text("\t# SSE instructions (Linux and *BSD do), you can set\n", CU_TT_BLUE);
      gui_cf_append_text("\t# this to accelerate memory transfers. X.org 6.9 and\n", CU_TT_BLUE);
      gui_cf_append_text("\t# later will do this automatically.\n", CU_TT_BLUE);
      gui_cf_append_text("\tOption \"UseSSE\" \"yes\"\n", CU_TT_BLUE_BOLD);
   }
}

static void
conf_append_busid(Bool ishint)
{
   char buffer[32];
   int col = ishint ? CU_TT_BLUE : CU_TT_GREY;
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_BLACK_BOLD;

   if(isusb()) return;

   gui_cf_append_text("\n\t# \"BusID\" is mandatory for Dual Head mode.\n", col);

   if(sc.havenewhwinfo) {
      sprintf(buffer, "\tBusID \"PCI:%d:%d:%d\"\n", sc.pciBus, sc.pciDev, sc.pciFunc);
   } else {
      sprintf(buffer, "\tBusID \"PCI:%d:%d:%d\"\n", (sc.busid >> 16), (sc.busid >> 8) & 0xff, sc.busid & 0xff);
   }
   gui_cf_append_text(buffer, colb);
}

static void
conf_append_screen(int crtnum, Bool ishint, int hints)
{
   char buffer[16];
   int col = ishint ? CU_TT_BLUE : CU_TT_GREY;
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_BLACK_BOLD;

   if(isusb()) return;

   gui_cf_append_text("\n\t# \"Screen\" is mandatory for Dual Head mode.\n", col);
   gui_cf_append_text("\t# \"Screen 0\" is CRT2 (master), \"Screen 1\" is CRT1 (slave)\n", col);

   if(hints == CF_HINTS_GE) {
      gui_cf_append_text("\t# Note: The \"Screen\" keyword is only for running Dual Head\n", CU_TT_BLUE);
      gui_cf_append_text("\t# mode (no matter whether with or without Xinerama) on one\n", CU_TT_BLUE);
      gui_cf_append_text("\t# graphics card with two outputs. If you run Dual Head mode\n", CU_TT_BLUE);
      gui_cf_append_text("\t# with two graphics cards, LEAVE THIS STATEMENT OUT!\n", CU_TT_BLUE);
      gui_cf_append_text("\t# If you run Dual Head mode with a SiS card using both CRT1\n", CU_TT_BLUE);
      gui_cf_append_text("\t# and CRT2, place \"Screen\" 0 or 1 in the corresponding\n", CU_TT_BLUE);
      gui_cf_append_text("\t# \"Device\" sections. If you have a second physical graphics\n", CU_TT_BLUE);
      gui_cf_append_text("\t# card in your system that should work as a third head, do NOT \n", CU_TT_BLUE);
      gui_cf_append_text("\t# place a \"Screen\" statement in that card's \"Device\" section!\n", CU_TT_BLUE);
   }

   sprintf(buffer, "\tScreen %d\n", (crtnum == 1) ? 1 : 0);
   gui_cf_append_text(buffer, colb);
}

static void
conf_append_crt1_type(int hints)
{
   char *mode;

   if(!vbhavevb()) return;

   if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
      gui_cf_append_text("\n\t# [sisctrl] Set CRT1 device type (Note: overrides auto-detection)\n", CU_TT_RED);
   } else {
      gui_cf_append_text("\n\t# [sisctrl] Enable/disable CRT1 (Note: overrides auto-detection)\n", CU_TT_RED);
   }
   if(sc.CRT1isoff) {
      if((hints == CF_HINTS_MFB) || (hints == CF_HINTS_DHM)) {
         if(hints == CF_HINTS_MFB) mode = "MergedFB";
	 else mode = "Dual Head";
	 conf_ap_three("\t# CRT1 can't be disabled in ", mode, " mode.\n", CU_TT_BLUE);
	 if(sc.sdflags & SiS_SD_SUPPORTLCDA) {
	    gui_cf_append_text("\t# You can choose between \"VGA\" and \"LCD\"\n", CU_TT_BLUE);
	 }
	 if(hints == CF_HINTS_MFB) {
	    gui_cf_append_text("\t# If you use the \"auto\" parameter for the option\n", CU_TT_BLUE);
	    gui_cf_append_text("\t# \"MergedFB\" (see below), you can leave this out; \n", CU_TT_BLUE);
	    gui_cf_append_text("\t# MergedFB mode will then only be enabled, if a CRT1\n", CU_TT_BLUE);
	    gui_cf_append_text("\t# device is detected during X server start\n", CU_TT_BLUE);
	 }
	 gui_cf_insert_option("ForceCRT1Type", "VGA", CU_TT_BLUE_BOLD);
      } else {
         gui_cf_insert_option("ForceCRT1Type", "NONE", CU_TT_RED_BOLD);
      }
   } else if((sc.currentvbflags & CRT1_LCDA) && (sc.sdflags & SiS_SD_SUPPORTLCDA)) {
      gui_cf_insert_option("ForceCRT1Type", "LCD", CU_TT_RED_BOLD);
   } else {
      gui_cf_insert_option("ForceCRT1Type", "VGA", CU_TT_RED_BOLD);
   }
}

static void
conf_append_crt2_type(int hints)
{
   char forcecrt2type[] = "ForceCRT2Type";
   char *mode;

   if(!vbhavevb()) return;

   gui_cf_append_text("\n\t# [sisctrl] Set CRT2 device type (Note: overrides auto-detection)\n", CU_TT_RED);

   switch(sc.currentvbflags & CRT2_ENABLE) {

   case CRT2_LCD:
      gui_cf_insert_option(forcecrt2type, "LCD", CU_TT_RED_BOLD);
      break;

   case CRT2_VGA:
      gui_cf_insert_option(forcecrt2type, "VGA", CU_TT_RED_BOLD);
      break;

   case CRT2_TV:
      if(vbsupportsettvtype()) {
	 if((sc.currentvbflags & (TV_SVIDEO | TV_AVIDEO)) == (TV_SVIDEO | TV_AVIDEO))
	    gui_cf_insert_option(forcecrt2type, "SVIDEO+COMPOSITE", CU_TT_RED_BOLD);
	 else if(sc.currentvbflags & TV_SVIDEO)
	    gui_cf_insert_option(forcecrt2type, "SVIDEO", CU_TT_RED_BOLD);
	 else if(sc.currentvbflags & TV_AVIDEO)
	    gui_cf_insert_option(forcecrt2type, "COMPOSITE", CU_TT_RED_BOLD);
	 else if(sc.currentvbflags & TV_SCART)
	    gui_cf_insert_option(forcecrt2type, "SCART", CU_TT_RED_BOLD);
	 else if(sc.currentvbflags & TV_HIVISION)
	    gui_cf_insert_option(forcecrt2type, "HIVISION", CU_TT_RED_BOLD);
	 else if(sc.currentvbflags & TV_YPBPR) {
	    if(sc.currentvbflags & TV_YPBPR525I)
	       gui_cf_insert_option(forcecrt2type, "YPBPR480I", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_YPBPR525P)
	       gui_cf_insert_option(forcecrt2type, "YPBPR480P", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_YPBPR625I)
	       gui_cf_insert_option(forcecrt2type, "YPBPR576I", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_YPBPR625P)
	       gui_cf_insert_option(forcecrt2type, "YPBPR576P", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_YPBPR750P)
	       gui_cf_insert_option(forcecrt2type, "YPBPR720P", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_YPBPR1080I)
	       gui_cf_insert_option(forcecrt2type, "YPBPR1080I", CU_TT_RED_BOLD);
	    else
	       gui_cf_insert_option(forcecrt2type, "TV", CU_TT_RED_BOLD);
	 } else
	    gui_cf_insert_option(forcecrt2type, "TV", CU_TT_RED_BOLD);
      } else {
	 gui_cf_insert_option(forcecrt2type, "TV", CU_TT_RED_BOLD);
      }
      break;

   default:
      if((hints == CF_HINTS_MFB) || (hints == CF_HINTS_DHM)) {
         if(hints == CF_HINTS_MFB) mode = "MergedFB";
	 else mode = "Dual Head";
	 conf_ap_three("\t# CRT2 can't be disabled in ", mode, " mode.\n", CU_TT_BLUE);
	 gui_cf_append_text("\t# You must choose a device, for example:\n", CU_TT_BLUE);
	 gui_cf_insert_option(forcecrt2type, "SVIDEO", CU_TT_BLUE_BOLD);
      } else {
         gui_cf_insert_option(forcecrt2type, "NONE", CU_TT_RED_BOLD);
      }
   }
}

static void
conf_append_lcd_options(void)
{
   Bool alldef = TRUE;

   if(!vbhavevb()) return;

   if(sc.currentvbflags & (CRT2_LCD | CRT1_LCDA)) {

      if(sc.sdflags & SiS_SD_SUPPORTSCALE) {

	 gui_cf_append_text("\n\t# [sisctrl] LCD related options\n", CU_TT_RED);
	 if(!(sc.lcdscaler & 0x01)) {
	    gui_cf_insert_option("ScaleLCD", ((sc.lcdscaler & 0x02) ? "no" : "yes"), CU_TT_RED_BOLD);
	    alldef = FALSE;
	 }

	 if(sc.sdflags & SiS_SD_SUPPORTCENTER) {
	    if((sc.lcdscaler & 0x03) && (!(sc.lcdscaler & 0x04))) {
	       gui_cf_insert_option("CenterLCD", ((sc.lcdscaler & 0x08) ? "yes" : "no"), CU_TT_RED_BOLD);
	       alldef = FALSE;
	    }
	 }

	 if(alldef) {
	    gui_cf_append_text("\t# Currently all set to defaults\n", CU_TT_GREY);
	 }
      }

   }
}

static void
conf_append_tv_options(void)
{
   char chtvoverscan[]     = "CHTVOverscan";
   char chtvsoverscan[]    = "CHTVSuperOverscan";
   char sistvantiflicker[] = "SISTVAntiFlicker";
   char buffer[64];

   if(!vbhavevb()) return;

   if(sc.currentvbflags & CRT2_TV) {
      gui_cf_append_text("\n\t# [sisctrl] TV related options\n", CU_TT_RED);

      if(!(sc.currentvbflags & (TV_HIVISION | TV_YPBPR))) {

         if(sc.currentvbflags & TV_NTSC) {
	    if(sc.currentvbflags & TV_NTSCJ)
	       gui_cf_insert_option("TVStandard", "NTSC-J", CU_TT_RED_BOLD);
	    else
	       gui_cf_insert_option("TVStandard", "NTSC", CU_TT_RED_BOLD);
	 } else {
	    if(sc.currentvbflags & TV_PALM)
	       gui_cf_insert_option("TVStandard", "PAL-M", CU_TT_RED_BOLD);
	    else if(sc.currentvbflags & TV_PALN)
	       gui_cf_insert_option("TVStandard", "PAL-N", CU_TT_RED_BOLD);
	    else
	       gui_cf_insert_option("TVStandard", "PAL", CU_TT_RED_BOLD);
	 }

      } else if(sc.currentvbflags & TV_YPBPR) {

	 if(sc.sdflags & SiS_SD_SUPPORTYPBPRAR) {
	    if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR43)
	       gui_cf_insert_option("YPbPrAspectRatio", "4:3", CU_TT_RED_BOLD);
	    else if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR43LB)
	       gui_cf_insert_option("YPbPrAspectRatio", "4:3LB", CU_TT_RED_BOLD);
	    else if((sc.currentvbflags & TV_YPBPRAR) == TV_YPBPR169)
	       gui_cf_insert_option("YPbPrAspectRatio", "16:9", CU_TT_RED_BOLD);
	 }
      }

      if(vbsupportsoverscan()) {
	 switch(sc.choverscanstatus) {
	    case 1:
	       gui_cf_insert_option(chtvoverscan, "off", CU_TT_RED_BOLD);
	       gui_cf_insert_option(chtvsoverscan, "off", CU_TT_RED_BOLD);
	       break;
	    case 2:
	       gui_cf_insert_option(chtvoverscan, "on", CU_TT_RED_BOLD);
	       gui_cf_insert_option(chtvsoverscan, "off", CU_TT_RED_BOLD);
	       break;
	    case 3:
	       gui_cf_insert_option(chtvoverscan, "on", CU_TT_RED_BOLD);
	       gui_cf_insert_option(chtvsoverscan, "on", CU_TT_RED_BOLD);
	       break;
	 }
      }

      if(vbischrontel()) {

	 sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVCHCONTRAST, sc.atom_tco));
	 gui_cf_insert_option("CHTVContrast", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVCHTEXTENHANCE, sc.atom_tte));
	 gui_cf_insert_option("CHTVTextEnhance", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVCHLUMAFLICKERFILTER, sc.atom_tlf));
	 gui_cf_insert_option("CHTVLumaFlickerfilter", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVCHCHROMAFLICKERFILTER, sc.atom_tcf));
	 gui_cf_insert_option("CHTVChromaFlickerfilter", buffer, CU_TT_RED_BOLD);

	 gui_cf_insert_option("CHTVCVBSColor", newgetxvval(SDC_CMD_GETTVCHCVBSCOLOR, sc.atom_ccc) ?
				"yes" : "no", CU_TT_RED_BOLD);

      }

      if(vbissisbridge()) {

	 switch(newgetxvval(SDC_CMD_GETTVANTIFLICKER, sc.atom_taf)) {
	    case 0: gui_cf_insert_option(sistvantiflicker, "off", CU_TT_RED_BOLD); break;
	    case 1: gui_cf_insert_option(sistvantiflicker, "low", CU_TT_RED_BOLD); break;
	    case 2: gui_cf_insert_option(sistvantiflicker, "med", CU_TT_RED_BOLD); break;
	    case 3: gui_cf_insert_option(sistvantiflicker, "high", CU_TT_RED_BOLD); break;
	    case 4: gui_cf_insert_option(sistvantiflicker, "adaptive", CU_TT_RED_BOLD); break;
	 }

	 if(vbsupporttvsaturation()) {
	    sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVSATURATION, sc.atom_tsa));
	    gui_cf_insert_option("SISTVSaturation", buffer, CU_TT_RED_BOLD);
	 }

	 if(vbsupporttvedgeenhance()) {
	    sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVEDGEENHANCE, sc.atom_tee));
	    gui_cf_insert_option("SISTVEdgeEnhance", buffer, CU_TT_RED_BOLD);
	 }

	 sprintf(buffer, "%d", sd_readcolorcalib(sc.atom_coc, TRUE));
	 gui_cf_insert_option("SISTVColorCalibCoarse", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", sd_readcolorcalib(sc.atom_cof, FALSE));
	 gui_cf_insert_option("SISTVColorCalibFine", buffer, CU_TT_RED_BOLD);

	 gui_cf_insert_option("SISTVCFilter", newgetxvval(SDC_CMD_GETTVCFILTER, sc.atom_cfi) ?
				"on" : "off", CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", newgetxvval(SDC_CMD_GETTVYFILTER, sc.atom_yfi));
	 gui_cf_insert_option("SISTVYFilter", buffer, CU_TT_RED_BOLD);

      }

      if(vbsupporttvsize()) {

	 sprintf(buffer, "%d", sd_gettvposscale(SDC_CMD_GETTVXSCALE, sc.atom_txs));
	 gui_cf_insert_option("SISTVXScale", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", sd_gettvposscale(SDC_CMD_GETTVYSCALE, sc.atom_tys));
	 gui_cf_insert_option("SISTVYScale", buffer, CU_TT_RED_BOLD);

      }

      if(sc.sdflags & SiS_SD_SUPPORTTVPOS) {

	 sprintf(buffer, "%d", sd_gettvposscale(SDC_CMD_GETTVXPOS, sc.atom_tvx));
	 gui_cf_insert_option("TVXPosOffset", buffer, CU_TT_RED_BOLD);

	 sprintf(buffer, "%d", sd_gettvposscale(SDC_CMD_GETTVYPOS, sc.atom_tvy));
	 gui_cf_insert_option("TVYPosOffset", buffer, CU_TT_RED_BOLD);

      }

   }
}

static void
conf_append_gamma_enable_crt1(Bool ishint)
{
    gui_cf_insert_option("CRT1Gamma", (sc.gammaflags & 0x01) ? "on" : "off", ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD);
}

static void
conf_append_gamma_enable_crt2(Bool ishint)
{
    gui_cf_insert_option("CRT2Gamma", (sc.gammaflags & 0x02) ? "on" : "off", ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD);
}

static void
conf_append_gamma_bri_header(void)
{
   if(!(sc.sdflags & SiS_SD_CANSETGAMMA)) {
      gui_cf_append_text("\t# The driver does NOT write these to the hardware. Instead, for\n", CU_TT_GREY);
      gui_cf_append_text("\t# using brightness/contrast, you need to execute sisctrl with the\n", CU_TT_GREY);
      gui_cf_append_text("\t# -setgammabrightness (or -sg) switch at X server start. It is\n", CU_TT_GREY);
      gui_cf_append_text("\t# most convenient to do this by adding this to your ~/.xsession\n", CU_TT_GREY);
      gui_cf_append_text("\t# or ~/.xinitrc file:\n", CU_TT_GREY);
      gui_cf_append_text("\t#    sisctrl -sg\n", CU_TT_GREY);
      if(vbhavevb()) {
         gui_cf_append_text("\t# If you run Xinerama, do that twice for both screens. Add this:\n", CU_TT_GREY);
         gui_cf_append_text("\t#    sisctrl -screen 1 -sg\n", CU_TT_GREY);
      }
   }
}

static void
conf_append_gamma_bri_header_crt2(void)
{
   if(!(sc.sdflags & SiS_SD_CANSETGAMMA)) {
      gui_cf_append_text("\t# For CRT2, the driver can write these to hardware\n", CU_TT_GREY);
      gui_cf_append_text("\t# directly.\n", CU_TT_GREY);
   }
}

static void
conf_append_gamma_brightness(Bool ishint)
{
   char buffer[64];
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD;

   if(sc.usenewgamma) {

      sprintf(buffer, "%.3f %.3f %.3f",
		(float)sc.newgamma_bri[0],
		(float)sc.newgamma_bri[1],
		(float)sc.newgamma_bri[2]);

      conf_fix_comma(buffer);

      gui_cf_insert_option("Brightness", buffer, colb);

      sprintf(buffer, "%.3f %.3f %.3f",
		(float)sc.newgamma_con[0],
		(float)sc.newgamma_con[1],
		(float)sc.newgamma_con[2]);

      conf_fix_comma(buffer);

      gui_cf_insert_option("Contrast", buffer, colb);

   } else {

      if(sc.sd2flags & SiS_SD2_NEWGAMMABRICON) {
         gui_cf_append_text("\t# Note: \"GammaBrightness\" is deprecated. Your driver\n", colb);
         gui_cf_append_text("\t# already supports the new brightness/contrast\n", colb);
         gui_cf_append_text("\t# options. Leave this option out. SiSCtrl will then\n", colb);
         gui_cf_append_text("\t# automatically use the new system.\n", colb);
      }

      sprintf(buffer, "%.3f %.3f %.3f",
		(float)sc.gamma_max[0],
		(float)sc.gamma_max[1],
		(float)sc.gamma_max[2]);

      conf_fix_comma(buffer);

      gui_cf_insert_option("GammaBrightness", buffer, colb);
   }
}

static void
conf_append_crt2_gamma(void)
{
   char buffer[64];

   sprintf(buffer, "\tOption \"CRT2Gamma\" \"%.3f %.3f %.3f\"\n",
			sc.newgamma2r,
			sc.newgamma2g,
			sc.newgamma2b);
   conf_fix_comma(buffer);
   gui_cf_append_text(buffer, CU_TT_RED_BOLD);

   if(sc.sdinterface) {
      if(sc.usenewgamma) {
         sprintf(buffer, "\tOption \"CRT2Brightness\" \"%.3f %.3f %.3f\"\n",
			sc.newgamma_bri2[0],
			sc.newgamma_bri2[1],
			sc.newgamma_bri2[2]);
	 conf_fix_comma(buffer);
         gui_cf_append_text(buffer, CU_TT_RED_BOLD);
         sprintf(buffer, "\tOption \"CRT2Contrast\" \"%.3f %.3f %.3f\"\n",
			sc.newgamma_con2[0],
			sc.newgamma_con2[1],
			sc.newgamma_con2[2]);
	 conf_fix_comma(buffer);
         gui_cf_append_text(buffer, CU_TT_RED_BOLD);
      } else {
         if(sc.sd2flags & SiS_SD2_NEWGAMMABRICON) {
            gui_cf_append_text("\t# Note: \"CRT2GammaBrightness\" is deprecated. Your\n", CU_TT_RED_BOLD);
            gui_cf_append_text("\t# driver already supports the new brightness/contrast\n", CU_TT_RED_BOLD);
            gui_cf_append_text("\t# options. Leave this option out. SiSCtrl will then\n", CU_TT_RED_BOLD);
            gui_cf_append_text("\t# automatically use the new system.\n", CU_TT_RED_BOLD);
         }
         sprintf(buffer, "\tOption \"CRT2GammaBrightness\" \"%.3f %.3f %.3f\"\n",
			sc.gamma_bri2[0],
			sc.gamma_bri2[1],
			sc.gamma_bri2[2]);
	 conf_fix_comma(buffer);
         gui_cf_append_text(buffer, CU_TT_RED_BOLD);
      }
   } else {
      sprintf(buffer, "\tOption \"GammaBrightnessCRT2\" \"%.3f %.3f %.3f\"\n",
			sc.gamma_bri2[0],
			sc.gamma_bri2[1],
			sc.gamma_bri2[2]);
      conf_fix_comma(buffer);
      gui_cf_append_text(buffer, CU_TT_RED_BOLD);
   }

}

static void
conf_append_saturation(int crtnum, Bool ishint)
{
   char crtbuf[32], buffer[8];
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD;

   sprintf(crtbuf, "CRT%dSaturation", crtnum);
   sprintf(buffer, "%d", (crtnum == 1) ? sisctrl_gamma_get_sat_crt1() : sisctrl_gamma_get_sat_crt2(), 0);

   gui_cf_insert_option(crtbuf, buffer, colb);
}

static void
conf_append_xv_switchcrt(void)
{
   if(!vbhavevb()) return;

   if(((!(sc.sdflags & SiS_SD_SUPPORT2OVL)) || (sc.sd2flags & SiS_SD2_SUPPORT760OO))) {
      gui_cf_append_text("\n\t# Xv (video overlay) head selection\n", CU_TT_RED);
      gui_cf_insert_option("XvOnCRT2", newgetxvval(SDC_CMD_GETXVSWITCHCRT, sc.atom_swc) ?
				"yes" : "no", CU_TT_RED_BOLD);
   }
}

static void
conf_append_xv_options(Bool ishint)
{
   char buffer[64];
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD;

   sprintf(buffer, "%d", newgetrealxvval(SDC_CMD_GETXVCONTRAST, sc.atom_con));
   gui_cf_insert_option("XvDefaultContrast", buffer, colb);

   sprintf(buffer, "%d", newgetrealxvval(SDC_CMD_GETXVBRIGHTNESS, sc.atom_bri));
   gui_cf_insert_option("XvDefaultBrightness", buffer, colb);

   if(supportvihuesat()) {

      sprintf(buffer, "%d", newgetrealxvval(SDC_CMD_GETXVHUE, sc.atom_hue));
      gui_cf_insert_option("XvDefaultHue", buffer, colb);

      sprintf(buffer, "%d", newgetrealxvval(SDC_CMD_GETXVSATURATION, sc.atom_sat));
      gui_cf_insert_option("XvDefaultSaturation", buffer, colb);

   }

   if(sc.havedglgfx) {
      gui_cf_insert_option("XvDefaultDisableGfxLR", sc.backup_dgl ? "yes" : "no", colb);
   }
}

static void
conf_append_xv_gamma(Bool ishint)
{
   char buffer[64];
   int colb = ishint ? CU_TT_BLUE_BOLD : CU_TT_RED_BOLD;

   if(sc.sdflags & SiS_SD_SUPPORTXVGAMMA1) {

      if(!(sc.gammaflags & 0x04)) {

	 gui_cf_insert_option("XvGamma", "off", colb);

      } else {

	 sprintf(buffer, "%.3f %.3f %.3f",
	     (float)sc.xvgammared, (float)sc.xvgammagreen, (float)sc.xvgammablue);

	 conf_fix_comma(buffer);

         gui_cf_insert_option("XvGamma", buffer, colb);

      }
   }
}


/*******************************************
 *                  HINTS                  *
 *******************************************/

/* MergedFB */

static void
conf_append_mergedfb_hints(int hints)
{
   if(!vbhavevb()) return;
   if(hints != CF_HINTS_MFB) return;

   gui_cf_append_text("\n\t# For MergedFB mode, see the following options:\n\n", CU_TT_BLUE);

   gui_cf_append_text("\t# To enable MergedFB mode automatically if both CRT1 and CRT2\n", CU_TT_BLUE);
   gui_cf_append_text("\t# are detected, uncomment this:\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"MergedFB\" \"auto\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Alternatively, to enable MergedFB mode regardless of device\n", CU_TT_BLUE);
   gui_cf_append_text("\t# detection, uncomment this:\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"MergedFB\" \"on\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# \"CRT2Position\" selects the relative position of CRT2 to CRT1\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Valid choices are \"LeftOf\", \"RightOf\", \"Above\", \"Below\" and\n", CU_TT_BLUE);
   gui_cf_append_text("\t# \"Clone\" (Default: RightOf)\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"CRT2Position\" \"LeftOf\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Optional: Supported frequency ranges for your CRT2 device.\n", CU_TT_BLUE);
   if(vbsupportsvga2()) {
      gui_cf_append_text("\t# It is NOT recommended to set these. You might only need them\n", CU_TT_BLUE);
      gui_cf_append_text("\t# it you connect an old analog monitor to VGA2 which lacks DDC\n", CU_TT_BLUE);
      gui_cf_append_text("\t# support. HSync is in kHz, VRefresh in Hz. For example:\n", CU_TT_BLUE);
   } else {
      gui_cf_append_text("\t# You don't need these as your hardware does not support VGA\n", CU_TT_BLUE);
      gui_cf_append_text("\t# output via CRT2. For LCD and TV they are not needed anyway.\n", CU_TT_BLUE);
   }
   gui_cf_append_text("\t# Option \"CRT2HSync\" \"31-64\"\n", CU_TT_BLUE_BOLD);
   gui_cf_append_text("\t# Option \"CRT2VRefresh\" \"50-75\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Optional: The \"MetaModes\" select the (real) display modes to\n", CU_TT_BLUE);
   gui_cf_append_text("\t# be used for CRT1 (left of \"-\") and CRT2 (right of \"-\"). If\n", CU_TT_BLUE);
   gui_cf_append_text("\t# this option is not set, the driver will calculate MetaModes from\n", CU_TT_BLUE);
   gui_cf_append_text("\t# the display devices' largest supported modes. For instance: You\n", CU_TT_BLUE);
   gui_cf_append_text("\t# have a laptop with a 1280x800 LCD panel and a 1280x1024 external\n", CU_TT_BLUE);
   gui_cf_append_text("\t# monitor; then the following might be smart:\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"MetaModes\" \"1280x1024-1280x800 1280x800\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Optional: Basically, the virtual screen size will be calculated\n", CU_TT_BLUE);
   gui_cf_append_text("\t# from the largest MetaMode and will be rectangular, regardless\n", CU_TT_BLUE);
   gui_cf_append_text("\t# whether the two real display modes this largest MetaMode consists\n", CU_TT_BLUE);
   gui_cf_append_text("\t# of are of the same resolution or not. Usually, if a Metamode\n", CU_TT_BLUE);
   gui_cf_append_text("\t# consists of real display modes which are smaller than the virtual\n", CU_TT_BLUE);
   gui_cf_append_text("\t# screen, the driver will pan if the mouse reaches the visible\n", CU_TT_BLUE);
   gui_cf_append_text("\t# part's edges. If you don't want to pan, but instead want a\n", CU_TT_BLUE);
   gui_cf_append_text("\t# non-rectangular virtual screen with no panning, set this option:\n", CU_TT_BLUE);
   gui_cf_append_text("\t# (Following up on our example: If the LCD supports 1280x800 and\n", CU_TT_BLUE);
   gui_cf_append_text("\t# the external monitor 1280x1024, the virtual screen would be\n", CU_TT_BLUE);
   gui_cf_append_text("\t# 1024 in height. On the LCD, the driver would pan. If you set this\n", CU_TT_BLUE);
   gui_cf_append_text("\t# option, the virtual screen will be 800 in height on one side and\n", CU_TT_BLUE);
   gui_cf_append_text("\t# 1024 in height on the other, and there will be no panning. Please\n", CU_TT_BLUE);
   gui_cf_append_text("\t# read the documentation on this option; it requires a somewhat sane\n", CU_TT_BLUE);
   gui_cf_append_text("\t# set of MetaModes)\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"MergedNonRectangular\" \"yes\"\n\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Optional: To force the screen's DPI for MergedFB mode to a\n", CU_TT_BLUE);
   gui_cf_append_text("\t# fixed value, use this: (first parameter X, second parameter Y)\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"MergedDPI\" \"100 100\"\n", CU_TT_BLUE_BOLD);
}

static void
conf_append_mergedfb_options(int hints)
{
   gui_cf_append_text("\n\t# ...\n", CU_TT_GREY);
   gui_cf_append_text("\t# [your current options for MergedFB mode]\n", CU_TT_GREY);
   if(hints == CF_HINTS_GE) {
      if(vbsupportsvga2()) {
         gui_cf_append_text("\t# \"CRT2HSync\" and \"CRT2VRefresh\" are only required if\n", CU_TT_BLUE);
         gui_cf_append_text("\t# your analog display device (CRT2 VGA) does not support\n", CU_TT_BLUE);
         gui_cf_append_text("\t# DDC; they are not needed if CRT2 is LCD/DVI-D or TV.\n", CU_TT_BLUE);
      } else {
         gui_cf_append_text("\t# \"CRT2HSync\" and \"CRT2VRefresh\" are not needed.\n", CU_TT_BLUE);
      }
   }
   gui_cf_append_text("\t# ...\n", CU_TT_GREY);
}

/* Dual Head */

static void
conf_append_hint_busid_screen(hints)
{
   if(hints != CF_HINTS_DHM) return;

   conf_append_busid(TRUE);

   conf_append_screen(2, TRUE, hints);
}

static void
conf_append_second_monitor_section(hints)
{
   char buffer[64];
   if(hints != CF_HINTS_DHM) return;

   gui_cf_append_text("\n# For Dual Head mode, a second \"Monitor\" section is required.\n", CU_TT_BLUE);
   gui_cf_append_text("# The above \"Monitor\" section is for CRT2, while the following\n", CU_TT_BLUE);
   gui_cf_append_text("# one is for CRT1. For example:\n\n", CU_TT_BLUE);

   gui_cf_append_text("Section \"Monitor\"\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Monitor\"\n", CU_TT_BLUE);
   gui_cf_append_text("\t# tag in your \"Screen\" section for CRT1\n", CU_TT_BLUE);
   gui_cf_append_text("\tIdentifier \"IDENTIFIER\"\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\n\t# VertRefresh ??-?? (optional, used for CRT1 only)\n", CU_TT_BLUE);
   gui_cf_append_text("\t# HorizSync ??-?? (optional, used for CRT1 only)\n", CU_TT_BLUE);
   gui_cf_append_text("\t# Option \"DPMS\" (optional, used for CRT1 only)\n", CU_TT_BLUE);

   gui_cf_append_text("\n\t# Gamma correction for CRT1\n", CU_TT_BLUE);
   sprintf(buffer, "\tGamma %.3f %.3f %.3f\n",
			sc.currentgamma.red,
			sc.currentgamma.green,
			sc.currentgamma.blue);
   conf_fix_comma(buffer);
   gui_cf_append_text(buffer, CU_TT_RED_BOLD);

   gui_cf_append_text("EndSection\n\n", CU_TT_BLUE_BOLD);
}

static void
conf_append_second_device_section(hints)
{
   if(hints != CF_HINTS_DHM) return;

   gui_cf_append_text("\n# For Dual Head mode, a second \"Device\" section is required.\n", CU_TT_BLUE);
   gui_cf_append_text("# The above \"Device\" section is for CRT2, while the following\n", CU_TT_BLUE);
   gui_cf_append_text("# one is for CRT1. Note that not all options can be set in this\n", CU_TT_BLUE);
   gui_cf_append_text("# second \"Device\" section. Options that determine settings\n", CU_TT_BLUE);
   gui_cf_append_text("# which both heads need to know about are only allowed in the\n", CU_TT_BLUE);
   gui_cf_append_text("# \"Device\" section for CRT2.\n", CU_TT_BLUE);
   gui_cf_append_text("# Options that are allowed here are all \"XvDefault...\", all\n", CU_TT_BLUE);
   gui_cf_append_text("# gamma correction and color related options and a few more.\n", CU_TT_BLUE);
   gui_cf_append_text("# For example:\n\n", CU_TT_BLUE);

   gui_cf_append_text("Section \"Device\"\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Device\"\n", CU_TT_BLUE);
   gui_cf_append_text("\t# tag in your \"Screen\" section for CRT1\n", CU_TT_BLUE);
   gui_cf_append_text("\tIdentifier \"IDENTIFIER\"\n", CU_TT_BLUE_BOLD);

   conf_append_driver(TRUE);

   conf_append_busid(TRUE);

   conf_append_screen(1, TRUE, hints);

   conf_append_enablesisctrl_option(TRUE);

   gui_cf_append_text("\n\t# Enable/disable gamma correction for CRT1\n", CU_TT_BLUE);
   conf_append_gamma_enable_crt1(FALSE);

   if(sc.usenewgamma) {
      gui_cf_append_text("\n\t# Brightness/contrast for CRT1\n", CU_TT_BLUE);
   } else {
      gui_cf_append_text("\n\t# Gamma brightness for CRT1\n", CU_TT_BLUE);
   }
   conf_append_gamma_bri_header();
   conf_append_gamma_brightness(FALSE);

   if(sc.sdinterface && (sc.sd3flags & SiS_SD3_CRT1SATGAIN)) {
      gui_cf_append_text("\n\t# Color saturation for CRT1\n", CU_TT_BLUE);
      conf_append_saturation(1, FALSE);
   }

   gui_cf_append_text("\n\t# Xv (video overlay) related options for CRT1\n", CU_TT_BLUE);
   conf_append_xv_options(FALSE);
   conf_append_xv_gamma(FALSE);

   gui_cf_append_text("EndSection\n\n", CU_TT_BLUE_BOLD);
}

static void
conf_append_hint_dhm_footer(hints)
{
   if(hints != CF_HINTS_DHM) return;

   gui_cf_append_text("\nAdditionally, you need a second \"Screen\" section for CRT1 and ", CU_TT_BLUE);
   gui_cf_append_text("your \"ServerLayout\" section needs to be extended to know about the "
   		      "second screen. Add\n\n", CU_TT_BLUE);

   gui_cf_append_text("\tScreen \"SCREEN2\" rightof \"SCREEN1\"\n", CU_TT_BLUE_BOLD);

   gui_cf_append_text("\nwhere \"SCREENx\" are the Identifiers of the two screen sections. Instead of \"rightof\","
   		"\"leftof\", \"above\" and \"below\" can be selected.\n\n", CU_TT_BLUE);

   gui_cf_append_text("To enable Xinerama, add to your \"ServerLayout\" section\n\n", CU_TT_BLUE);

   gui_cf_append_text("\tOption \"Xinerama\" \"on\"\n\n", CU_TT_BLUE_BOLD);
}

/****************************************************************
 *                          GENERATORS                          *
 ****************************************************************/

/*******************************************
 *       Generate CURRENT for DHM          *
 *******************************************/

static void
conf_append_dhm_monitor_section(int crtnum, Bool fillnow, int hints)
{
   char buffer[128];
   char crtbuffer[8];
   char *name = NULL;
   char *opt = "optional";

   sprintf(crtbuffer, "CRT%d", crtnum);

   conf_ap_three("\n# \"Monitor\" section for ", crtbuffer, "\n", CU_TT_GREY);

   conf_append_section_monitor();

   if(crtnum == 1) {
      if(sc.havemonname1) name = sc.monname1;
   } else {
      if(sc.havemonname2) name = sc.monname2;
   }

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Monitor\"\n", CU_TT_GREY);
      conf_ap_three("\t# tag in your \"Screen\" section for", crtbuffer, "\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n\n", CU_TT_BLACK_BOLD);

   conf_append_hsvr_hint(hints);
   if(sc.sd2flags & SiS_SD2_NODDCSUPPORT) opt = "mandatory";
   conf_ap_three("\t# VertRefresh ??-?? (", opt, ")\n", CU_TT_GREY);
   conf_ap_three("\t# HorizSync ??-?? (", opt, ")\n", CU_TT_GREY);
   gui_cf_append_text("\t# Option \"DPMS\" (optional)\n", CU_TT_GREY);

   conf_ap_three("\n\t# Gamma correction for ", crtbuffer, "\n", fillnow ? CU_TT_RED : CU_TT_GREY);

   if(fillnow) {

      sprintf(buffer, "\tGamma %.3f %.3f %.3f\n",
			sc.currentgamma.red,
			sc.currentgamma.green,
			sc.currentgamma.blue);
      conf_fix_comma(buffer);

      gui_cf_append_text(buffer, CU_TT_RED_BOLD);

   } else {

      gui_cf_append_text("\t# Gamma ...\n", CU_TT_GREY);

   }

   conf_append_endsection();
}

static void
conf_append_dhm_device_section(int crtnum, Bool fillnow, int hints)
{
   char buffer[128];
   char crtbuffer[8];
   char *name = NULL;

   sprintf(crtbuffer, "CRT%d", crtnum);

   conf_ap_three("\n# \"Device\" section for ", crtbuffer, "\n", CU_TT_GREY);

   conf_append_section_device();

   if(crtnum == 1) {
      if(sc.havedevname1) name = sc.devname1;
   } else {
      if(sc.havedevname2) name = sc.devname2;
   }

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Device\"\n", CU_TT_GREY);
      conf_ap_three("\t# tag in your \"Screen\" section for", crtbuffer, "\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n", CU_TT_BLACK_BOLD);

   conf_append_driver(FALSE);

   conf_append_busid(FALSE);

   conf_append_screen(crtnum, FALSE, hints);

   conf_append_enablesisctrl_option(FALSE);

   if(crtnum == 2) conf_append_usesse(hints);

   /* Device types (only in CRT2 Device section!) */

   if(crtnum == 2) {

      /* CRT1 can't be off in DHM, so it's either VGA or LCD */
      conf_append_crt1_type(hints);

      conf_append_crt2_type(hints);

      conf_append_lcd_options();

      conf_append_tv_options();

   }

   /* Gamma */

   if((crtnum == 1) || (vbsupportgamma2())) {

      if(fillnow) {

         conf_ap_three("\n\t# [sisctrl] Enable/disable Gamma correction for ", crtbuffer, "\n",
			CU_TT_RED);

	 /* There is no "CRT2Gamma value value value" in DHM! (no "separate gamma corr") */
	 if(crtnum == 1)
	    conf_append_gamma_enable_crt1(FALSE);
	 else
	    conf_append_gamma_enable_crt2(FALSE);

         if(sc.usenewgamma) {
            conf_ap_three("\n\t# [sisctrl] Brightness/contrast for ", crtbuffer, "\n", CU_TT_RED);
         } else {
            conf_ap_three("\n\t# [sisctrl] Gamma brightness for ", crtbuffer, "\n", CU_TT_RED);
         }

	 conf_append_gamma_bri_header();

	 conf_append_gamma_brightness(FALSE);

      } else {

         conf_ap_three("\n\t# [sisctrl] Gamma correction related options for ", crtbuffer, "\n", CU_TT_GREY);
         gui_cf_append_text("\t# ...\n", CU_TT_GREY);

         if(sc.usenewgamma) {
            conf_ap_three("\n\t# [sisctrl] Brightness/contrast related options for ", crtbuffer, "\n", CU_TT_GREY);
            gui_cf_append_text("\t# ...\n", CU_TT_GREY);
         }

      }

   }

   /* Saturation */
   if(sc.sdinterface) {
      if( ((crtnum == 1) && (sc.sd3flags & SiS_SD3_CRT1SATGAIN)) ||
          ((crtnum == 2) && (sc.sd3flags & SiS_SD3_CRT2SATGAIN)) ) {
         if(fillnow) {
            conf_ap_three("\n\t# [sisctrl] Saturation for ", crtbuffer, "\n", CU_TT_RED);
            conf_append_saturation(crtnum, FALSE);
         } else {
            conf_ap_three("\n\t# [sisctrl] Saturation for ", crtbuffer, "\n", CU_TT_GREY);
            gui_cf_append_text("\t# ...\n", CU_TT_GREY);
         }
      }
   }

   /* Xv */

   if(!(sc.sd2flags & SiS_SD2_NOOVERLAY)) {

      if(crtnum == 2) {
         conf_append_xv_switchcrt();
      }

      conf_ap_three("\n\t# [sisctrl] Xv (video overlay) related options for ", crtbuffer, "\n",
			fillnow ? CU_TT_RED : CU_TT_GREY);

      if(fillnow) {

         conf_append_xv_options(FALSE);
         if(crtnum == 1) {
            conf_append_xv_gamma(FALSE);
         }


      } else {

         gui_cf_append_text("\t# ...\n", CU_TT_GREY);

      }

   }

   conf_append_endsection();
}

static void
conf_generate_current_dual_head(int hints)
{
   Bool fillnow;

   conf_append_header_basic(hints);

   conf_append_legend(hints);

   conf_append_dhm_header((sc.sdflags & SiS_SD_ISDHSECONDHEAD) ? 1 : 2, hints);

   /* This no longer works for the old Xv interface! */
#if 0
   if(sc.sdflags & SiS_SD_ISDHXINERAMA) {

      /* FIXME */
      fillnow = (sc.myscreen == DefaultScreen(sc.dpy)) ? TRUE : FALSE;

   } else {
#endif

      fillnow = (sc.sdflags & SiS_SD_ISDHSECONDHEAD) ? FALSE : TRUE;
#if 0
   }
#endif

   conf_append_dhm_monitor_section(2, fillnow, hints);

   conf_append_dhm_device_section(2, fillnow, hints);

   conf_append_dhm_monitor_section(1, !fillnow, hints);

   conf_append_dhm_device_section(1, !fillnow, hints);
}

/******************************************
 *     Generate CURRENT for MergedFB      *
 ******************************************/

static void
conf_append_mfb_monitor_section(int hints)
{
   char buffer[128];
   char *name = NULL;
   char *opt = "optional";
   Bool commongamma;

   commongamma = ((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) ? FALSE : TRUE;

   gui_cf_append_text("\n# The \"Monitor\" section\n", CU_TT_GREY);

   conf_append_section_monitor();

   if(sc.havemonname1) name = sc.monname1;

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Monitor\"\n", CU_TT_GREY);
      gui_cf_append_text("\t# tag in your \"Screen\" section\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n\n", CU_TT_BLACK_BOLD);

   conf_append_hsvr_hint(hints);
   if(sc.sd2flags & SiS_SD2_NODDCSUPPORT) opt = "mandatory";
   conf_ap_three("\t# VertRefresh ??-?? (", opt, ", used for CRT1 only)\n", CU_TT_GREY);
   conf_ap_three("\t# HorizSync ??-?? (", opt, ", used for CRT1 only)\n", CU_TT_GREY);
   gui_cf_append_text("\t# Option \"DPMS\" (optional)\n", CU_TT_GREY);

   if(vbsupportgamma2() && commongamma) {
      gui_cf_append_text("\n\t# Gamma correction for CRT1 and CRT2\n", CU_TT_RED);
   } else {
      gui_cf_append_text("\n\t# Gamma correction for CRT1\n", CU_TT_RED);
      if(hints == CF_HINTS_GE) {
         gui_cf_append_text("\t# CRT1 only because separate values for CRT2 are given\n", CU_TT_BLUE);
         gui_cf_append_text("\t# in the Device section\n", CU_TT_BLUE);
      }
   }

   sprintf(buffer, "\tGamma %.3f %.3f %.3f\n",
			sc.currentgamma.red,
			sc.currentgamma.green,
			sc.currentgamma.blue);
   conf_fix_comma(buffer);

   gui_cf_append_text(buffer, CU_TT_RED_BOLD);

   conf_append_endsection();
}

static void
conf_append_mfb_device_section(int hints)
{
   char buffer[128];
   char *name = NULL;
   Bool commongamma;

   commongamma = ((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) ? FALSE : TRUE;

   gui_cf_append_text("\n# The \"Device\" section\n", CU_TT_GREY);

   conf_append_section_device();

   if(sc.havedevname1) name = sc.devname1;

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Device\"\n", CU_TT_GREY);
      gui_cf_append_text("\t# tag in your \"Screen\" section\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n", CU_TT_BLACK_BOLD);

   conf_append_driver(FALSE);

   conf_append_enablesisctrl_option(FALSE);

   conf_append_usesse(hints);

   /* Device types */

   /* CRT1 can't be off in MergedFB mode, so it's either VGA or LCD */
   conf_append_crt1_type(hints);

   conf_append_crt2_type(hints);

   conf_append_lcd_options();

   conf_append_tv_options();

   /* MergedFB */

   conf_append_mergedfb_options(hints);

   /* Gamma */

   gui_cf_append_text("\n\t# [sisctrl] Enable/disable gamma correction for CRT1\n", CU_TT_RED);
   conf_append_gamma_enable_crt1(FALSE);

   if(vbsupportgamma2() && commongamma) {

      gui_cf_append_text("\n\t# [sisctrl] Enable/disable gamma correction for CRT2\n", CU_TT_RED);
      conf_append_gamma_enable_crt2(FALSE);

   }

   if(vbsupportgamma2() && commongamma) {

      if(sc.usenewgamma) {
         gui_cf_append_text("\n\t# [sisctrl] Brightness/contrast for CRT1 and CRT2\n", CU_TT_RED);
      } else {
         gui_cf_append_text("\n\t# [sisctrl] Gamma brightness for CRT1 and CRT2\n", CU_TT_RED);
      }

   } else {

      if(sc.usenewgamma) {
         gui_cf_append_text("\n\t# [sisctrl] Brightness/contrast for CRT1\n", CU_TT_RED);
      } else {
         gui_cf_append_text("\n\t# [sisctrl] Gamma brightness for CRT1\n", CU_TT_RED);
      }

   }

   conf_append_gamma_bri_header();

   conf_append_gamma_brightness(FALSE);

   if(vbsupportgamma2() && !commongamma) {

      gui_cf_append_text("\n\t# [sisctrl] Enable/disable/set gamma correction for CRT2\n", CU_TT_RED);
      conf_append_gamma_bri_header_crt2();
      conf_append_crt2_gamma();

   }

   /* Saturation */
   if(sc.sdinterface) {
      if(sc.sd3flags & SiS_SD3_CRT1SATGAIN) {
         gui_cf_append_text("\n\t# [sisctrl] Saturation for CRT1\n", CU_TT_RED);
         conf_append_saturation(1, FALSE);
      }
      if(sc.sd3flags & SiS_SD3_CRT2SATGAIN) {
         gui_cf_append_text("\n\t# [sisctrl] Saturation for CRT2\n", CU_TT_RED);
         conf_append_saturation(2, FALSE);
      }
   }

   /* Xv */

   if(!(sc.sd2flags & SiS_SD2_NOOVERLAY)) {

      gui_cf_append_text("\n\t# [sisctrl] Xv (video overlay) related options\n", CU_TT_RED);

      conf_append_xv_options(FALSE);

      conf_append_xv_gamma(FALSE);

   }

   conf_append_endsection();
}

static void
conf_generate_current_mergedfb(int hints)
{
   conf_append_header_basic(hints);

   conf_append_legend(hints);

   conf_append_mfb_monitor_section(hints);

   conf_append_mfb_device_section(hints);
}

/******************************************
 *   Generate CURRENT for Single Head     *
 ******************************************/

static void
conf_append_simple_monitor_section(int hints)
{
   char buffer[128];
   char forcrtbuf[32];
   char gforcrtbuf[32];
   char dforcrtbuf[32];
   char *name = NULL;
   Bool commongamma;
   Bool printgammahint = FALSE;
   int col;
   char *opt = "optional";

   forcrtbuf[0] = gforcrtbuf[0] = dforcrtbuf[0] = 0;

   commongamma = ((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) ? FALSE : TRUE;

   if(hints == CF_HINTS_DHM) {

      gui_cf_append_text("\n# In Dual Head mode, this \"Monitor\" section is only used\n", CU_TT_BLUE);
      gui_cf_append_text("# for CRT2. A second one, for CRT1, is added below.\n", CU_TT_BLUE);

      sprintf(forcrtbuf, "; used for CRT2 only");
      sprintf(dforcrtbuf, "; used for CRT2 only");
      sprintf(gforcrtbuf, "for CRT2");

      col = CU_TT_BLUE;

   } else if(vbhavevb()) {

      /* HSync, VRefresh will be used for ... */
      if(sc.currentvbflags & CRT2_VGA) {
         sprintf(forcrtbuf, "; used for CRT1 and CRT2");
      } else {
         sprintf(forcrtbuf, "; used for CRT1 only");
      }

      /* DPMS will be used for ... don't be too picky... */
      /* if(sc.currentvbflags & CRT2_ENABLE) { */
         sprintf(dforcrtbuf, "; used for CRT1 and CRT2");
      /* } */

      /* Gamma will be used for ... */
      if(vbsupportgamma2() && /*(sc.currentvbflags & CRT2_ENABLE) &&*/ commongamma) {
         sprintf(gforcrtbuf, "for CRT1 and CRT2");
      } else {
         sprintf(gforcrtbuf, "for CRT1");
         if(vbsupportgamma2()) printgammahint = TRUE;
      }

      col = CU_TT_RED;

   } else {

      col = CU_TT_RED;

   }

   gui_cf_append_text("\n", CU_TT_GREY);

   conf_append_section_monitor();

   if(sc.havemonname1) name = sc.monname1;

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Monitor\"\n", CU_TT_GREY);
      gui_cf_append_text("\t# tag in your \"Screen\" section\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n\n", CU_TT_BLACK_BOLD);

   conf_append_hsvr_hint(hints);
   if(sc.sd2flags & SiS_SD2_NODDCSUPPORT) opt = "mandatory";
   conf_ap_four("\t# VertRefresh ??-?? (", opt, forcrtbuf, ")\n", CU_TT_GREY);
   conf_ap_four("\t# HorizSync ??-?? (", opt, forcrtbuf, ")\n", CU_TT_GREY);
   conf_ap_three("\t# Option \"DPMS\" (optional", dforcrtbuf, ")\n", CU_TT_GREY);

   conf_ap_three("\n\t# Gamma correction ", gforcrtbuf, "\n", col);

   if(((hints == CF_HINTS_GE) || (hints == CF_HINTS_MFB)) && printgammahint) {
      gui_cf_append_text("\t# CRT1 only because separate values for CRT2 are given\n", CU_TT_BLUE);
      gui_cf_append_text("\t# in the Device section\n", CU_TT_BLUE);
   }

   if((hints == CF_HINTS_DHM) && !commongamma) {
      sprintf(buffer, "\tGamma %.3f %.3f %.3f\n",
			sc.newgamma2r,
			sc.newgamma2g,
			sc.newgamma2b);
   } else {
      sprintf(buffer, "\tGamma %.3f %.3f %.3f\n",
			sc.currentgamma.red,
			sc.currentgamma.green,
			sc.currentgamma.blue);
   }
   conf_fix_comma(buffer);
   gui_cf_append_text(buffer, CU_TT_RED_BOLD);

   conf_append_endsection();
}

static void
conf_append_simple_device_section(int hints)
{
   char buffer[128];
   char *name = NULL;
   Bool commongamma;

   commongamma = ((sc.HaveSepGC2) && (sc.gammaflags & 0x08)) ? FALSE : TRUE;

   if(hints == CF_HINTS_DHM) {

      gui_cf_append_text("\n# In Dual Head mode, this \"Device\" section is only used\n", CU_TT_BLUE);
      gui_cf_append_text("# for CRT2. A second one, for CRT1, is added below.\n", CU_TT_BLUE);

   }

   gui_cf_append_text("\n", CU_TT_GREY);

   conf_append_section_device();

   if(sc.havedevname1) name = sc.devname1;

   if(!name) {
      gui_cf_append_text("\t# Replace IDENTIFIER with the real one from the \"Device\"\n", CU_TT_GREY);
      gui_cf_append_text("\t# tag in your \"Screen\" section\n", CU_TT_GREY);
      name = "IDENTIFIER";
   }
   conf_ap_three("\tIdentifier \"", name, "\"\n", CU_TT_BLACK_BOLD);

   conf_append_driver(FALSE);

   conf_append_hint_busid_screen(hints);

   conf_append_enablesisctrl_option(FALSE);

   conf_append_usesse(hints);

   /* Device types */

   conf_append_crt1_type(hints);

   conf_append_crt2_type(hints);

   conf_append_lcd_options();

   conf_append_tv_options();

   conf_append_mergedfb_hints(hints);

   /* Gamma */

   if(hints != CF_HINTS_DHM) {

      gui_cf_append_text("\n\t# [sisctrl] Enable/disable Gamma correction for CRT1\n", CU_TT_RED);
      conf_append_gamma_enable_crt1(FALSE);

      if(vbsupportgamma2() && commongamma) {

         gui_cf_append_text("\n\t# [sisctrl] Enable/disable gamma correction for CRT2\n", CU_TT_RED);
         conf_append_gamma_enable_crt2(FALSE);

      }

      if(vbsupportgamma2() && commongamma) {

         if(sc.usenewgamma) {
            gui_cf_append_text("\n\t# [sisctrl] Brightness/contrast for CRT1 and CRT2\n", CU_TT_RED);
         } else {
            gui_cf_append_text("\n\t# [sisctrl] Gamma brightness for CRT1 and CRT2\n", CU_TT_RED);
         }

      } else {

	 if(sc.usenewgamma) {
            gui_cf_append_text("\n\t# [sisctrl] Brightness/contrast for CRT1\n", CU_TT_RED);
         } else {
            gui_cf_append_text("\n\t# [sisctrl] Gamma brightness for CRT1\n", CU_TT_RED);
         }

      }

      conf_append_gamma_bri_header();

      conf_append_gamma_brightness(FALSE);

      if(vbsupportgamma2() && !commongamma) {

         gui_cf_append_text("\n\t# [sisctrl] Enable/disable/set gamma correction for CRT2\n", CU_TT_RED);
         conf_append_gamma_bri_header_crt2();
         conf_append_crt2_gamma();

      }

      /* Saturation */
      if(sc.sdinterface) {
         if(sc.sd3flags & SiS_SD3_CRT1SATGAIN) {
            gui_cf_append_text("\n\t# [sisctrl] Saturation for CRT1\n", CU_TT_RED);
            conf_append_saturation(1, FALSE);
         }
         if(sc.sd3flags & SiS_SD3_CRT2SATGAIN) {
            gui_cf_append_text("\n\t# [sisctrl] Saturation for CRT2\n", CU_TT_RED);
            conf_append_saturation(2, FALSE);
         }
      }

   } else {	/* DHM HINTS */

      if(vbsupportgamma2()) {

         gui_cf_append_text("\n\t# [sisctrl] Enable/disable gamma correction for CRT2\n", CU_TT_RED);
         conf_append_gamma_enable_crt2(FALSE);

	 if(sc.usenewgamma) {

	    gui_cf_append_text("\n\t# [sisctrl] Brightness/contrast for CRT2\n", CU_TT_BLUE);

	    conf_append_gamma_bri_header();

	    gui_cf_append_text("\t# In Dual Head mode, the options \"Brightness\" and\n", CU_TT_BLUE);
	    gui_cf_append_text("\t# \"Contrast\" are preferred over \"CRT2Brightness\" and\n", CU_TT_BLUE);
	    gui_cf_append_text("\t# \"CRT2Contrast\" (although both variants are accepted).\n", CU_TT_BLUE);

	    if(commongamma) {
	       sprintf(buffer, "\tOption \"Brightness\" \"%.3f %.3f %.3f\"\n",
			(float)sc.newgamma_bri[0],
			(float)sc.newgamma_bri[1],
			(float)sc.newgamma_bri[2]);
	    } else {
	       sprintf(buffer, "\tOption \"Brightness\" \"%.3f %.3f %.3f\"\n",
			sc.newgamma_bri2[0],
			sc.newgamma_bri2[1],
			sc.newgamma_bri2[2]);
	    }
	    conf_fix_comma(buffer);
	    gui_cf_append_text(buffer, CU_TT_RED_BOLD);

	    if(commongamma) {
	       sprintf(buffer, "\tOption \"Contrast\" \"%.3f %.3f %.3f\"\n",
			(float)sc.newgamma_con[0],
			(float)sc.newgamma_con[1],
			(float)sc.newgamma_con[2]);
	    } else {
	       sprintf(buffer, "\tOption \"Contrast\" \"%.3f %.3f %.3f\"\n",
			sc.newgamma_con2[0],
			sc.newgamma_con2[1],
			sc.newgamma_con2[2]);
	    }
	    conf_fix_comma(buffer);
	    gui_cf_append_text(buffer, CU_TT_RED_BOLD);

	 } else {

	    gui_cf_append_text("\n\t# [sisctrl] Gamma brightness for CRT2\n", CU_TT_BLUE);

	    conf_append_gamma_bri_header();

	    gui_cf_append_text("\t# In Dual Head mode, the option \"GammaBrightness\" is\n", CU_TT_BLUE);
	    if(sc.sdinterface) {
	       gui_cf_append_text("\t# preferred over \"CRT2GammaBrightness\" (although both\n", CU_TT_BLUE);
	    } else {
	       gui_cf_append_text("\t# preferred over \"GammaBrightnessCRT2\" (although both\n", CU_TT_BLUE);
	    }
	    gui_cf_append_text("\t# are accepted equally).\n", CU_TT_BLUE);

	    if(sc.sd2flags & SiS_SD2_NEWGAMMABRICON) {
               gui_cf_append_text("\t# Note: \"GammaBrightness\" is deprecated. Your\n", CU_TT_BLUE_BOLD);
               gui_cf_append_text("\t# driver already supports the new brightness/contrast\n", CU_TT_BLUE_BOLD);
               gui_cf_append_text("\t# options. Leave this option out. SiSCtrl will then\n", CU_TT_BLUE_BOLD);
               gui_cf_append_text("\t# automatically use the new system.\n", CU_TT_BLUE_BOLD);
            }

	    if(commongamma) {
	       sprintf(buffer, "\tOption \"GammaBrightness\" \"%.3f %.3f %.3f\"\n",
			(float)sc.gamma_max[0],
			(float)sc.gamma_max[1],
			(float)sc.gamma_max[2]);
	    } else {
	       sprintf(buffer, "\tOption \"GammaBrightness\" \"%.3f %.3f %.3f\"\n",
			sc.gamma_bri2[0],
			sc.gamma_bri2[1],
			sc.gamma_bri2[2]);
	    }
	    conf_fix_comma(buffer);
	    gui_cf_append_text(buffer, CU_TT_RED_BOLD);

	 }

      }

      /* Saturation */
      if(sc.sdinterface) {
         if(sc.sd3flags & SiS_SD3_CRT2SATGAIN) {
            gui_cf_append_text("\n\t# [sisctrl] Saturation for CRT2\n", CU_TT_RED);
            conf_append_saturation(2, FALSE);
         }
      }

   }

   /* Xv */

   if(!(sc.sd2flags & SiS_SD2_NOOVERLAY)) {

      conf_append_xv_switchcrt();

      if(hints != CF_HINTS_DHM) {
         gui_cf_append_text("\n\t# [sisctrl] Xv (video overlay) related options\n", CU_TT_RED);
      } else {
         gui_cf_append_text("\n\t# [sisctrl] Xv (video overlay) related options for CRT2\n", CU_TT_BLUE);
      }

      conf_append_xv_options(FALSE);

      if(hints != CF_HINTS_DHM) {
         conf_append_xv_gamma(FALSE);
      }

   }

   conf_append_endsection();

   conf_append_second_monitor_section(hints);

   conf_append_second_device_section(hints);

}

static void
conf_generate_current_simple(int hints)
{
   conf_append_header_basic(hints);

   conf_append_header_extended(hints);

   conf_append_legend(hints);

   conf_append_simple_monitor_section(hints);

   conf_append_simple_device_section(hints);

   conf_append_hint_dhm_footer(hints);
}

/******************************************
 *                 main()                 *
 ******************************************/

void
cf_generate_current(int hints)
{

   gui_cf_clear_text();

   if(sc.sdflags & SiS_SD_ISDUALHEAD) {

      conf_generate_current_dual_head(hints);

   } else if(sc.sdflags & SiS_SD_ISMERGEDFB) {

      conf_generate_current_mergedfb(hints);

   } else {

      conf_generate_current_simple(hints);

   }
}





