#include <assert.h>
#include <stdio.h>
#include <string.h>
#include "../xposix/xposix.h"
#include "drs_p.h"

/* drs_resize_array() is used to determine how many dr_t structures to
 * adr to your "info" array when you have to realloc(). */
static const size_t drs_size_delta = 128;

static void
drs_resize_array(drs_t drs)
{
    unsigned long int dir_entries_max = 0;
    size_t count = 0;
    assert(drs);
    dir_entries_max = drs_get_directory_entries_max();
    count = drs->count_max + drs_size_delta;
    if( count > dir_entries_max ) {
        fprintf(stderr, "***\n");
        fprintf(stderr,
                "*** Error: You are only allowed %lu directory entries.\n",
                dir_entries_max);
        fprintf(stderr, "***\n");
        exit(1);
    }
    drs->info = xrealloc(drs->info, count * sizeof(dr_t));
    drs->count_max += drs_size_delta;
}

/* Returns 1 if "c" is a "d-character" according to Annex A of the ISO
 * 9660 Standard. */
static int
is_d_character(char c)
{
    switch( c ) {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
    case 'G':
    case 'H':
    case 'I':
    case 'J':
    case 'K':
    case 'L':
    case 'M':
    case 'N':
    case 'O':
    case 'P':
    case 'Q':
    case 'R':
    case 'S':
    case 'T':
    case 'U':
    case 'V':
    case 'W':
    case 'X':
    case 'Y':
    case 'Z':
    case '_':
        return 1;
    }
    return 0;
}

drs_t
drs_new(void)
{
    drs_t rv = (drs_t)xmalloc(sizeof(*rv));
    rv->info = NULL;
    rv->count = 0;
    rv->count_max = 0;
    rv->current = 0;
    return rv;
}

void
drs_delete(drs_t drs)
{
    assert(drs);
    free(drs->info);
#ifndef NDEBUG
    drs->info = 0;
#endif
    free(drs);
}

size_t
drs_get_current(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->current;
}

int
drs_set_current(drs_t drs, size_t current)
{
    int rv = 0;
    assert(drs);
    if( (drs->info != NULL) && (current <= drs->count) ) {
        drs->current = current;
        rv = 1;
    }
    return rv;
}

void
drs_push_back(drs_t drs)
{
    int info_was_null;
    assert(drs);
    info_was_null = (drs->info == NULL);
    if( drs->count + 1 >= drs->count_max ) {
        /* Adding one more entry would extend beyond the end of the array. */
        drs_resize_array(drs);
    }
    /* If (drs->info == NULL) (which happens the very first time
     *  drs_push_back() is called), you should just leave drs->count
     *  and drs->curent to their initialized value of zero.
     *  Otherwise, you need to increment drs->count and set
     *  drs->current to point to the new entry. */
    if( !info_was_null ) {
        ++drs->count;
        drs->current = drs->count;
    }
    memset((void*)&drs->info[drs->current], 0, sizeof(dr_t));
}

extern void
drs_dump_to_file(drs_t drs, FILE* f)
{
    char* buf = NULL;
    size_t i = 0;
    assert(drs);
    assert(f);
    buf = xmalloc(drs_get_current_as_string_min_buf_size());
    for( i = 0 ; ; ++i ) {
        if( !drs_set_current(drs, i) ) {
            break;
        }
        fprintf(f, "===================================="
                   "====================================\n");
        drs_get_current_as_string(drs, buf);
        fprintf(f, "%s", buf);
    }
    if( i != 0 ) {
        fprintf(f, "===================================="
                   "====================================\n");
    }
    free(buf);
}


void
drs_get_current_as_string(drs_t drs, char* buf)
{
    const char* fmt_general = "\
length_of_directory_record: %u\n\
length_of_extended_attribute_record: %u\n\
location_of_extent: %lu\n\
data_length: %lu\n\
years_since_1900: %u\n\
month: %u\n\
day: %u\n\
hour: %u\n\
minute: %u\n\
second: %u\n\
offset_from_GMT: %u\n\
file_flags: %u\n\
file_unit_size: %u\n\
interleave_gap_size: %u\n\
volume_sequence_number: %lu\n\
length_of_file_identifier: %u\n\
file_identifier: \"%s\"\n\
";

    const char* fmt_special = "\
length_of_directory_record: %u\n\
length_of_extended_attribute_record: %u\n\
location_of_extent: %lu\n\
data_length: %lu\n\
years_since_1900: %u\n\
month: %u\n\
day: %u\n\
hour: %u\n\
minute: %u\n\
second: %u\n\
offset_from_GMT: %u\n\
file_flags: %u\n\
file_unit_size: %u\n\
interleave_gap_size: %u\n\
volume_sequence_number: %lu\n\
length_of_file_identifier: %u\n\
file_identifier: 0x%x\n\
";

    const char* file_id = drs_get_file_identifier(drs);

    if( (file_id[0] == '\0') || (file_id[0] == '\1') ) {
        /* Account for the special cases of the "." and ".." directories. */
        sprintf(buf,
                fmt_special,
                drs_get_length_of_directory_record(drs),
                drs_get_length_of_extended_attribute_record(drs),
                drs_get_location_of_extent(drs),
                drs_get_data_length(drs),
                drs_get_years_since_1900(drs),
                drs_get_month(drs),
                drs_get_day(drs),
                drs_get_hour(drs),
                drs_get_minute(drs),
                drs_get_second(drs),
                drs_get_offset_from_GMT(drs),
                drs_get_file_flags(drs),
                drs_get_file_unit_size(drs),
                drs_get_interleave_gap_size(drs),
                drs_get_volume_sequence_number(drs),
                drs_get_length_of_file_identifier(drs),
                file_id[0]);
    } else {
        /* Account for the general cases. */
        sprintf(buf,
                fmt_general,
                drs_get_length_of_directory_record(drs),
                drs_get_length_of_extended_attribute_record(drs),
                drs_get_location_of_extent(drs),
                drs_get_data_length(drs),
                drs_get_years_since_1900(drs),
                drs_get_month(drs),
                drs_get_day(drs),
                drs_get_hour(drs),
                drs_get_minute(drs),
                drs_get_second(drs),
                drs_get_offset_from_GMT(drs),
                drs_get_file_flags(drs),
                drs_get_file_unit_size(drs),
                drs_get_interleave_gap_size(drs),
                drs_get_volume_sequence_number(drs),
                drs_get_length_of_file_identifier(drs),
                file_id);
    }
}

size_t
drs_get_current_as_string_min_buf_size(void)
{
    return 2048;
}

/*
 * Getters.
 */


unsigned char
drs_get_length_of_directory_record(drs_t drs)
{
    /* There are 33 bytes in the fixed part of the directory record. */
    unsigned long len = 33;
    assert(drs);
    assert(drs->info);
    /* Add to len, the number of bytes used by the variable part of
     * the directory record.*/
    len += drs_get_length_of_file_identifier(drs);
    len += drs_get_length_of_padding_field(drs);
    return len;
}

unsigned char
drs_get_length_of_extended_attribute_record(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].length_of_extended_attribute_record;
}

unsigned long int
drs_get_location_of_extent(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].location_of_extent;
}

unsigned long int
drs_get_data_length(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].data_length;
}

unsigned char
drs_get_years_since_1900(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].years_since_1900;
}

unsigned char
drs_get_month(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].month;
}

unsigned char
drs_get_day(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].day;
}

unsigned char
drs_get_hour(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].hour;
}

unsigned char
drs_get_minute(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].minute;
}

unsigned char
drs_get_second(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].second;
}

unsigned char
drs_get_offset_from_GMT(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].offset_from_GMT;
}

unsigned char
drs_get_file_flags(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].file_flags;
}

unsigned char
drs_get_file_unit_size(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].file_unit_size;
}

unsigned char
drs_get_interleave_gap_size(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].interleave_gap_size;
}

unsigned long int
drs_get_volume_sequence_number(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].volume_sequence_number;
}

unsigned char
drs_get_length_of_file_identifier(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].length_of_file_identifier;
}

const char*
drs_get_file_identifier(drs_t drs)
{
    assert(drs);
    assert(drs->info);
    return drs->info[drs->current].file_identifier;
}

int
drs_get_length_of_padding_field(drs_t drs)
{
    int rv = 0;
    unsigned char len = drs_get_length_of_file_identifier(drs);
    if( len % 2 == 0 ) {
        /* Pad if the file identifier has an even length. */
        rv = 1;
    }
    return rv;
}

unsigned long
drs_get_directory_entries_max()
{
    return 0xffffUL;
}



/*
 * Setters.
 */

void
drs_set_length_of_extended_attribute_record(drs_t drs, 
                   unsigned char length_of_extended_attribute_record)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].length_of_extended_attribute_record =
        length_of_extended_attribute_record;
}

void
drs_set_location_of_extent(drs_t drs, unsigned long int location_of_extent)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].location_of_extent = location_of_extent;
}

void
drs_set_data_length(drs_t drs, unsigned long int data_length)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].data_length = data_length;
}

void
drs_set_years_since_1900(drs_t drs, unsigned char years_since_1900)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].years_since_1900 = years_since_1900;
}

void
drs_set_month(drs_t drs, unsigned char month)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].month = month;
}

void
drs_set_day(drs_t drs, unsigned char day)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].day = day;
}

void
drs_set_hour(drs_t drs, unsigned char hour)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].hour = hour;
}

void
drs_set_minute(drs_t drs, unsigned char minute)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].minute = minute;
}

void
drs_set_second(drs_t drs, unsigned char second)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].second = second;
}

void
drs_set_offset_from_GMT(drs_t drs, unsigned char offset_from_GMT)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].offset_from_GMT = offset_from_GMT;
}

void
drs_set_file_flags(drs_t drs, unsigned char file_flags)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].file_flags = file_flags;
}

void
drs_set_file_unit_size(drs_t drs, unsigned char file_unit_size)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].file_unit_size = file_unit_size;
}

void
drs_set_interleave_gap_size(drs_t drs, unsigned char interleave_gap_size)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].interleave_gap_size = interleave_gap_size;
}

void
drs_set_volume_sequence_number(drs_t drs,
                               unsigned long int volume_sequence_number)
{
    assert(drs);
    assert(drs->info);
    drs->info[drs->current].volume_sequence_number = volume_sequence_number;
}

void
drs_set_file_identifier(drs_t drs, const char* file_identifier)
{
    int has_dot = 0;
    int is_invalid = 0;
    size_t i = 0;
    size_t len = 0;
    size_t char_count = 0;

    assert(drs);
    assert(drs->info);
    assert(file_identifier);

    /*
     * Special Cases
     */

    /* You store "." and ".." (the "this" and "parent" directories) on
     * an ISO file system by specifying a length of 0x01 and a
     * directory identifier of 0x00 and 0x01 respectively. */

    /* The "this" directory. */
    if( strcmp(file_identifier, ".") == 0 ) {
        drs->info[drs->current].length_of_file_identifier = 1;
        drs->info[drs->current].file_identifier[0] = 0;
        return;
    }

    /* The "parent" directory. */
    if( strcmp(file_identifier, "..") == 0 ) {
        drs->info[drs->current].length_of_file_identifier = 1;
        drs->info[drs->current].file_identifier[0] = 1;
        return;
    }

    /*
     * General Case
     */

    len = strlen(file_identifier);

    for( i = 0 ; i < len ; ++i ) {
        if( is_d_character(file_identifier[i]) ) {
            ++char_count;
            continue;
        }
        if( (file_identifier[i] == '.') && !has_dot ) {
            /* You are allowed one dot character, but it does not
             * count against your alloted 30 characters. */
            has_dot = 1;
            continue;
        }
        is_invalid = 1;
        break;
    }

    if( is_invalid || (char_count > 30) || (char_count == 0) ) {
        fprintf(stderr, "***\n");
        fprintf(stderr,
                "*** Error: Invalid file name: \"%s\"\n",
                file_identifier);
        fprintf(stderr, "***\n");
        exit(1);
    }

    /* Generate the full string including implied '.' characters and a
     * file version of ";1". */
    {
        char* s = drs->info[drs->current].file_identifier;
        strcpy(s, file_identifier);
        if( !has_dot ) {
            strcat(s, ".");
        }
        strcat(s, ";1");
        drs->info[drs->current].length_of_file_identifier = strlen(s);
    }
}
