#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#else
#include <sys/signal.h>
#endif
#include <netinet/in.h>

#include "sftp.h"

u_int32_t version = SFTP_VERSION_OLD;

sftp_channel *channel[MAXCHANNEL] = {NULL};

static int
do_chdir(char *dir) {
	char *str;

	if (dir == NULL) {
		dir = getenv("HOME");
		if (dir == NULL) {
			str = "$HOME does not exist";
			send_message(1, _message(ERROR, str, strlen(str)+1));
			return -1;
		}
	}
	else if (dir[0] == '~') {
		dir = tildeexpand(dir);
		if (dir == NULL) {
			str = "No such directory";
			send_message(1, _message(ERROR, str, strlen(str)+1));
			return -1;
		}
	}
	if (chdir(dir) < 0) {
		if (errno == EACCES)
			str = "Permission denied";
		else
			str = "No such directory";
		send_message(1, _message(ERROR, str, strlen(str)+1));
		return -1;
	}
	send_message(1, _message(SUCCESS, NULL, 0));
	return 0;
}

static int
do_getdir(char *dir) {
	char buf[256];
	if (getcwd(buf, sizeof(buf)) != NULL) {
		send_message(1, _message(TELLDIR, buf, strlen(buf)+1));
		return 0;
	}
	else {
		char *str = "Failed to get current directory";
		send_message(1, _message(TELLDIR, str, strlen(str)+1));
		return -1;
	}
}

static int
do_version(u_int32_t ver) {
	u_int32_t myversion = ntohl(SFTP_MY_VERSION);
	send_message(1, _message(VERSION, &myversion, 4));
	version = ntohl(ver);
	return 0;
}

/* Returns the number of channels that still have data to send */
static int
do_output() {
	sftp_channel_t c;
	int total = 0, ret;
	for (c=1; c<MAXCHANNEL; c++)
		if (channel[c] && channel[c]->waiting_to_send) {
			ret = channel[c]->send_data(c);
			if (ret < 0) {
				channel[c]->close(c);
				continue;
			}
			if (channel[c]->waiting_to_send)
				total++;
		}
	return total;
}

void
reap() {
        int status;
        int pid;
        while ((pid = wait3(&status, WNOHANG, NULL)) > 0);
}

int
main() {
	message m;
	sftp_channel_t c;
	int nwaiting, ret;
	struct sigaction act;
	sigset_t set;

	sigemptyset(&set);
	act.sa_handler = reap;
	act.sa_mask = set;
	act.sa_flags = SA_RESTART;
	sigaction (SIGCHLD, &act, NULL);

	fwrite("xsftp", 1, strlen("xsftp") + 1, stdout);
	fflush(stdout);
	while (1) {
		nwaiting = do_output();
		if (nwaiting > 0) {
			ret = query_message(0, &m);
			if (ret < 0)
				continue;
		}
		else {
			ret = recv_message(0, &m);
			if (ret < 0)
				exit(-1);
		}
		if (m.channel != 0) {
				c = m.channel;
				if (channel[c] == NULL)
					continue;
				ret = channel[c]->accept_data(c, m);
				if (ret < 0)
					channel[c]->close(c);
		}
		else
			switch (m.command) {
				case REQUEST:
					c = new_sendfile_channel(m.data);
					break;
				case SENDFILE:
					c = new_recvfile_channel(*(sftp_channel_t *)m.data);
					break;
				case EXEC:
					c = new_exec_channel(m.data, m.len);
					break;
				case CHDIR:
					do_chdir(m.data);
					break;
				case GETDIR:
					do_getdir(m.data);
					break;
				case VERSION:
					do_version(*(u_int32_t *)m.data);
					break;
				case CLOSE:
					exit(-1);
			}
		if (m.data)
			free(m.data);
	}
}
