/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "smscoding.h"
#include "smsudh.h"

#include <charsets.h>
#include <helper.h>
#include <intincl.h>
#include <timeincl.h>
#include <gtincl.h>

#include <string.h>
#include <stdlib.h>
#include <unistd.h>

static
uint16_t sms_new_sequence_id () {
  /* this does not have to be good random (linear would be ok)
   * just not the same on every call
   */
  srand(time(NULL));
  return rand()&UINT16_MAX;    
}

static
unsigned int sms_header_size (struct sms_pdu_ud_header** header) {
  unsigned int headersize = 1;
  unsigned int i = 0;

  if (header == NULL) {
    return 0;
  } else {
    for (; header[i] != NULL; ++i) {
      headersize += header[i]->len + 2;
    }
  }
  return headersize;
}

static
void sms_data_insert_header (struct sms_pdu_ud_header** header,
			     char* userdata)
{
  unsigned int size = sms_header_size(header);
  unsigned int i = 0;
  unsigned int k = 2;
  unsigned int l = 0;
  char* temp = NULL;

  if (userdata == NULL || size == 0)
    return;
  temp = mem_alloc(2*size+1,1);
  sprintf(temp,"%02X",size-1);
  for (; header[i] != NULL; ++i) {
    sprintf(temp + k,"%02X",header[i]->type);
    sprintf(temp + k + 2,"%02X",header[i]->len);
    for (; l < header[i]->len; ++l)
      sprintf(temp + k + 4 + 2*l,"%02X",header[i]->data[l]);
    k += 4 + 2*header[i]->len;
  }
  memcpy(userdata,temp,2*size);
  mem_realloc(temp,0);
}

static
char* sms_data_gsm_encode (struct sms_pdu_ud_header** header,
			   ucs4char_t* input)
{
  char* retval;
  gsmchar_t* temp;
  unsigned char first, end;

  unsigned int headersize = sms_header_size(header);
  unsigned int enc_len = 0;
  unsigned int inc = 0;
  unsigned int outc = headersize*2;
  unsigned int state = (headersize*8)/7;
  
  if (headersize > 140) return NULL;

  temp = convert_to_gsm(input);
  enc_len = strlen((char*)temp);
  retval = mem_alloc(2 + (140*2) + 1, 1);
  sms_data_insert_header(header,retval+2);

  if (ucs4len(input) > 0) {
    /* 7bit-GSM octet encoding
     */
    if (state%8 != 0) { //state%8 == 0 equals headersize%7 == 0
      // do the septet padding
      sprintf(retval+2 + outc, "%02X", ((temp[inc] & 0x7F) << (7-(state%8))) & 0xFF);
      outc += 2;
      ++state;
    }
    for (; inc < enc_len && state < (140*8/7); ++inc, ++state) {
      if (state%8 != 7) {
	//the character are 7bit
	//shifting dependent on state (0-7)
	first = ((temp[inc+1] & 0x7F) << (7-(state%8))) & 0xFF;
	end = (temp[inc] & 0x7F) >> (state%8);
	//storing as hexstring
	sprintf(retval+2 + outc, "%02X", (first | end));
	outc += 2;
      }
    }

    //Return NULL to indicate error during encoding.
    //In this case, input string was too large to fit.
    if (inc != enc_len) {
      mem_realloc(retval,0);
      return NULL;
    }
  }

  /* the user data length must contain the septet count
   * of all userdata (including header)
   */
  mem_realloc(temp,3);
  sprintf((char*)temp,"%02X",state);
  memcpy(retval,temp,2);
  mem_realloc(temp,0);

  return retval;
}

static
char* sms_data_ucs2_encode (struct sms_pdu_ud_header** header,
			    ucs4char_t* input)
{
  char* retval;
  char* tmp;
  unsigned int enc_len = 0;
  unsigned int headersize = sms_header_size(header);
  
  if (headersize > 140) return NULL;

  tmp = convert_to_ucs2_hexstring(input);
  enc_len = strlen(tmp)/2;
  retval = mem_alloc((141*2)+1,1);

  if (headersize) sms_data_insert_header(header,retval+2);
  memcpy(retval+2+strlen(retval+2),tmp,strlen(tmp));
  mem_realloc(tmp,3);
  sprintf(tmp,"%02X",strlen(retval+2)/2);
  memcpy(retval,tmp,2);
  mem_realloc(tmp,0);

  return retval;
}

#define SMS_PARTS_SUGGESTED_MAX 10
char** sms_data_encode (enum sms_encoding charset,
			struct sms_pdu_ud_header** header,
			ucs4char_t* input)
{
  char** retval = NULL;

  struct sms_pdu_ud_header** h = NULL;
  unsigned int hcount = 0;
  unsigned int count = 0;
  unsigned int parts = 0;
  unsigned int i = 0;
  unsigned int k;

  char* delayMessage = _("Delaying for %2d seconds, press Ctrl+C to abort.");
  ucs2char_t* input2 = NULL;
  ucs4char_t* input_tmp;
  uint16_t sequence = sms_new_sequence_id();

  /* get rid of compiler warning */
  header = header;

  switch (charset) {
  case SMS_CHARSET_GSM:
    /* If more than 160 septets are needed:
     * Text data must be aligned at septet:
     * if (H % 7) != 0: minus 1 septet
     *
     * H = 1+6
     * (140-H)*8/7 - ((H%7)?1:0) = 152 Septets
     */
    count = gsm_count(input,0);
    if (count <= 160) parts = 1;
    else while (i < ucs4len(input)) {
      i += gsm_count(input+i,152);
      parts += 1;
    }
    break;
  case SMS_CHARSET_UCS2:
    /* If more than 70 characters are needed (H=1+6):
     * 140-(H/2 + H%2)*2 = 140-(3+1)*2 = 132 Oktets
     *                                 = 66 UCS-2 characters
     */
    input2 = convert_to_ucs2(input);
    count = ucs2len(input2);
    if (count <= 70) parts = 1;
    else {
      parts = count/66;
      if (count%66) ++parts;
    }
    break;
  case SMS_CHARSET_8BIT:
    //no break
  default:
    return NULL; //not supported
  }
  fprintf(stderr,"%s: ",_("Notice"));
  fprintf(stderr,
	  ngettext("This message has %d character",
		   "This message has %d characters",
		   ucs4len(input)),
	  ucs4len(input));
  fprintf(stderr,"%s"," ");
  fprintf(stderr,
	  ngettext("and will use %d part.\n",
		   "and will use %d parts.\n",
		   parts),
	  parts);
  if (parts > SMS_PARTS_SUGGESTED_MAX) {
    fprintf(stderr, _("%s: The suggested maximum parts count is %d.\n"),
	    _("Warning"),SMS_PARTS_SUGGESTED_MAX);
    for (i=10; i>0; --i) {
      fprintf(stderr, delayMessage, i);
      fprintf(stderr,"\r");
      sleep(1);
    }
   for (i=strlen(delayMessage)-2; i>0; --i)
      fprintf(stderr," ");
    fprintf(stderr,"\r");
  }
  if (parts > 255) {
    fprintf(stderr,_("%s: message part count exceeds maximum of 255\n"),
	    _("Error"));
    fprintf(stderr,_("%s: SMS text is too long (max. %d characters).\n"),
	    _("Error"),
	    (charset==SMS_CHARSET_GSM) ? 255*140*8/7 : 255*140/2);
    if (charset == SMS_CHARSET_GSM) {
      fprintf(stderr,"%s\n",
	      _("Please be aware that some characters "
	        "are encoded as 14bit (instead of 7bit), "
	        "e.g. the euro character."));
    }
    return NULL;
  }

  retval = mem_alloc(sizeof(*retval)*(parts+1),0);
  for (i=0; i<=parts; ++i) retval[i] = NULL;
  if (parts > 1) {
    hcount += 1;
    h = mem_alloc((hcount+1)*sizeof(*h),0);
    for (i=0; i<=hcount; ++i) h[i] = NULL;

    count = 0;
    switch (charset) {
    case SMS_CHARSET_GSM:
      for (i=0; i<parts; ++i) {
	h[0] = sms_udh_multipart16_get(sequence,i+1,parts);
	k = gsm_count(input+count,152);
	input_tmp = ucs4ndup(input+count,k);
	retval[i] = sms_data_gsm_encode(h,input_tmp);
	input_tmp = mem_realloc(input_tmp,0);
	if (retval[i] == NULL) { //encoding failed
	  if (i != 0)
	    do { mem_realloc(retval[--i],0); } while (i != 0);
	  return mem_realloc(retval,0);
	}
	count += k;
      }
      break;
    case SMS_CHARSET_UCS2:
      k = 66;
      input = convert_from_ucs2(input2);
      for (i=0; i<parts; ++i) {
	h[0] = sms_udh_multipart16_get(sequence,i+1,parts);
	input_tmp = ucs4ndup(input+count,k);
	retval[i] = sms_data_ucs2_encode(h,input_tmp);
	input_tmp = mem_realloc(input_tmp,0);
	count += k;
      }
      mem_realloc(input,0);
    break;
    case SMS_CHARSET_8BIT:
      //no break
    default:
    return NULL; //not supported
    }
  } else {
    switch (charset) {
    case SMS_CHARSET_GSM:
      retval[0] = sms_data_gsm_encode(NULL,input);
      if (retval[0] == NULL) //encoding failed
	return mem_realloc(retval,0);
      break;
    case SMS_CHARSET_UCS2:
      retval[0] = sms_data_ucs2_encode(NULL,input);
      break;
    case SMS_CHARSET_8BIT:
      //no break
    default:
      return NULL; //not supported
    }
  }
  
  return retval;
}
