/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "common.h"
#include "helpers.h"
#include <locale.h>
#include <string.h>
#include <stdlib.h>

#define _GNU_SOURCE
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#else
# include <gnugetopt/getopt.h>
#endif

int phone_init (void) {
  char* ack;
  int retval=0;

  tty_write_command("\rATZ");
  ack=tty_read("ATZ");
  if (!strcmp(ack,"OK")) {
    retval=1;
  } else if (!strcmp(ack,"ERROR")) {
    retval=0;
  } else {
    mem_realloc(ack,0);
    ack=tty_read("ATZ");
    if (!strcmp(ack,"OK")) {
      retval=1;
    } else if (!strcmp(ack,"ERROR")) {
      retval=0;
    } else {
      errexit("Unexpected return code: %s\n",ack);
    }
  }
  mem_realloc(ack,0);
  return retval;;
}

int main (int argc, char** argv) {
  char empty[] = ""; // to be used at init with all char pointers

  struct parameters myparams;
  struct smsopts mysmsopts;

  char* myFILE;
  char* myPIPE;
  char* myttyport;
  speed_t myttyspeed;
  unsigned int ignore_serial_bits=0;
  unsigned int timeout=10;
    
  int did_something=0;
  char scmxx_action=0;
  char scmxx_ftype=0;
  char direct_sms=0;
        
  char* temp;
  char* ausgabe;

  struct option myoptions[]={
    {"remove",0,0,'r'},{"send",0,0,'s'},{"get",0,0,'g'},
    {"bitmap",2,0,'B'},{"midi",2,0,'M'},{"vcal",2,0,'C'},{"vcard",2,0,'F'},
    {"pbook",1,0,'P'},{"sms",2,0,'S'},
    {"smsmem",1,0,0},
    {"text",1,0,'t'},{"unicode",0,0,0},{"number",1,0,'n'},{"direct",0,0,0},{"flash",0,0,0},{"srr",0,0,0},
    {"device",1,0,'d'},{"baud",1,0,'b'},
    {"out",1,0,'o'},{"pipe",1,0,'p'},
    {"pin",1,0,0},
    {"info",0,0,'i'},{"set-time",0,0,0},{"set-smsc",0,0,0},
    {"help",0,0,'h'},{"version",0,0,0},
    {"verbose",0,0,'v'},{"quiet",0,0,'q'},
    {"reset",0,0,0},{"ignore-serial-bits",0,0,0},{"device-timeout",1,0,0},
    {0,0,0,0}
  };
  const char myoptions_short[]="rsgB::M::C::F::P:S::t:n:d:b:o:p:ivqh";
  int option_index,arg;

  //start setup
  //begin init global vars	
  memset(PIN,0,sizeof(PIN));
  VERBOSE_LEVEL=0;
  mytty=-1;
  //end init global vars    

  myttyport=NULL;
  myparams.text=empty;
  myparams.number=empty;
  myparams.mem=empty;
  myparams.slot=0;
  mysmsopts.flash=0;
  mysmsopts.srr=0;
  mysmsopts.unicode=0;
  myFILE=empty;
  myPIPE=empty;

  //using env. var. SCMXX_BAUD
  if ((temp=getenv("SCMXX_BAUD"))==NULL) {
    myttyspeed=tty_speed(TTYSPEED);
  } else {
    myttyspeed=tty_speed(temp);
  }
  //setting program's locale
  setlocale(LC_ALL,"");
	
  if (argc==1) {
    help(argv[0]);
    exit(0);
  }

  /* First, we look for parameters that have no dependencies
   * and that do not need an open device
   */
  while ((arg=getopt_long(argc,argv,myoptions_short,myoptions,&option_index)) != -1) {
    switch (arg){
    case 'd':
      if (optarg) {
	myttyport=optarg;
      }
      break;
    case 'b':
      if (optarg) {
	myttyspeed=tty_speed(optarg);
      }
      break;
    case 't':
      if (optarg) {
	myparams.text=optarg;
      } else {
	errexit("Zero length text makes no sense.\n");
      }
      break;
    case 'n':
      if (optarg) {
	myparams.number=optarg;
      }
    case 'o':
      if (optarg) {
	myFILE=optarg;
      }
      break;
    case 'p':
      if (optarg) {
	myPIPE=optarg;
      }
      break;
    case 'v':
      VERBOSE_LEVEL++;
      break;
    case 'q':
      VERBOSE_LEVEL--;
      break;
    case 'h':
      help(argv[0]);
      exit(0);
      break;
    case 0:
      if (!strcmp(myoptions[option_index].name,"pin")) {
	if (optarg && strlen(optarg)<sizeof(PIN)) {
	  strcpy(PIN,optarg);
	}
      } else if (!strcmp(myoptions[option_index].name,"unicode")) {
	mysmsopts.unicode=1;
      } else if (!strcmp(myoptions[option_index].name,"direct")) {
	direct_sms=1;
      } else if (!strcmp(myoptions[option_index].name,"srr")) {
	mysmsopts.srr=1;
      } else if (!strcmp(myoptions[option_index].name,"flash")) {
	mysmsopts.flash=1;
      } else if(!strcmp(myoptions[option_index].name,"version")) {
	version();
	exit(0);
      } else if(!strcmp(myoptions[option_index].name,"ignore-serial-bits")) {
	ignore_serial_bits=1;
      } else if(!strcmp(myoptions[option_index].name,"device-timeout")) {
	if (optarg) {
	  if ((timeout=atoi(optarg))<1) {
	    timeout=1;
	  }
	}
      }
      break;
    }
  }
  //now open the port
  //using env. var. SCMXX_TTY
  if ((myttyport==NULL || !strlen(myttyport)) &&
      (myttyport=getenv("SCMXX_TTY"))==NULL) {
    tty_open(TTYPORT,myttyspeed,timeout,ignore_serial_bits);
  } else {
    tty_open(myttyport,myttyspeed,timeout,ignore_serial_bits);
  }

  /* All error from getopt_long were already shown
   */
  opterr=0;

  /* Now there is the reset option that needs to run before normal init
   */
  optind=0;
  while ((arg=getopt_long(argc,argv,myoptions_short,myoptions,&option_index)) != -1) {
    switch (arg){
    case 0:
      if(!strcmp(myoptions[option_index].name,"reset")) {
	/* sending a sync frame
	 * the phone may not respond without it because it thinks that all
	 * send commands are still data from a previous run
	 */
	myprintf(0,"Setting reset frame...");
	tty_write("\x1a\r\n",3,"");
	myprintf(0,"done\n");
	exit(0);
      }
      break;
    }
  }

  /* Now perform initial phone setup
   */
  //testing the device
  if (phone_init()) {
    myprintf(0,"OK, a modem device is present.\n");
  } else {
    errexit("ERROR, cannot communicate with device.\n");
  }
  //disabling command echo
  ausgabe=tty_write_read("ATE0");
  if (strcmp(ausgabe,"OK")) {
    errexit("ERROR, aborting\n");
  }
  mem_realloc(ausgabe,0);
  
  //enabling advanced error response
  //not needed everywhere but this reduces code duplication
  mem_realloc(tty_write_read("AT+CMEE=2"),0);
  
  //changing to GSM charset
  mem_realloc(tty_write_read("AT+CSCS=GSM"),0);

#define SCMXX_ACTION_REMOVE 1
#define SCMXX_ACTION_SEND   2
#define SCMXX_ACTION_GET    4

  /* Now process the remaining options
   * Part 1: Options that run an inline action
   *         and options that are a dependency to others
   */
  optind=0;
  while ((arg=getopt_long(argc,argv,myoptions_short,myoptions,&option_index)) != -1) {
    switch (arg){
    case 'r':
      scmxx_action |= SCMXX_ACTION_REMOVE;
      break;
    case 's': 
      scmxx_action |= SCMXX_ACTION_SEND;
      break;
    case 'g':
      scmxx_action |= SCMXX_ACTION_GET;
      break;
    case 0:
      if (!strcmp(myoptions[option_index].name,"set-time")) {
	set_time();
	did_something=1;
      } else if (!strcmp(myoptions[option_index].name,"set-smsc")) {
	if (myparams.number!=NULL && strlen(myparams.number)) {
	  set_smsc(myparams.number);
	  did_something=1;
	} else {
	  errexit("You must define a number with the --number option.\n");
	}
      } else if (!strcmp(myoptions[option_index].name,"smsmem")) {
	if (optarg) {
	  myparams.mem=optarg;
	}
      }
      break;
    }
  }

  /* Now process the remaining options
   * Part 2: the real actions
   */
  optind=0;
  while ((arg=getopt_long(argc,argv,myoptions_short,myoptions,&option_index)) != -1) {
    switch (arg){
    case 'i':
      info(myFILE);
      exit(0);
      break;
    }
  }
  temp=get_vendor();
  if (temp!=NULL && strcasecmp(temp,"SIEMENS")) {
    myprintf(0,"WARNING: Phones from this vendor were not confirmed to be working with this software!\n");
  }
  ausgabe=get_model();
  myprintf(0,"Detected %s %s\n",temp,ausgabe);
  if (!(!strcmp(ausgabe,"S35i") || !strcmp(ausgabe,"M35i") || !strcmp(ausgabe,"C35i")
	|| !strcmp(ausgabe,"SL42") || !strcmp(ausgabe,"SLIN")
	|| !strcmp(ausgabe,"SL45") || !strcmp(ausgabe,"SLIK")
	|| !strcmp(ausgabe,"C45") || !strcmp(ausgabe,"ME45")
	|| !strcmp(ausgabe,"S45") || !strcmp(ausgabe,"S45i")
	|| !strcmp(ausgabe,"M50") || !strcmp(ausgabe,"MT50")  || !strcmp(ausgabe,"M50I")
	|| !strcmp(ausgabe,"S25")
	//there are even some phones from north america
	|| !strcmp(ausgabe,"S46"))
	) {
    myprintf(0,"WARNING: this phone was not confirmed to be working with this software.\n");
  }
  mem_realloc(temp,0);
  mem_realloc(ausgabe,0);

#define SCMXX_FTYPE_BITMAP 0x01
#define SCMXX_FTYPE_MIDI   0x03
#define SCMXX_FTYPE_VCAL   0x05
#define SCMXX_FTYPE_VCARD  0x07
#define SCMXX_FTYPE_PBOOK  0x02
#define SCMXX_FTYPE_SMS    0x04

  if (!did_something && !scmxx_action) {
    errexit ("You must specify a valid action.\n");
  }
  optind=0;
  while ((arg=getopt_long(argc,argv,myoptions_short,myoptions,&option_index)) != -1) {
    switch (arg){
    case 'B':
      scmxx_ftype = SCMXX_FTYPE_BITMAP;
      if (optarg) {
	if(is_number(optarg)) {
	  myparams.slot=atoi(optarg);
	} else if (!strcmp(optarg,"all")) {
	  myparams.slot=-2;
	} else {
	  errexit("\"%s\" is not a valid %s slot.\n",optarg,"bitmap");
	}
      } else {
	myparams.slot = -1;
      }
      break;
    case 'M':
      scmxx_ftype = SCMXX_FTYPE_MIDI;
      if (optarg) {
	if(is_number(optarg)) {
	  myparams.slot=atoi(optarg);
	} else if (!strcmp(optarg,"all")) {
	  myparams.slot=-2;
	} else {
	  errexit("\"%s\" is not a valid %s slot.\n",optarg,"midi");
	}
      } else {
	myparams.slot = -1;
      }
      break;
    case 'C':
      scmxx_ftype = SCMXX_FTYPE_VCAL;
      if (optarg) {
	if(is_number(optarg)) {
	  myparams.slot=atoi(optarg);
	} else if (!strcmp(optarg,"all")) {
	  myparams.slot=-2;
	} else {
	  errexit("\"%s\" is not a valid %s slot.\n",optarg,"vcal");
	}
      } else {
	myparams.slot = -1;
      }
      break;
    case 'F':
      scmxx_ftype = SCMXX_FTYPE_VCARD;
      if (optarg) {
	if(is_number(optarg)) {
	  myparams.slot=atoi(optarg);
	} else if (!strcmp(optarg,"all")) {
	  myparams.slot=-2;
	} else {
	  errexit("\"%s\" is not a valid %s slot.\n",optarg,"vcf");
	}
      } else {
	myparams.slot = -1;
      }
      break;
    case 'P':
      scmxx_ftype = SCMXX_FTYPE_PBOOK;
      if (optarg) {
	myparams.mem=optarg;
      } else {
	errexit("You have to select a phonebook.\n");
      }
      break;
    case 'S':
      scmxx_ftype = SCMXX_FTYPE_SMS;
      if (optarg) {
	if (is_number(optarg)) {
	  myparams.slot=atoi(optarg);
	} else {
	  if(!strcmp(optarg,"all")) {
	    myparams.slot=-4;
	  } else if(!strcmp(optarg,"sent")) {
	    myparams.slot=-3;
	  } else if(!strcmp(optarg,"unsent")) {
	    myparams.slot=-2;
	  } else if(!strcmp(optarg,"read")) {
	    myparams.slot=-1;
	  } else {
	    myparams.slot=0;
	  }
	}
      }
      break;
    }
  }
  if (!did_something) {
    if (!scmxx_ftype) {
      errexit("No action taken. If you wanted to do something, you misinterpreted the syntax.\n");
    }
  } else {
    if (!scmxx_ftype) {
      exit(0);
    }
  }
  if (scmxx_action==SCMXX_ACTION_SEND) {
    if (scmxx_ftype==SCMXX_FTYPE_SMS  &&
	(strlen(myparams.text) || myparams.slot>0)) {
      handle_s35(myFILE,myPIPE,
		 (((scmxx_action&7)<<5)|
		  ((direct_sms&1)<<4)|
		  (scmxx_ftype&15))&255,
		 myparams,mysmsopts);
    }
    while (optind<argc) {
      handle_s35(argv[optind++],myPIPE,
		 (((scmxx_action&7)<<5)|
		  ((direct_sms&1)<<4)|
		  (scmxx_ftype&15))&255,
		 myparams,mysmsopts);
    }
  } else if (optind>=argc && !strlen(myparams.text)) {
    handle_s35(myFILE,myPIPE,
	       (((scmxx_action&7)<<5)|
		((direct_sms&1)<<4)|
		(scmxx_ftype&15))&255,
	       myparams,mysmsopts);
  }
  tty_close();
  exit(0);
}

void errexit(char *errmessage, ...){
  va_list arg_list;
  va_start(arg_list,errmessage);
  vfprintf(stderr,errmessage,arg_list);
  va_end(arg_list);
  tty_close();
  exit(1);
}

void myprintf(int verbose_level, char *output, ...){
  va_list arg_list;
  va_start(arg_list,output);
  if (verbose_level<=VERBOSE_LEVEL) {
    vfprintf(stderr,output,arg_list);
  }
  va_end(arg_list);
}
