/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
	 
#include "common.h"
#include "helpers.h"
#include "depincludes.h"
#include <string.h>
#include <stdlib.h>

char* get_value (char* at) {
  char* ausgabe;
    
  ausgabe=tty_write_read(at);
  if (strstr(ausgabe,"ERROR") == NULL) {
    mem_realloc(tty_read(at),0);
    return ausgabe;
  } else {
    return NULL;
  }
}

char* get_charset (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CSCS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSCS: \"",8)) {
    if ((temp=index(ausgabe+8,'"'))!=NULL) {
      temp=strn_dup(ausgabe+8,temp-ausgabe-8);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

void set_smsc (char* smsnr) {
  char at_command[128];
  char* ack;
  char buffer;

  memset(at_command,0,sizeof(at_command));
  if (!strlen(smsnr)) {
    errexit("No number specified.\n");
  } else if (!is_pnumber(smsnr,1)) {
    errexit("No valid number specified.\n");
  }
  new_at_command(at_command,"+CSCA");
  add_at_command(at_command,"=%s,",smsnr);
  buffer=smsnr[1];
  if (buffer=='+') {
    add_at_command(at_command,"145");
  } else {
    add_at_command(at_command,"129");
  }
  ack=tty_write_read(at_command);
  if (!strcmp(ack,"OK")) {
    myprintf(0,"SMS server number was set to %s\n",smsnr);
  } else {
    errexit("%s\n",ack);
  }
  mem_realloc(ack,0);
}

char* get_smsc (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CSCA?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSCA: \"",8)) {
    if ((temp=strstr(ausgabe+8,"\","))!=NULL) {
      temp=strn_dup(ausgabe+8,temp-ausgabe-8);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

void set_time (void) {
  char at_command[128];
  char* ack;
  time_t seconds;
  char timestr[128];
	
  memset(at_command,0,sizeof(at_command));
  memset(timestr,0,sizeof(timestr));
	
  new_at_command(at_command,"+CCLK");
  time(&seconds); //seconds since Unix
  //string formatting with conversion to local time
  strftime(timestr,sizeof(timestr),"%y/%m/%d,%H:%M:%S",localtime(&seconds));
  add_at_command(at_command,"=\"%s\"",timestr);
  ack=tty_write_read(at_command);
  if (!strcmp(ack,"OK")) {
    myprintf(0,"Time was synchronized\n");
  } else {
    errexit("Time could not be synchronized\n");
  }
  mem_realloc(ack,0);
}

struct tm* get_time (void) {
  char* ausgabe;
  struct tm* phonetime;
  
  ausgabe=get_value("AT+CCLK?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CCLK: \"",8)) {
    phonetime=mem_alloc(sizeof(struct tm),1);
    if (strptime(ausgabe+8,"%y/%m/%d,%H:%M:%S",phonetime)!=NULL) {
      mem_realloc(ausgabe,0);
      return phonetime;
    }
    mem_realloc(ausgabe,0);
    mem_realloc(phonetime,0);
  }
  return NULL;
}

inline char* get_vendor (void) {
  return get_value("AT+CGMI");
}

inline char* get_model (void) {
  return get_value("AT+CGMM");
}

inline char* get_revision (void) {
  return get_value("AT+CGMR");
}

inline char* get_phoneserial (void) {
  return get_value("AT+CGSN");
}

inline char* get_simserial (void) {
  return get_value("AT+CIMI");
}

char* get_simid (void) {
  char* ausgabe;

  ausgabe=get_vendor();
  if(ausgabe!=NULL && !strcasecmp(ausgabe,"SIEMENS")) {
    ausgabe=get_value("AT^SCID");
    if (ausgabe!=NULL && !strncmp(ausgabe,"^SCID: ",7)) {
      memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
      return mem_realloc(ausgabe,strlen(ausgabe)+1);
    }
  }
  return NULL;
}

char* get_operator (void) {
  char* ausgabe;
  char* temp;

  ausgabe=get_value("AT+COPS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+COPS: ",7)) {
    if ((temp=index(ausgabe+7,'"'))!=NULL) {
      ++temp;
      if (temp[strlen(temp)-1]=='"') {
	temp[strlen(temp)-1]=0;
      }
      temp=strn_dup(temp,strlen(temp));
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

char* get_battery (void) {
  char* ausgabe;
  char* temp;
    
  ausgabe=get_value("AT+CBC");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CBC: ",6)) {
    if ((temp=index(ausgabe+6,','))!=NULL) {
      temp=strn_dup(temp+1,strlen(temp+1));
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

int get_signal (int* ber_p) {
  char* ausgabe;
  char* temp;
  int signal = -1;

  if (ber_p != NULL) {
    *ber_p = -1;
  }    
  ausgabe=get_value("AT+CSQ");
  //example: +CSQ: 28,99
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CSQ: ",6)) {
    signal=atoi(ausgabe+6);
    if (signal==99) {
      signal=-1;
    } else {
      if ((signal=111-((signal-1)*2)) < 0) {
	signal = -1;
      }
    }
    if (ber_p != NULL) {
      if ((temp=index(ausgabe,','))!=NULL) {
	*ber_p=atoi(temp+1);
	if (*ber_p==7) {
	  *ber_p=255;
	} else if (*ber_p>=0 && *ber_p <7) {
	  *ber_p=2<<(*ber_p);
	} else {
	  // unknown value (the standard 99 means the same)
	  *ber_p=-1;
	}
      }
    }
    mem_realloc(ausgabe,0);
  }
  return signal;
}

char* get_filetypes (short readable) {
  char* ausgabe;

  ausgabe=get_vendor();
  if(ausgabe!=NULL && !strcasecmp(ausgabe,"SIEMENS")) {
    if (readable) {
      ausgabe=get_value("AT^SBNR=?");
    } else {
      ausgabe=get_value("AT^SBNW=?");
    }
    if (ausgabe!=NULL && (!strncmp(ausgabe,"^SBNW: ",7) || !strncmp(ausgabe,"^SBNR: ",7))) {
      memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
      return mem_realloc(ausgabe,strlen(ausgabe)+1);
    }
  }
  return NULL;
}

char* get_pbtypes (void) {
  char* ausgabe;

  ausgabe=get_vendor();
  if(ausgabe!=NULL && !strcasecmp(ausgabe,"SIEMENS")) {
    ausgabe=get_value("AT^SPBS=?");
    if (ausgabe!=NULL && !strncmp(ausgabe,"^SPBS: ",7)) {
      memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
      return mem_realloc(ausgabe,strlen(ausgabe)+1);
    }
  } else {
    ausgabe=get_value("AT+CPBS=?");
    if (ausgabe!=NULL && !strncmp(ausgabe,"+CPBS: ",7)) {
      memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
      return mem_realloc(ausgabe,strlen(ausgabe)+1);
    }
  }
  return NULL;
}

char* get_smsmemtypes (void) {
  char* ausgabe;

  ausgabe=get_value("AT+CPMS=?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CPMS: ",7)) {
    memmove(ausgabe,ausgabe+7,strlen(ausgabe+7)+1);
    return mem_realloc(ausgabe,strlen(ausgabe)+1);
  }
  return NULL;
}

char* get_netstatus (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;

  ausgabe=get_value("AT+CREG=?");
  if (index(ausgabe,'2')==NULL) {
    mem_realloc(ausgabe,0);
    ausgabe=get_value("AT+CREG?");
  } else {
    mem_realloc(ausgabe,0);
    /* Fix: firmware may forget to send LAC/IC (ME45 v23)
     * Provided by Dominik Neubauer
     */
    ausgabe=get_value("AT+CREG=2;+CREG?;+CREG=0");
  }
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CREG: ",7)) {
    if (areacode_p!=NULL) {
      *areacode_p=0;
    }
    if (areacode_p!=NULL) {
      *cellid_p=0;
    }
    index1=index(ausgabe+7,'"');
    if (index1!=NULL) {
      index2=index(++index1,'"');
      if (index2!=NULL && areacode_p!=NULL) {
	*areacode_p=hexstr2int(index1,index2-index1);
	index1=index(index2+1,'"');
	if (index1!=NULL) {
	  index2=index(++index1,'"');
	  if (index2!=NULL && cellid_p!=NULL) {
	    *cellid_p=hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe+7,','))!=NULL) {
      switch (*(index1+1)) {
      case '0':
	mem_realloc(ausgabe,0);
	return "not checked in, not seeking";
      case '1':
	mem_realloc(ausgabe,0);
	return "checked in";
      case '2':
	mem_realloc(ausgabe,0);
	return "not checked in, but seeking a network";
      case '3':
	mem_realloc(ausgabe,0);
	return "check-in denied by network";
      case '5':
	mem_realloc(ausgabe,0);
	return "registered, roaming";
      default:
	mem_realloc(ausgabe,0);
	return "unknown status";
      }
    }
  }
  return NULL;
}

char* get_gprs_status (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;

  ausgabe=get_value("AT+CGREG?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGREG: ",8)) {
    if (areacode_p!=NULL) {
      *areacode_p=0;
    }
    if (areacode_p!=NULL) {
      *cellid_p=0;
    }
    index1=index(ausgabe+8,'"');
    if (index1!=NULL) {
      index2=index(++index1,'"');
      if (index2!=NULL && areacode_p!=NULL) {
	*areacode_p=hexstr2int(index1,index2-index1);
	index1=index(index2+1,'"');
	if (index1!=NULL) {
	  index2=index(++index1,'"');
	  if (index2!=NULL && cellid_p!=NULL) {
	    *cellid_p=hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe+8,','))!=NULL) {
      switch (*(index1+1)) {
      case '0':
	mem_realloc(ausgabe,0);
	return "not registered, not searching";
      case '1':
	mem_realloc(ausgabe,0);
	return "registered, home network";
      case '2':
	mem_realloc(ausgabe,0);
	return "not registered, searching";
      case '3':
	mem_realloc(ausgabe,0);
	return "registration denied by network";
      case '5':
	mem_realloc(ausgabe,0);
	return "registered, roaming";
      default:
	mem_realloc(ausgabe,0);
	return "unknown status";
      }
    }
  }
  return NULL;
}

int get_gprs_attach (void) {
  char* ausgabe;

  ausgabe=get_value("AT+CGATT?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGATT: ",8)) {
    switch (ausgabe[8]) {
    case '0':
      mem_realloc(ausgabe,0);
      return 0;
    case '1':
      mem_realloc(ausgabe,0);
      return 1;
    }
  }
  return -1;
}

char* get_gprs_class (void) {
  char* ausgabe;

  ausgabe=get_value("AT+CGCLASS?");
  if (ausgabe!=NULL && !strncmp(ausgabe,"+CGCLASS: ",10)) {
    memmove(ausgabe,ausgabe+10,strlen(ausgabe+10)+1);
    return mem_realloc(ausgabe,strlen(ausgabe)+1);
  }
  return NULL;
}

void info(char* file){
  char *temp;
  char* work;
  int i,j;
  struct tm* ltime;

  FILE* fd;
  if (!strlen(file) || !strcmp(file,"-")) {
    fd=stdout;
  } else {
    fd=fdopen(open_myFile_rw(file),"w+");
  }

  fprintf(fd,"Phone related information:\n");
  fprintf(fd,"Vendor:\t\t%s\n",temp=get_vendor());
  mem_realloc(temp,0);
  fprintf(fd,"Model:\t\t%s\n",temp=get_model());
  mem_realloc(temp,0);
  fprintf(fd,"Revision:\t%s\n",temp=get_revision());
  mem_realloc(temp,0);
  fprintf(fd,"IMEI:\t\t%s\n",temp=get_phoneserial());
  mem_realloc(temp,0);
  fprintf(fd,"Battery:\t%s%%\n",temp=get_battery());
  mem_realloc(temp,0);
  fprintf(fd,"Charset:\t%s\n",temp=get_charset());
  temp=mem_realloc(temp,81);
  ltime=get_time();
  if (ltime!=NULL && strftime(temp,80,nl_langinfo(D_T_FMT),ltime)) {
    fprintf(fd,"Time:\t\t%s\n",temp);
    mem_realloc(ltime,0);
  }
  mem_realloc(temp,0);

  fprintf(fd,"\nSIM card related information:\n");
  fprintf(fd,"IMSI:\t\t%s\n",temp=get_simserial());
  mem_realloc(temp,0);
  fprintf(fd,"card ID:\t%s\n",temp=get_simid());
  mem_realloc(temp,0);

  fprintf(fd,"\nNetwork related information:\n");
  temp=get_netstatus(&i,&j);
  if (temp!=NULL) {
    fprintf(fd,"Status:\t\t%s\n",temp);
    if (i) {
      fprintf(fd,"Area code:\t%04X\n",i);
      if (j) {
	fprintf(fd,"Cell ID:\t%04X\n",j);
      }
    }
  }
  temp=get_operator();
  if (temp!=NULL) {
    fprintf(fd,"Operator:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  temp=get_smsc();
  if (temp!=NULL) {
    fprintf(fd,"SMS Server:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  if ((i=get_signal(&j)) >= 0) {
    fprintf(fd,"Signal:\t\t%d dBm\n",-i);
    if (j >= 0) {
      if (j <= 128) {
	fprintf(fd,"BER:\t\t<=%d,%d%%\n",j/10,j%10);
      } else {
	fprintf(fd,"BER:\t\t>12,8%%\n");
      }
    }
  }
  temp=get_gprs_class();
  if (temp!=NULL) {
    fprintf(fd,"GPRS class:\t");
    temp=(char *)strtok(temp,"()\",");
    while (temp!=0) {
      fprintf(fd,"%s",temp);
      temp=(char *)strtok(NULL,"()\",");
      if (temp!=0) {
	fprintf(fd,", ");
      }
    }
    fprintf(fd,"\n");
  }
  mem_realloc(temp,0);
  temp=get_gprs_status(NULL,NULL);
  if (temp!=NULL) {
    fprintf(fd,"GRPS status:\t%s",temp);
    i=get_gprs_attach();
    if (i==0) {
      fprintf(fd,", detached\n");
    } else if (i==1) {
      fprintf(fd,", attached\n");
    }
  }

  fprintf(fd,"\nSlot information (readable, not all may be writeable/deletable):\n");
  temp=get_filetypes(1);
  if (temp!=NULL) {
    temp=(char *)strtok(temp,"()\",");
    while(temp!=0){
      if (!strcmp(temp,"bmp")) {
	fprintf(fd,"Bitmap ");
      } else if (!strcmp(temp,"mid")) {
	fprintf(fd,"Midi ");
      } else if (!strcmp(temp,"vcs")) {
	fprintf(fd,"vCal ");
      } else if (!strcmp(temp,"vcf")) {
	fprintf(fd,"vCard ");
      }
      fprintf(fd,"(%s):\t%s\n",temp,strtok(NULL,"()\","));
      temp=(char *)strtok(NULL,"()\",");
    }
  }

  temp=get_pbtypes();
  if (temp!=NULL) {
    fprintf(fd,"Phonebooks:\t");
    temp=(char *)strtok(temp,"()\",");
    while(temp!=0){
      fprintf(fd,"%s",temp);
      temp=(char *)strtok(NULL,"()\",");
      if (temp!=0) {
	fprintf(fd,", ");
      }
    }
    fprintf(fd,"\n");
  }
    
  temp=get_smsmemtypes();
  if (temp!=NULL) {
    fprintf(fd,"SMS storages:\t");
    work=(char *)malloc(strlen(temp)+1);
    memset(work,0,strlen(temp)+1);
    temp=(char *)strtok(temp,"()\",");
    while(temp!=0){
      if (strstr(work,temp)==NULL) {
	if (strlen(work)) {
	  sprintf(&work[strlen(work)],", ");
	}
	sprintf(&work[strlen(work)],temp);
      }
      temp=(char *)strtok(NULL,"()\",");
    }
    fprintf(fd,"%s\n",work);
    free(work);
  }

  if (strlen(file) && strcmp(file,"-")) {
    fclose(fd);
  }
}
