/*--------------------------------*-C-*---------------------------------*
 * File:	init.c
 *----------------------------------------------------------------------*
 * $Id: init.c,v 1.8 2000/03/26 07:47:01 mason Exp $
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (C) 1992      John Bovey, University of Kent at Canterbury <jdb@ukc.ac.uk>
 *				- original version
 * Copyright (C) 1994      Robert Nation <nation@rocket.sanders.lockheed.com>
 * 				- extensive modifications
 * Copyright (C) 1998,1999 Geoff Wing <gcw@pobox.com>
 * 				- extensive modifications
 * Copyright (C) 1999      D J Hawkey Jr <hawkeyd@visi.com>
 *				- QNX support
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *---------------------------------------------------------------------*/
/*
 * Initialisation routines.
 */

#include "../config.h"		/* NECESSARY */
#include "rxvt.h"		/* NECESSARY */
#include "init.h"

static ttymode_t tio;

static const char * const def_colorName[] =
{
    COLOR_FOREGROUND,
    COLOR_BACKGROUND,
/* low-intensity colors */
    "Black",			/* 0: black             (#000000) */
#ifndef NO_BRIGHTCOLOR
    "Red3",			/* 1: red               (#CD0000) */
    "Green3",			/* 2: green             (#00CD00) */
    "Yellow3",			/* 3: yellow            (#CDCD00) */
    "Blue3",			/* 4: blue              (#0000CD) */
    "Magenta3",			/* 5: magenta           (#CD00CD) */
    "Cyan3",			/* 6: cyan              (#00CDCD) */
    "AntiqueWhite",		/* 7: white             (#FAEBD7) */
/* high-intensity colors */
    "Grey25",			/* 8: bright black      (#404040) */
#endif				/* NO_BRIGHTCOLOR */
    "Red",			/* 1/9: bright red      (#FF0000) */
    "Green",			/* 2/10: bright green   (#00FF00) */
    "Yellow",			/* 3/11: bright yellow  (#FFFF00) */
    "Blue",			/* 4/12: bright blue    (#0000FF) */
    "Magenta",			/* 5/13: bright magenta (#FF00FF) */
    "Cyan",			/* 6/14: bright cyan    (#00FFFF) */
    "White",			/* 7/15: bright white   (#FFFFFF) */
#ifndef NO_CURSORCOLOR
    COLOR_CURSOR_BACKGROUND,
    COLOR_CURSOR_FOREGROUND,
#endif				/* ! NO_CURSORCOLOR */
    NULL,			/* Color_pointer                  */
    NULL			/* Color_border                   */
#ifndef NO_BOLDUNDERLINE
  , NULL,			/* Color_BD                       */
    NULL			/* Color_UL                       */
#endif				/* ! NO_BOLDUNDERLINE */
#ifdef KEEP_SCROLLCOLOR
  , COLOR_SCROLLBAR,
    COLOR_SCROLLTROUGH
#endif				/* KEEP_SCROLLCOLOR */
};

#ifdef MULTICHAR_SET
/* Multicharacter font names, roman fonts sized to match */
static const char * const def_mfontName[] =
{
    MFONT_LIST
};
#endif				/* MULTICHAR_SET */

static const char * const def_fontName[] =
{
    NFONT_LIST
};

/*----------------------------------------------------------------------*/
/* substitute system functions */
#if defined(__svr4__) && ! defined(_POSIX_VERSION)
/* INTPROTO */
int
getdtablesize(void)
{
    struct rlimit   rlim;

    getrlimit(RLIMIT_NOFILE, &rlim);
    return rlim.rlim_cur;
}
#endif
/*----------------------------------------------------------------------*/
/* EXTPROTO */
void
init_vars(void)
{
    TermWin.ncol = 80;
    TermWin.nrow = 24;
    TermWin.mapped = 0;
    TermWin.int_bwidth = INTERNALBORDERWIDTH;
    TermWin.ext_bwidth = EXTERNALBORDERWIDTH;
    TermWin.saveLines = SAVELINES;
    TermWin.fontset = NULL;
    scrollBar.win = 0;
#if (MENUBAR_MAX)
    menuBar.win = 0;
#endif
#ifndef NO_NEW_SELECTION
    selection_style = NEW_SELECT;
#else
    selection_style = OLD_SELECT;
#endif
}

/*----------------------------------------------------------------------*/
/* EXTPROTO */
const char    **
init_resources(int argc, const char * const *argv)
{
    int             i, r_argc;
    char           *val;
    const char     *tmp;
    const char    **cmd_argv, **r_argv, **t2;

/*
 * Look for -exec option.  Find => split and make cmd_argv[] of command args
 */
    for (r_argc = 0; r_argc < argc; r_argc++)
	if (!STRCMP(argv[r_argc], "-e") || !STRCMP(argv[r_argc], "-exec"))
	    break;
    r_argv = (const char **) MALLOC(sizeof(char *) * (r_argc + 1));
    for (i = 0; i < r_argc; i++)
	r_argv[i] = (const char *) argv[i];
    r_argv[i] = NULL;
    if (r_argc == argc)
	cmd_argv = NULL;
    else {
	cmd_argv = (const char **) MALLOC(sizeof(char *) * (argc - r_argc));
	for (i = 0; i < argc - r_argc - 1; i++)
	    cmd_argv[i] = (const char *) argv[i + r_argc + 1];
	cmd_argv[i] = NULL;
    }

/* clear all resources */
    for (i = 0, t2 = (const char **)&rs; i < sizeof(rs) / sizeof(char *); i++)
	t2[i] = NULL;

    rs.name = r_basename(argv[0]);
    if (cmd_argv != NULL && cmd_argv[0] != NULL)
	rs.iconName = rs.title = r_basename(cmd_argv[0]);
/*
 * Open display, get options/resources and create the window
 */
    if ((rs.display_name = getenv("DISPLAY")) == NULL)
	rs.display_name = ":0";

    get_options(r_argc, r_argv);

    FREE(r_argv);

#ifdef LOCAL_X_IS_UNIX
    if (rs.display_name[0] == ':') {
	val = MALLOC(5 + STRLEN(rs.display_name));
	STRCPY(val, "unix");
	STRCAT(val, rs.display_name);
	Xdisplay = XOpenDisplay(val);
	FREE(val);
    }
#endif
	
    if (Xdisplay == NULL
	&& (Xdisplay = XOpenDisplay(rs.display_name)) == NULL) {
	print_error("can't open display %s", rs.display_name);
	exit(EXIT_FAILURE);
    }
#ifdef INEXPENSIVE_LOCAL_X_CALLS
    /* it's hard to determine further if we're on a local display or not */
    if (rs.display_name[0] == ':'
	|| STRNCMP(rs.display_name, "unix:", 5))
	display_is_local = 1;
    else
	display_is_local = 0;
#endif

    extract_resources(Xdisplay, rs.name);

#if ! defined(XTERM_SCROLLBAR) && ! defined(NEXT_SCROLLBAR)
    if (!(Options & Opt_scrollBar_floating))
	sb_shadow = SHADOW;
#endif

/*
 * set any defaults not already set
 */
    if (!rs.title)
	rs.title = rs.name;
    if (!rs.iconName)
	rs.iconName = rs.title;
    if (rs.saveLines && (i = atoi(rs.saveLines)) >= 0)
	TermWin.saveLines = (int16_t) i;
#ifndef NO_FRILLS
    if (rs.int_bwidth && (i = atoi(rs.int_bwidth)) >= 0)
	TermWin.int_bwidth = (int16_t) i;
    if (rs.ext_bwidth && (i = atoi(rs.ext_bwidth)) >= 0)
	TermWin.ext_bwidth = (int16_t) i;
#endif

/* no point having a scrollbar without having any scrollback! */
    if (!TermWin.saveLines)
	Options &= ~Opt_scrollBar;

#ifdef PRINTPIPE
    if (!rs.print_pipe)
	rs.print_pipe = PRINTPIPE;
#endif
    if (!rs.cutchars)
	rs.cutchars = CUTCHARS;
#ifndef NO_BACKSPACE_KEY
    if (!rs.backspace_key)
# ifdef DEFAULT_BACKSPACE
	key_backspace = DEFAULT_BACKSPACE;
# else
	key_backspace = "DEC";	/* can toggle between \033 or \177 */
# endif
    else {
	val = strdup(rs.backspace_key);
	Str_trim(val);
	Str_escaped(val);
	key_backspace = val;
    }
#endif
#ifndef NO_DELETE_KEY
    if (!rs.delete_key)
# ifdef DEFAULT_DELETE
	key_delete = DEFAULT_DELETE;
# else
	key_delete = "\033[3~";
# endif
    else {
	val = strdup(rs.delete_key);
	Str_trim(val);
	Str_escaped(val);
	key_delete = val;
    }
#endif

    if (rs.selectstyle) {
	if (STRNCASECMP(rs.selectstyle, "oldword", 7) == 0)
	    selection_style = OLD_WORD_SELECT;
#ifndef NO_OLD_SELECTION
	else if (STRNCASECMP(rs.selectstyle, "old", 3) == 0)
	    selection_style = OLD_SELECT;
#endif
    }

#ifndef NO_BOLDFONT
    if (rs.font[0] == NULL && rs.boldFont != NULL) {
	rs.font[0] = rs.boldFont;
	rs.boldFont = NULL;
    }
#endif
    for (i = 0; i < NFONTS; i++) {
	if (!rs.font[i])
	    rs.font[i] = def_fontName[i];
#ifdef MULTICHAR_SET
	if (!rs.mfont[i])
	    rs.mfont[i] = def_mfontName[i];
#endif
    }

#ifdef XTERM_REVERSE_VIDEO
/* this is how xterm implements reverseVideo */
    if (Options & Opt_reverseVideo) {
	if (!rs.color[Color_fg])
	    rs.color[Color_fg] = def_colorName[Color_bg];
	if (!rs.color[Color_bg])
	    rs.color[Color_bg] = def_colorName[Color_fg];
    }
#endif

    for (i = 0; i < NRS_COLORS; i++)
	if (!rs.color[i])
	    rs.color[i] = def_colorName[i];

#ifndef XTERM_REVERSE_VIDEO
/* this is how we implement reverseVideo */
    if (Options & Opt_reverseVideo)
	SWAP_IT(rs.color[Color_fg], rs.color[Color_bg], tmp);
#endif

/* convenient aliases for setting fg/bg to colors */
    color_aliases(Color_fg);
    color_aliases(Color_bg);
#ifndef NO_CURSORCOLOR
    color_aliases(Color_cursor);
    color_aliases(Color_cursor2);
#endif				/* NO_CURSORCOLOR */
    color_aliases(Color_pointer);
    color_aliases(Color_border);
#ifndef NO_BOLDUNDERLINE
    color_aliases(Color_BD);
    color_aliases(Color_UL);
#endif				/* NO_BOLDUNDERLINE */

    return cmd_argv;
}

/*----------------------------------------------------------------------*/
/* EXTPROTO */
void
init_env(void)
{
    int             i;
    unsigned int    u;
    char           *val;
/* these don't need to be static but do so to placate some mem checkers */
    static char    *env_windowid, *env_display, *env_term;

#ifdef DISPLAY_IS_IP
/* Fixup display_name for export over pty to any interested terminal
 * clients via "ESC[7n" (e.g. shells).  Note we use the pure IP number
 * (for the first non-loopback interface) that we get from
 * network_display().  This is more "name-resolution-portable", if you
 * will, and probably allows for faster x-client startup if your name
 * server is beyond a slow link or overloaded at client startup.  Of
 * course that only helps the shell's child processes, not us.
 *
 * Giving out the display_name also affords a potential security hole
 */
    rs.display_name = (const char *) val = network_display(rs.display_name);
    if (val == NULL)
#endif				/* DISPLAY_IS_IP */
	val = XDisplayString(Xdisplay);
    if (rs.display_name == NULL)
	rs.display_name = val;	/* use broken `:0' value */

    i = STRLEN(val);
    env_display = MALLOC((i + 9) * sizeof(char));
    sprintf(env_display, "DISPLAY=%s", val);

    /* avoiding the math library:
     * i = (int)(ceil(log10((unsigned int)TermWin.parent[0]))) */
    for (i = 0, u = (unsigned int)TermWin.parent[0]; u; u /= 10, i++);
    MAX_IT(i, 1);
    env_windowid = MALLOC((i + 10) * sizeof(char));
    sprintf(env_windowid, "WINDOWID=%u", (unsigned int)TermWin.parent[0]);

/* add entries to the environment:
 * @ DISPLAY:   in case we started with -display
 * @ WINDOWID:  X window id number of the window
 * @ COLORTERM: terminal sub-name and also indicates its color
 * @ TERM:      terminal name
 * @ TERMINFO:	path to terminfo directory
 */
    putenv(env_display);
    putenv(env_windowid);
#ifdef RXVT_TERMINFO
    putenv("TERMINFO=" RXVT_TERMINFO);
#endif
    if (Xdepth <= 2)
	putenv("COLORTERM=" COLORTERMENV "-mono");
    else
	putenv("COLORTERM=" COLORTERMENVFULL);
    if (rs.term_name != NULL) {
	env_term = MALLOC((STRLEN(rs.term_name) + 6) * sizeof(char));
	sprintf(env_term, "TERM=%s", rs.term_name);
	putenv(env_term);
    } else
	putenv("TERM=" TERMENV);

#ifdef HAVE_UNSETENV
/* avoid passing old settings and confusing term size */
    unsetenv("LINES");
    unsetenv("COLUMNS");
    unsetenv("TERMCAP");	/* terminfo should be okay */
#endif				/* HAVE_UNSETENV */
}


/*----------------------------------------------------------------------*/
/*
 * This is more or less stolen straight from XFree86 xterm.
 * This should support all European type languages.
 */
/* EXTPROTO */
void
init_xlocale(void)
{
    char           *locale = NULL;

#if !defined(NO_XSETLOCALE) || !defined(NO_SETLOCALE)
    locale = setlocale(LC_CTYPE, "");
#endif
#ifdef USE_XIM
    if (locale == NULL)
	print_error("Setting locale failed.");
    else {
    /* To avoid Segmentation Fault in C locale */
	setTermFontSet(0);
# ifdef MULTICHAR_SET
	if (STRCMP(locale, "C"))
# endif
	    XRegisterIMInstantiateCallback(Xdisplay, NULL, NULL, NULL,
					   IMInstantiateCallback, NULL);
    }
#endif
}
/*----------------------------------------------------------------------*/
/* EXTPROTO */
void
init_command(const char * const *argv)
{
/*
 * Initialize the command connection.
 * This should be called after the X server connection is established.
 */

    xa_compound_text = XInternAtom(Xdisplay, "COMPOUND_TEXT", False);
    xa_multiple = XInternAtom(Xdisplay, "MULTIPLE", False);
    xa_targets = XInternAtom(Xdisplay, "TARGETS", False);
    xa_text = XInternAtom(Xdisplay, "TEXT", False);
    xa_timestamp = XInternAtom(Xdisplay, "TIMESTAMP", False);
/* Enable delete window protocol */
    wmDeleteWindow = XInternAtom(Xdisplay, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(Xdisplay, TermWin.parent[0], &wmDeleteWindow, 1);

#ifdef OFFIX_DND
/* Enable OffiX Dnd (drag 'n' drop) protocol */
    DndProtocol = XInternAtom(Xdisplay, "DndProtocol", False);
    DndSelection = XInternAtom(Xdisplay, "DndSelection", False);
#endif				/* OFFIX_DND */
#ifdef TRANSPARENT
    xrootpmapid = XInternAtom(Xdisplay, "_XROOTPMAP_ID", False);
#endif

/* get number of available file descriptors */
#if defined(_POSIX_VERSION) || ! defined(__svr4__)
    num_fds = (int)sysconf(_SC_OPEN_MAX);
#else
    num_fds = getdtablesize();
#endif

#ifdef META8_OPTION
    meta_char = (Options & Opt_meta8 ? 0x80 : 033);
#endif
    get_ourmods();
    if (!(Options & Opt_scrollTtyOutput))
	PrivateModes |= PrivMode_TtyOutputInh;
    if (Options & Opt_scrollKeypress)
	PrivateModes |= PrivMode_Keypress;
#ifndef NO_BACKSPACE_KEY
    if (STRCMP(key_backspace, "DEC") == 0)
	PrivateModes |= PrivMode_HaveBackSpace;
#endif
/* add value for scrollBar */
    if (scrollbar_visible()) {
	PrivateModes |= PrivMode_scrollBar;
	SavedModes |= PrivMode_scrollBar;
    }
    if (menubar_visible()) {
	PrivateModes |= PrivMode_menuBar;
	SavedModes |= PrivMode_menuBar;
    }
    greek_init();

    Xfd = XConnectionNumber(Xdisplay);

    if ((cmd_fd = run_command(argv)) < 0) {
	print_error("aborting");
	exit(EXIT_FAILURE);
    }
}
/*----------------------------------------------------------------------*/
/* INTPROTO */
void
Get_Colours(void)
{
    int             i;

    for (i = 0; i < (Xdepth <= 2 ? 2 : NRS_COLORS); i++) {
	const char     *msg = "can't load color \"%s\"";
	XColor          xcol;

	if (!rs.color[i])
	    continue;

	if (!XParseColor(Xdisplay, Xcmap, rs.color[i], &xcol)
	    || !rXAllocColor(Xdisplay, Xcmap, &xcol)) {
	    print_error(msg, rs.color[i]);
#ifndef XTERM_REVERSE_VIDEO
	    if (i < 2 && (Options & Opt_reverseVideo)) {
		rs.color[i] = def_colorName[!i];
	    } else
#endif
	        rs.color[i] = def_colorName[i];
	    if (!rs.color[i])
		continue;
	    if (!XParseColor(Xdisplay, Xcmap, rs.color[i], &xcol)
		|| !rXAllocColor(Xdisplay, Xcmap, &xcol)) {
		print_error(msg, rs.color[i]);
		switch (i) {
		case Color_fg:
		case Color_bg:
		/* fatal: need bg/fg color */
		    print_error("aborting");
		    exit(EXIT_FAILURE);
		    /* NOTREACHED */
		    break;
#ifndef NO_CURSORCOLOR
		case Color_cursor2:
		    xcol.pixel = PixColors[Color_fg];
		    break;
#endif				/* ! NO_CURSORCOLOR */
		case Color_pointer:
		    xcol.pixel = PixColors[Color_fg];
		    break;
		default:
		    xcol.pixel = PixColors[Color_bg];	/* None */
		    break;
		}
	    }
	}
	PixColors[i] = xcol.pixel;
    }

    if (Xdepth <= 2 || !rs.color[Color_pointer])
	PixColors[Color_pointer] = PixColors[Color_fg];
    if (Xdepth <= 2 || !rs.color[Color_border])
	PixColors[Color_border] = PixColors[Color_fg];

/*
 * get scrollBar/menuBar shadow colors
 *
 * The calculations of topShadow/bottomShadow values are adapted
 * from the fvwm window manager.
 */
#ifdef KEEP_SCROLLCOLOR
    if (Xdepth <= 2) {		/* Monochrome */
	PixColors[Color_scroll] = PixColors[Color_fg];
	PixColors[Color_topShadow] = PixColors[Color_bg];
	PixColors[Color_bottomShadow] = PixColors[Color_bg];
    } else {
	XColor          xcol, white;

    /* bottomShadowColor */
	xcol.pixel = PixColors[Color_scroll];
	XQueryColor(Xdisplay, Xcmap, &xcol);

	xcol.red = ((xcol.red) / 2);
	xcol.green = ((xcol.green) / 2);
	xcol.blue = ((xcol.blue) / 2);

	if (!rXAllocColor(Xdisplay, Xcmap, &xcol)) {
	    print_error("can't allocate %s", "Color_bottomShadow");
	    xcol.pixel = PixColors[minCOLOR];
	}
	PixColors[Color_bottomShadow] = xcol.pixel;

    /* topShadowColor */
# ifdef PREFER_24BIT
	white.red = white.green = white.blue = (unsigned short) ~0;
	rXAllocColor(Xdisplay, Xcmap, &white);
/*        XFreeColors(Xdisplay, Xcmap, &white.pixel, 1, ~0); */
# else
	white.pixel = WhitePixel(Xdisplay, Xscreen);
	XQueryColor(Xdisplay, Xcmap, &white);
# endif

	xcol.pixel = PixColors[Color_scroll];
	XQueryColor(Xdisplay, Xcmap, &xcol);

	xcol.red = max((white.red / 5), xcol.red);
	xcol.green = max((white.green / 5), xcol.green);
	xcol.blue = max((white.blue / 5), xcol.blue);

	xcol.red = min(white.red, (xcol.red * 7) / 5);
	xcol.green = min(white.green, (xcol.green * 7) / 5);
	xcol.blue = min(white.blue, (xcol.blue * 7) / 5);

	if (!rXAllocColor(Xdisplay, Xcmap, &xcol)) {
	    print_error("can't allocate %s", "Color_topShadow");
	    xcol.pixel = PixColors[Color_White];
	}
	PixColors[Color_topShadow] = xcol.pixel;
    }
#endif				/* KEEP_SCROLLCOLOR */
}
/*----------------------------------------------------------------------*/
/* color aliases, fg/bg bright-bold */
/* INTPROTO */
void
color_aliases(int idx)
{
    if (rs.color[idx] && isdigit(*(rs.color[idx]))) {
	int             i = atoi(rs.color[idx]);

	if (i >= 8 && i <= 15) {	/* bright colors */
	    i -= 8;
#ifndef NO_BRIGHTCOLOR
	    rs.color[idx] = rs.color[minBrightCOLOR + i];
	    return;
#endif
	}
	if (i >= 0 && i <= 7)	/* normal colors */
	    rs.color[idx] = rs.color[minCOLOR + i];
    }
}
/*----------------------------------------------------------------------*/
/*
 * Probe the modifier keymap to get the Meta (Alt) and Num_Lock settings
 * Use resource ``modifier'' to override the modifier
 */
/* INTPROTO */
void
get_ourmods(void)
{
    int             i, j, k, m;
    int             got_meta, got_numlock;
    XModifierKeymap *map;
    KeyCode        *kc;
    unsigned int    modmasks[] =
			{ Mod1Mask, Mod2Mask, Mod3Mask, Mod4Mask, Mod5Mask };

    got_meta = got_numlock = m = 0;
    if (rs.modifier
	&& rs.modifier[0] == 'm'
	&& rs.modifier[1] == 'o'
	&& rs.modifier[2] == 'd'
	&& rs.modifier[3] >= '1' && rs.modifier[3] <= '5'
	&& !rs.modifier[4]) {
	ModMetaMask = modmasks[(rs.modifier[3] - '1')];
	got_meta = 1;
    }
    map = XGetModifierMapping(Xdisplay);
    kc = map->modifiermap;
    for (i = 3; i < 8; i++) {
	k = i * map->max_keypermod;
	for (j = 0; j < map->max_keypermod; j++, k++) {
	    if (kc[k] == 0)
		break;
	    switch (XKeycodeToKeysym(Xdisplay, kc[k], 0)) {
	    case XK_Num_Lock:
		if (!got_numlock) {
		    ModNumLockMask = modmasks[i - 3];
		    got_numlock = 1;
		}
		break;
	    case XK_Meta_L:
	    case XK_Meta_R:
		if (rs.modifier
		    && !STRNCASECMP(rs.modifier, "meta", 4)) {
		    ModMetaMask = modmasks[i - 3];
		    got_meta = 1;
		    break;
		}
	    /* FALLTHROUGH */
	    case XK_Alt_L:
	    case XK_Alt_R:
		if (rs.modifier
		    && !STRNCASECMP(rs.modifier, "alt", 3)) {
		    ModMetaMask = modmasks[i - 3];
		    got_meta = 1;
		    break;
		}
		m = modmasks[i - 3];
		break;
	    case XK_Super_L:
	    case XK_Super_R:
		if (rs.modifier
		    && !STRNCASECMP(rs.modifier, "super", 5)) {
		    ModMetaMask = modmasks[i - 3];
		    got_meta = 1;
		}
		break;
	    case XK_Hyper_L:
	    case XK_Hyper_R:
		if (rs.modifier
		    && !STRNCASECMP(rs.modifier, "hyper", 5)) {
		    ModMetaMask = modmasks[i - 3];
		    got_meta = 1;
		}
	    /* FALLTHROUGH */
	    default:
		break;
	    }
	}
	if (got_meta && got_numlock)
	    break;
    }
    XFreeModifiermap(map);
    if (!got_meta && m)
	ModMetaMask = m;
}

/*----------------------------------------------------------------------*/
/* Create_Windows() - Open and map the window */
/* EXTPROTO */
void
Create_Windows(int argc, const char * const *argv)
{
    Cursor          cursor;
    XClassHint      classHint;
    XWMHints        wmHint;
#ifdef PREFER_24BIT
    XSetWindowAttributes attributes;

    Xdepth = DefaultDepth(Xdisplay, Xscreen);
    Xcmap = DefaultColormap(Xdisplay, Xscreen);
    Xvisual = DefaultVisual(Xdisplay, Xscreen);
/*
 * If depth is not 24, look for a 24bit visual.
 */
    if (Xdepth != 24) {
	XVisualInfo     vinfo;

	if (XMatchVisualInfo(Xdisplay, Xscreen, 24, TrueColor, &vinfo)) {
	    Xdepth = 24;
	    Xvisual = vinfo.visual;
	    Xcmap = XCreateColormap(Xdisplay, RootWindow(Xdisplay, Xscreen),
				    Xvisual, AllocNone);
	}
    }
#endif

/* grab colors before netscape does */
    Get_Colours();

    change_font(1, NULL);
    szhints_set();

/* parent window - reverse video so we can see placement errors
 * sub-window placement & size in resize_subwindows()
 */

#ifdef PREFER_24BIT
    attributes.background_pixel = PixColors[Color_fg];
    attributes.border_pixel = PixColors[Color_border];
    attributes.colormap = Xcmap;
    TermWin.parent[0] = XCreateWindow(Xdisplay, Xroot,
				   szHint.x, szHint.y,
				   szHint.width, szHint.height,
				   TermWin.ext_bwidth,
				   Xdepth, InputOutput,
				   Xvisual,
				   CWBackPixel | CWBorderPixel | CWColormap,
				   &attributes);
#else
    TermWin.parent[0] = XCreateSimpleWindow(Xdisplay, Xroot,
					 szHint.x, szHint.y,
					 szHint.width, szHint.height,
					 TermWin.ext_bwidth,
					 PixColors[Color_border],
					 PixColors[Color_fg]);
#endif
    xterm_seq(XTerm_title, rs.title);
    xterm_seq(XTerm_iconName, rs.iconName);
/* ignore warning about discarded `const' */
    classHint.res_name = (char *) rs.name;
    classHint.res_class = (char *) APL_CLASS;
    wmHint.input = True;
    wmHint.initial_state = (Options & Opt_iconic ? IconicState : NormalState);
    wmHint.window_group = TermWin.parent[0];
    wmHint.flags = (InputHint | StateHint | WindowGroupHint);

    XSetWMProperties(Xdisplay, TermWin.parent[0], NULL, NULL, (char **) argv,
		     argc, &szHint, &wmHint, &classHint);

    XSelectInput(Xdisplay, TermWin.parent[0],
		 (KeyPressMask | FocusChangeMask
		  | VisibilityChangeMask
		  | StructureNotifyMask));

/* vt cursor: Black-on-White is standard, but this is more popular */
    TermWin_cursor = XCreateFontCursor(Xdisplay, XC_xterm);
    {
	XColor          fg, bg;

	fg.pixel = PixColors[Color_pointer];
	XQueryColor(Xdisplay, Xcmap, &fg);
	bg.pixel = PixColors[Color_bg];
	XQueryColor(Xdisplay, Xcmap, &bg);
	XRecolorCursor(Xdisplay, TermWin_cursor, &fg, &bg);
    }

/* cursor (menuBar/scrollBar): Black-on-White */
    cursor = XCreateFontCursor(Xdisplay, XC_left_ptr);

/* the vt window */
    TermWin.vt = XCreateSimpleWindow(Xdisplay, TermWin.parent[0],
				     0, 0,
				     szHint.width, szHint.height,
				     0,
				     PixColors[Color_fg],
				     PixColors[Color_bg]);
#ifdef DEBUG_X
    XStoreName(Xdisplay, TermWin.vt, "vt window");
#endif

    XDefineCursor(Xdisplay, TermWin.vt, TermWin_cursor);
    XSelectInput(Xdisplay, TermWin.vt,
		 (ExposureMask | ButtonPressMask | ButtonReleaseMask |
		  Button1MotionMask | Button3MotionMask));

/* scrollBar: size doesn't matter */
    scrollBar.win = XCreateSimpleWindow(Xdisplay, TermWin.parent[0],
					0, 0,
					1, 1,
					0,
					PixColors[Color_fg],
					PixColors[Color_bg]);
#ifdef DEBUG_X
    XStoreName(Xdisplay, scrollBar.win, "scrollbar");
#endif

    XDefineCursor(Xdisplay, scrollBar.win, cursor);
    XSelectInput(Xdisplay, scrollBar.win,
		 (ExposureMask | ButtonPressMask | ButtonReleaseMask |
		  Button1MotionMask | Button2MotionMask | Button3MotionMask));

    { /* ONLYIF MENUBAR */
	create_menuBar(cursor);
    }

#ifdef XPM_BACKGROUND
    if (rs.backgroundPixmap != NULL && !(Options & Opt_transparent)) {
	const char     *p = rs.backgroundPixmap;

	if ((p = STRCHR(p, ';')) != NULL) {
	    p++;
	    scale_pixmap(p);
	}
	set_bgPixmap(rs.backgroundPixmap);
	scr_touch(True);
    }
# ifdef XPM_BUFFERING
    else {
	set_bgPixmap("");
	scr_touch(True);
    }
# endif
#endif

/* graphics context for the vt window */
    {
	XGCValues       gcvalue;

	gcvalue.font = TermWin.font->fid;
	gcvalue.foreground = PixColors[Color_fg];
	gcvalue.background = PixColors[Color_bg];
	gcvalue.graphics_exposures = 0;
	TermWin.gc = XCreateGC(Xdisplay, TermWin.vt,
			       GCForeground | GCBackground |
			       GCFont | GCGraphicsExposures,
			       &gcvalue);
    }
}
/*----------------------------------------------------------------------*/
/*
 * Run the command in a subprocess and return a file descriptor for the
 * master end of the pseudo-teletype pair with the command talking to
 * the slave.
 */
/* INTPROTO */
int
run_command(const char * const *argv)
{
    int             i, cfd;

    if ((cfd = get_ptytty()) < 0)
	return -1;

/* install exit handler for cleanup */
#ifdef HAVE_ATEXIT
    atexit(clean_exit);
#else
# if defined (__sun__)
    on_exit(clean_exit, NULL);	/* non-ANSI exit handler */
# endif
#endif

/*
 * Close all unused file descriptors.
 * We don't want them, we don't need them.
 */
    for (i = 0; i < num_fds; i++) {
	if (i == STDERR_FILENO || i == cfd || i == tty_fd || i == Xfd)
	    continue;
	close(i);
    }

    signal(SIGHUP, Exit_signal);
#ifndef __svr4__
    signal(SIGINT, Exit_signal);
#endif
    signal(SIGQUIT, Exit_signal);
    signal(SIGTERM, Exit_signal);
    signal(SIGCHLD, Child_signal);

/* need to trap SIGURG for SVR4 (Unixware) rlogin */
/* signal (SIGURG, SIG_DFL); */

#ifndef __QNX__
/* spin off the command interpreter */
    switch (cmd_pid = fork()) {
    case -1:
	print_error("can't fork");
	return -1;
    case 0:
	close(cfd);		/* only keep tty_fd open */
	close(Xfd);
#ifndef DEBUG_TTY
	close(STDERR_FILENO);
#endif
	run_child(argv);
	exit(EXIT_FAILURE);
    /* NOTREACHED */
    default:
	close(tty_fd);		/* keep STDERR_FILENO, cmd_fd, Xfd open */
	break;
    }
#else				/* __QNX__ uses qnxspawn() */
    fchmod(tty_fd, 0622);
    fcntl(tty_fd, F_SETFD, FD_CLOEXEC);
    fcntl(cmd_fd, F_SETFD, FD_CLOEXEC);

    if (run_child(argv) == -1)
	exit(EXIT_FAILURE);
#endif
/*
 * Reduce num_fds to what we use, so select() is more efficient
 */
    num_fds = max(STDERR_FILENO, cfd);
    MAX_IT(num_fds, Xfd);
    num_fds++;			/* counts from 0 */

    privileged_utmp(SAVE);
    return cfd;
}

/* ------------------------------------------------------------------------- *
 *                          CHILD PROCESS OPERATIONS                         *
 * ------------------------------------------------------------------------- */
/*
 * The only open file descriptor is the slave tty - so no error messages.
 * returns are fatal
 */
/* INTPROTO */
int
run_child(const char *const *argv)
{
    int             fd;
    char           *login;

#ifndef __QNX__
/* ---------------------------------------- */
# ifdef HAVE_SETSID
    setsid();
# endif
# if defined(HAVE_SETPGID)
    setpgid(0, 0);
# elif defined(HAVE_SETPGRP)
    setpgrp(0, 0);
# endif
/* ---------------------------------------- */
# ifdef TIOCNOTTY
    fd = open("/dev/tty", O_RDWR | O_NOCTTY);
    D_TTY((stderr, "run_child(): Voiding tty associations: previous=%s", fd < 0 ? "no" : "yes"));
    if (fd >= 0) {
	ioctl(fd, TIOCNOTTY, 0);	/* void tty associations */
	close(fd);
    }
# endif
/* ---------------------------------------- */
    fd = open("/dev/tty", O_RDWR | O_NOCTTY);
    D_TTY((stderr, "run_child(): /dev/tty has controlling tty? %s", fd < 0 ? "no (good)" : "yes (bad)"));
    if (fd >= 0)
	close(fd);		/* ouch: still have controlling tty */
/* ---------------------------------------- */
# if defined(TIOCSCTTY)
    if (ioctl(tty_fd, TIOCSCTTY, 0) < 0 && errno != EINVAL)
# elif defined(TIOCSETCTTY)
    if (ioctl(tty_fd, TIOCSETCTTY, 0) < 0)
# endif
/* ---------------------------------------- */
/*
 * If we can't force the only open file descriptor to be the controlling
 * terminal, close it and open a new one: _some_ systems make it automatically
 * the controlling terminal
 */
    {
	close(tty_fd);
	tty_fd = open(ttydev, O_RDWR, 0);
	D_TTY((stderr, "run_child(): couldn't set controlling terminal, trying again: %s", tty_fd < 0 ? "no (bad)" : "yes (good)"));
    }
/* ---------------------------------------- */
    fd = open("/dev/tty", O_WRONLY);
    D_TTY((stderr, "run_child(): do we have controlling tty now: %s", fd < 0 ? "no (fatal)" : "yes (good)"));
    if (fd < 0)
	return -1;		/* fatal */
    close(fd);
/* ---------------------------------------- */
# if 0
    close(tty_fd);
    tty_fd = open(ttydev, O_RDWR, 0);
    D_TTY((stderr, "run_child(): reopening tty: %s", tty_fd < 0 ? "no (fatal)" : "yes (good)"));
    if (tty_fd < 0)
	return -1;
# endif
    fprintf(stderr, "run_child(): tcgetpgrp(): %d  getpgrp(): %d\n", tcgetpgrp(tty_fd), getpgrp());
/* ---------------------------------------- */
/* Reopen stdin, stdout and stderr over the tty file descriptor */
    dup2(tty_fd, STDIN_FILENO);
    dup2(tty_fd, STDOUT_FILENO);
    dup2(tty_fd, STDERR_FILENO);
    if (tty_fd > 2)
	close(tty_fd);
#endif				/* ! __QNX__ */

    SET_TTYMODE(STDIN_FILENO, &tio);	/* initialize terminal attributes */

    if (Options & Opt_console) {	/* be virtual console, fail silently */
#ifdef TIOCCONS
	unsigned int    on = 1;

	ioctl(STDIN_FILENO, TIOCCONS, &on);
#elif defined (SRIOCSREDIR)
	fd = open(CONSOLE, O_WRONLY, 0);
	if (fd < 0 || ioctl(fd, SRIOCSREDIR, 0) < 0) {
	    if (fd >= 0)
		close(fd);
	}
#endif				/* SRIOCSREDIR */
    }

    tt_winsize(STDIN_FILENO);	/* set window size */

/* reset signals and spin off the command interpreter */
    signal(SIGINT, SIG_DFL);
    signal(SIGQUIT, SIG_DFL);
    signal(SIGCHLD, SIG_DFL);
/*
 * mimick login's behavior by disabling the job control signals
 * a shell that wants them can turn them back on
 */
#ifdef SIGTSTP
    signal(SIGTSTP, SIG_IGN);
    signal(SIGTTIN, SIG_IGN);
    signal(SIGTTOU, SIG_IGN);
#endif				/* SIGTSTP */

#ifndef __QNX__
/* command interpreter path */
    if (argv != NULL) {
# ifdef DEBUG_CMD
	int             i;

	for (i = 0; argv[i]; i++)
	    fprintf(stderr, "argv [%d] = \"%s\"\n", i, argv[i]);
# endif
	execvp(argv[0], (char *const *)argv);
	/* no error message: STDERR is closed! */
    } else {
	const char     *argv0, *shell;

	if ((shell = getenv("SHELL")) == NULL || *shell == '\0')
	    shell = "/bin/sh";

	argv0 = (const char *)r_basename(shell);
	if (Options & Opt_loginShell) {
	    login = MALLOC((STRLEN(argv0) + 2) * sizeof(char));

	    login[0] = '-';
	    STRCPY(&login[1], argv0);
	    argv0 = login;
	}
	execlp(shell, argv0, NULL);
	/* no error message: STDERR is closed! */
    }
#else				/* __QNX__ uses qnxspawn() */
    {
	char            iov_a[10] = {-1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
	char           *command = NULL, fullcommand[_MAX_PATH];
	char          **arg_v, *arg_a[2] = {NULL, NULL};

	if (argv != NULL) {
	    if (access(argv[0], X_OK) == -1) {
		if (strchr(argv[0], '/') == NULL) {
		    searchenv(argv[0], "PATH", fullcommand);
		    if (fullcommand[0] != '\0')
			command = fullcommand;
		}
		if (access(command, X_OK) == -1)
		    return -1;
	    } else
		command = argv[0];
	    arg_v = argv;
	} else {
	    if ((command = getenv("SHELL")) == NULL || *command == '\0')
		command = "/bin/sh";

	    arg_a[0] = my_basename(command);
	    if (Options & Opt_loginShell) {
		login = MALLOC((strlen(arg_a[0]) + 2) * sizeof(char));
		login[0] = '-';
		STRCPY(&login[1], arg_a[0]);
		arg_a[0] = login;
	    }
	    arg_v = arg_a;
	}
	iov_a[0] = iov_a[1] = iov_a[2] = tty_fd;
	cmd_pid = qnx_spawn(0, 0, 0, -1, -1, _SPAWN_SETSID | _SPAWN_TCSETPGRP,
			    command, arg_v, environ, iov_a, 0);
	if (login)
	    FREE(login);
	close(tty_fd);
	return cmd_pid;
    }
#endif
    return -1;
}

/* ------------------------------------------------------------------------- *
 *                  GET PSEUDO TELETYPE - MASTER AND SLAVE                   *
 * ------------------------------------------------------------------------- */
/*
 * On failure, returns -1.
 * If successful, file descriptors cmd_fd and tty_fd point to the master
 * and slave parts respectively; and ttydev is the name of the slave.
 */
/* INTPROTO */
int
get_ptytty(void)
{
    int             pfd;
    char           *ptydev;

#ifdef PTYS_ARE_OPENPTY
    char            tty_name[] = "/dev/pts/????";

    if (openpty(&pfd, &tty_fd, tty_name, NULL, NULL) != -1) {
	ttydev = strdup(tty_name);
	goto Found_pty;
    }
#endif
#ifdef PTYS_ARE__GETPTY
    ptydev = ttydev = _getpty(&pfd, O_RDWR | O_NDELAY | O_NOCTTY, 0622, 0);
    if (ptydev != NULL)
	goto Found_pty;
#endif
#ifdef PTYS_ARE_GETPTY
    while ((ptydev = getpty()) != NULL)
	if ((pfd = open(ptydev, O_RDWR | O_NOCTTY, 0)) >= 0) {
	    ttydev = ptydev;
	    goto Found_pty;
	}
#endif
#if defined(HAVE_GRANTPT) && defined(HAVE_UNLOCKPT)
# if defined(PTYS_ARE_GETPT) || defined(PTYS_ARE_PTMX)
    {
	extern char    *ptsname();

#  ifdef PTYS_ARE_GETPT
	if ((pfd = getpt()) >= 0) {
#  else
	if ((pfd = open("/dev/ptmx", O_RDWR | O_NOCTTY, 0)) >= 0) {
#  endif
	    if (grantpt(pfd) == 0	/* change slave permissions */
		&& unlockpt(pfd) == 0) {	/* slave now unlocked */
		ptydev = ttydev = ptsname(pfd);	/* get slave's name */
		changettyowner = 0;
		goto Found_pty;
	    }
	    close(pfd);
	}
    }
# endif
#endif
#ifdef PTYS_ARE_PTC
    if ((pfd = open("/dev/ptc", O_RDWR | O_NOCTTY, 0)) >= 0) {
	ptydev = ttydev = ttyname(pfd);
	goto Found_pty;
    }
#endif
#ifdef PTYS_ARE_CLONE
    if ((pfd = open("/dev/ptym/clone", O_RDWR | O_NOCTTY, 0)) >= 0) {
	ptydev = ttydev = ptsname(pfd);
	goto Found_pty;
    }
#endif
#ifdef PTYS_ARE_NUMERIC
    {
	int             idx;
	char           *c1, *c2;
	char            pty_name[] = "/dev/ptyp???";
	char            tty_name[] = "/dev/ttyp???";

	ptydev = pty_name;
	ttydev = tty_name;

	c1 = &(pty_name[sizeof(pty_name) - 4]);
	c2 = &(tty_name[sizeof(tty_name) - 4]);
	for (idx = 0; idx < 256; idx++) {
	    sprintf(c1, "%d", idx);
	    sprintf(c2, "%d", idx);
	    if (access(ttydev, F_OK) < 0) {
		idx = 256;
		break;
	    }
	    if ((pfd = open(ptydev, O_RDWR | O_NOCTTY, 0)) >= 0) {
		if (access(ttydev, R_OK | W_OK) == 0) {
		    ttydev = strdup(tty_name);
		    goto Found_pty;
		}
		close(pfd);
	    }
	}
    }
#endif
#ifdef PTYS_ARE_SEARCHED
    {
	int             len;
	const char     *c1, *c2;
	char            pty_name[] = "/dev/pty??";
	char            tty_name[] = "/dev/tty??";

	len = sizeof(pty_name) - 3;
	ptydev = pty_name;
	ttydev = tty_name;

# define PTYCHAR1	"pqrstuvwxyz"
# define PTYCHAR2	"0123456789abcdef"
	for (c1 = PTYCHAR1; *c1; c1++) {
	    ptydev[len] = ttydev[len] = *c1;
	    for (c2 = PTYCHAR2; *c2; c2++) {
		ptydev[len + 1] = ttydev[len + 1] = *c2;
		if ((pfd = open(ptydev, O_RDWR | O_NOCTTY, 0)) >= 0) {
		    if (access(ttydev, R_OK | W_OK) == 0) {
			ttydev = strdup(tty_name);
			goto Found_pty;
		    }
		    close(pfd);
		}
	    }
	}
    }
#endif

    print_error("can't open pseudo-tty");
    return -1;

  Found_pty:
    cmd_fd = pfd;
    fcntl(cmd_fd, F_SETFL, O_NDELAY);
    privileged_ttydev(SAVE);
    if (tty_fd == -1 && (tty_fd = open(ttydev, O_RDWR | O_NOCTTY, 0)) < 0) {
	print_error("can't open slave tty %s", ttydev);
	close(pfd);
	return -1;
    }
#if defined(PTYS_ARE_PTMX) && defined(I_PUSH)
/*
 * Push STREAMS modules:
 *    ptem: pseudo-terminal hardware emulation module.
 *    ldterm: standard terminal line discipline.
 *    ttcompat: V7, 4BSD and XENIX STREAMS compatibility module.
 */
    if (!changettyowner) {
	D_TTY((stderr, "get_ptytty(): STREAMS pushing"));
	ioctl(tty_fd, I_PUSH, "ptem");
	ioctl(tty_fd, I_PUSH, "ldterm");
	ioctl(tty_fd, I_PUSH, "ttcompat");
    }
#endif

    get_ttymode(&tio);

    return cmd_fd;
}

/* ------------------------------------------------------------------------- *
 *                            GET TTY CURRENT STATE                          *
 * ------------------------------------------------------------------------- */
/* get_ttymode() */
/* INTPROTO */
void
get_ttymode(ttymode_t *tio)
{
#ifdef HAVE_TERMIOS_H
/*
 * standard System V termios interface
 */
    if (GET_TERMIOS(STDIN_FILENO, tio) < 0) {
    /* return error - use system defaults */
	tio->c_cc[VINTR] = CINTR;
	tio->c_cc[VQUIT] = CQUIT;
	tio->c_cc[VERASE] = CERASE;
	tio->c_cc[VKILL] = CKILL;
	tio->c_cc[VSTART] = CSTART;
	tio->c_cc[VSTOP] = CSTOP;
	tio->c_cc[VSUSP] = CSUSP;
# ifdef VDSUSP
	tio->c_cc[VDSUSP] = CDSUSP;
# endif
# ifdef VREPRINT
	tio->c_cc[VREPRINT] = CRPRNT;
# endif
# ifdef VDISCRD
	tio->c_cc[VDISCRD] = CFLUSH;
# endif
# ifdef VWERSE
	tio->c_cc[VWERSE] = CWERASE;
# endif
# ifdef VLNEXT
	tio->c_cc[VLNEXT] = CLNEXT;
# endif
    }
    tio->c_cc[VEOF] = CEOF;
    tio->c_cc[VEOL] = VDISABLE;
# ifdef VEOL2
    tio->c_cc[VEOL2] = VDISABLE;
# endif
# ifdef VSWTC
    tio->c_cc[VSWTC] = VDISABLE;
# endif
# ifdef VSWTCH
    tio->c_cc[VSWTCH] = VDISABLE;
# endif
# if VMIN != VEOF
    tio->c_cc[VMIN] = 1;
# endif
# if VTIME != VEOL
    tio->c_cc[VTIME] = 0;
# endif

/* input modes */
    tio->c_iflag = (BRKINT | IGNPAR | ICRNL | IXON
# ifdef IMAXBEL
		    | IMAXBEL
# endif
	);

/* output modes */
    tio->c_oflag = (OPOST | ONLCR);

/* control modes */
    tio->c_cflag = (CS8 | CREAD);

/* line discipline modes */
    tio->c_lflag = (ISIG | ICANON | IEXTEN | ECHO | ECHOE | ECHOK
# if defined (ECHOCTL) && defined (ECHOKE)
		    | ECHOCTL | ECHOKE
# endif
	);
# else				/* HAVE_TERMIOS_H */

/*
 * sgtty interface
 */

/* get parameters -- gtty */
    if (ioctl(STDIN_FILENO, TIOCGETP, &(tio->sg)) < 0) {
	tio->sg.sg_erase = CERASE;	/* ^H */
	tio->sg.sg_kill = CKILL;	/* ^U */
    }
    tio->sg.sg_flags = (CRMOD | ECHO | EVENP | ODDP);

/* get special characters */
    if (ioctl(STDIN_FILENO, TIOCGETC, &(tio->tc)) < 0) {
	tio->tc.t_intrc = CINTR;	/* ^C */
	tio->tc.t_quitc = CQUIT;	/* ^\ */
	tio->tc.t_startc = CSTART;	/* ^Q */
	tio->tc.t_stopc = CSTOP;	/* ^S */
	tio->tc.t_eofc = CEOF;		/* ^D */
	tio->tc.t_brkc = -1;
    }
/* get local special chars */
    if (ioctl(STDIN_FILENO, TIOCGLTC, &(tio->lc)) < 0) {
	tio->lc.t_suspc = CSUSP;	/* ^Z */
	tio->lc.t_dsuspc = CDSUSP;	/* ^Y */
	tio->lc.t_rprntc = CRPRNT;	/* ^R */
	tio->lc.t_flushc = CFLUSH;	/* ^O */
	tio->lc.t_werasc = CWERASE;	/* ^W */
	tio->lc.t_lnextc = CLNEXT;	/* ^V */
    }
/* get line discipline */
    ioctl(STDIN_FILENO, TIOCGETD, &(tio->line));
# ifdef NTTYDISC
    tio->line = NTTYDISC;
# endif				/* NTTYDISC */
    tio->local = (LCRTBS | LCRTERA | LCTLECH | LPASS8 | LCRTKIL);
#endif				/* HAVE_TERMIOS_H */

/*
 * Debugging
 */
#ifdef DEBUG_TTYMODE
#ifdef HAVE_TERMIOS_H
/* c_iflag bits */
    fprintf(stderr, "Input flags\n");

/* cpp token stringize doesn't work on all machines <sigh> */
# define FOO(flag,name)			\
    if ((tio->c_iflag) & flag)		\
	fprintf (stderr, "%s ", name)

/* c_iflag bits */
    FOO(IGNBRK, "IGNBRK");
    FOO(BRKINT, "BRKINT");
    FOO(IGNPAR, "IGNPAR");
    FOO(PARMRK, "PARMRK");
    FOO(INPCK, "INPCK");
    FOO(ISTRIP, "ISTRIP");
    FOO(INLCR, "INLCR");
    FOO(IGNCR, "IGNCR");
    FOO(ICRNL, "ICRNL");
    FOO(IXON, "IXON");
    FOO(IXOFF, "IXOFF");
# ifdef IUCLC
    FOO(IUCLC, "IUCLC");
# endif
# ifdef IXANY
    FOO(IXANY, "IXANY");
# endif
# ifdef IMAXBEL
    FOO(IMAXBEL, "IMAXBEL");
# endif
    fprintf(stderr, "\n");

# undef FOO
# define FOO(entry, name)					\
    fprintf(stderr, "%-8s = %#04o\n", name, tio->c_cc [entry])

    FOO(VINTR, "VINTR");
    FOO(VQUIT, "VQUIT");
    FOO(VERASE, "VERASE");
    FOO(VKILL, "VKILL");
    FOO(VEOF, "VEOF");
    FOO(VEOL, "VEOL");
# ifdef VEOL2
    FOO(VEOL2, "VEOL2");
# endif
# ifdef VSWTC
    FOO(VSWTC, "VSWTC");
# endif
# ifdef VSWTCH
    FOO(VSWTCH, "VSWTCH");
# endif
    FOO(VSTART, "VSTART");
    FOO(VSTOP, "VSTOP");
    FOO(VSUSP, "VSUSP");
# ifdef VDSUSP
    FOO(VDSUSP, "VDSUSP");
# endif
# ifdef VREPRINT
    FOO(VREPRINT, "VREPRINT");
# endif
# ifdef VDISCRD
    FOO(VDISCRD, "VDISCRD");
# endif
# ifdef VWERSE
    FOO(VWERSE, "VWERSE");
# endif
# ifdef VLNEXT
    FOO(VLNEXT, "VLNEXT");
# endif
    fprintf(stderr, "\n");
# undef FOO
# endif				/* HAVE_TERMIOS_H */
#endif				/* DEBUG_TTYMODE */
}
/*----------------------- end-of-file (C source) -----------------------*/
