/*****************************************************************************/

/*
 *	usbdrvlinux.c  --  Linux USB driver interface.
 *
 *	Copyright (C) 1999-2000
 *          Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please note that the GPL allows you to use the driver, NOT the radio.
 *  In order to use the radio, you need a license from the communications
 *  authority of your country.
 *
 *
 *  History:
 *   0.1  23.06.99  Created
 *
 */

/*****************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef WITH_USBDEVFS

#define _GNU_SOURCE
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <math.h>
#include <dirent.h>

#include "usbdevfs.h"

/* --------------------------------------------------------------------- */

#define __u8  u_int8_t
#define __u16 u_int16_t

struct usb_device_descriptor_x {
        __u8  bLength;
        __u8  bDescriptorType;
        __u16 bcdUSB;
        __u8  bDeviceClass;
        __u8  bDeviceSubClass;
        __u8  bDeviceProtocol;
        __u8  bMaxPacketSize0;
        __u16 idVendor;
        __u16 idProduct;
        __u16 bcdDevice;
        __u8  iManufacturer;
        __u8  iProduct;
        __u8  iSerialNumber;
        __u8  bNumConfigurations;
};

/* --------------------------------------------------------------------- */

/*
 * Parse and show the different USB descriptors.
 */
static void usb_show_device_descriptor(struct usb_device_descriptor_x *desc)
{
        printf("  Length              = %2d%s\n", desc->bLength,
                desc->bLength == USB_DT_DEVICE_SIZE ? "" : " (!!!)");
        printf("  DescriptorType      = %02x\n", desc->bDescriptorType);

        printf("  USB version         = %x.%02x\n",
                desc->bcdUSB >> 8, desc->bcdUSB & 0xff);
        printf("  Vendor:Product      = %04x:%04x\n",
                desc->idVendor, desc->idProduct);
        printf("  MaxPacketSize0      = %d\n", desc->bMaxPacketSize0);
        printf("  NumConfigurations   = %d\n", desc->bNumConfigurations);
        printf("  Device version      = %x.%02x\n",
                desc->bcdDevice >> 8, desc->bcdDevice & 0xff);

        printf("  Device Class:SubClass:Protocol = %02x:%02x:%02x\n",
                desc->bDeviceClass, desc->bDeviceSubClass, desc->bDeviceProtocol);
        switch (desc->bDeviceClass) {
        case 0:
                printf("    Per-interface classes\n");
                break;
        case 9:
                printf("    Hub device class\n");
                break;
        case 0xff:
                printf("    Vendor class\n");
                break;
        default:
                printf("    Unknown class\n");
        }
}

/* ---------------------------------------------------------------------- */

static int fd = -1;

/* ---------------------------------------------------------------------- */

void usb_close(void)
{
	if (fd != -1)
		close(fd);
	fd = -1;
}

int usb_init(const char *usbbus, u_int16_t vendorid, u_int16_t productid)
{
        static const char *default_usbbus = "/proc/bus/usb/";
        struct dirent *de, *de2;
	DIR *d, *d2;
        struct usb_device_descriptor_x devdesc;
        unsigned char buf[256];
	struct stat statbuf;

	usb_close();
	if (!usbbus)
		usbbus = default_usbbus;
	if (stat(usbbus, &statbuf)) {
		fprintf(stderr, "cannot open %s, %s (%d)\n", usbbus, strerror(errno), errno);
                return -1;
	}
	if (!S_ISDIR(statbuf.st_mode)) {
		if ((fd = open(usbbus, O_RDWR)) == -1) {
			fprintf(stderr, "cannot open %s, %s (%d)\n", usbbus, strerror(errno), errno);
			return -1;
		}
		return 0;
	}
        d = opendir(usbbus);
        if (!d) {
                fprintf(stderr, "cannot open %s, %s (%d)\n", usbbus, strerror(errno), errno);
                return -1;
        }
        while ((de = readdir(d))) {
                if (de->d_name[0] < '0' || de->d_name[0] > '9')
                        continue;
                snprintf(buf, sizeof(buf), "%s%s/", usbbus, de->d_name);
                if (!(d2 = opendir(buf)))
                        continue;
                while ((de2 = readdir(d2))) {
                        if (de2->d_name[0] == '.')
                                continue;
                        snprintf(buf, sizeof(buf), "%s%s/%s", usbbus, de->d_name, de2->d_name);
                        if ((fd = open(buf, O_RDWR)) == -1) {
                                fprintf(stderr, "cannot open %s, %s (%d)\n", buf, strerror(errno), errno);
                                continue;
                        }
			if (read(fd, &devdesc, sizeof(devdesc)) != sizeof(devdesc)) {
				fprintf(stderr, "cannot read device descriptor %s (%d)\n", strerror(errno), errno);
				close(fd);
				continue;
			}
//			usb_show_device_descriptor(&devdesc);
			if ((devdesc.idVendor == vendorid || vendorid == 0xffff) &&
			    (devdesc.idProduct == productid || productid == 0xffff)) {
				return 0;
			}
                        close(fd);
                }
                closedir(d2);
        }
        closedir(d);
	fd = -1;
	return -1;
}

int usb_control(unsigned char requesttype, unsigned char request,
		unsigned short value, unsigned short index, unsigned short length, void *data)
{
        struct usb_proc_ctrltransfer ctrl;
	int i;

	ctrl = (struct usb_proc_ctrltransfer){ requesttype, request, value, index, length, 5000, data };
	i = ioctl(fd, USB_PROC_CONTROL, &ctrl);
	if (i < 0) {
		return -1;
	}
	return i;
}

int usb_bulk(unsigned int ep, unsigned int dlen, void *data)
{
	struct usb_proc_bulktransfer bulk;
	int i;

	bulk = (struct usb_proc_bulktransfer){ ep, dlen, 5000, data };
	i = ioctl(fd, USB_PROC_BULK, &bulk);
	if (i < 0) {
		return -1;
	}
	return i;
}

int usb_resetep(unsigned int ep)
{
	int i;

	i = ioctl(fd, USB_PROC_RESETEP, &ep);
	if (i < 0) {
		return -1;
	}
	return 0;
}

int usb_setconfiguration(unsigned int configuration)
{
	int i;

	i = ioctl(fd, USB_PROC_SETCONFIGURATION, &configuration);
	if (i < 0) {
		return -1;
	}
	return 0;
}

int usb_setinterface(unsigned int intf, unsigned int altsetting)
{
	struct usb_proc_setinterface setif;
	int i;

	setif = (struct usb_proc_setinterface) { intf, altsetting };
	i = ioctl(fd, USB_PROC_SETINTERFACE, &setif);
	if (i < 0) {
		return -1;
	}
	return 0;
}

int usb_getdevicedescriptor(struct usb_device_descriptor_x *desc)
{
	return usb_control(0x80, 6, 0x100, 0, sizeof(*desc), desc);
}

int usb_fd(void)
{
	return fd;
}

/* ---------------------------------------------------------------------- */

#endif /*usbdevfs*/
