/* $Id: hash.h,v 1.12 2005/12/25 10:08:35 rockyb Exp $
   Copyright (C) 1995, 1999, 2002, 2004, 2005 Free Software Foundation, Inc.
   Written by Greg McGary <gkm@gnu.org> <greg@mcgary.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
*/

/** \file hash.h
 *
 *  \brief Declarations for GNU Make hash tables.
 */

#ifndef _HASH_H_
#define _HASH_H_

#include "types.h"
#include <stdio.h>
#include <ctype.h>

typedef void (*free_fn_t) (void *p_item);
typedef unsigned long (*hash_func_t) (void const *p_key);
typedef int (*hash_cmp_func_t) (void const *p_x, void const *p_y);
typedef void (*hash_map_func_t) (void const *p_item);
typedef void (*hash_map_arg_func_t) (void const *p_item, void *p_arg);

/** \brief hash table structure. We use hash tables for things like
    variables, targets, directories and functions. */
typedef struct {
  void **ht_vec;
  unsigned long   ht_size;	 /**< total number of slots (power of 2) */
  unsigned long   ht_capacity;	 /**< usable slots, limited by 
				      loading-factor */
  unsigned long   ht_fill;	 /**< items in table */
  unsigned long   ht_empty_slots;/**< empty slots not including deleted 
				      slots */
  unsigned long   ht_collisions; /**< # of failed calls to comparison 
				      function */
  unsigned long   ht_lookups;	 /**< # of queries */
  unsigned int    ht_rehashes;	 /**< # of times we've expanded table */
  hash_func_t     ht_hash_1;	 /**< primary hash function */
  hash_func_t     ht_hash_2;	 /**< secondary hash function */
  hash_cmp_func_t ht_compare;	 /**< comparison function */
} hash_table_t;

hash_table_t files;

typedef int (*qsort_cmp_t) (void const *, void const *);

void hash_init(hash_table_t *p_ht, unsigned long size,
	       hash_func_t hash_1, hash_func_t hash_2, 
	       hash_cmp_func_t hash_cmp);

/*! Load an array of items into `ht'.  */
void hash_load (hash_table_t *ht, void *item_table, unsigned long cardinality, 
		unsigned long size);

/*! Returns the address of the table slot matching `key'.  If `key' is
   not found, return the address of an empty slot suitable for
   inserting `key'.  The caller is responsible for incrementing
   ht_fill on insertion.  */
void **hash_find_slot (hash_table_t *p_ht, void const *p_key);

void *hash_find_item (hash_table_t *p_ht, void const *p_key);
void *hash_insert (hash_table_t *p_ht, void *p_item);
void *hash_insert_at (hash_table_t *p_ht, void *p_item, void const *p_slot);
void *hash_delete (hash_table_t *p_ht, void const *p_item);
void *hash_delete_at (hash_table_t *p_ht, void const *p_slot);
void hash_delete_items (hash_table_t *p_ht);

/*! Free just the items in hash tables ht. */
void hash_free_items (hash_table_t *p_ht, free_fn_t free_fn);

/*! Free memory allocated in hash tables ht. If b_free_items, free the items
  in ht too. */
void hash_free (hash_table_t *p_ht, free_fn_t free_fn);

/*! run map() on every vacant hash item in use in ht. */
void hash_map (hash_table_t *p_ht, hash_map_func_t map);

/*! run map(arg) on every vacant hash item in use in ht. */
void hash_map_arg (hash_table_t *p_ht, hash_map_arg_func_t map, 
		       void *arg);

/*! print hash statistics: percent in use, rehashes and collisions. */
void hash_print_stats (hash_table_t *p_ht, FILE *p_out_FILE);

/*! Dump all items into a NULL-terminated vector.  Use the
   user-supplied vector_0, or a malloc one if vector_0 is NULL.  */
void **hash_dump (hash_table_t *p_ht, void **pp_vector_0, qsort_cmp_t compare);

extern void *hash_deleted_item;
#define HASH_VACANT(item) ((item) == 0 || (void *) (item) == hash_deleted_item)


/* hash and comparison macros for case-sensitive string keys. */

#define STRING_HASH_1(KEY, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  while (*++_key_) \
    (RESULT) += (*_key_ << (_key_[1] & 0xf)); \
} while (0)
#define return_STRING_HASH_1(KEY) do { \
  unsigned long _result_ = 0; \
  STRING_HASH_1 ((KEY), _result_); \
  return _result_; \
} while (0)

#define STRING_HASH_2(KEY, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  while (*++_key_) \
    (RESULT) += (*_key_ << (_key_[1] & 0x7)); \
} while (0)
#define return_STRING_HASH_2(KEY) do { \
  unsigned long _result_ = 0; \
  STRING_HASH_2 ((KEY), _result_); \
  return _result_; \
} while (0)

#define STRING_COMPARE(X, Y, RESULT) do { \
  RESULT = strcmp ((X), (Y)); \
} while (0)
#define return_STRING_COMPARE(X, Y) do { \
  return strcmp ((X), (Y)); \
} while (0)


#define STRING_N_HASH_1(KEY, N, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  int _n_ = (N); \
  if (_n_) \
    while (--_n_ && *++_key_) \
      (RESULT) += (*_key_ << (_key_[1] & 0xf)); \
  (RESULT) += *++_key_; \
} while (0)
#define return_STRING_N_HASH_1(KEY, N) do { \
  unsigned long _result_ = 0; \
  STRING_N_HASH_1 ((KEY), (N), _result_); \
  return _result_; \
} while (0)

#define STRING_N_HASH_2(KEY, N, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  int _n_ = (N); \
  if (_n_) \
    while (--_n_ && *++_key_) \
      (RESULT) += (*_key_ << (_key_[1] & 0x7)); \
  (RESULT) += *++_key_; \
} while (0)
#define return_STRING_N_HASH_2(KEY, N) do { \
  unsigned long _result_ = 0; \
  STRING_N_HASH_2 ((KEY), (N), _result_); \
  return _result_; \
} while (0)

#define STRING_N_COMPARE(X, Y, N, RESULT) do { \
  RESULT = strncmp ((X), (Y), (N)); \
} while (0)
#define return_STRING_N_COMPARE(X, Y, N) do { \
  return strncmp ((X), (Y), (N)); \
} while (0)

#ifdef HAVE_CASE_INSENSITIVE_FS

/* hash and comparison macros for case-insensitive string _key_s. */

#define ISTRING_HASH_1(KEY, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  while (*++_key_) \
    (RESULT) += ((isupper (*_key_) ? tolower (*_key_) : *_key_) << (_key_[1] & 0xf)); \
} while (0)
#define return_ISTRING_HASH_1(KEY) do { \
  unsigned long _result_ = 0; \
  ISTRING_HASH_1 ((KEY), _result_); \
  return _result_; \
} while (0)

#define ISTRING_HASH_2(KEY, RESULT) do { \
  unsigned char const *_key_ = (unsigned char const *) (KEY) - 1; \
  while (*++_key_) \
    (RESULT) += ((isupper (*_key_) ? tolower (*_key_) : *_key_) << (_key_[1] & 0x7)); \
} while (0)
#define return_ISTRING_HASH_2(KEY) do { \
  unsigned long _result_ = 0; \
  ISTRING_HASH_2 ((KEY), _result_); \
  return _result_; \
} while (0)

#define ISTRING_COMPARE(X, Y, RESULT) do { \
  RESULT = strcmpi ((X), (Y)); \
} while (0)
#define return_ISTRING_COMPARE(X, Y) do { \
  return strcmpi ((X), (Y)); \
} while (0)

#else

#define ISTRING_HASH_1(KEY, RESULT) STRING_HASH_1 ((KEY), (RESULT))
#define return_ISTRING_HASH_1(KEY) return_STRING_HASH_1 (KEY)

#define ISTRING_HASH_2(KEY, RESULT) STRING_HASH_2 ((KEY), (RESULT))
#define return_ISTRING_HASH_2(KEY) return_STRING_HASH_2 (KEY)

#define ISTRING_COMPARE(X, Y, RESULT) STRING_COMPARE ((X), (Y), (RESULT))
#define return_ISTRING_COMPARE(X, Y) return_STRING_COMPARE ((X), (Y))

#endif

/* hash and comparison macros for integer _key_s. */

#define INTEGER_HASH_1(KEY, RESULT) do { \
  (RESULT) += ((unsigned long)(KEY)); \
} while (0)
#define return_INTEGER_HASH_1(KEY) do { \
  unsigned long _result_ = 0; \
  INTEGER_HASH_1 ((KEY), _result_); \
  return _result_; \
} while (0)

#define INTEGER_HASH_2(KEY, RESULT) do { \
  (RESULT) += ~((unsigned long)(KEY)); \
} while (0)
#define return_INTEGER_HASH_2(KEY) do { \
  unsigned long _result_ = 0; \
  INTEGER_HASH_2 ((KEY), _result_); \
  return _result_; \
} while (0)

#define INTEGER_COMPARE(X, Y, RESULT) do { \
  (RESULT) = X - Y; \
} while (0)
#define return_INTEGER_COMPARE(X, Y) do { \
  int _result_; \
  INTEGER_COMPARE (X, Y, _result_); \
  return _result_; \
} while (0)

/*! hash and comparison macros for address keys. */

#define ADDRESS_HASH_1(KEY, RESULT) INTEGER_HASH_1 (((unsigned long)(KEY)) >> 3, (RESULT))
#define ADDRESS_HASH_2(KEY, RESULT) INTEGER_HASH_2 (((unsigned long)(KEY)) >> 3, (RESULT))
#define ADDRESS_COMPARE(X, Y, RESULT) INTEGER_COMPARE ((X), (Y), (RESULT))
#define return_ADDRESS_HASH_1(KEY) return_INTEGER_HASH_1 (((unsigned long)(KEY)) >> 3)
#define return_ADDRESS_HASH_2(KEY) return_INTEGER_HASH_2 (((unsigned long)(KEY)) >> 3)
#define return_ADDRESS_COMPARE(X, Y) return_INTEGER_COMPARE ((X), (Y))

#endif /* _HASH_H_ */
