#!/usr/bin/perl -sw
##
## Razor::Agent -- UI routines for razor agents.
##
## Copyright (c) 2001, Vipul Ved Prakash.  All rights reserved.
## This code is free software; you can redistribute it and/or modify
## it under the same terms as Perl itself.
##
## $Id: Agent.pm,v 1.5 2001/12/26 02:29:21 vipul Exp $

package Razor::Agent;

use lib qw(lib);
use strict;
use Razor::Client;
use Getopt::Long; 
use Razor::String qw(hash headers);
use Data::Dumper;
use Mail::Header;
use base qw(Razor::Client);


sub options { 
    my ($agent) = @_;
    my %self;
    Getopt::Long::Configure ("no_ignore_case");
    GetOptions(
        's'   => \$self{simulate},
        'd'   => \$self{debug},
  'verbose'   => \$self{debug},
        'v'   => \$self{version},
        'h'   => \$self{usage},
        'H'   => \$self{printhash},
      'sig'   => \$self{sig},
        'S'   => \$self{sig},
     'mbox'   => \$self{mbox},
        'M'   => \$self{mbox},
        'n'   => \$self{negative},
   'conf=s'   => \$self{config},
 'config=s'   => \$self{config},
        'f'   => \$self{foreground},
     'noml'   => \$self{noml},
    );
    if ($self{usage}) { 
        print "$agent [-s] [-d] [-H] [-M] [-h] [-conf=filename] filename | mbox | signatures\n";
        print "             -s  Simulate Only.\n";
        print "             -d  Turn on debugging.\n";
        print "             -H  Compute and print signature.\n";
        print "             -f  Stay in foreground.\n" if $agent eq "razor-report";
        print "    -S |  --sig  Accept a lists of signatures to check on the command line\n";
        print "    -M | --mbox  Accept a mailbox name on the command line\n";
        print "             -v  Print version number and exit\n";
        print "             -h  Print this usage message.\n";
        print "see $agent(1) manpage for details.\n";
        exit 1;
    } elsif ($self{mbox} && $self{sig}) { 
        print "--mbox and --sig are mutually exclusive.\n";
        exit 1;
    } elsif ($self{version}) { 
        print "Razor Client Tools $Razor::Client::VERSION, protocol version $Razor::Version::PROTOCOL\n"; 
        exit 1;
    } 
    return \%self;
} 



sub compute_sigs { 

    my $self = shift;
    my (@sigs, $hash);

    if ($$self{sig})  { 
        push @sigs, @ARGV;
    } elsif ($$self{mbox}) {
        my $mbox = shift @ARGV;
        open MBOX, $mbox || die $!;
        {
            my @mailbox = <MBOX>;
            close MBOX;
            exit 1 unless @mailbox;
            my $line; my @message;
            foreach $line (@mailbox) {
                if ($line =~ /^From / && $mailbox[0] ne $line) {
                    $hash = $self->local_check(\@message);
                    push @sigs, $hash if $hash;
                    splice @message;
                    push @message, $line;
                } else {
                    push @message, $line;
                }
            }
            if ($hash = $self->local_check(\@message)) { 
                push @sigs, $hash;
            }
         }
         $$self{sig} = 1;
    } else { 
        my @spam = <>; 
        if ($hash = $self->local_check(\@spam)) { 
            push @sigs, $hash;
        }
    }
    if ($$self{printhash}) { 
        $" = "\n"; print "@sigs\n";
        exit 1;
    }
    return @sigs;
}


sub raise_error { 
    my $self = shift;
    my $str = $self->errstr();
    my ($code) = $str =~ /Razor Error (\d+):/;
    $code = 255 unless $code; 
    print "FATAL: $str";
    exit $code;
}


sub local_check {
    my ($self, $mail) = @_;
    my $headers = headers ($mail);
    my $hash = hash ($mail);
    my $head = new Mail::Header ($headers);
    if ($self->{whitelist}) {  # read the whitelist and do checks
        my $whitelist = $self->read_whitelist;
        my $iswl = 0;
        for (keys %$whitelist) { 
            my $sh = $_;
            if ($sh ne 'sha1') { 
                my $fc = $head->get ($sh); 
                next unless $fc;
                my @list = @{$$whitelist{$sh}};
                for (@list) {
                    my $address = $_; 
                    if ($fc =~ /$address/i) { 
                        $self->debug ("Address <$address> is whitelisted; ignoring.");
                        return 0
                    }
                }
            } elsif ($sh eq 'sha1') { 
                my @list = @{$$whitelist{$sh}}; 
                for (@list) { 
                    my $wlhash = $_;
                    if ($hash eq $wlhash) { 
                        $self->debug ("Signature $wlhash is whitelisted; ignoring.");
                        return 0
                    }
                }
            }
        }
    }
    return $hash;
}



sub read_whitelist { 
    my ($self) = @_; 
    my %whitelist;
    my $whitelist = $self->{whitelist}; 
    unless (open WL, $whitelist) { 
        $self->debug ("Unable to open the $self->{whitelist}\n");
        exit 1; 
    }
    my @WL = <WL>;
    close WL;
    for (@WL) { 
        s/^\s*//;
        next if /^#/;
        chomp;
        my ($type, $value) = split /\s+/, $_, 2; 
        $type =~ y/A-Z/a-z/;
        push @{$whitelist{$type}}, $value if ($type && $value);
    }
    return \%whitelist;
}   


1;


