/* 
 * Copyright (c) 2001 Secure Software Solutions
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "c-tokens.h"
#include "report.h"

Hash        database = (Hash)NULL;
char *      current_file;
stack_t *   current_frame = (stack_t *)NULL;

static int                  depths[DEPTH_COUNT];
static int                  ungotten_token  = -1;
static int                  toctou_count = 0;
static toctou_t *           toctous = (toctou_t *)NULL;
static accumulator_t *      accumulators = (accumulator_t *)NULL;

static void analyze_variable(void);
static void analyze_identifier(void);

static
int accumulate_text(accumulator_t *acc, char *text)
{
    int     length;
    char *  new_text;

    length = strlen(text);
    if (*(acc->text) == (char *)NULL)
    {
        *(acc->text) = (char *)malloc(length + 1);
        acc->length  = 0;
    }
    else
    {
        new_text = (char *)realloc(*(acc->text), acc->length + length + 1);
        if (new_text == (char *)NULL)
            return 0;
        *(acc->text) = new_text;
    }

    memcpy(*acc->text + acc->length, text, length);
    acc->length += length;
    *(*acc->text + acc->length) = '\0';
    return 1;
}

static
void push_accumulator(char **text)
{
    accumulator_t * acc;

    acc = (accumulator_t *)malloc(sizeof(accumulator_t));
    acc->text    = text;
    acc->length  = (*text == (char *)NULL ? 0 : strlen(*text));
    acc->next    = accumulators;
    accumulators = acc;

    accumulate_text(acc, yytext);
}

static
void pop_accumulator(void)
{
    accumulator_t * acc;

    acc = accumulators;
    accumulators = acc->next;
    if (*(acc->text) != (char *)NULL)
    {
        if (accumulators != (accumulator_t *)NULL)
        {
            accumulators->length = 0;
            accumulate_text(accumulators, *(acc->text));
        }
        acc->length -= strlen(yytext);
        *(*acc->text + acc->length) = '\0';
    }
    free(acc);
}

static
void unget_token(int token)
{
    ungotten_token = token;
    switch (token)
    {
        case '(': depths[DEPTH_PARENTHESIS]--;  break;
        case ')': depths[DEPTH_PARENTHESIS]++;  break;
        case '[': depths[DEPTH_BRACKET]--;  break;
        case ']': depths[DEPTH_BRACKET]++;  break;
        case '{': depths[DEPTH_BRACE]--;  break;
        case '}': depths[DEPTH_BRACE]--;  break;
    }
}

static
int get_token(void)
{
    int token;

    if (ungotten_token != -1)
    {
        token = ungotten_token;
        ungotten_token = -1;
    }
    else
    {
        if (!(token = yylex()))
            return 0;
        if (accumulators != (accumulator_t *)NULL)
            accumulate_text(accumulators, yytext);
    }

    switch (token)
    {
        case '(': depths[DEPTH_PARENTHESIS]++;  break;
        case ')': depths[DEPTH_PARENTHESIS]--;  break;
        case '[': depths[DEPTH_BRACKET]++;  break;
        case ']': depths[DEPTH_BRACKET]--;  break;
        case '{': depths[DEPTH_BRACE]++;  break;
        case '}': depths[DEPTH_BRACE]--;  break;
    }

    return token;
}

static
void scan_tokens(processorfn_t fn, void *arg)
{
    int token;

    while ((token = get_token()) != 0)
    {
        if (fn != NULL)
            if (!fn(token, arg))
                break;

        switch (token)
        {
            case CHAR:
                analyze_variable();
                break;

            case IDENTIFIER:
                analyze_identifier();
                break;
        }
    }
}

static
int check_buffer(int token, void *arg)
{
    charscan_t *    data;

    data = (charscan_t *)arg;
    if (token == ';' || token == '{')
        return 0;
    if (data->skip)
        return 1;

    if (token == '[' && data->last_token == IDENTIFIER)
        data->depth = depths[DEPTH_BRACE];
    else if (data->last_token == '[')
    {
        if (token != ']')
        {
            if (data->depth)
                log_staticbuffer(StaticLocalBuffer, data->lineno, High);
            else
                log_staticbuffer(StaticGlobalBuffer, data->lineno, Low);
            data->skip = 1;
        }
    }
    data->last_token = token;
    return 1;
}

/*
 * XXX: This function can be cleaned up to prevent more false positives.
 *      Currently running lex.yy.c through this yields false positives due to
 *      pointer initializers, i.e. char *foo = bar[15]; - MMessier, 09-May-2001
 */
static
void analyze_variable(void)
{
    charscan_t  data;

    /* If we're processing arguments right now, we don't want to check for
     * stack variables because they're not really stack variables
     */
    if (current_frame != (stack_t *)NULL && current_frame->next != (stack_t *)NULL)
        return;
    if (depths[DEPTH_PARENTHESIS] || depths[DEPTH_BRACKET])
        return;

    data.lineno = lex_lineno;
    data.last_token = CHAR;
    data.depth = depths[DEPTH_BRACE];
    data.skip = 0;
    scan_tokens(check_buffer, (void *)&data);
}

static
int check_format_string(char *fmt)
{
    char *  c;

    for (c = strchr(fmt, '%');  c != (char *)NULL;  c = strchr((fmt = c + 1), '%'))
    {
        int done = 0, precision = 0;

        for (fmt = c++;  !done && *c;  c++)
        {
            switch (*c)
            {
                case '#':
                case '-':
                case ' ':
                case '+':
                    break;
                default:
                    done = 1;
                    c--;
                    break;
            }
        }

        if (*c == '*')
            c++;
        else if (isdigit(*c))
            while (isdigit(*++c));

        if (*c == '.')
        {
            precision = 1;
            if (*c == '*')
                c++;
            else
                while (isdigit(*++c));
        }

        switch (*c)
        {
            case 'L':
            case 'j':
            case 't':
            case 'z':
                c++;
                break;

            case 'h':
                if (*++c == 'h')
                    c++;
                break;

            case 'l':
                if (*++c == 'l')
                    c++;
                break;
        }

        if (*c == 's' && !precision)
            return 1;
    }

    return 0;
}

static
int check_argument(int token, void *arg)
{
    int         advance = 0;
    argscan_t * data;

    data = (argscan_t *)arg;
    if (token == ',' || token == ')')
    {
        int i;

        advance = 1;
        for (i = 0;  i < DEPTH_COUNT;  i++)
        {
            if (data->depths[i] != depths[i])
            {
                if (i != DEPTH_PARENTHESIS || token != ')' ||
                    data->depths[i] != depths[i] + 1)
                {
                    advance = 0;
                }
            }
        }
    }

    if (advance)
    {
        if (data->current != (argument_t *)NULL)
        {
            pop_accumulator();
            current_frame->argc++;
            if (data->tail != (argument_t *)NULL)
                data->tail->next = data->current;
            else
                current_frame->argv = data->current;
            data->tail = data->current;
            data->current = (argument_t *)NULL;
        }
        if (token == ')' || token == ';')
            return 0;
        return 1;
    }

    if (data->current == (argument_t *)NULL)
    {
        data->current = (argument_t *)malloc(sizeof(argument_t));
        data->current->is_constant = 1;
        data->current->contains_ps = 0;
        data->current->yytext      = (char *)NULL;
        data->current->next        = (argument_t *)NULL;

        push_accumulator(&(data->current->yytext));
    }

    if (data->current->is_constant)
    {
        if (token != STRING_CONST)
            data->current->is_constant = 0;
        else if (!data->current->contains_ps)
            data->current->contains_ps = check_format_string(yytext);
    }

    return 1;
}

static
void scan_arguments(void)
{
    int         i;
    argscan_t   data;

    data.tail = (argument_t *)NULL;
    data.current = (argument_t *)NULL;
    for (i = 0;  i < DEPTH_COUNT;  i++)
        data.depths[i] = depths[i];

    scan_tokens(check_argument, (void *)&data);
}

static
argument_t *get_argument(int number)
{
    argument_t *    arg;

    if (number < 0 || number > current_frame->argc)
        return (argument_t *)NULL;

    for (arg = current_frame->argv;  --number > 0;  arg = arg->next);
    return arg;
}

static
int push_identifier(char *identifier, int lineno)
{
    Vuln_t *    data;
    stack_t *   frame;

    if (database == NULL || (data = HashGet(database, identifier)) == NULL)
        return 0;

    frame = (stack_t *)malloc(sizeof(stack_t));
    frame->identifier = identifier;
    frame->data       = data;
    frame->lineno     = lineno;
    frame->argc       = 0;
    frame->argv       = (argument_t *)NULL;
    frame->next       = current_frame;
    current_frame     = frame;

    return 1;
}

static
void pop_identifier(void)
{
    stack_t *       frame;
    argument_t *    arg;
    argument_t *    next;

    frame = current_frame;
    current_frame = frame->next;

    free(frame->identifier);
    for (arg = frame->argv;  arg != (argument_t *)NULL;  arg = next)
    {
        next = arg->next;
        if (arg->yytext != (char *)NULL)
            free(arg->yytext);
        free(arg);
    }
    free(frame);
}

static
void record_toctou(int argn, int lineno, int use)
{
    toctou_t *      toctou;
    argument_t *    arg;

    if ((arg = get_argument(argn)) == (argument_t *)NULL)
        return;

    toctou = (toctou_t *)malloc(sizeof(toctou_t));
    toctou->lineno = lineno;
    toctou->data   = current_frame->data;
    toctou->key    = strdup(arg->yytext);
    toctou->use    = use;
    toctou->next   = toctous;
    toctous        = toctou;
    toctou_count++;
}

static
void analyze_identifier(void)
{
    int             analyzed, lineno, token;
    char *          identifier;
    Vuln_t *        data;
    argument_t *    arg;

    lineno     = lex_lineno;
    identifier = strdup(yytext);

    if (!push_identifier(identifier, lineno))
    {
        free(identifier);
        return;
    }

    /* looking only for function calls here */
    if ((token = get_token()) != '(')
    {
        if (flags & INCLUDE_ALL_REFERENCES)
            log_vulnerability(Reference, Medium);
        pop_identifier();
        unget_token(token);
        return;
    }

    scan_arguments();
    data = current_frame->data;
    analyzed = 0;

    /* If there's an info record, it's always a vulnerability.  Log it */
    if (data->Info != (Info_t *)NULL)
    {
        analyzed++;
        log_vulnerability(Info, data->Info->Severity);
    }

    if (data->FSProblem != (FSProblem_t *)NULL)
    {
        analyzed++;
        if ((arg = get_argument(data->FSProblem->Arg)) != (argument_t *)NULL)
            if (!arg->is_constant)
                log_vulnerability(FSProblem, data->FSProblem->Severity);
    }

    if (data->BOProblem != (BOProblem_t *)NULL)
    {
        analyzed++;
        if (data->BOProblem->FormatArg > 0)
            if ((arg = get_argument(data->BOProblem->FormatArg)) != (argument_t *)NULL)
                if (!arg->is_constant || arg->contains_ps)
                    log_vulnerability(BOProblem, data->BOProblem->Severity);
        if (data->BOProblem->SrcBufArg > 0)
            if ((arg = get_argument(data->BOProblem->SrcBufArg)) != (argument_t *)NULL)
                if (!arg->is_constant)
                    log_vulnerability(BOProblem, data->BOProblem->Severity);
    }

    if (data->InputProblem != (InputProblem_t *)NULL)
    {
        analyzed++;
        if ((arg = get_argument(data->InputProblem->Arg)) != (argument_t *)NULL)
            if (!arg->is_constant)
                log_vulnerability(InputProblem, data->InputProblem->Severity);
    }

    if (data->RaceCheck > 0)
    {
        analyzed++;
        record_toctou(data->RaceCheck, lineno, 0);
    }
    if (data->RaceUse > 0)
    {
        analyzed++;
        record_toctou(data->RaceUse, lineno, 1);
    }
    if (data->Input > 0)
    {
        analyzed++;
        if (flags & INPUT_MODE)
            record_input();
    }

    if (!analyzed)
        log_vulnerability(None, Default);

    pop_identifier();
}

static
int toctou_sort(const void *p1, const void *p2)
{
    toctou_t *  t1 = *(toctou_t **)p1;
    toctou_t *  t2 = *(toctou_t **)p2;

    if (strcmp(t1->key, t2->key) == 0)
        if (t1->use != t2->use)
            return (t1->use ? 1 : -1);

    if (t1->lineno == t2->lineno)
        return 0;
    return (t1->lineno < t2->lineno ? -1 : 1);
}

static
void process_toctou(void)
{
    int         check = -1, i = 0, start;
    char *      name = (char *)NULL;
    toctou_t *  toctou;
    toctou_t ** table;

    if (!toctou_count)
        return;

    /* build a table for sorting and sort it, first by name and then by check
     * vs. use.  sort checks ahead of uses
     */
    table = (toctou_t **)malloc(sizeof(toctou_t *) * toctou_count);
    for (toctou = toctous;  toctou != (toctou_t *)NULL;  toctou = toctou->next)
        table[i++] = toctou;
    qsort(table, toctou_count, sizeof(toctou_t *), toctou_sort);

    /* Go over toctou records and match them up */
    start = 0;
    for (i = 0;  i < toctou_count;  i++)
    {
        if (name == (char *)NULL || strcmp(table[i]->key, name) != 0)
        {
            if (name != (char *)NULL)
                log_toctou(table, start, i - 1, check);
            name = table[i]->key;
            check = (table[i]->use ? -1 : i);
            start = i;
        }
    }

    if (name != (char *)NULL)
        log_toctou(table, start, i - 1, check);

    /* cleanup */
    for (i = 0;  i < toctou_count;  i++)
    {
        free(table[i]->key);
        free(table[i]);
    }
    toctous = (toctou_t *)NULL;
    toctou_count = 0;
}

void process_file(char *filename, FILE *fd)
{
    int             i;
    accumulator_t * acc;

    /* (Re-)Initialize state */
    yyin = fd;
    lex_lineno = 1;
    current_file = strdup(filename);
    for (i = 0;  i < DEPTH_COUNT;  depths[i++] = 0);

    /* Process the file */
    scan_tokens((processorfn_t)NULL, NULL);
    process_toctou();

    /* Cleanup */
    while ((acc = accumulators) != (accumulator_t *)NULL)
    {
        if (*(acc->text) != (char *)NULL)
            free(*(acc->text));
        pop_accumulator();
    }
}
