/*
 * Placed into the public domain in 2002
 * by the author, Curt Sampson <cjs@cynic.net>.
 */

char RCSID[] = "$Id: randread.c,v 1.14 2002/09/11 06:25:46 cjs Exp $";
char VERSION[] = "@(#) randread v0.2 2002-09-11   "
    "This software is public domain.\n";

#include <sys/types.h>

#include <sys/disklabel.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "filelist.h"

/* Number of microseconds in a second. */
#define	USEC	(1000*1000)

/*
 * Parameters.
 *
 * Note that the start and end blocks are limited to values between 0 and
 * 2^31. With the default 8K block size, this limits the size of the seek
 * range to 16 TB. However, you can use a larger block size if you need a
 * larger seek range.
 */
static FILELIST flist;
static int block_size = 8192;
static int io_write_mode = 0;
static int io_length = 1;	/* # of blocks to read/write with each I/O. */
static int seek_count = 512;
static int file_count = 0;

/* The buffer to which we do our I/O.  Allocated in main(). */
static void *io_buffer;

/*
 * Do one block of I/O, read or write depending on the current mode.
 * Return 0 on success, -1 on failure.
 */
int do_io(int filedes)
{
    int retval;

    if (io_write_mode)
	retval = write(filedes, io_buffer, (size_t) block_size);
    else
	retval = read(filedes, io_buffer, (size_t) block_size);

#ifdef DEBUG
	fprintf(stderr, "%s %filedes=%d, buffer=0x%lx, size=%d, offset=%ld\n",
	    io_write_mode ? "write" : "read", filedes, (long) io_buffer,
	    block_size, (long) offset);
#endif
    if (retval == -1) {
	fprintf(stderr, "Error %s %s: %s\n",
	    io_write_mode ? "writing" : "reading",
	    "(unknown file)", strerror(errno));
	return -1;
    }
    if (retval != block_size) {
	fprintf(stderr, "Error %s %d bytes from %s instead of %d.\n",
	    io_write_mode ? "wrote" : "read",
	    retval, "(unknown file)", block_size);
	return -1;
    }
    return 0;
}

int run_test()
{
    long total_range;
    int count;
    long blockno;
    int fd;
    struct timeval start, end;
    long total_time;
    int up_to_block;

    count = seek_count;
    total_range = filelist_totalblocks(flist) - io_length;

    gettimeofday(&start, NULL);
    srandom(start.tv_usec);

    printf("%d %s of %d x %0.2f KB blocks (%0.2f KB)\n", seek_count,
	io_write_mode ? "writes" : "reads",
	io_length, block_size / 1024.0,
	(io_length * block_size) / 1024.0);
    printf("  totalling %d blocks (%0.2f MB)\n", seek_count * io_length,
	(seek_count * io_length / 1024.0 * block_size / 1024.0));
    printf("  from %ld blocks (%0.2f MB) in %d files.\n",
	filelist_totalblocks(flist),
	(total_range / 1024.0) * block_size / 1024.0, file_count);

    gettimeofday(&start, NULL);
    while (count-- > 0) {
	blockno = random() % total_range;
	up_to_block = blockno + io_length;
	while (blockno < up_to_block) {
	    fd = filelist_getfiledesc(flist, blockno++);
	    if (do_io(fd) == -1) {
		return 1;
	    }
	}
    }
    gettimeofday(&end, NULL);
    filelist_delete(flist);

    total_time = (end.tv_sec - start.tv_sec) * USEC;
    total_time += (end.tv_usec - start.tv_usec);
    if (end.tv_usec < start.tv_usec)
	total_time += USEC;

    printf("%d %s in %ld.%06ld sec.", seek_count,
	io_write_mode ? "writes" : "reads",
	total_time / USEC, total_time % USEC);
    printf(" (%ld usec/%s", total_time / seek_count,
	io_write_mode ? "write" : "read");
    if (total_time > USEC) {	/* Avoid divide by zero error. */
	printf(", %ld %s/sec", seek_count / (total_time / USEC),
	    io_write_mode ? "writes" : "reads");
	printf(", %0.2f KB/sec", (float) ((double) seek_count * io_length
	    / total_time * USEC * block_size / 1024.0));
    }
    printf(")\n");

    return 0;
}

void usage()
{
    fprintf(stderr, "Usage: randread [-b blocksize] [-c count] [-l length]"
	" [-W] file[:start-end] ...\n");
}

int
main(int argc, char **argv)
{
    /* Misc. */
    int optchar;
    int status;

    /* Process arguments. */
    while ((optchar = getopt(argc, argv, "b:c:l:s:W")) != -1) {
	switch (optchar) {
	case 'b':
	    block_size = atoi(optarg);
	    break;
	case 'c':
	    seek_count = atoi(optarg);
	    break;
	case 'l':
	    io_length = atoi(optarg);
	    break;
	case 'W':
	    io_write_mode = 1;
	    break;
	default:
	    usage();
	    return 2;
	}
    }
    argc -= optind;
    argv += optind;

    if (argc <= 0) {
	usage();
	exit(2);
    }

    flist = filelist_create(block_size, io_write_mode ? O_WRONLY : O_RDONLY);
    if (flist == NULL) {
	perror("Couldn't create filelist: ");
	exit(1);
    }
    while (argc > 0) {
	status = filelist_add(flist, *argv);
	if (status < 0) {
	    fprintf(stderr, "Can't add file %s: status %d, errno %d\n",
		*argv, status, errno);
	    exit(1);
	}
	file_count++;
	argc--;
	argv++;
    }

    /* Allocate the buffer we'll use for I/O. */
    io_buffer = malloc(block_size);
    if (io_buffer == NULL) {
	fprintf(stderr, "Can't allocate %d bytes of memory: %s\n",
	    block_size, strerror(errno));
	return 1;
    }

    return run_test();
}

