/*
 * Placed into the public domain in 2002
 * by the author, Curt Sampson <cjs@cynic.net>.
 */

#include <fcntl.h>
#include <stddef.h>
#include <unistd.h>

#include <check.h>
#include "filelist.h"

START_TEST(test_create)
{
    FILELIST list;

    list = filelist_create(65536, O_RDONLY);
    fail_unless(list != NULL, "filelist created");
    fail_unless(filelist_blocksize(list) == 65536, "blocksize");
    filelist_delete(list);
    filelist_delete(NULL);
}
END_TEST

/*
 * Unfortunately, this can't work on systems where O_RDONLY = 0. Oh well.
 *
START_TEST(test_open_flags)
{
    FILELIST list;

    list = filelist_create(8192, O_RDONLY);
    fail_unless(list != NULL, "O_RDONLY");
    filelist_delete(list);

    list = filelist_create(8192, O_WRONLY);
    fail_unless(list != NULL, "O_WRONLY");
    filelist_delete(list);

    list = filelist_create(8192, O_RDWR);
    fail_unless(list != NULL, "O_RDWR");
    filelist_delete(list);

    list = filelist_create(8192, 0);
    fail_unless(list == NULL, "0");
}
END_TEST
*/


START_TEST(test_add)
{
    FILELIST list;

    list = filelist_create(4096, O_RDONLY);
    fail_unless(filelist_blocksize(list) == 4096, "blocksize");
    fail_unless(filelist_totalblocks(list) == 0, "totalblocks 0");

    fail_unless(filelist_add(list, "test_file") == 0, "add test_file");
    fail_unless(filelist_totalblocks(list) == 16, "totalblocks 16");

    fail_unless(filelist_add(list, "filelist_test:0-2") == 0,
	"add filelist_test:0-2");
    fail_unless(filelist_totalblocks(list) == 18, "totalblocks 18");

    filelist_delete(list);
}
END_TEST

START_TEST(test_getblockentry)
{
    FILELIST list;
    struct filelist_entry *entry;
    long offset = -54321;

    list = filelist_create(4096, O_RDONLY);
    fail_unless(filelist_add(list, "test_file") == 0, "add test_file");
    fail_unless(filelist_add(list, "filelist_test:0-2") == 0,
	"add filelist_test:0-2");
    fail_unless(filelist_add(list, "filelist_test:1-3") == 0,
	"add filelist_test:1-3");

    entry = filelist_getblockentry(list, -1, &offset);
    fail_unless(entry == NULL, "block -1");
    fail_unless(offset == -54321, "block -1 offset");

    entry = filelist_getblockentry(list, 20, &offset);
    fail_unless(entry == NULL, "block 20");
    fail_unless(offset == -54321, "block 20 offset");

    entry = filelist_getblockentry(list, 87654321, &offset);
    fail_unless(entry == NULL, "block 87654321");
    fail_unless(offset == -54321, "block 87654321 offset");

    entry = filelist_getblockentry(list, 0, &offset);
    fail_unless(!strcmp(entry->path, "test_file"), "block 0");
    fail_unless(offset == 0, "block 0 offset");

    entry = filelist_getblockentry(list, 1, &offset);
    fail_unless(!strcmp(entry->path, "test_file"), "block 1");
    fail_unless(offset == 1, "block 1 offset");

    entry = filelist_getblockentry(list, 15, &offset);
    fail_unless(!strcmp(entry->path, "test_file"), "block 15");
    fail_unless(offset == 15, "block 15 offset");

    entry = filelist_getblockentry(list, 16, &offset);
    fail_unless(!strcmp(entry->path, "filelist_test"), "block 16");
    fail_unless(offset == 0, "block 16 offset");

    entry = filelist_getblockentry(list, 17, &offset);
    fail_unless(!strcmp(entry->path, "filelist_test"), "block 17");
    fail_unless(offset == 1, "block 17 offset");

    entry = filelist_getblockentry(list, 18, &offset);
    fail_unless(!strcmp(entry->path, "filelist_test"), "block 17");
    fail_unless(offset == 1, "block 18 offset");

    entry = filelist_getblockentry(list, 19, &offset);
    fail_unless(!strcmp(entry->path, "filelist_test"), "block 17");
    fail_unless(offset == 2, "block 19 offset");

    filelist_delete(list);
}
END_TEST

/*
 * For this particular test, test_file is expected to be all zeros,
 * except for an 'A' at the beginning of the 4th 1024 byte block.
 */
START_TEST(test_getfiledesc)
{
    FILELIST list;
    int fd;
    off_t offset;
    ssize_t size;
    char buf[1];

    list = filelist_create(1024, O_RDONLY);
    filelist_add(list, "filelist_test:0-2");
    filelist_add(list, "test_file");

    fd = filelist_getfiledesc(list, 6);
    fail_unless(fd >= 0, "fd is valid");

    offset = lseek(fd, 0, SEEK_CUR);
    fail_unless(offset == 4096, "offset == 4096");

    size = read(fd, buf, 1);
    fail_unless(size == 1, "read size == 1");
    fail_unless(buf[0] == 'A', "buf[0] == 'A'");

    filelist_delete(list);
}
END_TEST

Suite *get_suite(void) 
{ 
    Suite *s = suite_create("randread"); 
    TCase *tc_all = tcase_create("filelist");
    suite_add_tcase(s, tc_all);

    tcase_add_test(tc_all, test_create); 
    tcase_add_test(tc_all, test_add);
    tcase_add_test(tc_all, test_getblockentry);
    tcase_add_test(tc_all, test_getfiledesc);

    return s; 
}

int main(void) 
{ 
    int num_failed; 

    Suite *s = get_suite(); 
    SRunner *sr = srunner_create (s); 
    srunner_run_all (sr, CK_NORMAL); 
    num_failed = srunner_ntests_failed (sr); 
    srunner_free (sr); 
    suite_free (s); 

    return num_failed;
}
