/*
Copyright (C) 1996-1997 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// net_main.c

/*
-------------------------------------------------------------------------------

Code change by Viagenie Inc.
Florent Parent and Andre Cormier

send comments to quake-v6@viagenie.qc.ca

The change uses AF_INET6 to check if an ipv6 stack as been installed. So this
code should compile on any ipv4 only stack as on any ipv6 kame stack. It's
been coded under freebsd 3.2 RELEASE with Kame stable and solaris 8.

The ipv6 code as been inspired by the ipv6 patch submitted by Pontus Lidman
he can be joined at <pontus@lysator.liu.se>.

any other implemantation as not been tested.

-------------------------------------------------------------------------------
*/

#include "quakedef.h"

#ifdef __sun__
/* Sun's model_t in sys/model.h conflicts w/ Quake's model_t */
#define model_t sunmodel_t
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <arpa/inet.h>
#include <errno.h>

#ifdef __sun__
#include <unistd.h>
#include <sys/filio.h>
#undef model_t
#endif

#ifdef NeXT
#include <libc.h>
#endif

#ifndef AF_INET6

#define IPV4_ONLY_STACK
typedef struct sockaddr_in sockaddress_t;
#else
typedef struct sockaddr_in6 sockaddress_t;
#ifdef __sun__
#define s6_addr32       _S6_un._S6_u32
#endif __sun__
#ifdef __FreeBSD__
#define s6_addr32       __u6_addr.__u6_addr32
#endif __FreeBSD__

#endif AF_INET6

netadr_t	net_local_adr;

netadr_t	net_from;
sizebuf_t	net_message;
int			net_socket;			// non blocking, for receives
int			net_send_socket;	// blocking, for sends

#define	MAX_UDP_PACKET	8192
byte		net_message_buffer[MAX_UDP_PACKET];

int gethostname (char *, int);
int close (int);

//=============================================================================

void NetadrToSockadr (netadr_t *a, sockaddress_t *s)
{
#ifdef IPV4_ONLY_STACK
	memset (s, 0, sizeof(*s));
	s->sin_family = AF_INET;

	*(int *)&s->sin_addr = *(int *)&a->ip;
	s->sin_port = a->port;
#else
	memset (s, 0, sizeof(*s));
	//	s->sin6_family = a->family;

	s->sin6_family = AF_UNSPEC;
	s->sin6_addr.s6_addr32[0] = a->ip[0];
	s->sin6_addr.s6_addr32[1] = a->ip[1];
	s->sin6_addr.s6_addr32[2] = a->ip[2];
	s->sin6_addr.s6_addr32[3] = a->ip[3];
	s->sin6_port = a->port;
	s->sin6_len = sizeof(struct sockaddr_in6);
#endif
}

void SockadrToNetadr (sockaddress_t *s, netadr_t *a)
{
#ifdef IPV4_ONLY_STACK
	*(int *)&a->ip = *(int *)&s->sin_addr;
	a->port = s->sin_port;
#else
	a->ip[0] = s->sin6_addr.s6_addr32[0];
	a->ip[1] = s->sin6_addr.s6_addr32[1];
	a->ip[2] = s->sin6_addr.s6_addr32[2];
	a->ip[3] = s->sin6_addr.s6_addr32[3];
	a->port = s->sin6_port;
	a->family = s->sin6_family;
#endif
}

qboolean	NET_CompareBaseAdr (netadr_t a, netadr_t b)
{
	if (a.ip[0] == b.ip[0] && a.ip[1] == b.ip[1] && a.ip[2] == b.ip[2] && a.ip[3] == b.ip[3])
		return true;
	return false;
}


qboolean	NET_CompareAdr (netadr_t a, netadr_t b)
{
	if (a.ip[0] == b.ip[0] && a.ip[1] == b.ip[1] && a.ip[2] == b.ip[2] && a.ip[3] == b.ip[3] && a.port == b.port)
		return true;
	return false;
}

#ifdef IPV4_ONLY_STACK
char    *NET_AdrToString (netadr_t a)
{
	static  char    s[64];

	sprintf (s, "%i.%i.%i.%i:%i", a.ip[0], a.ip[1], a.ip[2], a.ip[3], ntohs(a.port));

	return s;
}

char    *NET_BaseAdrToString (netadr_t a)
{
	static  char    s[64];

	sprintf (s, "%i.%i.%i.%i", a.ip[0], a.ip[1], a.ip[2], a.ip[3]);

	return s;
}
#else
char    *NET_AdrToString (netadr_t a)
{
	static  char    s[64];
	char *base;

	base=NET_BaseAdrToString(a);
	sprintf(s,"%s,%d",base,ntohs(a.port));
	return s;
}

char    *NET_BaseAdrToString (netadr_t a)
{
	static  char    s[BUFSIZ];
	struct sockaddr_storage ss;

	ss.__ss_len = sizeof(ss);
	ss.__ss_family = AF_UNSPEC;
	memcpy(ss.__ss_pad1, &(a.ip), sizeof(a.ip));
	if (getnameinfo((struct sockaddr *) &ss,sizeof(ss),s,sizeof(s),NULL,0,NI_NUMERICHOST)) {
		strcpy(s,"<invalid>");
	}
	return s;
}

#endif IPV4_ONLY_STACK

/*
=============
NET_StringToAdr

idnewt
idnewt:28000
192.246.40.70
192.246.40.70:28000
=============
*/
#ifdef IPV4_ONLY_STACK
qboolean	NET_StringToAdr (char *s, netadr_t *a)
{
	struct hostent	*h;
	sockaddress_t sadr;
	char	*colon;
	char	copy[128];
	
	
	memset (&sadr, 0, sizeof(sadr));
	sadr.sin_family = AF_INET;
	
	sadr.sin_port = 0;

	strcpy (copy, s);
	// strip off a trailing :port if present
	for (colon = copy ; *colon ; colon++)
		if (*colon == ':')
		{
			*colon = 0;
			sadr.sin_port = htons(atoi(colon+1));	
		}
	
	if (copy[0] >= '0' && copy[0] <= '9')
	{
		*(int *)&sadr.sin_addr = inet_addr(copy);
	}
	else
	{
		if (! (h = gethostbyname(copy)) )
			return 0;
		*(int *)&sadr.sin_addr = *(int *)h->h_addr_list[0];
	}
	
	SockadrToNetadr (&sadr, a);

	return true;
}
#else

qboolean        NET_StringToAdr (char *s, netadr_t *a)
{       

	struct addrinfo hints;
	struct addrinfo *resultp;
	char *space;
	char *ports=NULL;
	char copy[128];
	int err;
	struct sockaddr_storage ss;
	struct sockaddr_in6 *ss6;
	struct sockaddr_in *ss4;

	memset(&hints,0,sizeof(hints));
	hints.ai_socktype=SOCK_DGRAM;
	hints.ai_family=PF_UNSPEC;

	strcpy(copy,s);
	for (space=copy; *space; space++) {
	   if (*space==' ') {
	      *space=0;
	      ports=space+1;
	   }
	}

        if ((err=getaddrinfo(copy,ports,&hints,&resultp)))
          {
            // Error
              Con_Printf ("NET_StringToAdr: string %s: %s",s, gai_strerror(err));
            return 0;
          }

        switch (resultp->ai_family) {
        case AF_INET:
          // convert to ipv6 addr
          memset(&ss,0,sizeof(ss));
          ss6=(struct sockaddr_in6 *) &ss;
          ss4=(struct sockaddr_in *) resultp->ai_addr;
          ss6->sin6_family=AF_INET6;
          ss6->sin6_addr.s6_addr32[0]=0;
          ss6->sin6_addr.s6_addr32[1]=0;
          ss6->sin6_addr.s6_addr32[2]=htonl(0xffff);
          ss6->sin6_addr.s6_addr32[3]=ss4->sin_addr.s_addr;
          break;
        case AF_INET6:
          memcpy(&ss,resultp->ai_addr,sizeof(struct sockaddr_in6));
          break;
        default:
            Con_Printf ("NET_StringToAdr: string %s: protocol family %d not supported",s, resultp->ai_family);     
            return 0;
        }
        SockadrToNetadr ((struct sockaddr_in6 *) &ss, a);
        
        return true;
}
#endif


// Returns true if we can't bind the address locally--in other words, 
// the IP is NOT one of our interfaces.
qboolean NET_IsClientLegal(netadr_t *adr)
{
	sockaddress_t sadr;
	int newsocket;

#if 0
	if (adr->ip[0] == 127)
		return false; // no local connections period

	NetadrToSockadr (adr, &sadr);

	if ((newsocket = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
		Sys_Error ("NET_IsClientLegal: socket:", strerror(errno));

	sadr.sin_port = 0;

	if( bind (newsocket, (void *)&sadr, sizeof(sadr)) == -1) 
	{
		// It is not a local address
		close(newsocket);
		return true;
	}
	close(newsocket);
	return false;
#else
	return true;
#endif
}


//=============================================================================

qboolean NET_GetPacket (void)
{
	int 	ret;
	sockaddress_t	from;
	int		fromlen;

	fromlen = sizeof(from);
	ret = recvfrom (net_socket, net_message_buffer, sizeof(net_message_buffer), 0, (struct sockaddr *)&from, &fromlen);
	if (ret == -1) {
		if (errno == EWOULDBLOCK)
			return false;
		if (errno == ECONNREFUSED)
			return false;
		Sys_Printf ("NET_GetPacket: %s\n", strerror(errno));
		return false;
	}

	net_message.cursize = ret;
	SockadrToNetadr (&from, &net_from);

	return ret;
}

//=============================================================================

void NET_SendPacket (int length, void *data, netadr_t to)
{
	int ret;
	sockaddress_t	addr;

	NetadrToSockadr (&to, &addr);

/*
	Con_Printf("sendpacket: family = %d, len = %d", addr.sin6_family, addr.sin6_len);
*/
	ret = sendto (net_socket, data, length, 0, (struct sockaddr *)&addr, sizeof(addr) );
	if (ret == -1) {
		if (errno == EWOULDBLOCK)
			return;
		if (errno == ECONNREFUSED)
			return;
		Sys_Printf ("NET_SendPacket: %s\n", strerror(errno));
	}
}

//=============================================================================

int UDP_OpenSocket (int port)
{
	int newsocket;

#ifdef IPV4_ONLY_STACK
	sockaddress_t address;
	qboolean _true = true;
	int i;

	if ((newsocket = socket (PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1)
		Sys_Error ("UDP_OpenSocket: socket:", strerror(errno));
	if (ioctl (newsocket, FIONBIO, (char *)&_true) == -1)
		Sys_Error ("UDP_OpenSocket: ioctl FIONBIO:", strerror(errno));

	address.sin_family = AF_INET;

//ZOID -- check for interface binding option
	if ((i = COM_CheckParm("-ip")) != 0 && i < com_argc) {
		address.sin_addr.s_addr = inet_addr(com_argv[i+1]);
		Con_Printf("Binding to IP Interface Address of %s\n",
				inet_ntoa(address.sin_addr));
	} else
		address.sin_addr.s_addr = INADDR_ANY;
	if (port == PORT_ANY)
		address.sin_port = 0;
	else
		address.sin_port = htons((short)port);
	if( bind (newsocket, (void *)&address, sizeof(address)) == -1)
		Sys_Error ("UDP_OpenSocket: bind: %s", strerror(errno));
#else
	struct addrinfo hints, *res;
	int Error;
	char Buf[BUFSIZ], *Service, *Host;
	qboolean _true = true;
	int i;

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_UNSPEC;
	hints.ai_socktype = SOCK_DGRAM;
	hints.ai_protocol = IPPROTO_UDP;
	hints.ai_flags = AI_PASSIVE;

	if ((i = COM_CheckParm("-ip")) != 0 && i < com_argc) {
		Host = com_argv[i+1];
	} else {
		Host = "0::0";
	}
	Con_Printf("Binding to IP Interface Address of %s\n", Host);

	if (port == PORT_ANY)
		Service = NULL;
	else {
		sprintf(Buf, "%5d", port);
                Service = Buf;
        }

	if(Error = getaddrinfo(Host, Service, &hints, &res))
		Sys_Error ("UDP_OpenSocket: getaddrinfo: %s", gai_strerror(Error));

	if ((newsocket = socket(res->ai_family, res->ai_socktype, res->ai_protocol)) == -1)
		Sys_Error ("UDP_OpenSocket: socket:", strerror(errno));
	if (ioctl (newsocket, FIONBIO, (char *)&_true) == -1)
		Sys_Error ("UDP_OpenSocket: ioctl FIONBIO:", strerror(errno));

	if (bind(newsocket, res->ai_addr, res->ai_addrlen) < 0)
		Sys_Error ("UDP_OpenSocket: bind: %s", strerror(errno));

	freeaddrinfo(res);

#endif

	return newsocket;
}

void NET_GetLocalAddress (void)
{
	char	buff[MAXHOSTNAMELEN];
	sockaddress_t	address;
	int		namelen;

	gethostname(buff, MAXHOSTNAMELEN);
	buff[MAXHOSTNAMELEN-1] = 0;

	NET_StringToAdr (buff, &net_local_adr);

	namelen = sizeof(address);
	if (getsockname (net_socket, (struct sockaddr *)&address, &namelen) == -1)
		Sys_Error ("NET_Init: getsockname:", strerror(errno));
#ifdef IPV4_ONLY_STACK
	net_local_adr.port = address.sin_port;
#else
	net_local_adr.port = address.sin6_port;
#endif

	Con_Printf("IP address %s\n", NET_AdrToString (net_local_adr) );
}

/*
====================
NET_Init
====================
*/
void NET_Init (int port)
{
	//
	// open the single socket to be used for all communications
	//
	net_socket = UDP_OpenSocket (port);

	//
	// init the message buffer
	//
	net_message.maxsize = sizeof(net_message_buffer);
	net_message.data = net_message_buffer;

	//
	// determine my name & address
	//
	NET_GetLocalAddress ();

	Con_Printf("UDP Initialized\n");
}

/*
====================
NET_Shutdown
====================
*/
void	NET_Shutdown (void)
{
	close (net_socket);
}

