/*
*  File      : xwindow.cpp
*  Written by: alinden@gmx.de
*  Copyright : GPL
*
*  This is constructed for every normal window
*/

#include "defs.h"
#include "defaults.h"
#include "qapp.h"
#include "rubber.h"
#include "moc_xwindow.cpp"

xwindow::xwindow(Window w, QWidget *parent) : QWidget(parent) 
{
	dt = QApplication::desktop();
	maxstate = 0;
	mrb = NULL;
	clientid = w;
	actpal = TRUE;
	urgpal = FALSE;
	cmapwins = NULL;
	sstate = FALSE;
	winfo = NULL;
	ncmaps = 0;
	withdrawnstate = unmapped = TRUE;

	// get ClassHint
	get_classhint();
	clientname = res_class;

	// get flags for WM_COMMAND
	
	if(clientname == "unnamed")  // workaround for some Qt versions
		clientname = res_name;
		
	char **argv;
	int argc;
	if(XGetCommand(qt_xdisplay(), w, &argv, &argc) && argc)
	{
		int ncm=0;
		while(1)
		{
			command += argv[ncm];

			if(argc > ++ncm)
				command += ' ';
			else
				break;
		}
		
		if(clientname.isEmpty() && ! command.isEmpty())
		{
			char *base;
			if((base = strrchr(argv[0], '/')) == NULL)
				base = argv[0];
			else
				base++;

			clientname = base;
		}	
		XFreeStringList(argv);
	}
	clientname = clientname.stripWhiteSpace();
	pflags = qapp::cprops[clientname];
	
	if(pflags & qapp::SmallFrame)
	{
		uborderh = defaults::lowerborderheight;
		borderh = 2*uborderh;
	}
	else
	{
		uborderh = defaults::windowbuttonsize;
		borderh = defaults::windowbuttonsize+defaults::lowerborderheight;
	}
	
	setBackgroundColor(QColor(Qt::black));
	
	// check for nonrectangular window
	shaped = (qapp::servershapes)?(query_shape()):(FALSE);

	if(shaped || (pflags & qapp::NoResize))
		borderh -= defaults::lowerborderheight;

	// save window geometry
	Window root;
	uint bwidth,depth;
	XGetGeometry(qt_xdisplay(), w, &root, &pos_x, &pos_y, &init_w, &init_h, &bwidth, &depth);
	base_w = init_w;
	base_h = init_h;
	init_h += borderh;
	
	// reparent
	XSetWindowBorderWidth(qt_xdisplay(), w, 0);
	XSetWindowBorderWidth(qt_xdisplay(), winId(), 0);
	XReparentWindow(qt_xdisplay(), w, winId(), 0, uborderh);
	XAddToSaveSet(qt_xdisplay(), w);

	// get TransientForHint
	transfor = None;
	XGetTransientForHint(qt_xdisplay(), w, &transfor);

	// set Font
	setFont(defaults::borderfont);

	// get colormap and check for WM_COLORMAP_WINDOWS property
	get_colormaps();

	// get WM_CLIENT_MACHINE
	XTextProperty prop;
	if(XGetWMClientMachine(qt_xdisplay(), w, &prop) && prop.nitems && prop.format == 8)
	{
		clientfqdn = (char *)prop.value;

		if(defaults::showclientmachines)
		{
			int pos = clientfqdn.find('.');
			if(pos == -1)
				clientmachine = clientfqdn;
			else	
				clientmachine = clientfqdn.left(pos);
		}
	}	

	// get WMHints
	get_wmhints();

	// get WMNormalhints
	get_wmnormalhints();
	
	int cw = init_w;
	int ch = init_h;
	getsize(&cw, &ch);
		
	// window position

	if(wmnflags & USPosition)
	{
		if(wmnflags & PWinGravity && 
			(wingrav == SouthWestGravity || wingrav == SouthEastGravity))
				pos_y -= uborderh;
	}

	// a transient window with program specified position looks like a dialog box,
	// otherwise use autoplacement
	
	else if(transfor == None || ! (wmnflags & PPosition))
	{
		if(qapp::next_x+cw > dt->width())
		{
			pos_x = 0;
			if(cw < dt->width())
				qapp::next_x = 2*defaults::windowbuttonsize;
		}
		else
		{
			pos_x = qapp::next_x;
			qapp::next_x += 2*defaults::windowbuttonsize;
		}	

		int sy,ey;
		if(defaults::toolbar_top)
		{
			sy = defaults::tb_height;
			ey = dt->height();
		}
		else
		{
			sy = 0;
			ey = dt->height()-defaults::tb_height;
		}
		
		if(qapp::next_y+ch > ey)
		{
			pos_y = sy;
			if(ch < dt->height())
				qapp::next_y = sy+2*defaults::windowbuttonsize;
		}
		else
		{
			pos_y = qapp::next_y;
			qapp::next_y += 2*defaults::windowbuttonsize;
		}
	}	

	// move and resize
	
	XResizeWindow(qt_xdisplay(), clientid, cw, ch-borderh);
	setGeometry(pos_x, pos_y, cw, ch);

    	// overwrite Qt-defaults because we need SubstructureNotifyMask
	
	XSelectInput(qt_xdisplay(), winId(),
  		 KeyPressMask | KeyReleaseMask |
  		 ButtonPressMask | ButtonReleaseMask |
  		 KeymapStateMask |
   		 ButtonMotionMask |
  		 PointerMotionMask |
		 EnterWindowMask | LeaveWindowMask |
  		 FocusChangeMask |
  		 ExposureMask |
		 StructureNotifyMask |
		 SubstructureRedirectMask |
		 SubstructureNotifyMask);

	XSetWindowAttributes attr;
	attr.event_mask = ColormapChangeMask|PropertyChangeMask;
	XChangeWindowAttributes(qt_xdisplay(), w, CWEventMask, &attr);

	// get WM protocols
	getwmprotocols();

	// create window borders
	create_wborder();

	// add client to lookup tables
	
	qapp::cwindows.insert(w, this);
	qapp::pwindows.insert(winId(), this);

	// get WM_NAME and set window title
	get_wmname();

	if(shaped)
		reshape();

	// init autofocus timer
	
	focustimer = new QTimer(this);
	CHECK_PTR(focustimer);
	connect(focustimer, SIGNAL(timeout()), SLOT(raise()));
	
	if(! urgent && ! defaults::starturgent)
	{
		setinactive();
	}
	else 
	{
		seturgent();
	}	

	map();

#ifdef DEBUGMSG
	cerr << "class xwindow constructed (WId:" << winId() << ")\n";
#endif	
}

void xwindow::create_wborder(void)
{
	lbdr = NULL;
	ubdr = NULL;
	layout = new QVBoxLayout(this);
	CHECK_PTR(layout);
	layout->setMargin(0);

	if(pflags & qapp::SmallFrame)
	{
		midmove = new wframe(this, "smallframe");
		CHECK_PTR(midmove);
		midmove->setFixedHeight(defaults::lowerborderheight);
		layout->add(midmove);
		
		connect(midmove, SIGNAL(left_press(QMouseEvent *)), SLOT(press_move(QMouseEvent *)));
		connect(midmove, SIGNAL(left_release(QMouseEvent *)), SLOT(release_move(QMouseEvent *)));
		connect(midmove, SIGNAL(right_press()), SLOT(show_info()));
		connect(midmove, SIGNAL(right_release()), SLOT(release_info()));
		connect(midmove, SIGNAL(mid_press()), SLOT(show_info()));
		connect(midmove, SIGNAL(mid_release()), SLOT(release_info()));
		connect(midmove, SIGNAL(mouse_move(QMouseEvent *)), SLOT(move_move(QMouseEvent *)));
	}
	else
	{
		ubdr = new uborder((transfor == None), this, "upperborder");
		CHECK_PTR(ubdr);
		layout->add(ubdr);
		midmove = ubdr->midframe;
		
		if(transfor == None)
		{
			connect(ubdr->leftframe, SIGNAL(right_press()), SLOT(t_maximize()));
			connect(ubdr->leftframe, SIGNAL(left_press()), SLOT(iconify()));
		}
		connect(ubdr->rightframe, SIGNAL(press()), SLOT(wdestroy()));
		connect(midmove, SIGNAL(right_press()), SLOT(s_maximize()));
		connect(midmove, SIGNAL(left_press(QMouseEvent *)), SLOT(press_move(QMouseEvent *)));
		connect(midmove, SIGNAL(left_release(QMouseEvent *)), SLOT(release_move(QMouseEvent *)));
		connect(midmove, SIGNAL(mid_press()), SLOT(show_info()));
		connect(midmove, SIGNAL(mid_release()), SLOT(release_info()));
		connect(midmove, SIGNAL(mouse_move(QMouseEvent *)), SLOT(move_move(QMouseEvent *)));
	}	
	layout->addStretch();

	if(! shaped && ! (pflags & qapp::NoResize))
	{
		lbdr = new lborder(this, "lowerborder");
		CHECK_PTR(lbdr);
		layout->add(lbdr);
		
		connect(lbdr->leftframe, SIGNAL(press(QMouseEvent *)), SLOT(press_leftresize(QMouseEvent *)));
		connect(lbdr->leftframe, SIGNAL(release(QMouseEvent *)), SLOT(release_leftresize(QMouseEvent *)));
		connect(lbdr->leftframe, SIGNAL(mouse_move(QMouseEvent *)), SLOT(move_leftresize(QMouseEvent *)));
		connect(lbdr->rightframe, SIGNAL(press(QMouseEvent *)), SLOT(press_rightresize(QMouseEvent *)));
		connect(lbdr->rightframe, SIGNAL(release(QMouseEvent *)), SLOT(release_rightresize(QMouseEvent *)));
		connect(lbdr->rightframe, SIGNAL(mouse_move(QMouseEvent *)), SLOT(move_rightresize(QMouseEvent *)));
		connect(lbdr->midframe, SIGNAL(press(QMouseEvent *)), SLOT(press_midresize(QMouseEvent *)));
		connect(lbdr->midframe, SIGNAL(release(QMouseEvent *)), SLOT(release_midresize(QMouseEvent *)));
		connect(lbdr->midframe, SIGNAL(mouse_move(QMouseEvent *)), SLOT(move_midresize(QMouseEvent *)));
	}	
}

void xwindow::getsize(int *pw, int *ph)   // adjust for possible width and height including border
{
	int w = *pw;
	int h = *ph;

	if(inc_w == 0)
		w = init_w;

	if(inc_h == 0)
		h = init_h;

	if(w > max_w)
		w = max_w;

	if(h > max_h)
		h = max_h;

	if(w < min_w)
		w = min_w;

	if(h < min_h)
		h = min_h;

	if(inc_w > 1)
	{
		int i = (w-base_w)/inc_w;
		w = base_w+i*inc_w;
		
		if(w < min_w)
			w += inc_w;
	}

	if(inc_h > 1)	
	{
		int j = (h-base_h)/inc_h;
		h = base_h+j*inc_h;
		
		if(h < min_h)
			h += inc_h;
	}	
	*pw = w;
	*ph = h;
}

void xwindow::resize_request(int cx, int cy, int cw, int ch)  // client requested resize
{
	if(cy < 0)
		cy = 0;
		
	ch += borderh;
	getsize(&cw, &ch);

	setGeometry(cx, cy, cw, ch);
	XResizeWindow(qt_xdisplay(), clientid, cw, ch-borderh);
	maxstate = 0;
}

void xwindow::resize_client(int px, int py, int pw, int ph)  // dimensions include borders
{
	int nw = width();
	int nh = height();

	if(px != x() || py != y() || pw != nw || ph != nh)  // parent
		setGeometry(px, py, pw, ph);

	if(pw != nw || ph != nh) // client
	{
#ifdef DEBUGMSG
		cerr << "resize child window (WId:" << clientid << ")\n";
#endif
		XResizeWindow(qt_xdisplay(), clientid, pw, ph-borderh);
	}
}

void xwindow::t_maximize(void)
{
	if(qapp::smode)
	{
		focus_mouse();
		return;
	}
	
	if(maxstate != 1)  // maximize
	{
		if(maxstate == 0)
		{
			icx = x();
			icy = y();
			icw = width();
			ich = height();
		}
		int cw = dt->width();
		int ch = dt->height()-defaults::tb_height-1;
		getsize(&cw, &ch);
		
		resize_client(0, defaults::toolbar_top?defaults::tb_height+1:0, cw, ch);
		maxstate = 1;
	}
	else
	{
		resize_client(icx, icy, icw, ich);
		maxstate = 0;
	}
	raise();
}

void xwindow::s_maximize(void)
{
	if(qapp::smode)
	{
		focus_mouse();
		return;
	}
	
	if(maxstate != 2)  // maximize
	{
		if(maxstate == 0)
		{
			icx = x();
			icy = y();
			icw = width();
			ich = height();
		}
		int cw = dt->width();
		int ch = dt->height();
		getsize(&cw, &ch);
		
		resize_client(0, 0, cw, ch);
		maxstate = 2;
	}
	else
	{
		resize_client(icx, icy, icw, ich);
		maxstate = 0;
	}
	raise();
}

void xwindow::press_move(QMouseEvent *event)
{
	if(mrb != NULL)
		return;
		
	mousepos = event->pos()+midmove->pos();  // offset
	midmove->grabMouse(QCursor(Qt::sizeAllCursor));
	mrb = new rubber(0, "move-rubberband");
	move_move(event);   // draw frame
}

void xwindow::release_move(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	delete mrb;
	mrb = NULL;
	midmove->releaseMouse();
	move(event->globalPos()-mousepos);
	maxstate = 0;
	raise();
}

void xwindow::move_move(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	QPoint p = event->globalPos()-mousepos;
	mrb->draw(p.x(), p.y(), width(), height());
}

void xwindow::press_leftresize(QMouseEvent *event)
{
	if(mrb != NULL)
		return;
		
	mousepos = event->globalPos();
	lbdr->leftframe->grabMouse(QCursor(Qt::sizeAllCursor));
	mrb = new rubber(base_w, base_h, inc_w, inc_h, 0, "left-rubberband");
	move_leftresize(event);   // draw frame
}

void xwindow::release_leftresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	delete mrb;
	mrb = NULL;
	lbdr->leftframe->releaseMouse();
	QPoint dpos = event->globalPos()-mousepos;
	int resw = width()-dpos.x();
	int resh = height()+dpos.y();

	getsize(&resw, &resh);
	int resx = x()-resw+width();
	
	if(resx > x()+width())
		resx = x()+width();

	resize_client(resx, y(), resw, resh);
	maxstate = 0;
	raise();
}

void xwindow::move_leftresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
	
	QPoint dpos = event->globalPos()-mousepos;
	int resw = width()-dpos.x();
	int resh = height()+dpos.y();

	getsize(&resw, &resh);
	int resx = x()-resw+width();
	
	if(resx+min_w > x()+width())
		resx = x()+width()-min_w;
		
	mrb->draw(resx, y(), resw, resh);
}

void xwindow::press_rightresize(QMouseEvent *event)
{
	if(mrb != NULL)
		return;
		
	mousepos = event->globalPos();
	lbdr->rightframe->grabMouse(QCursor(Qt::sizeAllCursor));
	mrb = new rubber(base_w, base_h, inc_w, inc_h, 0, "right-rubberband");
	move_rightresize(event);   // draw frame
}

void xwindow::release_rightresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	delete mrb;
	mrb = NULL;	
	lbdr->rightframe->releaseMouse();
	QPoint dpos = event->globalPos()-mousepos;
	int resw = width()+dpos.x();
	int resh = height()+dpos.y();
	
	getsize(&resw, &resh);
	resize_client(x(), y(), resw, resh);
	maxstate = 0;
	raise();
}

void xwindow::move_rightresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	QPoint dpos = event->globalPos()-mousepos;
	int resw = width()+dpos.x();
	int resh = height()+dpos.y();

	getsize(&resw, &resh);
	mrb->draw(x(), y(), resw, resh);
}

void xwindow::press_midresize(QMouseEvent *event)
{
	if(mrb != NULL)
		return;
		
	mousepos = event->globalPos();
	lbdr->midframe->grabMouse(QCursor(Qt::sizeAllCursor));
	mrb = new rubber(base_w, base_h, inc_w, inc_h, 0, "mid-rubberband");
	move_midresize(event);   // draw frame
}

void xwindow::release_midresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	delete mrb;
	mrb = NULL;
	lbdr->midframe->releaseMouse();
	QPoint dpos = event->globalPos()-mousepos;
	int resh = height()+dpos.y();
	int resw = width();
		
	getsize(&resw, &resh);	
	resize_client(x(), y(), resw, resh);
	maxstate = 0;
	raise();
}

void xwindow::move_midresize(QMouseEvent *event)
{
	if(mrb == NULL)
		return;
		
	QPoint dpos = event->globalPos()-mousepos;
	int resh = height()+dpos.y();
	int resw = width();

	getsize(&resw, &resh);
	mrb->draw(x(), y(), resw, resh);
}

void xwindow::show_info(void)
{
	if(winfo != NULL)
		return;

	winfo = new QWidget(0, "winfo");
	XSetWindowBorderWidth(qt_xdisplay(), winfo->winId(), 1);

	QLabel *left = new QLabel(winfo);
	CHECK_PTR(left);
	left->setAutoResize(TRUE);
	QLabel *right = new QLabel(winfo);
	CHECK_PTR(right);
	right->setAutoResize(TRUE);
	QHBoxLayout *hl = new QHBoxLayout(winfo);
	CHECK_PTR(hl);

	int cx,cw,ch;
	cx = x()+(qapp::adesk*dt->width());
	cw = (inc_w > 1)?((width()-base_w)/inc_w):(width());
	ch = (inc_h > 1)?((height()-base_h)/inc_h):(height());

	QString gm;
	QTextOStream tsg(&gm);
	tsg << cw << 'x' << ch << '+' << cx << '+' << y();

	QString txtl,txtr;
	QTextOStream tsl(&txtl);
	QTextOStream tsr(&txtr);

	tsl << " Title"      << '\n'
	    << " Name"       << '\n' 
	    << " Class"      << '\n'
	    << " Location"   << '\n'
	    << " Invocation" << '\n'
	    << " Geometry";

	tsr << ": " << wmname.left(100)     << " \n"
	    << ": " << res_name.left(100)   << " \n"
	    << ": " << res_class.left(100)  << " \n"
	    << ": " << clientfqdn.left(100) << " \n"
	    << ": " << command.left(100)    << " \n"
	    << ": " << gm << ' ';

	left->setText(txtl);
	right->setText(txtr);
	hl->add(left);
	hl->add(right);
	winfo->move((dt->width()/2)-(winfo->width()/2), (dt->height()/2)-(winfo->height()/2));
	winfo->show();
}

void xwindow::release_info(void)
{
	if(winfo != NULL)
	{
		delete winfo;
		winfo = NULL;
	}	
}

void xwindow::unscreen(void)
{
	if(sstate)  // screen mode
	{
		resize_client(scx, scy, scw, sch);
		sstate = FALSE;
	}
}

void xwindow::setinactive(void)
{

	if(actpal || urgpal)
	{
		setPalette(*qapp::ipal);  // set inactive colors
		stopautofocus();
		actpal = FALSE;
	}

	if(urgpal)
	{
		qapp::pb->change_palette(QApplication::palette(), this);
		urgpal = FALSE;
	}	
}

void xwindow::setactive(void)
{
	if(urgpal || ! actpal)
	{
		setPalette(QApplication::palette());  // set standard colors
		actpal = TRUE;
	}

	if(urgpal)
	{
		qapp::pb->change_palette(QApplication::palette(), this);
		urgpal = FALSE;
	}	
}

void xwindow::seturgent(void)
{
	if(! urgpal)
	{
		setPalette(*qapp::upal);
		qapp::pb->change_palette(*qapp::upal, this);
		urgpal = TRUE;
	}
}

void xwindow::raise()
{
	if(mrb != NULL)
		return;
		
#ifdef DEBUGMSG
	cerr << "raise (WId:" << winId() << ")\n";
#endif

	XRaiseWindow(qt_xdisplay(), winId());
}

void xwindow::map(void)
{
	withdrawnstate = FALSE;	

	if(map_iconic)  // InitialState WMHint
	{
		qapp::pb->add(this);
		set_clientstate(IconicState);

		if(urgent || urgpal)
		{
			urgpal = FALSE;
			seturgent();
		}	
	}	
	else
	{
		map_normal();
		qapp::pb->remove(this);
	}	
}

void xwindow::map_normal(void)
{
	if(! isVisible())
	{
#ifdef DEBUGMSG
		cerr << "map client\n";
#endif	
		unmapped = FALSE;
		
		show();
		XMapWindow(qt_xdisplay(), clientid);
		set_clientstate(NormalState);
		XSync(qt_xdisplay(), FALSE);
	}	
	raise();
}

void xwindow::unmap(void)
{
	stopautofocus();

	if(! isVisible())
		return;

	unmapped = TRUE;	
	
	hide();
	XUnmapWindow(qt_xdisplay(), clientid);
}

void xwindow::iconify(void)  // transition to iconic
{
	unmap();
	withdrawnstate = FALSE;
	set_clientstate(IconicState);
	qapp::pb->add(this);  // add to procbar
}

void xwindow::withdraw(void)
{
	unmap();
	withdrawnstate = TRUE;
	set_clientstate(WithdrawnState);
	qapp::pb->remove(this);
	
#ifdef DEBUGMSG
	cerr << "changed to withdrawn (WId:" << winId() << ")\n";
#endif
}

void xwindow::focus_mouse(void)  // map and set mouse (generates enter event -> focus)
{
	int pw,ph;
	
	if(qapp::smode)  // screen mode
	{
		if(! isVisible())
			return;
			
		if(sstate == FALSE)
		{
			scx = x();
			scy = y();
			scw = width();
			sch = height();

			pw = dt->width();
			ph = dt->height()+uborderh+defaults::lowerborderheight;
			setGeometry(0, -uborderh, pw, ph);
			
			if(inc_w > 1)
			{
				int i = (pw-base_w)/inc_w;
				pw = base_w+i*inc_w;
			}

			if(inc_h > 1)	
			{
				int j = (ph-base_h)/inc_h;
				ph = base_h+j*inc_h;
			}	

			XResizeWindow(qt_xdisplay(), clientid, pw, ph-borderh);
			sstate = TRUE;
		}
		raise();
		return;
	}
	map();	

	int mid = x()+(width()/2);
	
	if(mid > dt->width())
	{
		qapp::pg->change_desk(qapp::adesk+(x()/(dt->width()-1)));
	}	
	else if(mid < 0)
	{
		qapp::pg->change_desk(qapp::adesk+(x()/(dt->width()+1))-1);
	}	

	XWarpPointer(qt_xdisplay(), None, winId(), 0, 0, 0, 0, width()/2, uborderh/2);
}

void xwindow::setchildfocus(long timestamp) // set to active and focus to child
{
#ifdef DEBUGMSG
	cerr << "change active client to (WId:" << winId() << ")\n";
#endif

	setactive();

	if(inputfield)  // InputHint
		XSetInputFocus(qt_xdisplay(), clientid, RevertToPointerRoot, CurrentTime);
		
	if(prot_takefocus)  // WM_TAKE_FOCUS protocol
		send_wmprotocol(qapp::wm_take_focus, timestamp);
}

void xwindow::stopautofocus(void)
{
	if(defaults::autofocustime)
		focustimer->stop();
}

void xwindow::startautofocus(void)  // start autofocus timer
{
	if(! defaults::autofocustime || mrb != NULL)
		return;
		
	// do not raise if another window appears within this window

	Window w1,w2,*wins;
	uint nwins,i;
	XWindowAttributes attr;
	
	if(XQueryTree(qt_xdisplay(), qt_xrootwin(), &w1, &w2, &wins, &nwins) && nwins)
	{
		for(i=0; i < nwins; i++)  // find current window
			if(wins[i] == winId())
				break;
				
		int cx2 = x()+width(); int cy2 = y()+height();
		
		while(++i < nwins)
		{
			XGetWindowAttributes(qt_xdisplay(), wins[i], &attr);

			if(attr.map_state !=  IsViewable)
					continue;

			int nx2 = attr.x+attr.width; int ny2 = attr.y+attr.height;
			
			if(attr.x >= x() && nx2 <= cx2 && attr.y >= y() && ny2 <= cy2)
			{
				XFree(wins);
				return;
			}	
		}	
		XFree(wins);
	}	
	focustimer->start(defaults::autofocustime, TRUE);
}

void xwindow::getwmprotocols(void)
{
	Atom *protocols;
	int nprot,i;
	
	prot_delete = FALSE;
	prot_takefocus = FALSE;

	if(XGetWMProtocols(qt_xdisplay(), clientid, &protocols, &nprot))
	{
		for(i=0; i < nprot; i++)
		{
			if(protocols[i] == qapp::wm_delete)
			{
				prot_delete = TRUE;
			}	
			else if(protocols[i] == qapp::wm_take_focus)
			{
				prot_takefocus = TRUE;
			}
		}
		XFree(protocols);
	}
}

void xwindow::send_wmprotocol(long data0, long data1)  // send protocol message to child window
{
	XEvent event;
	
	memset(&event, 0, sizeof(event));
	event.xclient.type = ClientMessage;
	event.xclient.window = clientid;
	event.xclient.message_type = qapp::wm_protocols;
	event.xclient.format = 32;
	event.xclient.data.l[0] = data0;
	event.xclient.data.l[1] = data1;
	
	XSendEvent(qt_xdisplay(), clientid, False, 0L, &event);
}

void xwindow::set_clientstate(int state)
{
	ulong data[2];
	data[0] = (ulong)state;
	data[1] = (ulong)None;

	XChangeProperty(qt_xdisplay(), clientid, qapp::wm_state, qapp::wm_state, 32, PropModeReplace, (uchar *)data, 2);
}

void xwindow::wdestroy(void)  // destroy client
{
	if(prot_delete)  // soft way
	{
		send_wmprotocol(qapp::wm_delete, CurrentTime);
#ifdef DEBUGMSG
		cerr << "soft kill\n";
#endif
	}
	else 
	{ 
		XKillClient(qt_xdisplay(), clientid);
		XSync(qt_xdisplay(), FALSE);
#ifdef DEBUGMSG
		cerr << "hard kill\n";
#endif		
	}	
}

void xwindow::get_colormaps(void)
{
	Window *w;
	int nm=0;

	XWindowAttributes attr;
	XGetWindowAttributes(qt_xdisplay(), clientid, &attr);
	cmap = attr.colormap;

	if(! XGetWMColormapWindows(qt_xdisplay(), clientid, &w, &nm))
		return;

	if(nm > 1000)
	{
		cerr << "More than 1000 colormaps for " << clientname << " - property rejected\n";
		return;
	}	
	
	if(ncmaps) 
		delete cmapwins;

	if(nm == 0)
	{
		cmapwins = NULL;
		ncmaps = 0;
		
		return;
	}

#ifdef DEBUGMSG
	cerr << "client has WM_COLORMAP_WINDOWS property\n";
#endif	
		
	cmapwins = new Window(nm);
	CHECK_PTR(cmapwins);
	ncmaps = nm;
	
	for(int i=0; i < nm; i++) 
		cmapwins[i] = w[i];
		
	XFree(w);
}

void xwindow::setcmapfocus(void)
{
	XWindowAttributes attr;
	bool installed = FALSE;

	for(int i = ncmaps-1; i >= 0; i--)
	{
		Window w = cmapwins[i];
		
		if(w == clientid)
			installed = TRUE;
			
		XGetWindowAttributes(qt_xdisplay(), w, &attr);
		qapp::install_colormap(attr.colormap);
	}	
    	if(! installed)
   		qapp::install_colormap(cmap);
}

bool xwindow::query_shape(void)
{
	int ns,order;

	XFree(XShapeGetRectangles(qt_xdisplay(), clientid, ShapeBounding, &ns, &order));
	XShapeSelectInput(qt_xdisplay(), clientid, ShapeNotifyMask);
	
	if(ns > 1)
		return TRUE;

	return FALSE;
}

void xwindow::reshape(void)
{
#ifdef DEBUGMSG
	cerr << "reshaping client\n";
#endif

	shaped = TRUE;
	XShapeCombineShape(qt_xdisplay(), winId(), ShapeBounding, 0, uborderh, clientid, ShapeBounding, ShapeSet);

	XRectangle tr;
	
	tr.x = tr.y = 0;
	tr.width = width();
	tr.height = uborderh;
	XShapeCombineRectangles(qt_xdisplay(), winId(), ShapeBounding, 0, 0, &tr, 1, ShapeUnion, Unsorted);
}

void xwindow::get_classhint(void)
{
	XClassHint ch;
	
	if(XGetClassHint(qt_xdisplay(), clientid, &ch))
	{
		res_name = ch.res_name;
		res_class = ch.res_class;
		
		XFree(ch.res_name);
		XFree(ch.res_class);
		
#ifdef DEBUGMSG
	cerr << "res_name set to " << res_name << '\n';
#endif

	}	
}

void xwindow::set_pflags(int flags)
{
	int oldflags = pflags;
	pflags = flags;
	
       	if((flags & ~(qapp::Sticky|qapp::WindowListSkip)) != (oldflags & ~(qapp::Sticky|qapp::WindowListSkip)))  // rebuild window frame
	{
	
#ifdef DEBUGMSG
	cerr << "rebuilding window frame (WId:" << winId() << ")\n";
#endif	

		delete mrb;
		delete winfo;
		delete midmove;
		delete ubdr;
		delete lbdr;
		delete layout;
		
		wtitle.resize(0, 0);
		int ch = height()-borderh;
		
		if(pflags & qapp::SmallFrame)
		{
			uborderh = defaults::lowerborderheight;
			borderh = 2*uborderh;
		}
		else
		{
			uborderh = defaults::windowbuttonsize;
			borderh = defaults::windowbuttonsize+defaults::lowerborderheight;
		}
	
		if(shaped || (pflags & qapp::NoResize))
			borderh -= defaults::lowerborderheight;

		get_wmnormalhints();
		resize(width(), ch+borderh);
		XMoveWindow(qt_xdisplay(), clientid, 0, uborderh);
		
		create_wborder();
		wmname = "";
		get_wmname();

		if(shaped)
			reshape();

		if(ubdr != NULL)
			ubdr->show();
		else
			midmove->show();

		if(lbdr != NULL)
			lbdr->show();
		
	}

	if(! (oldflags & qapp::Sticky) && (flags & qapp::Sticky))  // move to current desk
	{
		int dx = x()/dt->width();

		if(x() < 0)
			dx--;
			
		move(x()-(dx*dt->width()), y());
	}
}

void xwindow::get_wmnormalhints(void)
{
	XSizeHints *xsizehints;
	if((xsizehints = XAllocSizeHints()) == NULL)
		sleep(1);

#ifdef DEBUGMSG
	cerr << "reading WMNormalHints (WId:" << winId() << ")\n";
#endif	
		
	long hints;
	if(XGetWMNormalHints(qt_xdisplay(), clientid, xsizehints, &hints) == 0)
		xsizehints->flags = 0;

	wmnflags = xsizehints->flags;
	wingrav = xsizehints->win_gravity;
		
	bool pminsize = FALSE;
	
	// set max,min and base size, results include window borders

	if(wmnflags & PMinSize) 
	{
		min_w = xsizehints->min_width;
		min_h = xsizehints->min_height+borderh;
		pminsize = TRUE;   // to get base size if unspecified
	} 
	else 
	{
		min_w = 4;
		min_h = borderh;
	}
	if(wmnflags & PMaxSize)
	{
		max_w = xsizehints->max_width;
		max_h = xsizehints->max_height+borderh;
		
		if(max_w > dt->width())
			max_w = dt->width();

		if(max_h > dt->height())
			max_h = dt->height();
	}
	else
	{
		max_w = dt->width();
		max_h = dt->height();
	}
	
	if(wmnflags & PBaseSize)
	{
		base_w = xsizehints->base_width;
		base_h = xsizehints->base_height+borderh;

		if(! pminsize)  // get from base if unspecified
		{
			min_w = base_w;
			min_h = base_h;
		}
	}
	else if(pminsize)
	{
		base_w = xsizehints->min_width;
		base_h = xsizehints->min_height;
	}
	
	if(wmnflags & PResizeInc)
	{
		inc_w = xsizehints->width_inc;
		inc_h = xsizehints->height_inc;
	}
	else
		inc_w = inc_h = 1;
		
	XFree(xsizehints);
}		

void xwindow::get_wmhints(void)  // get WMHints
{
	XWMHints *xwmhints;
	map_iconic = FALSE;
	inputfield = TRUE;
	urgent = FALSE;
	
	if((xwmhints = XGetWMHints(qt_xdisplay(), clientid)) != NULL)
	{
#ifdef DEBUGMSG
	cerr << "reading WMHints (WId:" << winId() << ")\n";
#endif	

		if(xwmhints->flags & StateHint && xwmhints->initial_state == IconicState)
			map_iconic = TRUE;

		if(! (xwmhints->flags & InputHint))  // focus
			inputfield = FALSE;
			
		if(xwmhints->flags & IconPixmapHint)
			get_servericon(xwmhints->icon_pixmap, (xwmhints->flags & IconMaskHint)?(xwmhints->icon_mask):(None));

		if(xwmhints->flags & (1L << 8))
		{
			urgent = TRUE;
			seturgent();  // set urgent colors on window border and icon
		}
		else if(urgpal)
		{
			if(actpal)
				setactive();
			else
				setinactive();
		}
		XFree(xwmhints);
	}
}

void xwindow::get_wmname(void)  // get WM_NAME and ICON_NAME and set caption and border pixmap
{
	char *name;
	QString oldwname = wmname;
	QString oldiname = icname;
	
	if(XFetchName(qt_xdisplay(), clientid, &name) && name != NULL)
	{
		wmname = name;
		XFree(name);
		
	}
	else  // use class hints
		wmname = res_name;

	if(XGetIconName(qt_xdisplay(), clientid, &name) && name != NULL)
	{
		icname = name;
		XFree(name);
	}
	else 
		icname = wmname;

	if(wmname != oldwname) 
	{
		
#ifdef DEBUGMSG
	cerr << "WM_NAME set to " << wmname << '\n';
#endif

		// create window title pixmap
		
		if(! (pflags & qapp::SmallFrame))
		{
			QFontMetrics fm(font());
			int twidth = fm.width(wmname);  // pixel width of text
			int txtstart = 0;

			wtitle.resize(twidth, defaults::windowbuttonsize-3);
			wtitle.fill(dt->foregroundColor());

			QBitmap mask(wtitle.width(), wtitle.height(), TRUE);  // all transparent
			QPainter painter(&mask);
			painter.setFont(font());

			painter.drawText(txtstart, -1, mask.width()-txtstart, mask.height(), AlignLeft|AlignVCenter, wmname);
			painter.end();
			wtitle.setMask(mask);

			midmove->set_backpix(&wtitle, transfor == None);
			midmove->repaint();
		}	
	}
	if(icname != oldiname)  // change icon text on procbar
		qapp::pb->change_text(&icname, this);
}

void xwindow::get_servericon(Pixmap icon, Pixmap mask)  // get pixmap from server and scale it
{
	int ix,iy;
	uint iw,ih,bwidth,depth;
	Window w;

	if(! XGetGeometry(qt_xdisplay(), icon, &w, &ix, &iy, &iw, &ih, &bwidth, &depth))
		return;
		
	QPixmap pix(iw, ih, (int)depth);
	pix.detach();
	GC gc = qt_xget_temp_gc(depth == 1);  // no XFreeGC necessary
	XCopyArea(qt_xdisplay(), icon, pix.handle(), gc, 0, 0, iw, ih, 0 ,0);
	
	if(mask != None)  // apply icon_mask
	{
		gc = qt_xget_temp_gc(TRUE);

		QBitmap bmap(iw, ih);
		bmap.detach();
		XCopyArea(qt_xdisplay(), mask, bmap.handle(), gc, 0, 0, iw, ih, 0, 0);
		pix.setMask(bmap);
	}
	
	// scale Pixmap

	QWMatrix m;
	double wh = defaults::tc_height-4;
	m.scale(wh/iw, wh/ih);
	wicon = pix.xForm(m);
}

void xwindow::state(bool on)
{
	if(on)
		map_normal();
	else
		iconify();
}

xwindow::~xwindow(void)
{
	delete cmapwins;
	delete mrb;
	delete winfo;
	
	qapp::pb->remove(this);  // remove from procbar
	qapp::cwindows.remove(clientid);
	qapp::pwindows.remove(winId());
	
#ifdef DEBUGMSG
	cerr << "class xwindow destructed (WId:" << winId() << ")\n";
#endif	
}
