/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhaccvector.h"
#include "transaction.h"

QHaccVector::QHaccVector(int init, int grow, int max) : QObject(){
  nosort=false;
  arrSize=init;
  growth=grow;
  maxFree=max;
  trans=new Transaction * [arrSize];
  topIndex=0;
  namesort=false;
}

QHaccVector::QHaccVector( const QHaccVector& copyMe ){
  topIndex=copyMe.topIndex;
  nosort=false;
  arrSize=topIndex;
  growth=copyMe.maxFree;
  maxFree=copyMe.growth;
  namesort=copyMe.namesort;
  trans=new Transaction * [arrSize];
  //shallow copy only
  for ( int i=0; i<arrSize; i++ ) trans[i]=copyMe.trans[i];
}

const QHaccVector& QHaccVector::operator=( const QHaccVector& copyMe ){
  if ( &copyMe!=this ){
    topIndex=copyMe.topIndex;
    nosort=false;
    arrSize=topIndex;
    growth=copyMe.maxFree;
    maxFree=copyMe.growth;
    namesort=copyMe.namesort;
    trans=new Transaction * [arrSize];
    //shallow copy only
    for ( int i=0; i<arrSize; i++ ) trans[i]=copyMe.trans[i];
  }
  return *this;
}

QHaccVector::~QHaccVector(){delete [] trans;}
int QHaccVector::getMaxFree() const { return maxFree; }
int QHaccVector::getGrowth() const { return growth; }
bool QHaccVector::isEmpty() const { return topIndex==0; }
Transaction * QHaccVector::getTrans( int i ) const {return trans[i];}
void QHaccVector::setLimits( int grow, int max ){
  growth=grow;
  maxFree=max;
}

void QHaccVector::sortOnName( bool b ){
  namesort=b;
  resort();
}

int QHaccVector::length() const{ return topIndex; }
int QHaccVector::length( QDate beforeThis, QDate afterThis ) const {
  // we want the number of trans between afterThis and beforeThis
  // we can't get this if we're sorted on names, so resort
  if ( namesort ) return 0;

  if ( beforeThis.isNull() && afterThis.isNull() ) return length();

  //else we need to find a subset
  // why not make this a binary search?
  bool checkEnd=!afterThis.isNull(); //don't waste time unless you have to
  int returner=0;

  for ( int i=0; i<topIndex; i++ ){
    QDate d=trans[i]->getDate();
    if ( checkEnd ) { if ( d <= beforeThis && d > afterThis ) returner++; }
    else if ( d <= beforeThis ) returner++;
  }

  return returner;
}

QHaccVector QHaccVector::getSubset( QDate afterThis, QDate beforeThis ) const {
  QHaccVector v;
  if ( namesort ) return v;

  v.setNosort( true ); //insertions will already be in sort order

  //FIXME: how about a binary search?
  // the algrithm is as follows: binary search to find /a/ transaction
  // from the given date. once you've found that one, look at the next
  // one to see if it's the same date. If it is, move your index one
  // and look at the next in line. Repeat until you have the last
  // trans from the given day. If it's the beforeThis, repeat the
  // procedure, but look at the previous transactions. Will this work?
  
  for ( int i=0; i<topIndex; i++ ){
    QDate d=trans[i]->getDate();
    if ( d <= beforeThis && d > afterThis ) v.add( trans[i] );
  }
  return v;
}

Transaction * QHaccVector::find( unsigned int findMe ) const {
  for ( int i=0; i<topIndex; i++ )
    if ( trans[i]->getID()==findMe ) return trans[i];
  return 0;
}

void QHaccVector::add( Transaction * t ){
  if ( t ){
    trans[topIndex++]=t;
    bool change=false;
    
    if ( topIndex == arrSize - 1 ) { //grow
      arrSize=arrSize+growth;
      change=true;
    }
    else if ( ( arrSize - topIndex ) > maxFree ){ //shrink
      if ( !loading ){
	arrSize=arrSize - growth;
	change=true;
      }
    } 
    if ( change ){
      Transaction ** newArr=new Transaction * [arrSize];
      for (int i=0;i<topIndex;i++) newArr[i]=trans[i];
      delete [] trans;
      trans=newArr;
    }
    
    connect( t, SIGNAL( dateChanged() ), SLOT( resort() ) );
    if ( !nosort ) resort();
  }
}

int QHaccVector::index( const Transaction * t ) const {
  int ret=-1;
  for ( int i=0; i<topIndex; i++ ) if ( t && trans[i] && trans[i]==t ) ret=i;
  return ret;
}

bool QHaccVector::remove( Transaction * t ){return remove( index( t ) ); }
bool QHaccVector::remove( int i ){
  if ( i!=-1 ){
    disconnect( trans[i], SIGNAL( dateChanged() ), this, SLOT( resort() ) );

    for ( int j=i+1; j<topIndex; j++ ) trans[j-1]=trans[j];
    trans[topIndex--]=0;

    return true;
  }
  return false;
}

void QHaccVector::clear(){
  disconnect();
  delete [] trans;
  trans=new Transaction * [arrSize];
  topIndex=0;
}

bool QHaccVector::contains( Transaction * t ) const {
  for ( int i=0; i<topIndex; i++ ) if ( trans[i]==t ) return true;
  return false;
}

void QHaccVector::resort(){ resort( 0, length()-1 ); }
void QHaccVector::resort( int first, int last ){
 // this is a quicksort
  if ( first<last ){
    int i,j; 
    i=first; 
    j=last;
    
    QString n=trans[first]->get( Transaction::PAYEE );
    QDate d=trans[first]->getDate();

    while( true ) {
      while( compare( true, trans[i], n, d ) ) i++;
      while( compare( false, trans[j], n, d ) ) j--;
      
      if( i>=j ) break;
      swap( i, j );
      i++;
      j--;
    }
    if( first < i-1 ) resort( first, i-1 );
    if( j+1 < last ) resort( j+1, last );  
  }
}

bool QHaccVector::compare( bool smaller, const Transaction * t,
			   QString n, QDate d ) const {
  if ( namesort ) {
    if ( smaller ) return t->get( Transaction::PAYEE )<n;
    else return n<t->get( Transaction::PAYEE );
  }
  else{
    if ( smaller ) return t->getDate()<d;
    else return d<t->getDate();
  }
}

void QHaccVector::swap( int i, int j ) {
  Transaction * temp=trans[i];
  trans[i] = trans[j];
  trans[j] = temp;
}

void QHaccVector::startLoad( int size ){
  nosort=true;
  loading=true;

  if ( size>0 ){
    //if a size hint was given, make a big array now for the coming load
    Transaction ** newArr=new Transaction * [arrSize+size];
    for ( int i=0; i<topIndex; i++ ) newArr[i]=trans[i];
    delete [] trans;
    trans=newArr;
  }
}

void QHaccVector::setNosort( bool b ){nosort=b;}
void QHaccVector::stopLoad(){
  nosort=false;
  loading=false;
  resort();
}
