/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan F Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "prefs.h"
#include "config.h"
#include "accounts.h"
#include "qhaccloader.h"
#include "qhaccacctman.h"

#include <time.h>
#include <iostream.h>

#include <qfile.h>

bool QHaccLoader::loadEngine( const char * path ){
  if ( !path ) { //make sure there is a home directory
    cerr<<"\nQHACC_HOME is not set. Please set it before running QHacc, "
	<<"or use the -f option."<<endl;
    return false;
  }

  Prefs * prefs=Prefs::getInstance();
  QFile f( getConfigName( path ) );

  if ( f.exists() && f.open( IO_ReadOnly ) ){
    QString read, left;
    QTextStream in( &f );
    while ( !in.eof() ){
      read=in.readLine();
      int i=read.find( '=' );
      left=read.left( i );

      if ( left == "ACCTFILES" ) loadAccounts( read.mid( i+1 ), path );
      else if ( left == "CSYMBOL" )
	prefs->setSymbol( read.mid( i+1, 1 ) );
      else if ( left=="DATEFORMAT" ){
	if ( read.mid( i+1 )=="AMERICAN" ) prefs->setDateFormat( AMERICAN );
	else prefs->setDateFormat( EUROPEAN );
      }    
      else if ( left=="DATESEPARATOR" )
	prefs->setDateSeparator( read.mid( i+1 ) );
      else if ( left=="TRANSTYPES" ){
	QString typeList=read.mid( i+1 ).simplifyWhiteSpace();
	int start=0;
	int end=0;
	QStrList strs;
	
	while( end!=-1 ){
	  end=typeList.find( ' ', start );
	  strs.append( typeList.mid( start, end-start ) );
	  start=end+1;
	}
	prefs->setTransTypes( strs );
      }
    }
    f.close();
  }
  
  return true;
}

bool QHaccLoader::saveEngine( const char * path, QTextStream& out ){
  // do this here because if the account saving segfaults for some reason, 
  // you don't want to have an open file handle and all your data buffered.
  // That will make recovery a pain in the as...er...neck
  QString aSaveString=saveAccounts( path );
  Prefs * prefs=Prefs::getInstance();

  out<<"ACCTFILES="<<aSaveString<<endl;
  out<<"DATEFORMAT="<<( prefs->getDateFormat()==AMERICAN ? 
			"AMERICAN" : "EUROPEAN" )<<endl;
  out<<"DATESEPARATOR="<<prefs->getDateSeparator()<<endl;
  out<<"CSYMBOL="<<prefs->getSymbol()<<endl;
  out<<"TRANSTYPES=";
  
  QStrList str=prefs->getTransTypes();
  for ( unsigned int i=0; i<str.count(); i++ ) out<<str.at( i )<<" ";
  out<<endl;
  
  return true;
}

QString QHaccLoader::getConfigName( const char * path ){
  return QString( path )+"/config";
}

void QHaccLoader::loadAccounts( const QString& astring, const char * path ){
  QString accts=astring.simplifyWhiteSpace();
  QHaccAccountManager * manager=QHaccAccountManager::getInstance();
  int na=0;
  int start=0;
  int end=0;
  
  while( end!=-1 ){
    end=accts.find( ' ', start );
    start=end+1;
    na++;
  }

  //QHaccAccountManager::getInstance()->load( accts, path, numAccounts  );

#ifdef QDEBUG
  cout<<"starting load"<<endl;
  time_t loadstart=time( NULL );
#endif
  QString * files=new QString[na];
  Account ** accounts=new Account * [na];
  unsigned int * acctIDs=new unsigned int[na];

  // get all the filenames
  for ( int loop=0; loop<na; loop++ ){
    end=astring.find( ' ', start );
    files[loop]=QString( path )+"/"+astring.mid( start, end-start );
    start=end+1;
  }

#ifdef QDEBUG
  time_t astart=time( 0 );
  cout<<"  loading accounts";
#endif 
  // first, go through and load all the accounts...
  for ( int loop=0; loop<na; loop++ ){
    accounts[loop]=Account::loadFromFile( files[loop] );
    acctIDs[loop]=accounts[loop]->getID();
  }

  // then make sure the manager knows about all these new accounts
  manager->load( accounts, na );

#ifdef QDEBUG
  cout<<" ("<<time( NULL )-astart<<" seconds)"<<endl;
#endif

  // next, go through and link all the accounts to their parents
  for ( int loop=0; loop<na; loop++ ) accounts[loop]->linkToParent( manager );
  manager->resort();

  // load the accounts' transactions
  loadTrans( manager, acctIDs, files, ".trans", na );

  // load all the memorized transactions
  for ( int loop=0; loop<na; loop++ ) accounts[loop]->loadMems( files[loop] );
  delete [] files;

#ifdef QDEBUG
  cout<<"completed load in "<<time( NULL )-loadstart<<" seconds"<<endl;
#endif
}

void QHaccLoader::loadTrans( QHaccAccountManager * manager, 
			     unsigned int * acctIDs, const QString * files,
			     const char * suffix, int acctCount ){
  int transCount[acctCount];
#ifdef QDEBUG
  time_t startTime=time( NULL ), tTime=0;
  cout<<"  loading transactions"<<endl;
  int numTrans=0;
#endif

  for ( int acct=0; acct<acctCount; acct++ ){
    Account * loadMe=manager->getByID( acctIDs[acct] );
    if ( loadMe ){
#ifdef QDEBUG
      cout<<"    starting tran load for "<<loadMe->getName()<<endl;
      tTime=time( NULL );
#endif
      
      QFile f( files[acct]+suffix );
      if ( f.open( IO_ReadOnly ) ){
	QTextStream in( &f );
	
	QString read=in.readLine();
	int j=read.find( '=' );
	transCount[acct]=read.mid( j+1 ).toInt();
	
	loadMe->startLoad( transCount[acct] );
	for ( int i=0; i<transCount[acct]; i++ ) {
	  QString read=in.readLine();
	  Transaction * addMe=new Transaction( read );
	  loadMe->addTrans( addMe );
	}
	loadMe->stopLoad();
	f.close();
      }
#ifdef QDEBUG
      numTrans+=loadMe->getNumTrans();
      cout<<"      loaded "<<loadMe->getNumTrans()<<" in "
	  <<time( NULL )-tTime<<" seconds"<<endl;
#endif
    }
  }
#ifdef QDEBUG
  cout<<"  loaded "<<manager->count()<<" accounts ("<<numTrans<<" trans) in "
      <<time( NULL )-startTime<<" seconds"<<endl
      <<"  pairing "<<numTrans<<" transactions ";
  tTime=time( NULL );
#endif  
  
  //pair everything up
  for ( int acct=0; acct<acctCount; acct++ ){
    Account * loadMe=manager->getByID( acctIDs[acct] );
    for ( int j=0; j<transCount[acct]; j++ )
      loadMe->getTrans( j )->load( manager);
  }
  
#ifdef QDEBUG
  cout<<"("<<time( NULL )-tTime<<" seconds)"<<endl;
#endif
}

QString QHaccLoader::saveAccounts( const char * path, bool quietly ){
  QString returnMe;
  QHaccAccountManager * accts=QHaccAccountManager::getInstance();

  int numAccounts=accts->count();
  for ( int i=0; i<numAccounts; i++ ){
    QString t2;
    QString filename="acct"+t2.setNum( i );
    
    if ( accts->get( i )->writeToFile( QString( path )+"/"+filename ) ){
      if ( !quietly )
	cout<<"wrote "<<accts->get( i )->getName()
	    <<" to "<<filename<<endl;
      returnMe.append( filename+" ");
    }
  }
  return returnMe;
}


void QHaccLoader::prune( const char * path, QDate limit ){
  QString qpath=QString( path );
  QString temp=qpath+"/config.archived";
  QFile f( temp );
  QString acctNums, acctSums;
  QHaccAccountManager * manager=QHaccAccountManager::getInstance();
  int numAccounts=manager->count();
  
  if ( f.open( IO_WriteOnly ) ){
    QTextStream out( &f );
    out<<"NUMACCTS="<<numAccounts<<endl;
    
    for ( int i=0; i<numAccounts; i++ ){
      QString filename=qpath+"/acct"+temp.setNum( i )+".archived";
      Account::writeTrans( manager->get( i )->prune( limit ), filename );    
      out<<manager->get( i )->getID()<<" ";
    }
    out<<endl;

    saveAccounts( path, true );
    f.close();
  }
  else cerr<<"Could not write archives!"<<endl;
}

void QHaccLoader::restore( const char * path ){
  QHaccAccountManager * manager=QHaccAccountManager::getInstance();
  QString temp, qpath=QString( path );
  QFile f( qpath+"/config.archived" );

  if ( f.open( IO_ReadOnly ) ){
    QTextStream in( &f );
    QString read=in.readLine();
    int j=read.find( '=' );
    int num=read.mid( j+1 ).toInt();
    read=in.readLine();

    QString * files=new QString[num];
    unsigned int * acctIDs=new unsigned int[num];
    float * sums=new float[num];
    
    int start=0,end=0;

    for ( int i=0; i<num; i++ ){
      files[i]=qpath+"/acct"+temp.setNum( i );
      end=read.find( ' ', start );
      acctIDs[i]=read.mid( start, end-start ).toUInt();
      start=end+1;
      
      // after the restore, we'll need to recalc the opening balances
      Account * a=manager->getByID( acctIDs[i] );
      if ( a ) sums[i]=a->getBal( Account::CURRENT );
    }

    // here's the actual restore
    loadTrans( manager, acctIDs, files, ".archived", num );

    // now, go back and make all the opening balances correct
    for ( int i=0; i<num; i++ ){
      Account * acct=manager->getByID( acctIDs[i] );
      if ( acct ){
	if ( acct->isLeftPlusAccount() )
	  acct->setOBal( acct->getBal( Account::OPENING )
			 -( acct->getBal( Account::CURRENT )-sums[i] ) );
	else acct->setOBal( acct->getBal( Account::OPENING )
			    +( acct->getBal( Account::CURRENT )-sums[i] ) );
      }
    }
    f.close();    
  }
  else cerr<<"Could not read archive config file!"<<endl;
}

