/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/


#include "prefs.h"
#include "config.h"
#include "accounts.h"
#include "qhaccacctman.h"
#include "qhaccdialogs.h"

#include <iostream.h>

#include <qlabel.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qstringlist.h>
#include <qfontdialog.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qcolordialog.h> 

const int QAccountDlg1::aCHECKING  =0;
const int QAccountDlg1::aSAVINGS   =1;
const int QAccountDlg1::aCREDIT    =2;
const int QAccountDlg1::aASSET     =3;
const int QAccountDlg1::aLIABILITY =4;
const int QAccountDlg1::aEQUITY    =5;
const int QAccountDlg1::aEXPENSE   =6;
const int QAccountDlg1::aREVENUE   =7;
const int QAccountDlg1::ARRAYSIZE  =8;
const char * QAccountDlg1::names[ARRAYSIZE]={ "Checking", "Savings", "Credit",
					      "Asset", "Liability", "Equity",
					      "Equity Expense",
					      "Equity Revenue" };

/*************************************/
/******** Account Edit Dialog ********/
/*************************************/

QAccountDlg::QAccountDlg ( Account * acct, QWidget * p, bool m,
			   const char * n ) : QTabDialog( p, n, m ){

  account=acct;

  if ( acct ) setCaption( acct->getName() );
  else setCaption( "New Account" );
  
  tab1=new QAccountDlg1( account, this );
  tab2=new QAccountDlg2( account, this );
  addTab( tab1, "Account" );
  addTab( tab2, "Institution" );
  
  resize( 300, 300 );
  setCancelButton( "Cancel" );
}

QAccountDlg::~QAccountDlg(){}
Account * QAccountDlg::getAccount() const { return account; }
void QAccountDlg::close(){
  ( isModal() ? QDialog::close( true ) : QDialog::close() );
}


void QAccountDlg::accept(){
  // get all the info from the tabs
  QString * info=new QString [Account::INFOARRAYSIZE];
  const QString * inst=tab2->getInstInfo();
  for ( int i=0; i<Account::INFOARRAYSIZE; i++ ) info[i]=QString( inst[i] );
  info[Account::NAME]=tab1->getName();
  info[Account::ID]=tab1->getNumber();
  delete [] inst;
  
  bool taxed=tab1->getTaxable();
  AccountType type=tab1->getType();
  float bal=tab1->getBal();
  
  QHaccAccountManager * manager=QHaccAccountManager::getInstance();

  if ( account ){ // if we have an account already, reset everything
    account->setInfo( info );
    account->setTaxable( taxed );
    account->setOBal( bal );
  }
  else{ // make a new account
    account=Account::create( type, 0, info, bal, taxed );
    manager->add( account );
  }

  account->setDefaultTransType( tab1->getDefaultTransType() );
  
  Account * pacc=manager->getByFullName( tab1->getParent() );
  if ( pacc ) account->setParent( pacc );
  
  delete [] info;
  QDialog::accept();
}


/* * * * * * * * * * * * * * * * */
/* * * * TAB 1 INFO* * * * * * * */
/* * * * * * * * * * * * * * * * */

QAccountDlg1::QAccountDlg1 ( const Account * acct, QWidget * p,
			     const char * n ) : QWidget( p, n ){
  
  QGridLayout * layout=new QGridLayout( this, 12, 5, 5 );
  
  name=new QLineEdit( this );
  number=new QLineEdit( this );  
  openBal=new QLineEdit( this );
  dttype=new QComboBox( this );
  taxed=new QCheckBox( this );
  
  AccountType type;
  QString * info;
  typeBox=new QButtonGroup();
  
  ids=new int[ARRAYSIZE];
    
  for ( int i=0; i<ARRAYSIZE; i++ ){
    QRadioButton * qrb=new QRadioButton( names[i], this );
    ids[i]=typeBox->insert( qrb );
    //we want to change the dttype if we select a checking or savings account
    if ( i==aCHECKING ) connect( qrb, SIGNAL( clicked() ),
				 SLOT( checkTypes() ) );
  }

  if ( acct ) {
    info=acct->getInfo();
    type=acct->getType();
    name->setText( info[Account::NAME] );
    number->setText( info[Account::ID] );
    openBal->setText( QString( 0 ).setNum( acct->getBal( Account::OPENING )
					   , 'f', 2 ) );
    taxed->setChecked( acct->isTaxable() );
  }
  else {
    info=new QString[ Account::INFOARRAYSIZE];
    type=ASSET;
    openBal->setText( "0.00" );
    taxed->setChecked( true );
  }
  
  parent=new QHaccChoiceEdit( "<top level>", this, false );
  QStrList parents=QHaccAccountManager::getInstance()->getFullNames();
  
  dttype->insertItem( "<none>" );
  dttype->insertItem( "<incr number>" );
  dttype->insertStrList( Prefs::getInstance()->getTransTypes() );

  if ( acct ){
    QString defttype=acct->getDefaultTransType();
    for( int i=0; i<dttype->count(); i++ ){
      if ( dttype->text( i )==defttype ) dttype->setCurrentItem( i );
    }

    const Account * pacc=acct->getParent();
    if ( pacc ) parent->setText( pacc->getName( Account::FULL ) );
    
    // go through strings and remove this account's name, and
    // names of any children of this account--we don't want
    // circular parentage
    QString s=acct->getName( Account::FULL );
    int length=s.length();
    
    for ( unsigned int i=0; i<parents.count(); i++ ){
      QString cmp=parents.at( i );
      if ( cmp.left( length )==s ) {
	parents.remove( cmp );
	i--;
      }
    }
  }
  parent->insertItems( parents );

  switch ( type ){
  case ASSET:
    typeBox->setButton( ids[aASSET] );
    break;
  case LIABILITY:
    typeBox->setButton( ids[aLIABILITY] );
    break;
  case EQUITY:
    typeBox->setButton( ids[aEQUITY] );
    break;
  case EXPENSE:
    typeBox->setButton( ids[aEXPENSE] );
    break;
  case REVENUE:
    typeBox->setButton( ids[aREVENUE] );
    break;
  }
  
  int row=0;
  layout->addMultiCellWidget( new QLabel( "Account Name",this ),
			      row, row, 0, 3 );
  layout->addMultiCellWidget( name, row, row, 4, 9 );
  row++;

  layout->addMultiCellWidget( new QLabel( "Account Number",this ), 
			      row, row, 0, 3 );
  layout->addMultiCellWidget( number, row, row, 4, 9 );
  row++;

  layout->addMultiCellWidget( new QLabel( "Opening Balance", this ),
			      row, row, 0, 3 );
  layout->addMultiCellWidget( openBal, row, row, 4, 9 );
  row++;

  layout->addMultiCellWidget( new QLabel( "Taxable Account",this ),
			      row, row, 0, 3 );
  layout->addMultiCellWidget( taxed, row, row, 4, 4, AlignCenter );
  row++;

  layout->addMultiCellWidget( new QLabel( "Account Type",this ),
			      row, row+1, 0, 3 );
  for ( int i=0; i<ARRAYSIZE; i++ ){
    layout->addMultiCellWidget( typeBox->find( ids[i] ), row, row, 4, 7 );
    row++;
  }

  layout->addMultiCellWidget( new QLabel( "Child Of", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( parent, row, row, 4, 9 );
  row++;

  layout->addMultiCellWidget( new QLabel( "Default Transaction Type", this ),
			      row, row, 0, 3 );
  layout->addMultiCellWidget( dttype, row, row, 4, 9 );
  
  delete [] info;
}

QAccountDlg1::~QAccountDlg1(){
  delete taxed;
  delete openBal;
  delete name;
  delete number;
}

float QAccountDlg1::getBal() const { return openBal->text().toFloat(); }
bool QAccountDlg1::getTaxable() const { return taxed->isChecked(); }
QString QAccountDlg1::getParent() const { return parent->text(); }
QString QAccountDlg1::getNumber() const { return number->text(); }
QString QAccountDlg1::getName() const { return name->text(); }
QString QAccountDlg1::getDefaultTransType() const {
  QString text=dttype->currentText();
  if ( text=="<none>" ) return "";
  return text;
}

void QAccountDlg1::checkTypes(){ dttype->setCurrentItem( 1 ); }

AccountType QAccountDlg1::getType() const{
  int i=typeBox->id( typeBox->selected() );

  if ( i==ids[aCHECKING] || i==ids[aSAVINGS] || i==ids[aASSET] ) return ASSET;
  else if ( i==ids[aCREDIT] || i==ids[aLIABILITY] ) return LIABILITY;
  else if ( i==ids[aEQUITY] ) return EQUITY;
  else if ( i==ids[aEXPENSE] ) return EXPENSE;
  else return REVENUE;

}

/* * * * * * * * * * * * * * * * */
/* * * * TAB 2 INFO* * * * * * * */
/* * * * * * * * * * * * * * * * */

QAccountDlg2::QAccountDlg2 ( const Account * a, QWidget * p, const char * n )
  : QWidget( p, n ){

  QGridLayout * layout=new QGridLayout( this, 14, 9, 5 );
  QString * info=0;
  if ( a ) info=a->getInfo();
  else info=new QString[Account::INFOARRAYSIZE];

  institution=new QLineEdit( info[Account::INSTITUTION], this );
  instaddr1=new QLineEdit( info[Account::INSTADDR1], this );
  instaddr2=new QLineEdit( info[Account::INSTADDR2], this );
  instcity=new QLineEdit( info[Account::INSTCITY], this );
  inststate=new QLineEdit( info[Account::INSTSTATE], this );
  instzip=new QLineEdit( info[Account::INSTZIP], this );
  instzip->setMaxLength( 10 );

  instcontact=new QLineEdit( info[Account::INSTCONTACT], this );
  instphone=new QLineEdit( info[Account::INSTPHONE], this );
  instphone->setMaxLength( 13 );
  instfax=new QLineEdit( info[Account::INSTFAX], this );
  instfax->setMaxLength( 13 );
  instemail=new QLineEdit( info[Account::INSTEMAIL], this );
  
  int row=0;
  layout->addMultiCellWidget( new QLabel("Name", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( institution, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("Address", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instaddr1, row, row, 4, 13 );
  row++;
  
  layout->addMultiCellWidget( new QLabel("", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instaddr2, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("City", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instcity, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("State", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( inststate, row, row, 4, 5 );
  layout->addMultiCellWidget( new QLabel("Zip", this ), row, row, 7, 8 );
  layout->addMultiCellWidget( instzip, row, row, 9, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("Contact", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instcontact, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("Phone", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instphone, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("Fax", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instfax, row, row, 4, 13 );
  row++;

  layout->addMultiCellWidget( new QLabel("Email", this ), row, row, 0, 3 );
  layout->addMultiCellWidget( instemail, row, row, 4, 13 );

  delete [] info;
}

QAccountDlg2::~QAccountDlg2(){}

QString * QAccountDlg2::getInstInfo() const{
  QString * returner=new QString[Account::INFOARRAYSIZE];

  returner[Account::INSTITUTION]=institution->text();
  returner[Account::INSTADDR1]=instaddr1->text();
  returner[Account::INSTADDR2]=instaddr2->text();
  returner[Account::INSTCITY]=instcity->text();
  returner[Account::INSTSTATE]=inststate->text();
  returner[Account::INSTZIP]=instzip->text();
  returner[Account::INSTCONTACT]=instcontact->text();
  returner[Account::INSTPHONE]=instphone->text();
  returner[Account::INSTFAX]=instfax->text();
  returner[Account::INSTEMAIL]=instemail->text();
  return returner;
}

/* * * * * * * * * * * * * * * * */
/* * * * * TRANS DIALOG  * * * * */
/* * * * * * * * * * * * * * * * */

QTransDlg::QTransDlg( Transaction * t, bool de, const QRect& r,
		      QWidget * p, const char * n, bool m )
  : QDialog( p, n, m ){

  owner=t->getAccount();
  init( t, r, de );
}

QTransDlg::QTransDlg( Account * acct, bool de, const QRect& r,
		      QWidget * p, const char * n, bool m )
  : QDialog( p, n, m ){
  
  owner=acct;
  init( 0, r, de );
}

void QTransDlg::init( Transaction * trans, QRect r, bool de ){
  mytrans=trans;
  prefs=Prefs::getInstance();
  doubleEntry=de;
  numSplits=0;
  accounts=new QHaccChoiceEdit * [numSplits];
  sums=new QHaccLineEdit * [numSplits];
  splitAccounts=new Account * [numSplits];

  if ( trans && !( doubleEntry || trans->isSingle() ) ) {
    // we need to get rid of all the splits, so
    // just drop the trans and make a new one
    Transaction * t=new Transaction();
    t->setAll( trans );
    trans->getAccount()->remTrans( trans );
    owner->addTrans( t );
    trans=t;
  }
  QGridLayout * layout=new QGridLayout( this, 50, 2 );
  
  num=new QHaccChoiceEdit( this, true);
  date=new QHaccDateEdit( this );
  payee=new QHaccChoiceEdit( this, true );
  sum=new QHaccLineEdit( this );
  memo=new QHaccLineEdit( this );
  
  connect( num, SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( date, SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( payee, SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( sum, SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( memo, SIGNAL( returnPressed() ), SLOT( accept() ) );
  
  layout->addMultiCellWidget( num, 0, 0, 0, 3 );
  layout->addMultiCellWidget( date, 0, 0, 4, 12 );
  layout->addMultiCellWidget( payee, 0, 0, 13, 40 );
  layout->addMultiCellWidget( sum, 0, 0, 41, 49 );

  if ( !mytrans ){
    setCaption( "Add Transaction to "+owner->getName() );
    tsum=0;
    tdate=QDate::currentDate();
    tnum=owner->getTransType();
    taccount="<new>";
  }
  else{
    setCaption( "Edit Transaction in "+owner->getName() );
    tsum=trans->getSum();
    tdate=trans->getDate();
    tnum=trans->get( Transaction::NUM );
    tpayee=trans->get( Transaction::PAYEE );
    tmemo=trans->get( Transaction::MEMO );
    taccount=trans->getPairAFName();
  }

  if ( doubleEntry ){
    QHaccAccountManager * man=QHaccAccountManager::getInstance();
    more=new QPushButton( "+", this );
    connect( more, SIGNAL( clicked() ), SLOT( addAnotherSplit() ) );

    account=new QHaccChoiceEdit( this, false );    
    connect( account, SIGNAL( returnPressed() ), SLOT( accept() ) );
    
    layout->addMultiCellWidget( account, 1, 1, 0, 20 );
    layout->addMultiCellWidget( more, 1, 1, 21, 24, 1 );
    layout->addMultiCellWidget( memo, 1, 1, 25, 49 );
    
    account->setText( taccount );
    
    account->insertItems( man->getFullNames() );
    
    if ( mytrans && mytrans->isSplit() ){ //SPLIT TRANSACTION
      account->setEnabled( false );
      sum->setEnabled( false );
      account->setText( "Please enter splits below" );
      
      QHaccVector v=mytrans->getSplits();
      numSplits=v.length();
      accounts=new QHaccChoiceEdit * [v.length()];
      sums=new QHaccLineEdit * [v.length()];
      splitAccounts=new Account * [v.length()];
      int row=2;

      // fill out the fields
      for ( int i=0; i<v.length(); i++ ){
	Transaction * t=v.getTrans( i );
	splitAccounts[i]=t->getAccount();
	accounts[i]=new QHaccChoiceEdit( splitAccounts[i]->getName( Account::FULL ), this, false );
	accounts[i]->insertItems( man->getFullNames() );
	
	sums[i]=new QHaccLineEdit( QString( 0 ).setNum( t->getSum(), 'f', 2 ),
				   this );

	layout->addMultiCellWidget( accounts[i], row, row, 0, 20 );
	layout->addMultiCellWidget( new QLabel( "Amount ", this ),
				    row, row, 25, 40, AlignRight );
	layout->addMultiCellWidget( sums[i], row, row, 41, 49 );
	
	connect( accounts[i], SIGNAL( returnPressed() ), SLOT( accept() ) );
	connect( accounts[i], SIGNAL( lostFocus() ), SLOT( matchAccounts() ) );

	connect( sums[i], SIGNAL( returnPressed() ), SLOT( accept() ) );
	connect( sums[i], SIGNAL( lostFocus() ), SLOT( changedSums() ) );
	row++;
      }
    }
    else{
      account->insertItem( "<new>" );
      account->setText( taccount );
    }
  } //DE trans
  else layout->addMultiCellWidget( memo, 1, 1, 0, 49 );
 
  num->insertItem( tnum );
  QStrList sts=prefs->getTransTypes();
  for ( unsigned int i=0; i<sts.count(); i++ ) num->insertItem( sts.at( i ) );
  num->setText( tnum );
  
  date->setDate( tdate ); 
  memo->setText( tmemo );
  sum->setText( QString( 0 ).setNum( tsum, 'f', 2 ) );
  payee->setText( tpayee );
  payee->insertItems( owner->getPayees() );
  
  setGeometry( r );
}

QTransDlg::~QTransDlg(){
  delete [] accounts;
  delete [] sums;
  delete [] splitAccounts;
}

void QTransDlg::matchAccounts(){
  QHaccAccountManager * man=QHaccAccountManager::getInstance();
  for ( int i=0; i<numSplits; i++ ){ //split transactions
    QString aname=accounts[i]->text();
    if ( aname=="<new>" ){
      QAccountDlg qd( 0, this, true );
      if ( qd.exec() ){
	splitAccounts[i]=qd.getAccount();
	accounts[i]->setText( splitAccounts[i]->getName() );
      }
    }
    else splitAccounts[i]=man->getByFullName( aname );
  }
}

void QTransDlg::changedSums(){
  float adder=0;
  for ( int i=0; i<numSplits; i++ ){
    if ( splitAccounts[i] ){
      adder+=owner->getModifiedSumFor( sums[i]->text().toFloat(),
				       splitAccounts[i] );
    }
  }
  sum->setText( QString( 0 ).setNum( adder, 'f', 2 ) );
}

void QTransDlg::accept() {
  // FIXME: maybe we should keep the old id, just so we don't waste it
  // also, what about already-reconciled accounts?
  Account ** oldAs=0;
  Transaction::RecoType * oldRs=0;
  int nt=0;
  if ( mytrans ) {
    mytrans->set( Transaction::NUM, num->text() );
    mytrans->set( Transaction::PAYEE, payee->text() );
    mytrans->set( Transaction::MEMO, memo->text() );
    mytrans->setDate( date->getDate() );
    mytrans->setSum( sum->text().toFloat() );

    // if we have old splits, just drop them. It's too inconvenient
    // to go through all the checking otherwise
    QHaccVector v=mytrans->getSplits();
    nt=v.length();
    oldAs=new Account * [nt];
    oldRs=new Transaction::RecoType[nt];
    for ( int i=0; i<nt; i++ ){
      const Transaction * t=v.getTrans( i );
      oldAs[i]=t->getAccount();
      oldRs[i]=t->getReconciled();
    }
    mytrans->dropSplits();
  }
  else mytrans=owner->createTrans( date->getDate(), num->text(),
				   payee->text(), memo->text(),
				   sum->text().toFloat() );
  if ( doubleEntry ){
    float * splitSums=new float[numSplits];
    float nsum=0;

    Transaction ** nts=new Transaction *[numSplits];
    for( int i=0; i<numSplits; i++ ) nts[i]=0;

    // SPLIT TRANSACTION stuff
    for ( int i=0; i<numSplits; i++ ){
      if ( splitAccounts[i] && ( sums[i]->text().toFloat() )!=0 ){
	splitSums[i]=splitAccounts[i]->getModifiedSumFor( sums[i]->text().toFloat(), owner );
	nts[i]=splitAccounts[i]->createBalancedTrans( mytrans, splitSums[i] );

	// go through the "new" transactions, and make
	// sure they are consistently reconciled
	// ( just in case we dropped some splits that
	// we really didn't want to )
	for ( int j=0; j<nt; j++ )
	  if ( oldAs[j]==splitAccounts[i] ) nts[i]->setReconciled( oldRs[j] );

	nsum+=splitSums[i];
      }
    }
    
    // make sure everyone knows about the new splits
    QHaccVector v=mytrans->getSplits();
    for ( int i=0; i<v.length(); i++ ) v.getTrans( i )->addSplits( v );

    delete [] splitSums;
    delete [] nts;
    if ( nt>0 ){
      delete [] oldAs;
      delete [] oldRs;
    }
    // end of SPLIT TRANSACTION stuff
  
    // regular old DOUBLE ENTRY transaction
    if( numSplits==0 ){
      Account * DEA=0;
      if ( account->text()=="<new>" ){
	QAccountDlg qd( 0, this, true );
	if ( qd.exec() ) DEA=qd.getAccount();
      }
      else DEA=QHaccAccountManager::getInstance()->getByFullName( account->text() );
      if ( DEA ) DEA->createBalancedTrans( mytrans );
    }
    // end of DOUBLE ENTRY TRANSACTION stuff

    else mytrans->setSum( nsum ); //this is the last step for SPLIT TRANS

  }
  
  QDialog::accept();
}

void QTransDlg::addAnotherSplit(){
  QRect newRect=geometry();
  newRect.setBottom( newRect.bottom()+fontMetrics().height() );
  QHaccAccountManager * man=QHaccAccountManager::getInstance();
  QString oldAccount, oldSum;

  int lastIndex=numSplits;
  numSplits++;

  if( numSplits==1 ){
    // for the first split, disable all the junk we won't use
    sum->setEnabled( false );
    account->setEnabled( false );
    oldAccount=account->text();
    oldSum=sum->text();
    matchAccounts();
    account->setText( "Please enter splits below" );
    
    // FIXME: we really should try to add TWO rows here,
    // since adding 1 row just copies what we already
    // have and we want the user to have room to add more
    // i.e., the first split is still just double entry
  }
  else{
    oldAccount="";
    oldSum="0.00";
    QHaccChoiceEdit ** newNames=new QHaccChoiceEdit * [numSplits];
    QHaccLineEdit ** newSums=new QHaccLineEdit * [numSplits];
    Account ** newAccts=new Account * [numSplits];

    for ( int i=0; i<lastIndex; i++ ){
      newNames[i]=accounts[i];
      newSums[i]=sums[i];
      newAccts[i]=splitAccounts[i];
    }
  
    QHaccChoiceEdit ** temp=accounts;
    QHaccLineEdit ** temp2=sums;
    Account ** temp3=splitAccounts;

    accounts=newNames;
    sums=newSums;
    splitAccounts=newAccts;

    delete [] temp;
    delete [] temp2;
    delete [] temp3;
  }

  // FIXME: why must we hide the transdlg here
  hide();
  QGridLayout * mylayout=( QGridLayout * ) layout();
  int row=numSplits+1;

  accounts[lastIndex]=new QHaccChoiceEdit( this, false );
  accounts[lastIndex]->insertItems( man->getFullNames() );
  accounts[lastIndex]->insertItem( "<new>" );
  accounts[lastIndex]->setText( oldAccount );

  splitAccounts[lastIndex]=0;

  sums[lastIndex]=new QHaccLineEdit( oldSum, this );
  
  connect( accounts[lastIndex], SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( accounts[lastIndex], SIGNAL( lostFocus() ),
	   SLOT( matchAccounts() ) );

  connect( sums[lastIndex], SIGNAL( returnPressed() ), SLOT( accept() ) );
  connect( sums[lastIndex], SIGNAL( lostFocus() ), SLOT( changedSums() ) );
  
  mylayout->addMultiCellWidget( accounts[lastIndex], row, row, 0, 20 );
  mylayout->addMultiCellWidget( new QLabel( "Amount ", this ),
				    row, row, 25, 40, AlignRight );
  mylayout->addMultiCellWidget( sums[lastIndex], row, row, 41, 49 );

  resize( newRect.width(), newRect.height() );
  show();
}

/* * * * * * * * * * * * * * * * */
/* * * * * PREFS DIALOG  * * * * */
/* * * * * * * * * * * * * * * * */

QHaccPrefsDlg::QHaccPrefsDlg( QWidget * p, const char * n )
  : QDialog( p, n, false ){

  Prefs * prefs=Prefs::getInstance();
  QGridLayout * layout=0;
#ifdef QGRAPHS
  layout=new QGridLayout( this, 19, 11, 5 );
#else
  layout=new QGridLayout( this, 22, 11, 5 );
#endif

  fonter=prefs->getFont();
  setFont( fonter );

  QPushButton * okay=new QPushButton( "Accept", this );
  QPushButton * quit=new QPushButton( "Dismiss", this );
  connect( okay, SIGNAL( clicked() ), SLOT( accept( ) ) );
  connect( quit, SIGNAL( clicked() ), SLOT( close( ) ) );

  mbtn=new QPushButton( "Main Color", this );
  mbtn->setPalette( prefs->getMainColor() );
  connect( mbtn, SIGNAL( clicked() ), SLOT( mColor() ) );
  
  abtn=new QPushButton( "Alt Color", this );
  abtn->setPalette( prefs->getAltColor() );
  connect( abtn, SIGNAL( clicked() ), SLOT( aColor( ) ) );
    
  fbtn=new QPushButton( "Change this Font", this );
  connect( fbtn, SIGNAL( clicked() ), SLOT( cFont( ) ) );

  symbol=new QLineEdit( prefs->getSymbol(), this );
  symbol->setAlignment( AlignCenter );
  symbol->setMaxLength( 1 );

  sep=new QLineEdit( prefs->getDateSeparator(), this );
  sep->setAlignment( AlignCenter );
  sep->setMaxLength( 1 );

  liner=new QButtonGroup( 2, Horizontal, "Bookkeeping Method", this );
  new QRadioButton( "Single Entry", liner );
  new QRadioButton( "Double Entry", liner );
  liner->setButton( prefs->getLines()==1 ? 0 : 1 );
  connect( liner, SIGNAL( clicked( int ) ), SLOT( dEntry() ) );

  dater=new QButtonGroup( 2, Horizontal, "Date Format", this );
  new QRadioButton( "American (MM/DD/YYYY)", dater );
  new QRadioButton( "European (DD/MM/YYYY)", dater );
  dater->setButton( prefs->getDateFormat()==AMERICAN ? 0 : 1 );

  int row=0;
  layout->addMultiCellWidget( mbtn, row, row+1, 1, 4 );
  layout->addMultiCellWidget( abtn, row, row+1, 6, 9 );
  row+=2;
  layout->addMultiCellWidget( liner, row, row+2, 1, 9 );
  row+=3;
  layout->addMultiCellWidget( dater, row, row+2, 1, 9 );
  row+=3;
  layout->addMultiCellWidget( new QLabel( "Date Separator", this ), 
			      row, row+1, 1, 4 ); 
  layout->addMultiCellWidget( sep, row, row+1, 5, 6 );

  row+=2;
  layout->addMultiCellWidget( new QLabel( "Currency Symbol", this ), 
			      row, row+1, 1, 4 ); 
  layout->addMultiCellWidget( symbol, row, row+1, 5, 6 );
  row+=2;
  layout->addMultiCellWidget( fbtn, row, row+1, 1, 9 );
  row+=2;

#ifdef QGRAPHS
  grapher=new QButtonGroup( 4, Horizontal, "Default Graph", this );
  new QRadioButton( "Line", grapher );
  new QRadioButton( "Bar", grapher );
  new QRadioButton( "Two Bar", grapher );
  pie=new QRadioButton( "Pie", grapher );
  pie->setEnabled( prefs->getLines()==2 );

  if ( prefs->getGraphType()==LINE ) grapher->setButton( 0 );
  else if ( prefs->getGraphType()==BAR ) grapher->setButton( 1 );
  else if ( prefs->getGraphType()==TWOBAR ) grapher->setButton( 2 );
  else grapher->setButton( 3 );


  decs=new QComboBox( false, this );
  decs->insertItem( "No Decorations" );
  decs->insertItem( "Some" );
  decs->insertItem( "More" );
  decs->insertItem( "Blind me!" );
  decs->setCurrentItem( ( int ) prefs->getDecoration() );

  layout->addMultiCellWidget( grapher, row, row, 1, 6 );  
  layout->addMultiCellWidget( decs, row, row, 7, 9 );
  row++;
#endif

  QStrList strs=prefs->getTransTypes();
  QString list;
  for ( unsigned int i=0; i<strs.count(); i++ ){
    list+=( strs.at( i ) );
    list+=" ";
  }

  transTypes=new QLineEdit( list, this );  
  layout->addMultiCellWidget( new QLabel( "Transaction Types",
					  this ),  row, row, 1, 4 ); 
  layout->addMultiCellWidget( transTypes, row, row, 5, 9 );

  row++;
  layout->addMultiCellWidget( okay, row, row+1, 1, 4 );
  layout->addMultiCellWidget( quit, row, row+1, 6, 9 );

  resize( 450, 450 );
  setCaption( "Preferences" );
}

QHaccPrefsDlg::~QHaccPrefsDlg(){
  delete mbtn;
  delete abtn;
  delete fbtn;
  delete symbol;
  delete sep;
  delete liner;
  delete transTypes;
#ifdef QGRAPHS
  delete grapher;
  delete decs;
#endif
}

void QHaccPrefsDlg::close(){ QDialog::close( true ); }

void QHaccPrefsDlg::accept(){
  Prefs * prefs=Prefs::getInstance();
  prefs->setSymbol( symbol->text() );
  prefs->setMainColor( mbtn->backgroundColor() );
  prefs->setAltColor( abtn->backgroundColor() );
  prefs->setDateSeparator( sep->text() );
  prefs->setFont( fonter );
  prefs->setDecoration( ( Decoration ) decs->currentItem() );

  QButton * checko=liner->selected();
  if ( checko->text()=="Single Entry" ) prefs->setLines( 1 );
  else prefs->setLines( 2 );

  checko=dater->selected();
  if ( checko->text()=="American (MM/DD/YYYY)" )
    prefs->setDateFormat( AMERICAN );
  else prefs->setDateFormat( EUROPEAN );

  checko=grapher->selected();
  if ( checko->text()=="Bar" ) prefs->setGraphType( BAR );
  else if ( checko->text()=="Two Bar" ) prefs->setGraphType( TWOBAR );
  else if ( checko->text()=="Pie" ) prefs->setGraphType( PIE );
  else prefs->setGraphType( LINE );
  
  QString types=transTypes->text();
  int start=0;
  int end=0;
  QStrList strs;
  
  while( end!=-1 ){
    end=types.find( ' ', start );
    strs.append( types.mid( start, end-start ) );
    start=end+1;
  }
  prefs->setTransTypes( strs );



  QDialog::accept();
}

void QHaccPrefsDlg::mColor(){
  QColor c=QColorDialog::getColor( mbtn->backgroundColor() );
  if ( c.isValid() ) mbtn->setPalette( c );
}
void QHaccPrefsDlg::aColor(){
  QColor c=QColorDialog::getColor( abtn->backgroundColor() );
  if ( c.isValid() ) abtn->setPalette( c );
}
void QHaccPrefsDlg::cFont(){
  bool ok;
  QFont f=QFontDialog::getFont( &ok, font(), 0 );
  if ( ok ){
    setFont( f );
    fonter=f;
  }
}
 
void QHaccPrefsDlg::dEntry(){
  QButton * b=liner->selected();
  bool de=b->text()=="Double Entry";

  // can't have pie graphs if you're using single-entry bookkeeping
  if ( !de && grapher->selected()==pie ) grapher->setButton( 0 );
  pie->setEnabled( de );
}
