/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include <iostream.h>

#include "prefs.h"
#include "config.h"
#include "accounts.h"
#include "transaction.h"
#include "qhaccrecwin.h"
#include "qhaccdialogs.h"
#include "qhaccacctman.h"
#include "qhaccacctchsr.h"

#ifdef QGRAPHS
#include "qhaccgrapher.h"
#endif

#include <qfont.h>
#include <qcolor.h>
#include <qevent.h>
#include <qpixmap.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qscrollbar.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>


QHaccAccountChooser::QHaccAccountChooser( QWidget * p, const char * c )
  : QFrame( p, c ){

  prefs=Prefs::getInstance();
  connect ( prefs, SIGNAL( changedFont( QFont ) ), SLOT( recalcSizes() ) );

  setFrameStyle( Box|Sunken );
  setLineWidth( 2 );
  setMidLineWidth( 0 );
  setMouseTracking( true );  
  setFocusPolicy( StrongFocus );
  manager=QHaccAccountManager::getInstance();

  for ( int i=0; i<manager->count(); i++ ){
    Account * acct=manager->get( i );
    connect( acct, SIGNAL( needRefresh() ), SLOT( vrepaint() ) );
    connect( acct, SIGNAL( balanceChanged( float ) ), SLOT( vrepaint() ) );
  }

  //active is the index of the ccount currently open
  numViewableLines=line=active=0;

  QGridLayout * layout=new QGridLayout( this, 10, 12, 0 );
  scroller=new QScrollBar( QScrollBar::Vertical, this );
  layout->addMultiCellWidget( scroller, 0, 10, 12, 12 );
  connect( scroller, SIGNAL( valueChanged( int ) ), SLOT( vrepaint() ) );
  connect( prefs, SIGNAL( changedAltColor( QColor ) ), SLOT( vrepaint() ) );
  connect( manager, SIGNAL( addedAccount( Account * ) ), SLOT( vrepaint() ) );

  popup=new QPopupMenu( this );
  popup->insertItem( "reconcile", this, SLOT( reconcile( ) ) );
#ifdef QGRAPHS
  popup->insertItem( "graph account", this, SLOT( chart() ) );
  popup->insertItem( "account report", this, SLOT( report() ) );
#endif
  popup->insertItem( "edit account", this, SLOT( editAcct( ) ) );
  popup->insertItem( "new account", this, SLOT( newAcct( ) ) );
  remChoice=popup->insertItem( "delete", this, SLOT( remAcct() ) );

}

QHaccAccountChooser::~QHaccAccountChooser(){
  delete scroller;
  delete popup;
}

void QHaccAccountChooser::chart(){
#ifdef QGRAPHS
  QHaccGrapher * qg=new QHaccGrapher( QHaccGrapher::GRAPH, this );
  qg->show();
  qg->accountOpened( manager->get( line+scroller->value() ) );
#endif
}

void QHaccAccountChooser::report(){
#ifdef QGRAPHS
  QHaccGrapher * qg=new QHaccGrapher( QHaccGrapher::REPORT, this );
  qg->show();
  qg->accountOpened( manager->get( line+scroller->value() ) );
#endif
}

void QHaccAccountChooser::vrepaint(){ repaint( false ); }

void QHaccAccountChooser::drawContents( QPainter * p ){
  QFont f=prefs->getFont();
  QFont fb=f;
  fb.setBold( true );
  QFontMetrics fm=QFontMetrics( f );
  int w=width()-scroller->width() ,h=height(), marg=margin()+lineWidth()+2,
    counter=0, limit=manager->count();
  float totalBalance=0;
  //these next two multiplications should equal roughly 100 (minus margins)
  int end1=( int )( w*0.65 );
  int end2=( int )( w*0.33 );

  QPixmap off( size() );
  QPainter offPainter( &off );
  
  offPainter.fillRect( 0, 0, w, h, QBrush( backgroundColor() ) );
  QPen rp=QPen( "red" );
  QPen bp=QPen( "black" );

  for ( int i=scroller->value(); i<limit; i++ ){
    if ( i==active ){ //paint a highlight bar behind the active account
      offPainter.fillRect( 0, ( i-scroller->value() )*fm.height()+marg,
			   w, fm.lineSpacing(),
			   QBrush( prefs->getAltColor() ) );
    }
    if ( counter==line || i==active ) offPainter.setFont( fb );
    else offPainter.setFont( f );

    Account * acct=manager->get( i );

    float bal=acct->getBal( Account::CURRENT );
    totalBalance+=bal;
    // this is actually correct, but it looks really bad (to me, at least)
    //if ( ( bal<0 && acct->isAsset() ) || 
    // ( bal>0 && !acct->isAsset() ) ) offPainter.setPen( rp );
    if ( bal>-1 && bal<-0.01 ) bal=0; // seeing -0.00 annoys me
    if ( bal<0 ) offPainter.setPen( rp );
    else offPainter.setPen( bp );

    //paint name
    int m2=10*acct->levelsDeep();
    offPainter.drawText( marg+5+m2,
			 marg+counter*fm.lineSpacing(), end1-m2-marg-5,
			 fm.lineSpacing(), AlignLeft, acct->getName() );

    //paint balance
    offPainter.drawText( end1, marg+counter*fm.lineSpacing(),
			 end2, fm.lineSpacing(), AlignRight,
			 QString( 0 ).setNum( bal, 'f', 2 ) );
    counter++;
  }

  offPainter.setFont( fb );
  offPainter.setPen( bp );
  offPainter.drawLine( 0, counter*fm.lineSpacing()+marg,
		       w, counter*fm.lineSpacing()+marg );

  if ( totalBalance<0 ) offPainter.setPen( rp );
  else offPainter.setPen( bp );
  //paint total
  offPainter.drawText( marg+5, marg+counter*fm.lineSpacing(), end1,
		       fm.lineSpacing(), AlignLeft, "Total" );
  
  offPainter.setFont( f );
  offPainter.drawText( end1, marg+counter*fm.lineSpacing(),
		       end2, fm.lineSpacing(), AlignRight,
		       QString( 0 ).setNum( totalBalance, 'f', 2 ) );

  p->drawPixmap( 0, 0, off );
}
void QHaccAccountChooser::focusInEvent ( QFocusEvent * ){ repaint( false ); }
void QHaccAccountChooser::focusOutEvent ( QFocusEvent * ){ repaint( false ); }
void QHaccAccountChooser::mouseMoveEvent( QMouseEvent * qme ){
  //FIXME: What's this 5 mean?
  line=( qme->y()-5 )/fontMetrics().lineSpacing();
  vrepaint();
}
void QHaccAccountChooser::mousePressEvent( QMouseEvent * qme ){
  if ( qme->button() == QMouseEvent::RightButton )
    popup->popup( QCursor::pos() );
}
void QHaccAccountChooser::editAcct(){
  if ( !manager->get( line+scroller->value() ) ) newAcct();
  else ( new QAccountDlg( manager->get( line+scroller->value() ), this ) )->show();

}
void QHaccAccountChooser::mouseDoubleClickEvent( QMouseEvent * ){editAcct();}
void QHaccAccountChooser::mouseReleaseEvent( QMouseEvent * ){
  int me=line+scroller->value();
  if ( me<manager->count() && me!=active ){
    active=me;
    emit changedAccount( manager->get( me ) );
    vrepaint();
  }
}

void QHaccAccountChooser::recalcSizes(){
  numViewableLines=height()/fontMetrics().lineSpacing();
  if ( numViewableLines<manager->count() )
    scroller->setRange( 0, 1+manager->count()-numViewableLines );
  else scroller->setRange( 0, 0 );
}
void QHaccAccountChooser::resizeEvent( QResizeEvent * ){recalcSizes();}

int QHaccAccountChooser::getIndex() const { return active; }
void QHaccAccountChooser::setIndex( int i ){
  if ( i<manager->count() ){
    active=i;
    emit changedAccount( manager->get( i ) );
    vrepaint();
  }
  else {
    active=0;
    emit changedAccount( 0 );
    vrepaint();
  }
}

void QHaccAccountChooser::reconcile(){
  ( new QHaccRecWin( manager->get( line+scroller->value() ) ) )->show();
}

void QHaccAccountChooser::newAcct(){
  QAccountDlg qd( 0, this, true );
  if ( qd.exec() ){
    Account * acct=qd.getAccount();
    connect( acct, SIGNAL( needRefresh() ), SLOT( vrepaint() ) );
    connect( acct, SIGNAL( balanceChanged( float ) ), SLOT( vrepaint() ) );

    if ( manager->count()==1 ) {
      active=0;
      emit changedAccount( manager->get( 0 ) );
      vrepaint();
    }
  }
}

void QHaccAccountChooser::remAcct(){
  int index=line+scroller->value();
  Account * a=manager->get( index );
  if ( a && !a->isInReco() ){
    if ( QMessageBox::warning( this, "Delete "+a->getName(), "Are you sure?",
			       "Delete Account", "Reconsider") == 0 ){
      manager->remove( a );
      if ( active==0 ) setIndex( 0 );
      else if ( active>=index ) setIndex( active-1 );
      
      if ( 1+manager->count()>numViewableLines )
	scroller->setRange( 0, 1+manager->count()-numViewableLines );
      vrepaint();
    }
  }
}

void QHaccAccountChooser::keyReleaseEvent( QKeyEvent * qke ){
  int key=qke->key();
  switch( key ){
  case Key_Space:
    setIndex( line+scroller->value() );
    break;
  case Key_Up:
    if ( line>0 ){
      line--;
      if ( line < scroller->value() ){
	scroller->setValue( scroller->value()-1 );
	//if we move the scroller, we'll implicitly move the line
	// so we need to correct for this
	line++;
      }
    }
    break;
  case Key_Down:
    if ( line+scroller->value() < manager->count() ){
      line++;
      if ( line+2 > numViewableLines ){
	scroller->setValue( scroller->value()+1 );
	//if we move the scroller, we'll implicitly move the line
	// so we need to correct for this
	line--;
      }
    }
    break;
  default:
    qke->ignore();
  }
  vrepaint();
}
