/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan F Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "qhacc.h"
#include "prefs.h"
#include "config.h"
#include "qhaccrecwin.h"
#include "qhaccloader.h"
#include "qhaccdialogs.h"
#include "qhaccacctchsr.h"
#ifdef QGRAPHS
#include "qhaccgrapher.h"
#endif

#include <iostream.h>
#include <time.h>

#include <qfile.h>
#include <qcolor.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpalette.h>
#include <qmenubar.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qpopupmenu.h>
#include <qtextstream.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qtextstream.h>

QHacc::QHacc( const char * home, QWidget * p, const char * n )
  : QWidget( p, n ){

  env=0;
  currAcct=0;
  load( home );
}

QHacc::~QHacc(){
  delete view;
  // everything gets deleted from QWidget, but it's important to
  // get the view deleted as soon as possible to avoid spurious
  // repaints that cause segfaults.
}

Account * QHacc::getAcct() const {return currAcct;}
void QHacc::accountOpened( Account * a ){
  Account * oldAcct=currAcct;
  if ( oldAcct ){
    // don't want extraneous signal interfering
    disconnect( oldAcct, SIGNAL( balanceChanged( float ) ),
		this, SLOT( updateBalance( float ) ) );
    
    disconnect( oldAcct, SIGNAL( removedTrans() ), view, SLOT( refresh() ) );
    disconnect( oldAcct, SIGNAL( addedTrans( Transaction * ) ),
		view, SLOT( refresh() ) );
  }
  
  if ( !a ){ //no accounts
    currAcct=0;
    view->accountOpened( 0 );
  }
  else {
    currAcct=a;
    updateBalance( currAcct->getBal( Account::CURRENT ) );
    
    connect( currAcct,SIGNAL( balanceChanged( float ) ),
	     this, SLOT( updateBalance( float ) ) );
    connect( currAcct, SIGNAL( removedTrans() ), view, SLOT( refresh() ) );
    connect( currAcct, SIGNAL( addedTrans( Transaction * ) ),
	     view, SLOT( refresh() ) );

    view->accountOpened( currAcct );
  }
}

void QHacc::about(){
  QString vline;
  vline.sprintf( "The Q Home Accountant v%s\nWritten by: Ryan Bobko\nryan@ostrich-emulators.cx", VERSION );
  QMessageBox::about( this, "About QHacc", vline );
}

void QHacc::aboutQt(){ QMessageBox::aboutQt( this, "About QT" );}

void QHacc::chart(){
#ifdef QGRAPHS
  ( new QHaccGrapher( QHaccGrapher::GRAPH, this ) )->show();
#endif
}

void QHacc::report(){
#ifdef QGRAPHS
  ( new QHaccGrapher( QHaccGrapher::REPORT, this ) )->show();
#endif
}

void QHacc::updateBalance( float f ){
  QString s="";
  if ( f<0 ) {
    s.append ( "-" );
    f=0-f;
  }

  s.append( Prefs::getInstance()->getSymbol() );
  balance->setText( s+QString( 0 ).setNum( f, 'f', 2 ) );

  // update the number of transactions
  // every time you update the balance
  Account * acct=getAcct();
  if ( acct ){
    QString type;
    if ( acct->isType( ASSET ) ) type=" Asset";
    else if ( acct->isType( LIABILITY) ) type=" Liability";
    else if ( acct->isType( EQUITY ) ) type=" Equity";

    int i=getAcct()->getNumTrans();
    transCount->setText( QString( 0 ).setNum( i ) + type +
			 ( i==1 ? " Transaction" : " Transactions" ) );
  }
}

void QHacc::changeFont( QFont f ){ setFont( f ); }
void QHacc::changeSymbol( QString c ){
  if ( getAcct() ) updateBalance( getAcct()->getBal( Account::CURRENT ) );
}

void QHacc::changePrefs(){
  QHaccPrefsDlg * qpd=new QHaccPrefsDlg( this );
  qpd->show();
}

void QHacc::saveAndQuit(){
  hide();
  save();
  qApp->quit();
}

void QHacc::save(){
  QFile f( QHaccLoader::getConfigName( env ) );

  if ( f.exists() && f.open( IO_WriteOnly ) ){
    Prefs * prefs=Prefs::getInstance();      
    QTextStream out( &f );
    QHaccLoader::saveEngine( env, out );
    
    QColor c=prefs->getMainColor();
    out<<"MAINCOLOR="<<c.red()<<" "<<c.green()<<" "<<c.blue()<<endl;
    c=prefs->getAltColor();
    out<<"ALTCOLOR="<<c.red()<<" "<<c.green()<<" "<<c.blue()<<endl;
    out<<"LINESPERTRANS="<<prefs->getLines()<<endl;
    out<<"SHOWINDEX="<<chooser->getIndex()<<endl;
    QRect recto=geometry();
    out<<"LOCATION="<<recto.x()<<" "<<recto.y()<<" "
       <<recto.width()<<" "<<recto.height()<<endl;
    out<<"FONT="<<prefs->getFont().rawName()<<endl;
#ifdef QGRAPHS
    out<<"DECORATION="<<( int ) prefs->getDecoration()<<endl;
    out<<"GRAPHTYPE="<<( int ) prefs->getGraphType()<<endl;
#endif
    f.close();
  }
}

void QHacc::initGUI( int showIndex ){
  QGridLayout * layout=new QGridLayout( this, 15, 12, 6 );
  Prefs * prefs=Prefs::getInstance();

  CHECK_PTR( layout );

  view=new QHaccView( 0, this );
  CHECK_PTR( view );

  chooser=new QHaccAccountChooser( this );
  CHECK_PTR( chooser );

  connect( chooser, SIGNAL( changedAccount( Account * ) ),
	   this, SLOT( accountOpened( Account * ) ) );

  connect( view, SIGNAL( changePrefs() ), SLOT( changePrefs() ) );

  connect( prefs, SIGNAL( changedSymbol( QString ) ),
	   this, SLOT( changeSymbol( QString ) ) );

  connect( prefs, SIGNAL( changedFont( QFont ) ),
	   this, SLOT( changeFont( QFont ) ) );

  setFont( prefs->getFont() );
  setFontPropagation( AllChildren );
  
  QPopupMenu * file=new QPopupMenu( this );
  CHECK_PTR( file );
  QPopupMenu * help=new QPopupMenu( this );
  CHECK_PTR( help );  
  QPopupMenu * acct=new QPopupMenu( this );
  CHECK_PTR( acct );
  prefMenu=new QPopupMenu( this );
  CHECK_PTR( prefMenu );
  QMenuBar * menu=new QMenuBar( this );
  CHECK_PTR( menu ); 


  balance=new QLabel( prefs->getSymbol()+"0.00", this );
  CHECK_PTR( balance );

  transCount=new QLabel( "0 Transactions", this );
  CHECK_PTR( transCount );

  layout->addMultiCellWidget( view, 2, 13, 3, 12 );
  layout->addMultiCellWidget( chooser, 2, 13, 0, 2 );
  
  layout->addWidget( new QLabel( "Balance", this ), 14, 9, AlignRight );
  layout->addWidget( balance, 14, 11 );
  layout->addWidget( transCount, 14, 3 );

  // make a nice menubar
  file->insertItem( "&Save Now", this, SLOT( save() ) );
  file->insertItem( "E&xit", this, SLOT( saveAndQuit() ), CTRL+Key_X );
  file->insertItem( "&Quit", qApp, SLOT( quit() ), CTRL+Key_Q );

  help->insertItem( "About QHacc", this, SLOT( about() ) );
  help->insertSeparator();
  help->insertItem( "About QT", this, SLOT( aboutQt() ) );
  
  //make a ref to prefs dialog
  prefMenu->insertItem( "Change &Prefs", this, SLOT( changePrefs() ) );
  prefMenu->insertItem( "Restore Defaults", prefs, SLOT( resetPrefs() ) );

  acct->insertItem( "&Reconcile", chooser, SLOT( reconcile() ) );
  acct->insertItem( "&Edit", chooser, SLOT( editAcct() ) );
  acct->insertItem( "&Create", chooser, SLOT( newAcct() ) );
  acct->insertItem( "&Delete", chooser, SLOT( remAcct() ) );

  menu->insertItem( "&File", file );
  menu->insertItem( "&Account", acct );
  menu->insertItem( "&Preferences", prefMenu );
#ifdef QGRAPHS
  QPopupMenu * graphs=new QPopupMenu( this );
  CHECK_PTR( graphs );
  graphs->insertItem( "&Graphs", this, SLOT( chart() ) );
  graphs->insertItem( "&Reports", this, SLOT( report() ) );
  menu->insertItem( "&Graphs", graphs );
#endif
  menu->insertSeparator();
  menu->insertItem( "&Help", help );
  menu->setSeparator( QMenuBar::InWindowsStyle );

  chooser->setIndex( showIndex );
  setMinimumSize( 600, 300 );
}

void QHacc::load( const char * home ){
  env=home;

  int showIndex=0;
  if ( QHaccLoader::loadEngine( home ) ){
    QFile f( QHaccLoader::getConfigName( env ) );

    
    if ( f.exists() && f.open( IO_ReadOnly ) ){
      Prefs * prefs=Prefs::getInstance();
      
      QString read, left;
      QTextStream in( &f );
      while ( !in.eof() ){
	read=in.readLine();
	int i=read.find( '=' );
	left=read.left( i );
	
	if ( left=="LINESPERTRANS" ) 
	  prefs->setLines( read.mid( i+1 ).toInt() );
	else if ( left=="SHOWINDEX" )
	  showIndex=read.mid( i+1 ).toInt();
	else if ( left=="LOCATION" ) {
	  i++;
	  int l=0, t=0, w=0, h=0, finder=read.find( ' ', i );
	  
	  l=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  t=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  w=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  h=read.mid( i, finder-i ).toInt();
	  
	  setGeometry( l, t, w, h );
	}
	else if ( left=="ALTCOLOR" ){
	  i++;
	  int r=0, g=0, b=0, finder=read.find( ' ', i );
	  r=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  g=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  b=read.mid( i, finder-i ).toInt();
	  prefs->setAltColor( QColor( r, g, b ) );
	}
	else if ( left=="MAINCOLOR" ){
	  i++;
	  int r=0, g=0, b=0, finder=read.find( ' ', i );
	  r=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  g=read.mid( i, finder-i ).toInt();
	  
	  i=finder+1;
	  finder=read.find( ' ', i );
	  b=read.mid( i, finder-i ).toInt();
	  
	  prefs->setMainColor( QColor( r, g, b ) );
	}
	else if ( left=="FONT" ){
	  QFont f;
	  f.setRawName( read.mid( i+1 ) );
	  prefs->setFont( f );
	}
#ifdef QGRAPHS
	else if ( left=="DECORATION" ){
	  prefs->setDecoration( ( Decoration )( read.mid( i+1 ).toInt() ) );
	}
	else if ( left=="GRAPHTYPE" ){
	  prefs->setGraphType( (GraphType)( read.mid( i+1 ).toInt() ) );
	}
#endif
      }
      f.close();
    }
    else cerr<<"Could not read the config file! "
	     <<"Starting with defaults..."<<endl;
    
    initGUI( showIndex );
  }
  else exit( 1 );
}

void QHacc::cron( const char * env, const char * extra ){

  QFile f2( extra );
  if ( f2.exists() && f2.open( IO_ReadOnly ) ){
    QTextStream in( &f2 );
    QString read, left;
    
    Transaction * addMe=new Transaction();
    QString accountName;
    while ( !in.eof() ){
      read=in.readLine();
      int i=read.find( '=' );
      left=read.left( i );

      if ( left=="NUM" )
	addMe->set( Transaction::NUM, read.mid( i+1 ) );
      else if ( left=="PAYEE" )
	addMe->set( Transaction::PAYEE, read.mid( i+1 ) );
      else if ( left=="MEMO" )
	addMe->set( Transaction::MEMO, read.mid( i+1 ) );
      else if ( read=="RECO=Y" ) addMe->setReconciled( Transaction::YES );
      else if ( left=="ACCTS" ){	  
	//account splits
	bool goodRead=true;
	int limit=read.mid( i+1 ).toInt();
	Account ** accts=new Account * [limit];
	float * sums=new float[limit];
	int splitbase=-1;
	
	for ( int j=0; j<limit; j++ ){
	  read=in.readLine();
	  read=read.simplifyWhiteSpace();
	  int k=read.find( ' ' );	  

	  accts[j]=QHaccAccountManager::getInstance()->get( read.mid( k+1 ) );
	  if ( !accts[j] ) {
	    goodRead=false;
	    cerr<<"Could not find account "<<read.mid( k+1 )<<endl;
	  }
	  else{
	    if ( read.left( k )=="SPLIT" ){
	      sums[j]=0;
	      splitbase=j;
	    }
	    else {
	      QString amt=read.left( k );
	      // amounts can be percentages or regular values
	      if ( amt.contains( '%' ) ){
		float actsum=accts[j]->getBal( Account::CURRENT );
		float percentage=amt.toFloat();
		sums[j]=actsum*percentage/100;
#ifdef QDEBUG
		cout<<*accts[j]<<": "<<percentage<<"% for "<<sums[j]<<endl;
#endif
	      }
	      else sums[j]=read.left( k ).toFloat();
	    }
	  }
	}

	if ( goodRead ){
	  if ( limit==2 && sums[0]!=sums[1] ){
	    cerr<<"Using first sum (non-split transactions cannot "
		<<"have different sums)"<<endl;
	    sums[1]=sums[0];
	  }
	  if ( splitbase==-1 ) splitbase=0;
 
	  addMe->setSum( sums[splitbase] );
	  accts[splitbase]->addTrans( addMe );
#ifdef QDEBUG
	  cout<<"adding cron trans"<<endl;
#endif
	  
	  for ( int j=0; j<limit; j++ ){
	    if ( j!=splitbase ){
	      Transaction * t=new Transaction( *addMe );
	      accts[j]->addTrans( t );
	      t->addSplit( addMe );
	      addMe->addSplit( t );
	      //FIX ME: shouldn't  this set the sums on both trans?
	      t->setSum( sums[j] );
#ifdef QDEBUG
	      cout<<"  "<<accts[j]->getName()<<" "<<t->getSum()<<endl;
#endif
	    }
	  }
#ifdef QDEBUG
	  cout<<"  "<<accts[splitbase]->getName()<<" "<<addMe->getSum()<<endl;
#endif
	}
	delete [] sums;
	delete [] accts;
      }
    }      

    QHaccLoader::saveAccounts( env, true );
    f2.close();
  }
  
  else{
    cerr<<"Could not parse transaction file!"<<endl;
    exit( 1 );
  }
}

void QHacc::unmanned( LType typer, const char * env, const char * stringo ){

  if ( !QHaccLoader::loadEngine( env ) ) exit( 1 );
  QHaccAccountManager * accounts=QHaccAccountManager::getInstance();

  int count=accounts->count();
  int numTrans=0;

  switch ( typer ){
  case RESTORE:
    QHaccLoader::restore( stringo );
    break;
  case ARCHIVE:
    QHaccLoader::prune( env, Prefs::getDateFromString( stringo ) );
    break;
  case BALANCES:
    for ( int i=0; i<count; i++ ) numTrans+=accounts->get( i )->getNumTrans();
    cout<<"-- "<<numTrans<<" Transactions in "<<count<<" Accounts"<<endl;
    for ( int i=0; i<count; i++ )cout<<*accounts->get( i )<<endl;
    break;
  case CRON:
    cron( env, stringo );
    break;
#ifdef QGRAPHS
  case REPORT:
    QList <Account> alist;
    alist.setAutoDelete( false );

    QHaccVector transactions;
    QString title=stringo;
    if ( strcmp( stringo, "Assets" )==0 )
      alist=accounts->getAccounts( ASSET );
    else if ( strcmp( stringo, "Liabilities" )==0 )
      alist=accounts->getAccounts( LIABILITY );
    else if ( strcmp( stringo, "Equities" )==0 ){
      alist=accounts->getAccounts( EQUITY );
      title="Equity Accounts";
    }
    else if ( strcmp( stringo, "Expenses" )==0 )
      alist=accounts->getAccounts( EXPENSE );
    else if ( strcmp( stringo, "Revenues" )==0 )
      alist=accounts->getAccounts( REVENUE );
    else{
      Account * a=accounts->get( stringo );
      if ( a ) alist.append( a );
    }
    
    int sizer=0;
    for( unsigned int i=0; i<alist.count(); i++ )
      sizer+=alist.at( i )->getNumTrans();
    transactions.startLoad( sizer );
    for( unsigned int i=0; i<alist.count(); i++ ){
      Account * a=alist.at( i );
      for( int j=0; j<a->getNumTrans(); j++ )
	transactions.add( a->getTrans( j ) );
    }
    transactions.stopLoad();

    cout<<Reporter::makeReport( title, transactions )<<endl;
    break;
#endif
  }

  QHaccLoader::saveAccounts( env, true );
}
