/************************* * * * * * * * * * * * * ***************************
    Copyright (c) 2000 Ryan Bobko
                       ryan@ostrich-emulators.cx

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.     
************************** * * * * * * * * * * * * **************************/

#include "config.h"
#include "accounts.h"
#include "qhaccacctman.h"

#include <iostream.h>
#include <sys/stat.h> 
#include <sys/types.h>

#include <qfile.h>

const int Account::INSTITUTION=  0;
const int Account::INSTADDR1=    1;
const int Account::INSTADDR2=    2;
const int Account::INSTCITY=     3;
const int Account::INSTSTATE=    4;
const int Account::INSTZIP=      5;
const int Account::INSTPHONE=    6;
const int Account::INSTFAX=      7;
const int Account::INSTEMAIL=    8;
const int Account::INSTCONTACT=  9;
const int Account::ID=           10;
const int Account::NAME=         11;
const int Account::INFOARRAYSIZE=12;

const int Account::UPGRADE      =0;
const int Account::DOWNGRADE    =1;

const int Account::OPENING      =0;
const int Account::CURRENT      =1;

const int Account::SHORT        =0;
const int Account::FULL         =1;

unsigned int Account::highID    =0;

//{"INSTITUTION","INSTADDR1","INSTADDR2","INSTCITY","INSTSTATE", "INSTZIP","INSTPHONE","INSTFAX", "INSTEMAIL","INSTCONTACT","NAME"};

/* * * * * * * * * * * */

/* * * * * * * * * * * */

Account::Account( const QString * _info, unsigned int i, float f, bool tx ){

  loading=beingRecod=false;
  taxable=tx;
  openBal=f;
  currBal=f;
  parent=0;
  parentID=0;
  nextTranID=0;

  if ( i==0 ) id=++highID;
  else {
    id=i;
    if ( id > highID ) highID=id;
  }
  info=new QString[INFOARRAYSIZE];
  for ( int i=0; i<INFOARRAYSIZE; i++ ) info[i]=QString( _info[i] );
}

Account::~Account(){
  disconnect(); //don't start sending out a lot of of refresh signals
#ifdef QDEBUG
  cout<<"clearing "<<transactions.length()<<" trans from "
      <<getName()<<endl;
#endif

  memtransactions.clear();
  int count=transactions.length();
  for ( int i=0; i<count; i++ ){
    Transaction * t=transactions.getTrans( i );
    transactions.remove( t );
    delete t;
  }
  transactions.clear();
}

ostream& operator<<( ostream& out, const Account& a ){
  out<<"("<<a.id<<") "<<a.getName()<<": "
     <<QString( 0 ).setNum( a.getBal( Account::CURRENT ), 'f', 2 )
     <<" in "<<a.getNumTrans()<<" Transactions";
  return out;
}

void Account::startReco(){ beingRecod=true; }
void Account::stopReco(){ beingRecod=false; }
bool Account::isInReco() const { return beingRecod; }

QString * Account::getInfo() const {
  QString * newInfo=new QString[INFOARRAYSIZE];
  if ( newInfo!=0 ) 
    for ( int i=0; i<INFOARRAYSIZE; i++ ) newInfo[i]=QString( info[i] );

  return newInfo;
}

unsigned int Account::getID() const { return id; }
const Account * Account::getParent() const { return parent; }
int Account::index( const Transaction * t ) const {
  return transactions.index( t );
}

void Account::setInfo( const QString * newInfo ){
  delete [] info;
  info=new QString[INFOARRAYSIZE];
  for ( int i=0; i<INFOARRAYSIZE; i++ ) info[i]=QString( newInfo[i] );
}

bool Account::isTaxable() const { return taxable; }
void Account::setTaxable( bool fl ) { taxable=fl; }

QString Account::getName( int typer ) const {
  if ( typer==SHORT ) return QString( info[NAME] );
  else{
    QString myname( info[NAME] );
    if ( parent ) return parent->getName( FULL )+"::"+myname;
    else return myname;
  }
}

float Account::getBal( int typer ) const {
  return ( typer==OPENING ? openBal: currBal );
}

void Account::setParent( const Account * a ){
  if ( a ) {
    parent=a;
    emit changedParent();
  }
}

void Account::setDefaultTransType( QString s ){ defaultTransType=s; }
QString Account::getDefaultTransType() const { return defaultTransType; }
QString Account::getTransType(){
  if ( defaultTransType=="<incr number>" )
    return QString().setNum( nextTranID+1 );
  else return defaultTransType;
}

bool Account::isTopLevel() const { return ( parent==0 ); }
int Account::levelsDeep() const {
  if ( isTopLevel() ) return 0;
  else return parent->levelsDeep()+1;
}
void Account::setOBal( float f ) {
  openBal=f;
  recalcBal();
}

void Account::alterReconciled( int change ){
  int num=transactions.length();
  for ( int i=0; i<num; i++ ){
    Transaction * t=transactions.getTrans( i );
    if ( t->isReconciled( Transaction::MAYBE ) ){
      // disconnect to avoid unnecessary repaints
      disconnect( t, SIGNAL( recChanged() ), this, SLOT( refreshSignal() ) );
      if ( change==UPGRADE ) t->setReconciled( Transaction::YES );
      else t->setReconciled( Transaction::NO );
      connect( t, SIGNAL( recChanged() ), this, SLOT( refreshSignal() ) );
    }
  }
  emit refreshSignal();
}

void Account::recalcBal(){
  currBal=openBal;
  for ( int i=0; i<transactions.length(); i++ )
    currBal+=getTrans( i )->getSum();
  if ( !isLeftPlusAccount() ) currBal=0-currBal;
}

void Account::newBal( float oldy, float newy ){
  if ( isType( ASSET ) ) currBal=(currBal-oldy+newy);
  else currBal=( currBal+oldy-newy );
  emit balanceChanged( currBal );
}

QStrList Account::getPayees() const {return payees;}
void Account::refreshSignal(){emit needRefresh();}

void Account::addTrans( Transaction * t ){
  t->setAccount( this );
  transactions.add( t );

  connect( t, SIGNAL( balanceChanged( float, float ) ), this, SLOT( newBal( float, float ) ) );
  connect( t, SIGNAL( infoChanged() ), this, SLOT( refreshSignal() ) );
  connect( t, SIGNAL( recChanged() ), this, SLOT( refreshSignal() ) );
  
  if ( !loading ) newBal( 0, t->getSum() );  
  if ( !payees.contains( t->get( Transaction::PAYEE ) ) )
    payees.inSort( t->get( Transaction::PAYEE ) );
  
  if ( defaultTransType=="<incr number>" ){
    int i=t->get( Transaction::NUM ).toInt();
    if ( i > nextTranID ) nextTranID=i;
  }

  emit addedTrans( t );  
}

Transaction * Account::createBalancedTrans( Transaction * tr, float sum ){
  // if the sum is 0, just use the model's sum
  // else, we're adding a split

  Transaction * newTrans=new Transaction();
  newTrans->setAll( tr );

  if ( sum!=0 ) newTrans->setSum( getModifiedSumFor( sum, tr->getAccount() ) );
  
  // FIXME: what about connecting to a slot called negateBal or useBal to keep
  // transactions in synch with their pairs. If so, connect it here

  // something like:
  //connect( t, SIGNAL( balanceChanged( float, float ) ), this, SLOT( newBal( float, float ) ) );

  // I think this should probably go in the addSplit fx of the transaction

  addTrans( newTrans );
  newTrans->addSplit( tr );
  tr->addSplit( newTrans );

  return newTrans;
}

Transaction * Account::createTrans( QDate d, QString num, QString payee,
				    QString memo, float sum,
				    Transaction::RecoType rt ){
  Transaction * t=new Transaction( d, num, payee, memo, sum );
  t->setReconciled( rt );
  addTrans( t );
  return t;
}

void Account::remTrans( int index ){
  remTrans( transactions.getTrans( index ) );
}

void Account::remTrans( Transaction * t ){
  float old=t->getSum();
  if ( transactions.remove( t ) ){
    memtransactions.remove( t );
    newBal( old, 0 );
    delete t;
    emit removedTrans();
  }
}

Transaction * Account::getTrans( int i ) {
  if ( i >= 0 && i < transactions.length() ) return transactions.getTrans( i );
  else return 0;
}

int Account::getNumTrans( QDate beforeThisDate ) const {
  if ( beforeThisDate.isNull() ) return transactions.length();
  return transactions.length( beforeThisDate );
}

bool Account::writeToFile( const QString& filename ){
  QFile f( filename+".info" );
  if ( f.open( IO_WriteOnly ) ){
    QTextStream out( &f );
    // {"INSTITUTION","INSTADDR1","INSTADDR2","INSTCITY","INSTSTATE", 
    // "INSTZIP","INSTPHONE","INSTFAX", "INSTEMAIL","INSTCONTACT","NAME"};
    out<<"ACCTID="<<getID()<<endl;
    out<<"PARENTID="<<( isTopLevel() ? 0 : getParent()->getID() )<<endl;
    out<<"TYPE="<<( int ) getType()<<endl;
    out<<"NAME="<<info[NAME]<<endl;
    out<<"INSTITUTION="<<info[INSTITUTION]<<endl;
    out<<"INSTADDR1="<<info[INSTADDR1]<<endl;
    out<<"INSTADDR2="<<info[INSTADDR2]<<endl;
    out<<"INSTCITY="<<info[INSTCITY]<<endl;
    out<<"INSTSTATE="<<info[INSTSTATE]<<endl;
    out<<"INSTZIP="<<info[INSTZIP]<<endl;
    out<<"INSTCONTACT="<<info[INSTCONTACT]<<endl;
    out<<"INSTPHONE="<<info[INSTPHONE]<<endl;
    out<<"INSTFAX="<<info[INSTFAX]<<endl;
    out<<"INSTEMAIL="<<info[INSTEMAIL]<<endl;
    out<<"ID="<<info[ID]<<endl;
    out<<"TAXABLE="<<( isTaxable() ? "Y" : "N" )<<endl;
    out<<"BAL="<<openBal<<endl;
    out<<"DEFAULTTRANSTYPE="<<getDefaultTransType()<<endl;
    writeMoreToFile( f );
    f.close();
    chmod( f.name(), S_IRUSR | S_IWUSR );

    writeTrans( memtransactions, filename+".mem" );
    return writeTrans( transactions, filename+".trans" );
  }
  return false;
}

bool Account::writeMoreToFile( QFile& f ){ return true; }

bool Account::writeTrans( const QHaccVector& v, const QString & filename ){
  QFile f( filename  );
  if ( f.open( IO_WriteOnly ) ){
    QTextStream out( &f );
    
    out<<"NUMTRANS="<<v.length()<<endl;
    for ( int i=0; i<v.length(); i++ )
      out<<v.getTrans( i )->getWriteableString()<<endl;
    
    chmod( f.name(), S_IRUSR | S_IWUSR );
    f.close();
    
    return true;
  }
  return false;
}

Account * Account::loadFromFile( const QString& filename ){
  QFile f( filename + ".info" );
  Account * returner=0;

  if ( f.open( IO_ReadOnly ) ){
    QString read, dttype="";
    int type=-1, id=-1;
    float bal=0;
    bool tax=false;
    QString * newInfo=new QString[INFOARRAYSIZE];
    unsigned int pid=0;
    
    QTextStream in( &f );
    while ( !in.eof() ){
      read=in.readLine();
      int j=read.find( '=' );
      
      QString cater=read.left( j );
      QString jread=read.mid( j+1 );

      if ( cater=="NAME" ) newInfo[NAME]=jread;
      else if ( cater=="INSTITUTION" ) newInfo[INSTITUTION]=jread;
      else if ( cater=="INSTCONTACT" ) newInfo[INSTCONTACT]=jread;
      else if ( cater=="INSTADDR1" ) newInfo[INSTADDR1]=jread;
      else if ( cater=="INSTADDR2" ) newInfo[INSTADDR2]=jread;
      else if ( cater=="INSTSTATE" ) newInfo[INSTSTATE]=jread;
      else if ( cater=="INSTPHONE" ) newInfo[INSTPHONE]=jread;
      else if ( cater=="INSTEMAIL" ) newInfo[INSTEMAIL]=jread;
      else if ( cater=="INSTCITY" ) newInfo[INSTCITY]=jread;
      else if ( cater=="INSTZIP" ) newInfo[INSTZIP]=jread;
      else if ( cater=="INSTFAX" ) newInfo[INSTFAX]=jread;
      else if ( cater=="ID" ) newInfo[ID]=jread;
      else if ( cater=="DEFAULTTRANSTYPE" ) dttype=jread;
      //the following aren't strings
      else if ( cater=="BAL" ) bal=jread.toFloat();
      else if ( cater=="PARENTID" ) pid=jread.toUInt();
      else if ( cater=="ACCTID" ) id=jread.toUInt();
      else if ( cater=="TYPE" ) type=jread.toInt();
      else if ( read=="TAXABLE=N" ) tax=false;
      else if ( read=="TAXABLE=Y" ) tax=true;
    }
    f.close();

    returner=Account::create( ( AccountType ) type, id, newInfo, bal, tax );
    returner->setDefaultTransType( dttype );
    returner->parentID=pid;
    delete [] newInfo;
  }

  return returner;
}

void Account::linkToParent( QHaccAccountManager * man ){
  parent=man->getByID( parentID );
}
bool Account::memTrans( Transaction * t ){
  bool alreadyIn=memtransactions.contains( t );
  if ( !alreadyIn ) memtransactions.add( t );
  return !alreadyIn;
}

QHaccVector Account::getMems() const { return memtransactions;}

void Account::addMemTrans( int i ){ 
  if ( i<memtransactions.length() ){
    Transaction * t=memtransactions.getTrans( i );
    addCloneTransFrom( t );
  }
}

void Account::addCloneTransFrom( Transaction * t ){ 
  MemorizedTransaction * addMe=new MemorizedTransaction( t->getWriteableString() );
  t->getAccount()->addTrans( addMe );
  addMe->load( QHaccAccountManager::getInstance() );
  emit needRefresh();
}

void Account::loadMems( const QString& filename ){
  QFile f( filename + ".mem" );
  
  if ( f.open( IO_ReadOnly ) ){
    QTextStream in( &f );
    QString read=in.readLine();
    int finder=read.find( '=' );
    int nt=read.mid( finder+1 ).toInt();
    
    for ( int i=0; i<nt; i++ ){
      Transaction * t=new Transaction ( in.readLine() );
      memtransactions.add( transactions.find( t->getID() ) );
      delete t;
    }
  }
}

Account * Account::create( AccountType ty, unsigned int _id,
			   const QString * _info, float f, bool tx ){
  Account * returner=0;
  switch( ty ){
  case ASSET:
    returner=new AssetAccount( _info, _id, f, tx );
    break;
  case LIABILITY:
    returner=new LiabilityAccount( _info, _id, f, tx );
    break;
  case EQUITY:
    returner=new EquityAccount( _info, _id, f, tx );
    break;
  case EXPENSE:
    returner=new ExpenseAccount( _info, _id, f, tx );
    break;
  case REVENUE:
    returner=new RevenueAccount( _info, _id, f, tx );
  }

  return returner;
}

void Account::startLoad( int i ){
  loading=true;
  transactions.startLoad( i );
}

void Account::stopLoad(){
  loading=false;
  transactions.stopLoad();
  recalcBal();
}

Transaction * Account::find( unsigned int i ) const {
  return transactions.find( i );
}

QHaccVector Account::getSubset( QDate s, QDate e ) const {
  return transactions.getSubset( s, e );
}

QHaccVector Account::getUnreconciled( float& sum, float& unsum,
				      QDate limit ) const {
  int num=getNumTrans( limit );
  sum=getBal( OPENING );
  unsum=0;
  
  QHaccVector v( num );
  v.setNosort( true );
  for ( int i=0; i<num; i++ ){
    Transaction * trans=transactions.getTrans( i );
    if ( trans->isReconciled() ) sum+=trans->getSum();
    else{
      unsum+=trans->getSum();
      v.add( trans );
    }
  }
  return v;
}

QHaccVector Account::prune( QDate limit ){
  // return the trans that'll get axed (but don't delete 'em)
  int num=getNumTrans( limit );
  float pruneBal=0;

  QHaccVector v( num, 0, 0 );
  v.setNosort( true );
  for ( int i=0; i<num; i++ ) v.add( transactions.getTrans( i ) );

  startLoad( 0 ); //don't do any sorting or calculating
  for ( int i=0; i<num; i++ ){
    Transaction * t=transactions.getTrans( 0 );
    pruneBal+=t->getSum();
    transactions.remove( t );
  }

  setOBal( getBal( OPENING )+pruneBal );
  stopLoad();

#ifdef QDEBUG
  cout<<getName()<<": "<<num<<" out, "<<getNumTrans()<<" left"<<endl;

  Transaction * t;
  cout<<"out:"<<endl;
  for ( int i=0; i<v.length(); i++ ){
    t=v.getTrans( i );
    QDate d=t->getDate();
    cout<<" "<<t->getID()<<" "<<t->get( Transaction::PAYEE )<<" "
	<<d.month()<<"/"<<d.day()<<"/"<<d.year()<<endl;
  }
  cout<<"in:"<<endl;
  for ( int i=0; i<transactions.length(); i++ ){
    t=transactions.getTrans( i );
    QDate d=t->getDate();
    cout<<" "<<t->getID()<<" "<<t->get( Transaction::PAYEE )<<" "
	<<d.month()<<"/"<<d.day()<<"/"<<d.year()<<endl;
  }
#endif

  return v;
}

float Account::getModifiedSumFor( float sum, const Account * a ) const {
  // here's where we worry about account types
  // and transactions' sums being consistent
  if ( a->isLeftPlusAccount()==isLeftPlusAccount() ) return 0-sum;
  else return sum;
}

/*****************************************/
/***** Derivative Account definitions ****/
/*****************************************/

LeftPlusAccount::LeftPlusAccount( const QString * info, unsigned int id,
				  float bal, bool tx )
  : Account( info, id, bal, tx ){};
LeftPlusAccount::~LeftPlusAccount(){}

bool LeftPlusAccount::isLeftPlusAccount() const { return true; }

/*****************************************/

RightPlusAccount::RightPlusAccount( const QString * info, unsigned int id,
				    float bal, bool tx )
  : Account( info, id, bal, tx ){};
RightPlusAccount::~RightPlusAccount(){}

bool RightPlusAccount::isLeftPlusAccount() const { return false; }

/*****************************************/

AssetAccount::AssetAccount( const QString * info, unsigned int id,
			    float bal, bool tx )
  : LeftPlusAccount( info, id, bal, tx ){}

AssetAccount::~AssetAccount(){}

AccountType AssetAccount::getType() const { return ASSET; }
bool AssetAccount::isType( AccountType type ) const { return type==ASSET; }

/*****************************************/

LiabilityAccount::LiabilityAccount( const QString * info, unsigned int id,
				    float bal, bool tx )
  : RightPlusAccount( info, id, bal, tx ){}

LiabilityAccount::~LiabilityAccount(){}

AccountType LiabilityAccount::getType() const { return LIABILITY; }
bool LiabilityAccount::isType( AccountType type ) const {
  return type==LIABILITY;
}

/*****************************************/

EquityAccount::EquityAccount( const QString * info, unsigned int _id,
			      float bal, bool tx )
  : RightPlusAccount( info, _id, bal, tx ){}

EquityAccount::~EquityAccount(){}

AccountType EquityAccount::getType() const { return EQUITY; }
bool EquityAccount::isType( AccountType type ) const { return type==EQUITY; }

/*************************************/

ExpenseAccount::ExpenseAccount( const QString * info, unsigned int _id,
			      float bal, bool tx )
  : EquityAccount( info, _id, bal, tx ){}

ExpenseAccount::~ExpenseAccount(){}

bool ExpenseAccount::isLeftPlusAccount() const { return true; }
AccountType ExpenseAccount::getType() const { return EXPENSE; }
bool ExpenseAccount::isType( AccountType type ) const {
  return ( type==EXPENSE || type==EQUITY );
}

/*************************************/

RevenueAccount::RevenueAccount( const QString * info, unsigned int _id,
			      float bal, bool tx )
  : EquityAccount( info, _id, bal, tx ){}

RevenueAccount::~RevenueAccount(){}

AccountType RevenueAccount::getType() const { return REVENUE; }
bool RevenueAccount::isType( AccountType type ) const {
  return ( type==REVENUE || type==EQUITY );
}

/*************************************/
