/***************************************************************************
                          rmath.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rmath.cpp 1998/08/27 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include <math.h>

#include "exprparser.h"
#include "rgraphdef.h"
#include "rlog.h"
#include "rmath.h"
#include "rstring.h"

#include "relement.h"

//
// Finds greatest common divider using Euclid's algorithm
//
static int
findGCD( int _a, int _b )
{
  int rem;

  while (_b != 0) {
    rem = _a % _b;
    _a = _b;
    _b = rem;
  }

  return _a;
}


// Compare two double values:
//
// return: true: values are equal
//         false: values are not equal
//
bool 
mtCompFloat(double _v1, double _v2, double _tol)
{
  double delta = _v2-_v1;
  
  if(delta>-_tol && delta<_tol) return true;
  else                          return false;
}



// Correct an angle (bring it to range of 0-360)
//   this function returns the new angle
//   is doesn't change the parameter
//
double
mtCorrAngle(double _ang)
{
  double ret=_ang;
  
  while(ret<   0.0) ret+=360.0;
  while(ret>=360.0) ret-=360.0;
  return ret;
}



// Correct an angle (bring it to range of 0-360)
//   this function changes the parameter
//
void
mtCorrAngle(double* _ang)
{
  while(*_ang<   0.0) *_ang+=360.0;
  while(*_ang>=360.0) *_ang-=360.0;
}


// Test if angle _ang is beteen _startAngle and _endAngle:
//   (in pos. direction)
//
bool
mtAngleIsBetween(double _ang, double _startAngle, double _endAngle)
{
  bool ret=false;

  if(_startAngle>_endAngle) {
    if(_ang>=_startAngle || _ang<=_endAngle) ret=true;
  }
  else {
    if(_ang>=_startAngle && _ang<=_endAngle) ret=true;
  }

  return(ret);
}



// Get Angle from point _cx, _cy
//   to point _px, _py
//
double 
mtGetAngle(double _x1, double _y1, double _x2, double _y2)
{
  double xdist, ydist, angle;

  xdist=_x2-_x1;
  ydist=_y2-_y1;

  if(mtCompFloat(xdist, 0.0)) {
    if(_y2<_y1) angle=270.0;
    else angle=90.0;
    return angle;
  }

  if(mtCompFloat(ydist, 0.0)) {
    if(_x2<_x1) angle=180.0;
    else angle=0.0;
    return angle;
  }

  angle = atan(ydist/xdist)*ARAD;

  if(angle<0.0) angle*=-1.0;

  if(_x2>_x1 && _y2>_y1) angle =   0.0 + angle;   // 1. Quadrant
  if(_x2<_x1 && _y2>_y1) angle = 180.0 - angle;   // 2. Quadrant
  if(_x2<_x1 && _y2<_y1) angle = 180.0 + angle;   // 3. Quadrant
  if(_x2>_x1 && _y2<_y1) angle = 360.0 - angle;   // 4. Quadrant

  return angle;
}



// Get Angle difference between _a1 and _a2
//
//   (Which angle must I add to _a1 to get _a2)
//
double 
mtGetAngleDiff(double _a1, double _a2) 
{
 double ret;

 if(_a1>=_a2) _a2+=360.0;
 ret = _a2-_a1;

 if(ret>=360.0) ret=0.0;

 return ret;
}



// Get Distance from point _cx, _cy
//   to point _px, _py
//
double 
mtGetDistance(double _x1, double _y1, double _x2, double _y2)
{
  double xd2 = (_x2-_x1)*(_x2-_x1);
  double yd2 = (_y2-_y1)*(_y2-_y1);
  double d   = xd2+yd2;
  if(d<0.0) d*=-1.0;
  return(sqrt(d));
}



// Rotate point _px/_py around center _cx/_cy with angle _angle
//
void  
mtRotatePoint(double _cx, double _cy, double* _px, double* _py, double _angle)
{
  double dist,      // temporary distance
        ang;     // temporary angle

  dist = mtGetDistance(_cx, _cy, *_px, *_py);
  ang  = mtCorrAngle(mtGetAngle(_cx, _cy, *_px, *_py)+_angle);
  *_px = _cx+cos(ang/ARAD)*dist;
  *_py = _cy+sin(ang/ARAD)*dist;
}



// Mirror point _px/_py on line (_x1,_y1 / _x2,_y2):
//
void  
mtMirrorPoint(double _x1, double _y1, 
              double _x2, double _y2, 
              double* _px, double* _py)
{
  double tmpDis,          // temporary distance
        tmpX, tmpY;      // temporary coordinate
  RElement tmpEl;        // temporary element
  double mirAngle;        // mirroring angle (angle of axis +90.0)
  
  tmpEl.createLine(_x1, _y1, _x2, _y2);
  
  mirAngle = mtCorrAngle(tmpEl.getDirection1()+90.0);

  tmpDis = tmpEl.getDistanceToPoint(*_px, *_py, true);
  tmpX = *_px + cos(mirAngle/ARAD)*tmpDis*2.0;
  tmpY = *_py + sin(mirAngle/ARAD)*tmpDis*2.0;
  
  // Mirrored to wrong side:
  //
  if(tmpEl.getDistanceToPoint(tmpX, tmpY, true) >= 1.9*tmpDis) {
    tmpX = *_px - cos(mirAngle/ARAD)*tmpDis*2.0;
    tmpY = *_py - sin(mirAngle/ARAD)*tmpDis*2.0;
  }
  *_px = tmpX;
  *_py = tmpY;
}



// Square:
//
double 
mtSqr(double _v)
{
  return _v*_v;
}


/** Round double to int
*/
int
mtRound( double _v )
{
  int ret;
  ret = (int)(_v);

  if( (_v-ret) >= 0.5 ) {
    ret++;
  }

  return ret;
}


/** Special atan function for ellipses
    originally coded by Carsten Neuberg
*/
double
mtAtan( double ak, double gk )
{
  double h;
  double atan1_result;

  if( (Absolute(gk)<1e-4) && (Absolute(ak)<1e-4) ) {
    atan1_result=0;return atan1_result;
  }

  if( Absolute(ak)<1e-4 ) {
    if (gk<0) h=-M_PI/2;
    else      h=M_PI/2;
  }
  else if(ak>0) {
    h=atan(gk/ak);
  }
  else {
    h=M_PI-atan(gk/Absolute(ak));
  }

  if( h<0 ) atan1_result=2*M_PI+h;
  else      atan1_result=h;

  return atan1_result;
}



/** Format a value to a string in the form of 13.235 or 1" 3/16'

  \param _v  value
  \param _exactness (e.g.: 0.001 or 1/128)
  \param _unit (e.g.: Mm, Cm, Inch, ...)
  \param _format (e.g.: Scientific, Decimal, Fractional)
*/
QString
mtFormatFloat( double _v,
               double _exactness,
               Unit _unit,
               ValueFormat _format,
               bool _showUnit )
{
  RLOG( "\nmtFormatFloat: " );

  RLOG( "\nUnit: " );
  RLOG( mtGetSignForUnit( _unit ) );

  QString ret="";
  QString sign="";
  if(_showUnit) sign = mtGetSignForUnit( _unit );

  // Show as fraction:
  if( _format==Fractional ) {
    int num,            // number of complete units   (x' 7/128")
        numinator,      // number of fractions    (x/128)
        denominator;    // (4/x)
    QString neg="";

    RLOG( "\nvalue: " );
    RLOG( _v );

    if(_v < 0) {
      neg = "-";
      _v = -_v;
    }

    num = (int)floor(_v);
    numinator = mtRound((_v-num)/_exactness);
    denominator = mtRound(1/_exactness);

    // Fraction is 1:
    if( numinator==denominator ) {
      numinator=denominator=0;
      ++num;
    }

    if( _unit==Foot ) {
      if( num!=0 || numinator==0 ) {
        ret.sprintf( "%d'", num );

        if( numinator!=0 ) {
          ret += " ";
        }
      }
      if( numinator!=0 ) {
        QString inches = mtFormatFloat( (double)(numinator*12)/denominator, _exactness, Inch, Fractional, true );
        ret += inches;
      }
      if( num!=0 || numinator!=0 ) {
        ret = neg + ret;
      }
      return ret;
    }

    // Simplify the fraction
    if( numinator!=0 && denominator!=0 ) {
      int gcd = findGCD( numinator, denominator );
      numinator = numinator / gcd;
      denominator = denominator / gcd;
    }

    if( num!=0 && numinator!=0 ) {
      ret.sprintf( "%s%d %d/%d%s", neg.latin1(), num, numinator, denominator, sign.latin1() );
    }
    else if( numinator!=0 ) {
      ret.sprintf( "%s%d/%d%s", neg.latin1(), numinator, denominator, sign.latin1() );
    }
    else if( num!=0 ) {
      ret.sprintf( "%s%d%s", neg.latin1(), num, sign.latin1() );
    }
    else {
      ret.sprintf( "0%s", sign.latin1() );
    }

    RLOG( "\nret: " ); RLOG( ret );
    return ret;
  }

  // Show scientific:
  else if( _format==Scientific ) {
    ret.sprintf( "%e%s", _v, sign.latin1()  );
    return ret;
  }

  // Decimal:
  else {
    RLOG( "\nDecimal" );
    RLOG( "\nValue: " );
    RLOG( _v );
    RLOG( "\nExactness: " );
    RLOG( _exactness );

    ret = strFloatToString( _v, _exactness );
    if(_showUnit) ret+=sign;

    /*
    int num,          // number of exactness-number in the value.
        dotPos,       // Position of dot
        digits,       // Number of digits after dot
        lastDig;      // last significant digit
    QString exaStr;
    //QString format;

    num = mtRound(_v/_exactness);
    if( _v - num*_exactness >= 0.5*_exactness ) num++;

    exaStr = strFloatToString( _exactness );

    / *
    exaStr.setNum( _exactness, 'f', 6 );

    RLOG( "\nexaStr: " );
    RLOG( exaStr );

    // Remove zeros and point at the end:
    for( int i = lastDig = exaStr.length()-1; i>=0; --i ) {
      if( exaStr.at(i)=='0' ) {
        lastDig=i-1;
      }
      else if( exaStr.at(i)=='.' ) {
        lastDig=i-1;
        break;
      }
      else {
        break;
      }
    }

    exaStr = exaStr.left( lastDig+1 );
    * /

    RLOG( "\nexaStr cut: " );
    RLOG( exaStr );

    dotPos = exaStr.find( '.' );
    if( dotPos==-1 ) {
      ret.sprintf( "%d", mtRound(num*_exactness) );
    }
    else {
      digits = exaStr.length() - dotPos - 1;
      ret = strFloatToStr( num*_exactness, digits );
      //format.sprintf( "%%.%df", digits );
      //ret.sprintf( format, num*_exactness );
    }

    //RLOG( "\nFormat: " );
    //RLOG( format );

    RLOG( "\n_v:" ); RLOG( _v );
    RLOG( "\n_exactness:" ); RLOG( _exactness );
    RLOG( "\nret: " ); RLOG( ret );
    */

    return ret;
  }
}



/** Format a value to a string in the form of 13.235 or 1" 3/16'

  \param _v  value
  \param _exactness (e.g.: 0.001 or 1/128)
  \param _unit (e.g.: Deg, Rad, Gra)
  \param _format (e.g.: DecimalDegrees, DegreesMinutesSeconds, Radians, Gradians, Surveyor)
*/
QString
mtFormatAngleFloat( double _v,
                    double _exactness,
                    AngleUnit _unit,
                    AngleFormat _format )
{
  double value=_v;
  QString ret="";

  // Transform to unit:
  switch( _unit ) {
    case Deg: value = _v;                      break;
    case Rad: value = _v / ( 180.0 / M_PI );   break;
    case Gra: value = _v / 0.9;                break;
  }

  // Round to exactness:
  //int num = mtRound( value / _exactness );
  //value = num*_exactness;

  switch(_format) {
    case DecimalDegrees:
    case Radians:
    case Gradians:
      ret = strFloatToString( value, _exactness );
      if(_format==DecimalDegrees) ret+="";
      if(_format==Radians)        ret+="r";
      if(_format==Gradians)       ret+="g";

      /*
      if(true) {
        QString exaStr;
        int dotPos;

        exaStr = strFloatToString( _exactness );
        dotPos = exaStr.find( '.' );
        if( dotPos==-1 ) {
          ret.sprintf( "%d", mtRound(num*_exactness) );
        }
        else {
          int digits = exaStr.length() - dotPos - 1;
          ret = strFloatToStr( num*_exactness, digits );
        }
      }
      */
      break;

    case DegreesMinutesSeconds:
      if(true) {
        int vDegrees, vMinutes;
        double vSeconds;
        QString degrees, minutes, seconds;

        vDegrees = (int)floor(value);
        vMinutes = (int)floor( (value - vDegrees) * 60.0 );
        vSeconds = (value - vDegrees - (vMinutes/60.0)) * 3600.0;

        seconds = strFloatToString( vSeconds, _exactness );

        if(seconds=="60") {
          seconds="0";
          ++vMinutes;
          if(vMinutes==60) {
            vMinutes=0;
            ++vDegrees;
          }
        }

        degrees.setNum( vDegrees );
        minutes.setNum( vMinutes );

        ret = degrees + " " + minutes + "' " + seconds + "\"";
      }
      break;

    case Surveyor:

      break;

  }

  return ret;

}



/** Gets the sign for a unit

    \param _unit The unit as enum
    \param _part Part one (') or two (")
*/
QString
mtGetSignForUnit( Unit _unit )
{
  switch( _unit ) {
    case None:       return "";
    case Inch:       return "\"";
    case Foot:       return "'";
    case Mile:       return "mi";
    case Millimeter: return "mm";
    case Centimeter: return "cm";
    case Meter:      return "m";
    case Kilometer:  return "km";
    case Microinch:  return "\"";
    case Mil:        return "mil";
    case Yard:       return "yd";
    case Angstrom:   return "A";
    case Nanometer:  return "nm";
    case Micron:     return "m";
    case Decimeter:  return "dm";
    case Decameter:  return "dam";
    case Hectometer: return "hm";
    case Gigameter:  return "Gm";
    case Astro:      return "astro";
    case Lightyear:  return "ly";
    case Parsec:     return "pc";
  }

  return "";
}


/** Checks if a unit is metric
*/
bool
mtIsUnitMetric( Unit _unit ) {
  switch( _unit ) {
    case Inch:
    case Foot:
    case Mile:
    case Yard: return false;

    default:   return true;
  }
}


/** Gets unit enum from string:
*/
Unit
mtStringToUnit( const QString &_unit, bool *ok )
{
  if(ok) *ok=true;

  if(_unit=="None")         return None;
  if(_unit=="Inch")         return Inch;
  if(_unit=="Foot")         return Foot;
  if(_unit=="Mile")         return Mile;
  if(_unit=="Millimeter")   return Millimeter;
  if(_unit=="Centimeter")   return Centimeter;
  if(_unit=="Meter")        return Meter;
  if(_unit=="Kilometer")    return Kilometer;
  if(_unit=="Microinch")    return Microinch;
  if(_unit=="Mil")          return Mil;
  if(_unit=="Yard")         return Yard;
  if(_unit=="Angstrom")     return Angstrom;
  if(_unit=="Nanometer")    return Nanometer;
  if(_unit=="Micron")       return Micron;
  if(_unit=="Decimeter")    return Decimeter;
  if(_unit=="Decameter")    return Decameter;
  if(_unit=="Hectometer")   return Hectometer;
  if(_unit=="Gigameter")    return Gigameter;
  if(_unit=="Astro")        return Astro;
  if(_unit=="Lightyear")    return Lightyear;
  if(_unit=="Parsec")       return Parsec;

  if(ok) *ok=false;
  return None;
}


/** Gets string from unit enum:
*/
QString
mtUnitToString( Unit _unit )
{
  switch( _unit ) {
    case None:         return "None";
    case Inch:         return "Inch";
    case Foot:         return "Foot";
    case Mile:         return "Mile";
    case Millimeter:   return "Millimeter";
    case Centimeter:   return "Centimeter";
    case Meter:        return "Meter";
    case Kilometer:    return "Kilometer";
    case Microinch:    return "Microinch";
    case Mil:          return "Mil";
    case Yard:         return "Yard";
    case Angstrom:     return "Angstrom";
    case Nanometer:    return "Nanometer";
    case Micron:       return "Micron";
    case Decimeter:    return "Decimeter";
    case Decameter:    return "Decameter";
    case Hectometer:   return "Hectometer";
    case Gigameter:    return "Gigameter";
    case Astro:        return "Astro";
    case Lightyear:    return "Lightyear";
    case Parsec:       return "Parsec";
  }

  return "None";
}



/** Gets value format enum from string:
*/
ValueFormat
mtStringToValueFormat( const QString &_format, bool *ok )
{
  if(ok) *ok=true;

  if(_format=="Scientific") return Scientific;
  if(_format=="Decimal")    return Decimal;
  if(_format=="Fractional") return Fractional;

  if(ok) *ok=false;
  return Decimal;
}


QString
mtValueFormatToString( ValueFormat _format )
{
  switch( _format ) {
    case Scientific: return "Scientific";
    case Decimal:    return "Decimal";
    case Fractional: return "Fractional";
  }
  return "Decimal";
}



/** Gets angle unit enum from string:
*/
AngleUnit
mtStringToAngleUnit( const QString &_angleUnit, bool *ok )
{
  if(ok) *ok=true;

  if(_angleUnit=="Deg") return Deg;
  if(_angleUnit=="Rad") return Rad;
  if(_angleUnit=="Gra") return Gra;

  if(ok) *ok=false;
  return Deg;
}



/** Gets angle unit enum from string:
*/
QString
mtAngleUnitToString( AngleUnit _angleUnit )
{
  switch( _angleUnit ) {
    case Deg:   return "Deg";
    case Rad:   return "Rad";
    case Gra:   return "Gra";
  }

  return "Deg";
}


/** Gets angle format enum from string:
*/
AngleFormat
mtStringToAngleFormat( const QString &_format, bool *ok )
{
  if(ok) *ok=true;

  if(_format.lower()=="decimaldegrees")   return DecimalDegrees;
  if(_format.lower()=="decimal degrees")  return DecimalDegrees;
  if(_format.lower()=="decimal_degrees")  return DecimalDegrees;
  if(_format.lower()=="decimal")          return DecimalDegrees;
  if(_format.lower()=="degrees")          return DecimalDegrees;
  if(_format=="DegMinSec")                return DegreesMinutesSeconds;
  if(_format=="/'/\"")                   return DegreesMinutesSeconds;
  if(_format=="Radians")                  return Radians;
  if(_format=="Gradians")                 return Gradians;
  if(_format=="Surveyor")                 return Surveyor;

  if(ok) *ok=false;
  return DecimalDegrees;
}



QString
mtAngleFormatToString( AngleFormat _format )
{
  switch( _format ) {
    case DecimalDegrees:        return "Decimal_Degrees";
    case DegreesMinutesSeconds: return "/'/\"";
    case Radians:               return "Radians";
    case Gradians:              return "Gradians";
    case Surveyor:              return "Surveyor";
  }
  return "Decimal_Degrees";
}



ScaleType
mtStringToScaleType( const QString &_type, bool *ok )
{
  if(ok) *ok=true;

  if(_type=="Absolute")       return Absolute;
  if(_type=="Engineering")    return Engineering;
  if(_type=="Architectural")  return Architectural;

  if(ok) *ok=false;
  return Absolute;
}



QString
mtScaleTypeToString( ScaleType _type )
{
  switch( _type ) {
    case Absolute:      return "Absolute";
    case Engineering:   return "Engineering";
    case Architectural: return "Architectural";
  }
  return "Absolute";
}


QPrinter::Orientation
mtStringToOrientation( const QString &_or, bool *ok )
{
  if(ok) *ok=true;

  if(_or=="Portrait")   return QPrinter::Portrait;
  if(_or=="Landscape")  return QPrinter::Landscape;

  if(ok) *ok=false;
  return QPrinter::Portrait;
}



QString
mtOrientationToString( QPrinter::Orientation _or )
{
  switch( _or ) {
    case QPrinter::Portrait:   return "Portrait";
    case QPrinter::Landscape:  return "Landscape";
  }
  return "Portrait";
}



PageFormat
mtStringToPageFormat( const QString &_pf, bool *ok )
{
  if(ok) *ok=true;

  if(_pf=="User")      return User;
  if(_pf=="Letter")    return Letter;
  if(_pf=="Legal")     return Legal;
  if(_pf=="Executive") return Executive;
  if(_pf=="A0")        return A0;
  if(_pf=="A1")        return A1;
  if(_pf=="A2")        return A2;
  if(_pf=="A3")        return A3;
  if(_pf=="A4")        return A4;
  if(_pf=="A5")        return A5;
  if(_pf=="A6")        return A6;
  if(_pf=="A7")        return A7;
  if(_pf=="A8")        return A8;
  if(_pf=="A9")        return A9;
  if(_pf=="A10")       return A10;
  if(_pf=="B0")        return B0;
  if(_pf=="B1")        return B1;
  if(_pf=="B2")        return B2;
  if(_pf=="B3")        return B3;
  if(_pf=="B4")        return B4;
  if(_pf=="B5")        return B5;
  if(_pf=="B6")        return B6;
  if(_pf=="B7")        return B7;
  if(_pf=="B8")        return B8;
  if(_pf=="B9")        return B9;
  if(_pf=="B10")       return B10;
  if(_pf=="C0")        return C0;
  if(_pf=="C1")        return C1;
  if(_pf=="C2")        return C2;
  if(_pf=="C3")        return C3;
  if(_pf=="C4")        return C4;
  if(_pf=="C5")        return C5;
  if(_pf=="C6")        return C6;
  if(_pf=="C7")        return C7;
  if(_pf=="C8")        return C8;
  if(_pf=="C9")        return C9;
  if(_pf=="C10")       return C10;
  if(_pf=="C5E")       return C5E;
  if(_pf=="Comm10E")   return Comm10E;
  if(_pf=="DLE")       return DLE;
  if(_pf=="Folio")     return Folio;
  if(_pf=="Ledger")    return Ledger;
  if(_pf=="Tabloid")   return Tabloid;
  if(_pf=="NPageSize") return NPageSize;

  if(ok) *ok=false;
  return User;
}



QString
mtPageFormatToString( PageFormat _pf )
{
  switch( _pf ) {
    case User:      return "User";
    case Letter:    return "Letter";
    case Legal:     return "Legal";
    case Executive: return "Executive";
    case A0:        return "A0";
    case A1:        return "A1";
    case A2:        return "A2";
    case A3:        return "A3";
    case A4:        return "A4";
    case A5:        return "A5";
    case A6:        return "A6";
    case A7:        return "A7";
    case A8:        return "A8";
    case A9:        return "A9";
    case A10:       return "A10";
    case B0:        return "B0";
    case B1:        return "B1";
    case B2:        return "B2";
    case B3:        return "B3";
    case B4:        return "B4";
    case B5:        return "B5";
    case B6:        return "B6";
    case B7:        return "B7";
    case B8:        return "B8";
    case B9:        return "B9";
    case B10:       return "B10";
    case C0:        return "C0";
    case C1:        return "C1";
    case C2:        return "C2";
    case C3:        return "C3";
    case C4:        return "C4";
    case C5:        return "C5";
    case C6:        return "C6";
    case C7:        return "C7";
    case C8:        return "C8";
    case C9:        return "C9";
    case C10:       return "C10";
    case C5E:       return "C5E";
    case Comm10E:   return "Comm10E";
    case DLE:       return "DLE";
    case Folio:     return "Folio";
    case Ledger:    return "Ledger";
    case Tabloid:   return "Tabloid";
    case NPageSize: return "NPageSize";
  }
  return "User";
}



double
mtGetPageWidthMM( PageFormat _pf )
{
  switch( _pf ) {
    case User:      return 0.0;
    case Letter:    return 215.9;
    case Legal:     return 215.9;
    case Executive: return 190.5;
    case A0:        return 841.0;
    case A1:        return 594.0;
    case A2:        return 420.0;
    case A3:        return 297.0;
    case A4:        return 210.0;
    case A5:        return 148.0;
    case A6:        return 105.0;
    case A7:        return 74.0;
    case A8:        return 52.0;
    case A9:        return 37.0;
    case A10:       return 26.0;
    case B0:        return 1000.0;
    case B1:        return 707.0;
    case B2:        return 500.0;
    case B3:        return 353.0;
    case B4:        return 250.0;
    case B5:        return 176.0;
    case B6:        return 125.0;
    case B7:        return 88.0;
    case B8:        return 62.0;
    case B9:        return 44.0;
    case B10:       return 31.0;
    case C0:        return 917.0;
    case C1:        return 648.0;
    case C2:        return 458.0;
    case C3:        return 324.0;
    case C4:        return 229.0;
    case C5:        return 162.0;
    case C6:        return 114.0;
    case C7:        return 81.0;
    case C8:        return 57.0;
    case C9:        return 40.0;
    case C10:       return 28.0;
    case C5E:       return 163.0;
    case Comm10E:   return 105.0;
    case DLE:       return 110.0;
    case Folio:     return 210.0;
    case Ledger:    return 432.0;
    case Tabloid:   return 279.0;
    case NPageSize: return 0.0;
  }
  return 0.0;
}


double
mtGetPageHeightMM( PageFormat _pf )
{
  switch( _pf ) {
    case User:      return 0.0;
    case Letter:    return 279.4;
    case Legal:     return 355.6;
    case Executive: return 254.0;
    case A0:        return 1189.0;
    case A1:        return 841.0;
    case A2:        return 594.0;
    case A3:        return 420.0;
    case A4:        return 297.0;
    case A5:        return 210.0;
    case A6:        return 148.0;
    case A7:        return 105.0;
    case A8:        return 74.0;
    case A9:        return 52.0;
    case A10:       return 37.0;
    case B0:        return 1414.0;
    case B1:        return 1000.0;
    case B2:        return 707.0;
    case B3:        return 500.0;
    case B4:        return 353.0;
    case B5:        return 250.0;
    case B6:        return 176.0;
    case B7:        return 125.0;
    case B8:        return 88.0;
    case B9:        return 62.0;
    case B10:       return 44.0;
    case C0:        return 1297.0;
    case C1:        return 917.0;
    case C2:        return 648.0;
    case C3:        return 458.0;
    case C4:        return 324.0;
    case C5:        return 229.0;
    case C6:        return 162.0;
    case C7:        return 114.0;
    case C8:        return 81.0;
    case C9:        return 57.0;
    case C10:       return 40.0;
    case C5E:       return 229.0;
    case Comm10E:   return 241.0;
    case DLE:       return 220.0;
    case Folio:     return 330.0;
    case Ledger:    return 279.0;
    case Tabloid:   return 432.0;
    case NPageSize: return 0.0;
  }
  return 0.0;
}






/** Calculates a mathematical expression using the
    expression parser classes from Giovanni Dicanio
*/
double
mtCalculate( const QString &_expression, double _default, bool *_ok )
{
  double ret;

  if( !_expression.isEmpty() ) {
    ExprParser parser;
    parser.setExpression( _expression.latin1() );
    ret = parser.getValue();

    if( !parser.errorOccurred() ) {
      if(_ok) *_ok=true;
      return ret;
    }
  }

  if(_ok) *_ok=false;
  return _default;
}


/** Returns the amount of mm for _v pieces of _unit
*/
double
mtGetMM( double _v, Unit _unit )
{
  switch( _unit ) {
    case None:       return _v*1.0;
    case Inch:       return _v*25.4;
    case Foot:       return _v*304.8;
    case Mile:       return _v*1609344.0;
    case Millimeter: return _v*1.0;
    case Centimeter: return _v*10.0;
    case Meter:      return _v*1000.0;
    case Kilometer:  return _v*1000000.0;
    case Microinch:  return _v*0.0000254;
    case Mil:        return _v*0.0254;
    case Yard:       return _v*914.4;
    case Angstrom:   return _v*0.0000001;
    case Nanometer:  return _v*0.000001;
    case Micron:     return _v*0.001;
    case Decimeter:  return _v*100.0;
    case Decameter:  return _v*10000.0;
    case Hectometer: return _v*100000.0;
    case Gigameter:  return _v*1000000000.0;
    case Astro:      return _v*149600000000000.0;
    case Lightyear:  return _v*9460731798000000000.0;
    case Parsec:     return _v*30857000000000000000.0;      // Parallaxensekunde
    default:         return _v*1.0;
  }
}


/** Gets a fraction string (1/128) from a float (0.0078125)
*/
QString
mtGetFractionString( double _v )
{
  QString ret;
  ret.setNum( mtRound(1.0/_v) );
  ret = "1/" + ret;

  return ret;
}




// EOF





