/***************************************************************************
                          rlayerwidget.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** rlayerwidget.cpp 1998/08/24 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include "rlayerwidget.h"

#include <qapplication.h>
#include <qkeycode.h>
#include <qmessagebox.h>
#include <qtoolbutton.h>
#include <qtooltip.h>

#include "rappwin.h"
#include "rgraphic.h"
#include "rinputdialog.h"
#include "rlayerdialog.h"
#include "rlayoutdef.h"
#include "rlistboxitem.h"
#include "rlog.h"
#include "rprgdef.h"
#include "rtoolbar.h"
#include "rxpmlayer.h"
#include "ryesnodialog.h"

// Constructor:
//
// _rw: read/write
//
RLayerWidget::RLayerWidget(bool _rw,
                           QWidget* _parent, 
                           const char* _name, 
                           WFlags _f)
:RWidget(_parent, _name, _f)
{
  graphic=0;

  //setFrameStyle(WinPanel|Raised);
  setFrameStyle(Panel|Raised);
  setLineWidth(1);

  // create label:
  lLabel = new QLabel( tr("Layer List"), this );
  lLabel->setAlignment( QLabel::AlignCenter|QLabel::AlignVCenter );
  lLabel->setFrameStyle( QLabel::WinPanel|QLabel::Raised );
  lLabel->setLineWidth( 2 );
  lLabel->setBackgroundColor( qApp->palette().normal().light() );

  // Create Tools:
  //
  bShowLayers = new QToolButton(this, "showlayers");
  bShowLayers->setPixmap(QPixmap(visiblelayer_xpm));
  QToolTip::add(bShowLayers, tr("Show all Layers"));
  connect(bShowLayers, SIGNAL(clicked()), this, SLOT(showLayers()));
  
  bHideLayers = new QToolButton(this, "hidelayers");
  bHideLayers->setPixmap(QPixmap(hiddenlayer_xpm));
  QToolTip::add(bHideLayers, tr("Hide all Layers"));
  connect(bHideLayers, SIGNAL(clicked()), this, SLOT(hideLayers()));
  
  if(_rw) {
    bAddLayer = new QToolButton(this, "addlayer");
    bAddLayer->setPixmap(QPixmap(addlayer_xpm));
    QToolTip::add(bAddLayer, tr("Add a Layer"));
    connect(bAddLayer, SIGNAL(clicked()), this, SLOT(addLayer()));
    
    bRemoveLayer = new QToolButton(this, "removelayer");
    bRemoveLayer->setPixmap(QPixmap(removelayer_xpm));
    QToolTip::add(bRemoveLayer, tr("Remove the selected Layer"));
    connect(bRemoveLayer, SIGNAL(clicked()), this, SLOT(removeLayer()));
  
    bRenameLayer = new QToolButton(this, "renamelayer");
    bRenameLayer->setPixmap(QPixmap(renamelayer_xpm));
    QToolTip::add(bRenameLayer, tr("Change attributes of\nthe selected layer"));
    connect(bRenameLayer, SIGNAL(clicked()), this, SLOT(renameLayer()));

    bRemoveEmptyLayers = new QToolButton(this, "removeemptylayers");
    bRemoveEmptyLayers->setPixmap(QPixmap(removeemptylayers_xpm));
    QToolTip::add(bRemoveEmptyLayers, tr("Remove all empty Layers"));
    connect(bRemoveEmptyLayers, SIGNAL(clicked()), this, SLOT(removeEmptyLayers()));
  }
  else {
    bAddLayer=0;
    bRemoveLayer=0;
    bRenameLayer=0;
    bRemoveEmptyLayers=0;
  }

  //fSep1 = new QFrame(this, "sep1");
  //fSep1->setFrameStyle(HLine|Sunken);
                             
  // Create the Layer-List:
  //
  layerList = new QListBox(this, "layerlist");
  //layerList->setAutoUpdate(true);
  layerList->setDragSelect(false);
  layerList->setMultiSelection(false);
  layerList->setSmoothScrolling(true);
  QToolTip::add(layerList, tr("Click = select a Layer\nDouble-click = show/ hide a layer"));

  // Pixmaps for Layerlist (Eyes)
  //
  pxmVisible = new QPixmap(visiblelayer_xpm);
  pxmHidden  = new QPixmap(hiddenlayer_xpm);

  // Connect signal "selected" with slot "switchLayer"
  //   Whenever a layer was selected the function
  //   "switchLayer" gets called 
  //
  connect(layerList, SIGNAL(selected(int)),
          this     , SLOT(switchLayer(int)));

  // Connect signal "highlight" with slot "selectLayer"
  //   Whenever a layer was highlighted the function
  //   "selectLayer" gets called 
  //
  connect(layerList, SIGNAL(highlighted(int)),
          this     , SLOT(selectLayer(int)));
}



// Destructor:
//
RLayerWidget::~RLayerWidget()
{

}



// Show layer widget and all widgets in it:
//
void 
RLayerWidget::show()
{
  RWidget::show();
}



// Resize event:
//
void 
RLayerWidget::resizeEvent(QResizeEvent* _ev)
{
  RWidget::resizeEvent(_ev);
  resizeChildren();
}



// Resize the children:
//
void
RLayerWidget::resizeChildren()
{
  if(lLabel) {
    lLabel->setGeometry( 1, 1,
                         width()-2, DEF_BSIZE0);
  }

  if(layerList) {
    layerList->setGeometry(0, DEF_BSIZE0*2+4,
                           width()-2, height()-DEF_BSIZE0*2-4);
  }

  int lx=2;

  if(bShowLayers) {
    bShowLayers->setGeometry(lx, DEF_BSIZE0+2,
                             DEF_BSIZE0, DEF_BSIZE0);
    lx+=DEF_BSIZE0;
  }

  if(bHideLayers) {
    bHideLayers->setGeometry(lx, DEF_BSIZE0+2,
                             DEF_BSIZE0, DEF_BSIZE0);
    lx+=DEF_BSIZE0;
  }

  if(bAddLayer) {
    bAddLayer->setGeometry(lx, DEF_BSIZE0+2,
                           DEF_BSIZE0, DEF_BSIZE0);
    lx+=DEF_BSIZE0;
  }

  if(bRemoveLayer) {
    bRemoveLayer->setGeometry(lx, DEF_BSIZE0+2,
                              DEF_BSIZE0, DEF_BSIZE0);
    lx+=DEF_BSIZE0;
  }

  if(bRenameLayer) {
    bRenameLayer->setGeometry(lx, DEF_BSIZE0+2,
                              DEF_BSIZE0, DEF_BSIZE0);
    lx+=DEF_BSIZE0;
  }

  if(bRemoveEmptyLayers) {
    bRemoveEmptyLayers->setGeometry(lx, DEF_BSIZE0+2,
                                    DEF_BSIZE0, DEF_BSIZE0);
  }

}


// Give a size hint:
//
QSize 
RLayerWidget::sizeHint() const
{
  int listWidth = layerList->sizeHint().width()+3;
  if(listWidth<DEF_MINLAYERLISTWIDTH) listWidth=DEF_MINLAYERLISTWIDTH;
  if(listWidth>DEF_MAXLAYERLISTWIDTH) listWidth=DEF_MAXLAYERLISTWIDTH;
  QSize ret(listWidth+DEF_BSIZE0+3, 
            layerList->sizeHint().height()+6);
  return ret;
}



// Set the pointer to a graphic and update the layers:
//
void 
RLayerWidget::setGraphic(RGraphic* _graphic) 
{ 
  graphic=_graphic; 
  updateLayerList();
}



// Update the layerlist:
//
void 
RLayerWidget::updateLayerList()
{
  if(layerList) {

    QString selStr;           // selected text
    int selInd;               // selected index
    RListBoxItem* selItem=0;  // Pointer to new selected item
    selInd = layerList->currentItem();
    if( selInd>=0 ) selStr = layerList->text( selInd );
    else            selStr = "default";

    layerList->clear();
    
    if(graphic) {
      int     i;
      RLayer* lay;           // Pointer to a layer in the graphic
      RListBoxItem* newItem; // Temp pointer to create a new item
      
      RLOG("\nupdateLayerList");
      RLOG("\nNumber of layers: ");
      RLOG(graphic->countLayers());
  
      //layerList->setAutoUpdate(false);
  
      for(i=0; i<graphic->countLayers(); ++i) {
        lay = graphic->getLayer(i);
        
        if(lay->getFlag(Y_USED) &&
           lay->getName()) {
           
          if(lay->getFlag(Y_VISIBLE)) {
            newItem = new RListBoxItem(lay->getName(), *pxmVisible);
          }
          else {
            newItem = new RListBoxItem(lay->getName(), *pxmHidden);
          }

          layerList->inSort(newItem);

          if( lay->getName()==selStr ) {
            selItem = newItem;
          }
	  
     	    RLOG("\nRLayerWidget::updateLayerList: Layer inserted: ");
      	  RLOG(lay->getName());
        }
      }
      //layerList->setAutoUpdate(true);
    }
    layerList->repaint();

    if( selItem ) layerList->setCurrentItem( selItem );
  }
}



// Switch a layer on / off:
//
void
RLayerWidget::switchLayer(int _index) 
{ 
  if(_index>=0) {
    if(layerList && graphic) {
    
      emit layerSwitched(layerList->text(_index)); 
      
      RListBoxItem* newItem;
      if(graphic->isLayerVisible(layerList->text(_index))) {
        newItem = new RListBoxItem(layerList->text(_index), *pxmVisible);
      }
      else {
        newItem = new RListBoxItem(layerList->text(_index), *pxmHidden);
      }
      layerList->changeItem(newItem, _index);

      graphic->repaint();
    }
  }
}



// Select a layer by index:
//
void
RLayerWidget::selectLayer(int _index) 
{ 
  if(_index>=0) {
    if(layerList && graphic) {
      emit layerSelected( layerList->text(_index) ); 
    }
  }
}



// Select a layer by name:
//
void
RLayerWidget::selectLayer(const QString& _name) 
{ 
  if(_name) {
    if(layerList && graphic) {
      int num=0;
      for(int i=0; i<(int)layerList->count(); ++i) {
        if( layerList->text(i) == _name ) {
          num=i;
          break;
        }
      }
      layerList->setCurrentItem(num);
      //selectLayer(num);
    }
  }
}



// Add a Layer:
//
void
RLayerWidget::addLayer()
{
  if(graphic) {

    // New dialog for new layer:
    //
    RLayer lay;                                // Layer for parameter livery
    lay.setColor( graphic->getCurrentColor() );
    lay.setWidth( graphic->getCurrentWidth() );
    lay.setStyle( graphic->getCurrentStyle() );

    RLayerDialog dlg(this, "xxx", &lay);
    dlg.setCaption( tr("Add layer") );
    if(dlg.exec()) {
      graphic->addLayer( dlg.getName(), dlg.getColor(), dlg.getWidth(), dlg.getStyle() );
      selectLayer( dlg.getName() );
      updateLayerList();
    }
  }
}




// Remove a Layer:
//
void
RLayerWidget::removeLayer()
{
  QString remLayer;
  remLayer = layerList->text( layerList->currentItem() );

  if(graphic && layerList && remLayer) {
    int onLayer = graphic->countElementsOnLayer(remLayer);
    QString mes;

    if(onLayer==1) {
      // There is one object\non Layer 'nnn'.\n\nDo you really want to delete it?
      //
      mes = QString("%1 %2 %3")
                   .arg(tr("There is one object\non Layer '"))
		   .arg(remLayer)
		   .arg(tr("'.\n\nDo you really want to delete it?"));
    }
    else {
      // There are %d objects\non Layer '%s'.\n\nDo you really want to delete them?
      //
      mes = QString("%1 %2 %3%4%5")
                   .arg(tr("There are"))
		   .arg(onLayer)
		   .arg(tr("objects\non Layer '"))
		   .arg(remLayer)
		   .arg(tr("'.\n\nDo you really want to delete them?"));
    }
  
    RYesNoDialog ydlg(mes,
                      tr("Yes"),
                      tr("No"),
                      this, 
                      "ydlg");
                      
    if(onLayer==0 || ydlg.exec()) {
      graphic->removeLayer(remLayer, true);
      graphic->repaint();
    }
  }
}



// Rename a Layer:
//
void
RLayerWidget::renameLayer()
{
  /*
  // Old dialog for renaming:

  const QString& renLayer;
  renLayer = layerList->text(layerList->currentItem()).latin1();

  if(graphic && layerList && renLayer) {
    char mes[256];

    sprintf(mes, 
            "%s '%s':",
            tr("Type a new name for\nLayer"), renLayer);

    RInputDialog idlg(mes, 255, renLayer, this, "idlg");
    if(idlg.exec()) {
      const QString& newLayer = idlg.getInput();
  
      graphic->renameLayer(renLayer, newLayer);
    }
  }
  */

  // New dialog for renaming and attributs:
  //
  if(graphic && layerList) {
    RLayer *lay;
    lay=graphic->layerCurrent();

    if(lay) {
      RLayerDialog dlg(this, "xxx", lay);
      dlg.setCaption( tr("Change layer attributes") );
      if(dlg.exec()) {
        lay->setColor(dlg.getColor());
        lay->setWidth(dlg.getWidth());
        lay->setStyle(dlg.getStyle());
        graphic->renameLayer(lay->getName(), dlg.getName());
      }
    }
  }
}



// Show all Layers:
//
void
RLayerWidget::showLayers()
{
  showHideAllLayers(true);
}



// Hide all Layers:
//
void
RLayerWidget::hideLayers()
{
  showHideAllLayers(false);
}



// Show (_show=true) or hide (_show=false) all Layers:
//
void 
RLayerWidget::showHideAllLayers(bool _show)
{
  if(layerList && graphic) {
    int i;                  // counter
    bool found=false;       // found layer to switch
    RListBoxItem* newItem;  // pointer for creation of new items
    QString itemText;       // text of current item
    
    for(i=0; i<(int)layerList->count(); ++i) {
      itemText=layerList->text(i);
      if( graphic->isLayerVisible( itemText )!=_show ) {
        found=true;
        emit layerSwitched( itemText ); 
        newItem = new RListBoxItem(itemText, 
                                   (_show ? *pxmVisible : *pxmHidden));
        layerList->changeItem(newItem, i);
      }
    }
    if(found) graphic->repaint();
  }
}



// Remove all layers without elements on it:
//
void 
RLayerWidget::removeEmptyLayers()
{
  if(graphic) {
    graphic->removeEmptyLayers();
  }
}



// EOF





