/***************************************************************************
                          relement.h  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/

/****************************************************************************
** relement.h 1998/08/24 A. Mustun RibbonSoft 
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#ifndef RELEMENT_H
#define RELEMENT_H

#include <qcolor.h>
#include <qstring.h>
#include <qnamespace.h>

#include "rflags.h"
#include "rgraphdef.h"
#include "rtypes.h"
#include "rprgdef.h"

#define E_VISIBLE      0x00000001       // Element is visible
#define E_ACTIVE       0x00000002       //            active (not in Undo memory)
#define E_REVERSED     0x00000004       // Arc is reversed (clockwise; G02)
#define E_TOOLOFFL     0x00000008       // Tool offset Left on
#define E_TOOLOFFR     0x00000010       // Tool offset Right on
#define E_TOOLOFFO     0x00000020       // Tool offset Off
#define E_CONV         0x00000040       // Element is already converted
#define E_CONTBEG      0x00000080       // Element is at contour beginning
#define E_CONTEND      0x00000100       // Element is at contour end
#define E_TANGENTIAL   0x00000200       // Startway is tangential
#define E_BROKEN       0x00000400       // Element is highlighted (broken contour)
#define E_BROKEN1      0x00000800       // Elements startpoint is highlighted (broken contour)
#define E_BROKEN2      0x00001000       // Elements endpoint   is highlighted (broken contour)
#define E_STRAIGHT     0x00002000       // Straight Text
#define E_ROUNDIN      0x00004000       // Round Text readable from center / dimension
#define E_ROUNDOUT     0x00008000       // Round Text readable from outside / dimension
#define E_LEFT         0x00010000       // Text alignment: left
#define E_RIGHT        0x00020000       // Text alignment: right
#define E_CENTER       0x00040000       // Text alignment: centered
#define E_TAGGED       0x00080000       // Element tagged
#define E_TAGSTP       0x00100000       // Elements startpoint tagged (stretch)
#define E_TAGEND       0x00200000       // Elements endpoint tagged (stretch)
#define E_UNDO         0x00400000       // Element is in undo memory
#define E_REDO         0x00800000       // Element is in redo memory
#define E_HORIZONTAL   0x01000000       // Horizontal dimension ! collision: E_MIDDLE
#define E_VERTICAL     0x02000000       // Vertical dimension ! collision: E_BOTTOM
#define E_DIAMETER     0x04000000       // Diameter dimension
#define E_RADIUS       0x08000000       // Radius dimension
#define E_ARROW        0x10000000       // Arrow dimension
#define E_FIXEDWIDTH   0x20000000       // Fixed width for texts
#define E_PROPORTIONAL 0x40000000       // Proportional flag for texts
#define E_TOP          0x80000000       // Top attachement flag for texts
#define E_MIDDLE       0x01000000       // Middle attachement flag for texts ! collision:E_HORIZONTAL
#define E_BOTTOM       0x02000000       // Bottom attachement flag for texts ! collision:E_VERTICAL


enum elementEnum { // Collectives (Groups):
                   T_NO, T_ALL, T_CIRCULAR, T_STRAIGHT, T_LINE_ARC, T_LONG,
                   
                   // Singles:
                   T_POINT, T_LINE, T_ARC, T_CIRCLE, T_ELLIPSE,
                   T_TEXT, T_RECT, T_RAPID,
                   T_STARTWAY, T_ENDWAY,
                   T_DIMENSION, T_HATCH };

class RGraphic;
class RPainter;

class RElement : public RFlags
{
public:
  RElement(RGraphic* _graphic=0, 
           unsigned int _flags=0);
  ~RElement();

  void  reset();

  elementEnum getElementTyp() const { return elementTyp; }
  void  setElementTyp(elementEnum _typ) { elementTyp=_typ; }

  void  setGraphic(RGraphic* _graphic) { graphic=_graphic; }
  RGraphic* getGraphic() { return graphic; }

  byte  getLayer() { return layer; }
  void  setLayer(byte _lay) { layer=_lay; }
  int   getContour() { return contour; }
  void  setContour(int _cont) { contour=_cont; }
  double getX1() const  { return x1; }
  void  setX1(double _x1)  { x1=_x1; }
  double getY1() const  { return y1; }
  void  setY1(double _y1)  { y1=_y1; }
  double getX2() const  { return x2; }
  void  setX2(double _x2)  { x2=_x2; }
  double getY2() const  { return y2; }
  void  setY2(double _y2)  { y2=_y2; }
  double getCx() const  { return cx; }
  void  setCx(double _cx)  { cx=_cx; }
  double getCy() const  { return cy; }
  void  setCy(double _cy)  { cy=_cy; }
  double getCr() const  { return cr; }
  void  setCr(double _cr)  { cr=_cr; }
  double getA1() const  { return a1; }
  void  setA1(double _a1)  { a1=_a1; }
  double getA2() const  { return a2; }
  void  setA2(double _a2)  { a2=_a2; }
  int   getWidth() { return width; }
  void  setWidth(int _width) { width=_width; }
  QColor getColor() { return color; }
  void  setColor(QColor _color) { color=_color; }
  void  setNamedColor(QString _name) { color.setNamedColor(_name); }
  Qt::PenStyle getStyle() { return style; }
  void  setStyle(Qt::PenStyle _style) { style=_style; }
  void  setDecNum( int _n ) { decNum=_n; }
  int   getDecNum() { return decNum; }

  void  setDestructionNumber(byte _unumD) { unumD = _unumD; }
  void  setCreationNumber   (byte _unumC) { unumC = _unumC; }
  byte  getDestructionNumber() { return unumD; }
  byte  getCreationNumber()    { return unumC; }

  QString getText() const { return text; }
  void  setText(const QString& _text) { text = _text; }

  uint  getTextMode() const { if     (getFlag(E_ROUNDOUT)) return E_ROUNDOUT;
                              else if(getFlag(E_ROUNDIN))  return E_ROUNDIN;  
                              else                         return E_STRAIGHT; }
  uint  getTextAlign() const { if     (getFlag(E_CENTER))  return E_CENTER;
                               else if(getFlag(E_RIGHT))   return E_RIGHT;
                               else                        return E_LEFT; }
  uint  getTextVAlign() const { if     (getFlag(E_MIDDLE))  return E_MIDDLE;
                                else if(getFlag(E_TOP))     return E_TOP;
                                else                        return E_BOTTOM; }

  double getLetterSpacing() const { return x2; }
  void  setLetterSpacing(double _ls) { x2=_ls; }

  double getTextHeight() const { return y2; }
  void  setTextHeight(double _th) { y2=_th; }

  double getWordSpacing() const { return cx; }
  void  setWordSpacing(double _ws) { cx=_ws; }

  double getLineDistance() const { return cy; }
  void  setLineDistance(double _ld) { cy=_ld; }

  int   getFont() const { return (int)a2; }
  void  setFont(int _fnt) { a2=(double)_fnt; }
  
  /*! Get Font flags <br> (E_STRAIGHT, E_ROUNDIN, E_ROUNDOUT, E_LEFT, E_RIGHT, E_CENTER)
  */
  uint getFontFlags() { return getFlags()&(E_STRAIGHT|E_ROUNDIN|E_ROUNDOUT|E_LEFT|E_RIGHT|E_CENTER|E_FIXEDWIDTH|E_PROPORTIONAL); }

  bool  getFixedWidth() { return getFlag(E_FIXEDWIDTH); }

  int     getNumberOfTextLines() const;
  QString getTextLine(int _num) const;

  double getDimDist() const  { return cr; }
  void  setDimDist(double _cr)  { cr=_cr; }

  QString getDimText() const;
  void    setDimText(const QString& _text) { text = _text; }

  bool  hasEndpoint() const;
  bool  isCircular() const;
  bool  isInGroup(int _typGroup) const;

  bool  compareWith(const RElement* _element, double _tolerance=DEF_MMTOL, bool _reverse=false);

  void  getDataFrom(const RElement* _element);
  void  getMeasuresFrom(const RElement* _element);
  void  getAttribFrom(const RElement* _element);

  double getAngleAmount() const;
  double getAngleFrom1To2() const;
  double getLength() const;
  
  double getCenterX() const;
  double getCenterY() const;
  double getMiddleX() const;
  double getMiddleY() const;
  double getMiddleDirection() const;
  double getDirection1() const;
  double getDirection2() const;
  void  getDistPoint(double* _resx, double* _resy, double _dist, bool _startPoint) const;
  void  getTouchPoint(double* _resx, double* _resy, double _mx, double _my) const;

  double getDistanceToPoint(double _px, double _py, bool _noLimit=false) const;
  double getDistanceFromPointToPoint(double _px, double _py) const;
  double getDistanceFromLineToPoint(double _px, double _py, bool _noLimit=false) const;
  double getDistanceFromArcToPoint(double _px, double _py, bool _noLimit=false) const;
  double getDistanceFromCircleToPoint(double _px, double _py) const;
  double getDistanceFromTextToPoint(double _px, double _py) const;

  double getRelAngleToPoint(double _px, double _py, bool _startPoint) const;
  
  /*! Get Intersection between this element and an other one:
      \param _el    The other element
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
      \param _onElements true: The intersections must be on both elements
                         false: The intersections can be outside the elements 
  */
  void  getIntersection(const RElement* _el, 
                        bool* _ires1, double* _ix1, double* _iy1, 
                        bool* _ires2, double* _ix2, double* _iy2,
                        bool* _itan,
                        bool _onElements=true ) const;
                        
  /*! Get Intersection between this line and an other one:
      \param _el    The other line
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the intersection is retruned in this variable
  */
  void  getIntersectionLineWithLine(const RElement* _el, 
                                    bool* _ires1, double* _ix1, double* _iy1) const;
                                    
  /*! Get Intersection between this line and an arc:
      \param _el    The arc
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
  */
  void  getIntersectionLineWithArc(const RElement* _el, 
                                   bool* _ires1, double* _ix1, double* _iy1,
                                   bool* _ires2, double* _ix2, double* _iy2,
                                   bool* _itan) const;
                                   
  /*! Get Intersection between this arc and an other one:
      \param _el    The other arc
      \param _ires1 true is in this variable returned if there is an intersection
      \param _ix1   X-Coordinate of the 1st intersection is retruned in this variable
      \param _iy1   Y-Coordinate of the 1st intersection is retruned in this variable
      \param _ires2 true is in this variable returned if there is a 2nd intersection (circles)
      \param _ix2   X-Coordinate of the 2nd intersection is retruned in this variable
      \param _iy2   Y-Coordinate of the 2nd intersection is retruned in this variable
      \param _itan  true is in this variable returned if the intersection is a tangent point
  */
  void  getIntersectionArcWithArc(const RElement* _el, 
                                  bool* _ires1, double* _ix1, double* _iy1,
                                  bool* _ires2, double* _ix2, double* _iy2,
                                  bool* _itan) const;
                                  
  /*! Get Gliding ("The point somewhere between the elements") between this element and an other:
      \param _el   The other element
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGliding(const RElement* _el, 
                   bool* _gres, double* _gx, double* _gy) const;
                   
  /*! Get Gliding between this line and an arc:
      \param _el   The arc
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGlidingLineWithArc(const RElement* _el, 
                              bool* _gres, double* _gx, double* _gy) const;
                              
  /*! Get Gliding between this arc and an other one:
      \param _el   The other arc
      \param _gres true is in this variable returned if there is a gliding
      \param _gx   X-Coordinate of the 1st gliding is retruned in this variable
      \param _gy   Y-Coordinate of the 1st gliding is retruned in this variable
  */
  void  getGlidingArcWithArc(const RElement* _el, 
                             bool* _gres, double* _gx, double* _gy) const;
                                  
  /*! Get all intersections (<=8) of all paralles (<=4) of the two elements (this, _el)
      \param _el The other element
      \param _dist Distance to parallels
      \param _ires Pointer to Array of eight bools (for returning if intersection n is valid)
      \param _ipx  Pointer to Array of eight doubles (for returning the x-values of the intersection points)
      \param _ipy  Pointer to Array of eight doubles (for returning the y-values of the intersection points)
  */
  void  getParallelIntersections(const RElement* _el, 
                                 double _dist,
                                 bool* _ires, double* _ipx, double* _ipy) const;
                                 
                        
  /*! Get the point which we must trim if the user clicked an element for trimming
      \param _mx Mouse coordinate x
      \param _my Mouse coordinate y
      \param _ix Intersection point x
      \param _iy Intersection point y
      \return true: trim point is endpoint
              false: trim point is startpoint
  */
  bool  getTrimPoint(double _mx, double _my, double _ix, double _iy);
  
  /*! Trim an element to a point
      \param _ep Trim endpoint?
      \param _ix1/_iy1 Trim to this point
      \param _ix2/_iy2 Trim circles also to this point
      \param _mang Angle from circle center to mouse click
  */
  void trim(bool _ep,
            double _ix1, double _iy1,
            double _ix2, double _iy2,
            double _mang);

  /*! Trim an element by a given amount
      \param _ep Trim endpoint?
      \param _amount Trim amount
  */
  void trim(bool _ep,
            double _amount);

  
  /*! Check if element was trimmed to zero
  */
  bool  isTrimmedToZero(bool _ep, double _ix1, double _iy1);
  bool  isTrimmedToZero(double _amount);
                                  
  bool  isPointOnElement(double _x, double _y, double _tol=DEF_MMTOL) const;
  bool  isPointInsideSubElements(double _x, double _y) const;
  bool  isParallelWith(const RElement* _el) const;

  bool  getParallel(RElement* _parallel, 
                    double _px, double _py, 
                    double _dist) const;
  void  getParallels(RElement* _parallel1, 
                     RElement* _parallel2, 
                     double _dist) const;

  bool  getBisector(RElement* _bisector, 
                    const RElement* _leg, 
                    double _px, double _py,
                    double _length) const;
  void  getBisectors(RElement* _bisector1, 
                     RElement* _bisector2, 
                     RElement* _bisector3, 
                     RElement* _bisector4, 
                     const RElement* _leg, 
                     double _ix, double _iy,
                     double _length         ) const;

  bool getTangent1(RElement* _tangent,
                   double _x, double _y,
                   double _px, double _py) const;
  bool getTangents1(RElement* _tangent1,
                    RElement* _tangent2,
                    double _x, double _y) const;

  bool getTangent2(RElement* _tangent,
                   const RElement* _secondCircle,
                   double _px, double _py) const;
  bool getTangents2(RElement* _tangent1,
                    RElement* _tangent2,
                    RElement* _tangent3,
                    RElement* _tangent4,
                    const RElement* _secondCircle) const;

  bool  getOrtho(RElement* _ortho,
                 double _px, double _py,
                 double _ang, double _length) const;

  void  forceDirection(bool _pos);

  void  recalculateBorders();
  void  recalculateTextBorders();
  void  recalculateAngles();
  void  recalculateEndpoints();
  double left() const   { return leftBorder; }
  double bottom() const { return bottomBorder; }
  double right() const  { return rightBorder; }
  double top() const    { return topBorder; }
  void  getBorders(double* _left, double* _bottom, double* _right, double* _top) const;

  bool  isOnScreen() const;

  void  createPoint(double _x1, double _y1, byte _layer=0);
  void  createLine(double _x1, double _y1, 
                   double _x2, double _y2, byte _layer=0);
  void  createArc(double _cx, double _cy, 
                  double _cr, 
                  double _a1, double _a2, 
                  bool _reversed, byte _layer=0);
  void  createArcPPP(double _x1, double _y1,
                     double _x2, double _y2,
                     double _x3, double _y3,
                     bool _circle=false,
                     byte _layer=0);
  void  createCircle(double _cx, double _cy, 
                     double _cr, 
                     double _a1, double _a2, 
                     bool _reversed, byte _layer=0);
  void  createPolylineElement(double _x1, double _y1,
                              double _x2, double _y2,
                              double _ab, byte _layer=0);
  void  createRect(double _x1, double _y1, 
                   double _x2, double _y2, byte _layer=0);
  void  createText(double _x, double _y,
                   int   _font,
                   const QString& _text,
                   unsigned int _flags,
                   double _textheight,
                   double _angle=0.0,
                   double _radius=0.0,
                   double _letterspace=DEF_AREAMAX,
                   double _wordspace=DEF_AREAMAX,
                   double _linedistance=DEF_AREAMAX,
                   byte _layer=0);
  void  createRapid(double _x1, double _y1, byte _layer=0);
  void  createRapid(double _x1, double _y1,
                    double _x2, double _y2, 
                    byte _layer=0);
  void  createDimension(double _v1, double _v2, double _v3, double _v4,
                        double _v5, double _v6,
                        double _pos,
                        uint _flags=0,
                        byte _layer=0);
  void  createHatching(double _factor, QString _pattern,
                        byte _layer=0);

  void  turn();
  
  /*! Draw the element
      \param _paint Pointer to RPainter drawing device
      \param _ox    Real offset in X
      \param _oy    Real offset in Y
      \param _prev  Element is in preview (no texts, no changing of color if tagged)
      \param _curr  Explicit use of current settings of painting device (no changing of color or mode)
  */
  void  draw(RPainter* _paint, 
             double _ox=0.0, 
             double _oy=0.0, 
             bool _prev=false,
             bool _curr=false);
  void  fastDraw(RPainter* _paint);
  void  drawSmooth(RPainter* _paint);

  void  drawSmoothLine(RPainter* _paint);
  void  drawSmoothArc(RPainter* _paint);

  void  move(double _x, double _y, bool _updBorder=true);
  void  rotate(double _angle, double _cx, double _cy, bool _updBorder=true);
  void  scale(double _fact, double _cx, double _cy, bool _updBorder=true);
  void  mirror(double _x1, double _y1, double _x2, double _y2, bool _updBorder=true);
  void  moveStartPoint(double _x, double _y, bool _updBorder=true);
  void  moveEndPoint(double _x, double _y, bool _updBorder=true);

  void  tag()   { setFlag(E_TAGGED); }
  void  untag() { delFlag(E_TAGGED); }

  void  createSubElements(int _num);
  void  deleteSubElements();
  void  setSubElement(int _i, RElement* _el);
  RElement* getSubElement(int _i);
  int   getSubElementNum() { return subElementNum; }

  void  outputDebug();
  
protected:
  void  prepareDrawing(RPainter* _paint, int _x, int _y, bool _curr=false);
  void  finishDrawing(RPainter* _paint, int _x, int _y);

    
private:

  elementEnum  elementTyp;        // Element typ:
                                  //   T_POINT, T_LINE, 
                                  //   T_ARC, T_CIRCLE
                                  //   T_TEXT
  RGraphic*    graphic;           // Pointer to graphic
  double       leftBorder,        // borders
               topBorder,         // 
               rightBorder,       //
               bottomBorder;      // 
  byte         layer;             // Layer number (0=no layer)
  int          contour;           // Contour number (-1=no contour)
  byte         unumD,             // UNDO-Number when element was deleted
               unumC;             // UNDO-Number when element was constructed
  double       x1,                // Start point || Reference point for text || Start radius for angle dimension extension line 1
               y1,                //             ||
               x2,                // End point   || Letter space for text || Start radius for angle dimension extension line 2
               y2,                //             || Textheight   for text
               cx,                // Center      || Word space   for text
               cy,                //             || Line dist.   for text
               cr,                // Radius      || Rad. 1st line for text || Pos. for dimension line || Factor for hatching
               cr2,               // Radius 2 (for ellipses)
               a1,                // Start angle || Ref. Angle   for text
               a2,                // End angle   || Font         for text
               ang;               // Base angle for ellipses
  byte         width;             // Line width
  QColor       color;             // Line color
  Qt::PenStyle style;             // Line Style (NoPen, SolidLine, DashLine, DotLine, DashDotLine, DashDotDotLine)
  QString      text;              // text || Alternate text for dimensions || Pattern for hatching
  RElement*    subElement;        // Pointer to array of elements (Border elements for hatchings)
  int          subElementNum;     // Number of sub elements
  int          decNum;            // deceleration number (reduced feedrate 1: reduce a few, 2: reduce more)

};


#endif

// EOF






