import unittest
from openbsd.utils import *

class uUtility(unittest.TestCase):
    def test_multiord(self):
        self.failUnlessEqual(multiord("a"), 0x61)
        self.failUnlessEqual(multiord(""), 0)
        self.failUnlessEqual(multiord("aa"), 24929)

    def test_multichar(self):
        self.failUnlessEqual(multichar(0x61, 1), "\x61")
        self.failUnlessEqual(multichar(0x61, 2), "\x00\x61")
        self.failUnlessEqual(multichar(257, 2), "\x01\x01")
        self.failUnlessEqual(multichar(257, 4), "\x00\x00\x01\x01")
        self.failUnlessRaises(ValueError, multichar, 257, 1)

    def test_checksumBasic(self):
        # FIXME: More unit tests here.
        tstr = "".join([chr(i) for i in range(9)])
        self.failUnlessEqual(cksum16(tstr), 60399)

    def test_findLongestSubsequence(self):
        fls = findLongestSubsequence
        self.failUnlessEqual(fls("ffaaaff", "a"), (2, 4))
        self.failUnlessEqual(fls("ffaaaffaaaa", "a"), (7, 10))
        self.failUnlessEqual(fls("ffaaaffaaa", "a"), (2, 4))
        self.failUnlessEqual(fls("aaaffaaa", "a"), (0, 2))
        self.failUnlessEqual(fls("aaa", "a"), (0, 2))


class uIPAddresses(unittest.TestCase):
    def test_ipToBytes(self):
        self.failUnlessEqual(   ipToBytes("255.255.0.1"),
                                "\xff\xff\x00\x01")
        self.failUnlessEqual(   ipToBytes("0.0.0.0"),
                                "\x00\x00\x00\x00")

    def test_ipToBytesErr(self):
        self.failUnlessRaises(ValueError, ipToBytes, "256.255.0.1")
        self.failUnlessRaises(ValueError, ipToBytes, "af.255.0.1")
        self.failUnlessRaises(ValueError, ipToBytes, "255.0.1")
        self.failUnlessRaises(ValueError, ipToBytes, "255.0.1.1.1")

    def test_ipToStr(self):
        self.failUnlessEqual(   ipToStr("\xff\xff\x00\x01"),
                                "255.255.0.1")
        self.failUnlessEqual(   ipToStr("\x00\x00\x00\x00"),
                                "0.0.0.0")

    def test_ipToStrErr(self):
        self.failUnlessRaises(  ValueError,
                                ipToStr,
                                "\xff\xff\xff\x00\x01")
        self.failUnlessRaises(  ValueError,
                                ipToStr,
                                "\x00\x00\x00")

    def test_ipFromPrefix(self):
        self.failUnlessEqual(ipFromPrefix(0), "0.0.0.0")
        self.failUnlessEqual(ipFromPrefix(8), "255.0.0.0")
        self.failUnlessEqual(ipFromPrefix(32), "255.255.255.255")
        self.failUnlessEqual(ipFromPrefix(9), "255.128.0.0")
        self.failUnlessEqual(ipFromPrefix(12), "255.240.0.0")

    def test_ipFromPrefixErr(self):
        self.failUnlessRaises(ValueError, ipFromPrefix, -1)
        self.failUnlessRaises(ValueError, ipFromPrefix, 33)

    def test_isIPAddr(self):
        self.failUnless(isIPAddr("10.0.0.0"))
        self.failIf(isIPAddr("ff::"))


class uIP6Addresses(unittest.TestCase):
    def test_ip6ToStr(self):
        self.failUnlessEqual(ip6ToStr("\xff"*16), ":".join(["ffff"]*8))
        self.failUnlessEqual(ip6ToStr("\x00"*15+"\x01"), "::1")
        self.failUnlessEqual(ip6ToStr("\x01\x02" + "\x00"*13+"\x01"), "102::1")

    def test_ip6ToStrErr(self):
        self.failUnlessRaises(ValueError, ip6ToStr, "\xff\xff\xff\x00\x01")

    def test_ipToBytes(self):
        self.failUnlessEqual(ip6ToBytes(":".join(["ffff"]*8)), "\xff"*16)
        self.failUnlessEqual(ip6ToBytes("::1"), "\x00"*15+"\x01")
        self.failUnlessEqual(ip6ToBytes("102::1"), "\x01\x02" + "\x00"*13+"\x01")

    def test_ip6ToBytesErr(self):
        self.failUnlessRaises(ValueError, ip6ToBytes, "0::0::1")
        self.failUnlessRaises(ValueError, ip6ToBytes, "1:0:0:1:1:1:1:1:1")

    def test_Roundtrips(self):
        addrs = [
            "0:9:6b:e0:ca:ce:0:1",
            "fe80::1"
        ]
        for i in addrs:
            self.failUnlessEqual(i, ip6ToStr(ip6ToBytes(i)))

    def test_ip6FromPrefix(self):
        self.failUnlessEqual(ip6FromPrefix(0), "::")
        self.failUnlessEqual(ip6FromPrefix(128),
                                "ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff")
        self.failUnlessEqual(ip6FromPrefix(8), "ff00::")
        self.failUnlessEqual(ip6FromPrefix(32), "ffff:ffff::")
        self.failUnlessEqual(ip6FromPrefix(9), "ff80::")
        self.failUnlessEqual(ip6FromPrefix(12), "fff0::")

    def test_ip6FromPrefixErr(self):
        self.failUnlessRaises(ValueError, ip6FromPrefix, -1)
        self.failUnlessRaises(ValueError, ip6FromPrefix, 129)

    def test_isIP6Address(self):
        self.failIf(isIP6Addr("10.0.0.0"))
        self.failUnless(isIP6Addr("ff::"))


class uEthAddresses(unittest.TestCase):
    def test_ethToStr(self):
        self.failUnlessEqual(ethToStr("\xff"*6), ":".join(["ff"]*6))
        self.failUnlessEqual(ethToStr("\x0f"*6), ":".join(["0f"]*6))
        self.failUnlessEqual(ethToStr("\x00"*6), ":".join(["00"]*6))

    def test_ethToStrErr(self):
        self.failUnlessRaises(ValueError, ethToStr, "\xff"*5)
        self.failUnlessRaises(ValueError, ethToStr, "\xff"*8)

    def test_ethToBytes(self):
        self.failUnlessEqual(ethToBytes(":".join(["ff"]*6)), "\xff"*6)
        self.failUnlessEqual(ethToBytes(":".join(["0f"]*6)), "\x0f"*6)
        self.failUnlessEqual(ethToBytes(":".join(["00"]*6)), "\x00"*6)

    def test_ethToBytesErr(self):
        self.failUnlessRaises(ValueError, ethToBytes, ":".join(["00"]*5))
        self.failUnlessRaises(ValueError, ethToBytes, ":".join(["00"]*7))

class uDoubleAssociation(unittest.TestCase):
    def setUp(self):
        self.d = DoubleAssociation()

    def test_setitem(self):
        self.d["one"] = 1
        self.failUnlessEqual(self.d["one"], 1)
        self.failUnlessEqual(self.d[1], "one")

    def test_initialisation(self):
        d = DoubleAssociation({"one": 1})
        self.failUnlessEqual(d["one"], 1)
        self.failUnlessEqual(d[1], "one")



