import os, unittest
import openbsd.ifconfig
import openbsd._ifconfig

MEDIAIF = "fxp0"
MEDIATYPE = "Ethernet"
MEDIASUBTYPE = "autoselect"
# Interface for testing creation and destruction. Should not exist.
CDIFNAME = "tun6"       


class uIfconfig(unittest.TestCase):
    def setUp(self):
        self.ic = openbsd.ifconfig.IFConfig()

    def test_has_key(self):
        self.failUnless(self.ic.has_key("lo0"))

    def test_getitem(self):
        self.failUnlessEqual(self.ic["lo0"].Name, "lo0")

    def test_keys(self):
        self.failUnless(self.ic.keys())

    def test_repr(self):
        repr(self.ic)


class uInterface(unittest.TestCase):
    def setUp(self):
        self.ic = openbsd.ifconfig.IFConfig()

    def test_getinfo(self):
        self.failUnless(self.ic["lo0"]._getinfo())



class uInterfaceCreateDestroy(unittest.TestCase):
    def setUp(self):
        self.ic = openbsd.ifconfig.IFConfig()

    def tearDown(self):
        try:
            self.ic.destroy(CDIFNAME)
        except openbsd.ifconfig.IfConfigError:
            pass

    if os.geteuid() == 0:
        def test_createdestroy(self):
            self.failIf(self.ic.has_key(CDIFNAME))
            self.ic.create(CDIFNAME)
            self.failUnless(self.ic.has_key(CDIFNAME))
            self.ic.destroy(CDIFNAME)
            self.failIf(self.ic.has_key(CDIFNAME))

        def test_addAddress(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("10.0.0.1", "255.0.0.0")
            ifadd = self.ic[CDIFNAME].addresses[0]["address"]["address"]
            self.failUnlessEqual(ifadd, "10.0.0.1")
            ifmask = self.ic[CDIFNAME].addresses[0]["netmask"]["address"]
            self.failUnlessEqual(ifmask, "255.0.0.0")
           
        def test_addAddressPrefix(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("10.0.0.1", 8)
            ifadd = self.ic[CDIFNAME].addresses[0]["address"]["address"]
            self.failUnlessEqual(ifadd, "10.0.0.1")
            ifmask = self.ic[CDIFNAME].addresses[0]["netmask"]["address"]
            self.failUnlessEqual(ifmask, "255.0.0.0")

        def test_addAddressErr(self):
            self.ic.create(CDIFNAME)
            self.failUnlessRaises(  ValueError,
                                    self.ic[CDIFNAME].addAddress,
                                    "10.0.0.1", 55)
            self.failUnlessRaises(  ValueError,
                                    self.ic[CDIFNAME].addAddress,
                                    "purple", 55)

        def test_addAddress6(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("fe::1")
            ifadd = self.ic[CDIFNAME].addresses[0]["address"]["address"]
            self.failUnlessEqual(ifadd, "fe::1")
            ifadd = self.ic[CDIFNAME].addresses[0]["netmask"]["address"]
            self.failUnlessEqual(ifadd, "ffff:ffff:ffff:ffff::")

        def test_addAddress6Prefix(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("fe::1", 64)
            ifadd = self.ic[CDIFNAME].addresses[0]["address"]["address"]
            self.failUnlessEqual(ifadd, "fe::1")
            ifadd = self.ic[CDIFNAME].addresses[0]["netmask"]["address"]
            self.failUnlessEqual(ifadd, "ffff:ffff:ffff:ffff::")

        def test_addAddress6Err(self):
            self.ic.create(CDIFNAME)
            self.failUnlessRaises(  ValueError,
                                    self.ic[CDIFNAME].addAddress,
                                    "ff::", 150)

        def test_delAddress(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("10.0.0.1", "255.0.0.0")
            self.ic[CDIFNAME].delAddress("10.0.0.1")
            self.failIf(self.ic[CDIFNAME].addresses)

        def test_delAddressErr(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("10.0.0.1", "255.0.0.0")
            self.failUnlessRaises(  openbsd.ifconfig.IfConfigError,
                                    self.ic[CDIFNAME].delAddress,
                                    "10.0.0.2")

        def test_delAddress6(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("fe::1")
            self.ic[CDIFNAME].delAddress("fe::1")
            self.failIf(self.ic[CDIFNAME].addresses)

        def test_delAddressErr6(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].addAddress("fe::1")
            self.failUnlessRaises(  openbsd.ifconfig.IfConfigError,
                                    self.ic[CDIFNAME].delAddress,
                                    "fe::2")

        def test_setAddress(self):
            self.ic.create(CDIFNAME)
            self.ic[CDIFNAME].setAddress("10.0.0.1")
            self.ic[CDIFNAME].setAddress("10.0.0.2")
            self.failUnlessEqual(len(self.ic[CDIFNAME].addresses), 1)
            add = self.ic[CDIFNAME].addresses[0]["address"]["address"]
            self.failUnlessEqual(add, "10.0.0.2")
            self.ic[CDIFNAME].setAddress("fe::01")
            self.failUnlessEqual(len(self.ic[CDIFNAME].addresses), 2)
            self.ic[CDIFNAME].setAddress("fe::02")
            self.failUnlessEqual(len(self.ic[CDIFNAME].addresses), 2)


class uMedia(unittest.TestCase):
    def setUp(self):
        ic = openbsd.ifconfig.IFConfig()
        try:
            self.iface = ic[MEDIAIF]
        except KeyError:
            self.iface = None

    def test_mtype(self):
        if self.iface:
            self.failUnlessEqual(self.iface.media.mtype, MEDIATYPE)

    def test_subtype(self):
        if self.iface:
            self.failUnlessEqual(self.iface.media.subtype, MEDIASUBTYPE)

    def test_getAllSubtypes(self):
        if self.iface:
            self.failUnless(self.iface.media.getAllSubtypes())

    def test_getAllOptions(self):
        if self.iface:
            self.iface.media.getAllOptions(MEDIASUBTYPE)

    def test_repr(self):
        if self.iface:
            repr(self.iface.media)


class uFlags(unittest.TestCase):
    def setUp(self):
        ic = openbsd.ifconfig.IFConfig()
        self.ifp = ic["lo0"]

    def test_get(self):
        self.failUnless(self.ifp.flags)

    if os.geteuid() == 0:
        def test_set(self):
            f = self.ifp.flags
            self.ifp.flags = self.ifp.flags & (~openbsd.ifconfig.IFF_UP)
            self.failIf(self.ifp.flags & openbsd.ifconfig.IFF_UP)
            self.ifp.flags = f

    def test_repr(self):
        repr(self.ifp.flags)
        

class uMTU(unittest.TestCase):
    def setUp(self):
        ic = openbsd.ifconfig.IFConfig()
        self.ifp = ic["lo0"]

    def test_get(self):
        self.ifp.mtu

    if os.geteuid() == 0:
        def test_set(self):
            f = self.ifp.mtu
            self.ifp.mtu = 800
            self.ifp.mtu
            self.failUnlessEqual(self.ifp.mtu, 800)
            self.ifp.mtu = f


class uMetric(unittest.TestCase):
    def setUp(self):
        ic = openbsd.ifconfig.IFConfig()
        self.ifp = ic["lo0"]

    def test_get(self):
        self.ifp.metric

    if os.geteuid() == 0:
        def test_set(self):
            f = self.ifp.metric
            self.ifp.metric = 800
            self.failUnlessEqual(self.ifp.metric, 800)
            self.ifp.metric = f
