/*
	Copyright (c) 2003, Nullcube Pty Ltd
	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions are met:

	*   Redistributions of source code must retain the above copyright notice, this
		list of conditions and the following disclaimer.
	*   Redistributions in binary form must reproduce the above copyright notice,
		this list of conditions and the following disclaimer in the documentation
		and/or other materials provided with the distribution.
	*   Neither the name of Nullcube nor the names of its contributors may be used to
		endorse or promote products derived from this software without specific prior
		written permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <stdio.h>
#include <pcap.h>
#include <errno.h>
#include <setjmp.h>
#include <signal.h>
#include <Python.h>
/*
 *  Todo:
 *      - pcap_file, pcap_next?
 */
extern int errno;

static PyObject *PcapError;
jmp_buf JENV;

static PyObject *open_live(PyObject *self, PyObject *args){
    char *device;
    int snaplen, promisc, to_ms;
    char ebuf[PCAP_ERRBUF_SIZE];
    pcap_t *ret;

    if (!PyArg_ParseTuple(args, "siii", &device, &snaplen, &promisc, &to_ms))
        return NULL;

    ret = pcap_open_live(device, snaplen, promisc, to_ms, ebuf);
    if (!ret){
        PyErr_SetString(PcapError, ebuf);
        return NULL;
    }
    return PyCObject_FromVoidPtr((void*)ret, NULL);
}

static PyObject *dump_open(PyObject *self, PyObject *args){
    PyObject *ptr;
    char *filename;
    pcap_dumper_t *dumper;
    pcap_t *pptr;

    if (!PyArg_ParseTuple(args, "Os", &ptr, &filename))
        return NULL;
    pptr = (pcap_t*)PyCObject_AsVoidPtr(ptr);
    dumper = pcap_dump_open(pptr, filename);
    if (!dumper){
        PyErr_SetString(PcapError, pcap_geterr(pptr));
        return NULL;
    }
    return PyCObject_FromVoidPtr((void*)dumper, NULL);
}

static PyObject *open_offline(PyObject *self, PyObject *args){
    char ebuf[PCAP_ERRBUF_SIZE];
    char *filename;
    pcap_t *ret;

    if (!PyArg_ParseTuple(args, "s", &filename))
        return NULL;

    ret = pcap_open_offline(filename, ebuf);
    if (!ret){
        PyErr_SetString(PcapError, ebuf);
        return NULL;
    }
    return PyCObject_FromVoidPtr((void*)ret, NULL);
}

static PyObject *dump_close(PyObject *self, PyObject *args){
    PyObject *dptr;
    if (!PyArg_ParseTuple(args, "O", &dptr))
        return NULL;

    pcap_dump_close((pcap_dumper_t*)PyCObject_AsVoidPtr(dptr));
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *closeptr(PyObject *self, PyObject *args){
    PyObject *ptr;
    if (!PyArg_ParseTuple(args, "O", &ptr))
        return NULL;

    pcap_close((pcap_t*)PyCObject_AsVoidPtr(ptr));
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *datalink(PyObject *self, PyObject *args){
    PyObject *ptr;
    int linktype; 
    if (!PyArg_ParseTuple(args, "O", &ptr))
        return NULL;

    linktype = pcap_datalink((pcap_t*)PyCObject_AsVoidPtr(ptr));
    return Py_BuildValue("i", linktype);
}


static void callback(u_char *user, const struct pcap_pkthdr *phdr, const u_char *data){
    PyObject *packetstr;
    PyObject *tstamptuple, *argstuple;

    packetstr = PyString_FromStringAndSize((const char*)data, phdr->caplen);
    if (packetstr == NULL)
        longjmp(JENV, 1);
    tstamptuple = Py_BuildValue("(ll)", (long)phdr->ts.tv_sec, (long)phdr->ts.tv_usec);
    if (tstamptuple == NULL)
        longjmp(JENV, 1);
    argstuple = Py_BuildValue("(OOl)", packetstr, tstamptuple, (long)phdr->len);
    if (argstuple == NULL)
        longjmp(JENV, 1);
    if (PyObject_CallObject((PyObject*)user, argstuple) == NULL)
        longjmp(JENV, 1);
	if (PyErr_CheckSignals())
        longjmp(JENV, 1);

    Py_DECREF(packetstr);
    Py_DECREF(tstamptuple);
    Py_DECREF(argstuple);
}

static PyObject *dispatch(PyObject *self, PyObject *args){
    PyObject *ptr, *callable;
    pcap_t *pptr;
    int cnt;
    if (!PyArg_ParseTuple(args, "OiO", &ptr, &cnt, &callable))
        return NULL;

    if (sigsetjmp(JENV, 1)){
        return NULL;
    }

    pptr = (pcap_t*)PyCObject_AsVoidPtr(ptr);
    if (pcap_dispatch(pptr, cnt, &callback, (u_char*)callable) < 0){
        PyErr_SetString(PcapError, pcap_geterr(pptr));
        return NULL;
    }
    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *loop(PyObject *self, PyObject *args){
    PyObject *ptr, *callable;
    pcap_t *pptr;
    int cnt;

    if (!PyArg_ParseTuple(args, "OiO", &ptr, &cnt, &callable))
        return NULL;

    if (setjmp(JENV)){
        return NULL;
    }
    pptr = (pcap_t*)PyCObject_AsVoidPtr(ptr);
    if (pcap_loop(pptr, cnt, &callback, (u_char*)callable) < 0){
        PyErr_SetString(PcapError, pcap_geterr(pptr));
        return NULL;
    }
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *inject(PyObject *self, PyObject *args){
    PyObject *ptr;
    char *packet;
    int len;
    int ret;

    if (!PyArg_ParseTuple(args, "Os#", &ptr, &packet, &len))
        return NULL;
    ret = pcap_inject((pcap_t*)PyCObject_AsVoidPtr(ptr), packet, len);
    if (ret < 0){
        PyErr_SetString(PcapError, strerror(errno));
        return NULL;
    }
    return Py_BuildValue("i", ret);
}


static PyObject *dump(PyObject *self, PyObject *args){
    PyObject *dptr; 
    struct pcap_pkthdr phdr;
    char *packet;
    int datalen;
    int len;
    u_int32_t sec, usec;

    if (!PyArg_ParseTuple(args, "Os#(ii)i", &dptr, &packet, &datalen, &sec, &usec, &len))
        return NULL;

    /* First, we re-construct a pkthdr: */
    phdr.ts.tv_sec = sec;
    phdr.ts.tv_usec = usec;
    phdr.caplen = datalen;
    phdr.len = len;
    pcap_dump((u_char*)PyCObject_AsVoidPtr(dptr), &phdr, packet);

    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *lookupdev(PyObject *self, PyObject *args){
    char ebuf[PCAP_ERRBUF_SIZE];
    char *devstr;
    PyObject *pdevstr;

    if (!PyArg_ParseTuple(args, ""))
        return NULL;

    devstr = pcap_lookupdev(ebuf);
    if (devstr == NULL){
        PyErr_SetString(PcapError, ebuf);
        return NULL;
    }
    pdevstr = PyString_FromString(devstr);
    if (pdevstr == NULL){
        PyErr_SetString(PcapError, "Could not retrieve device string.");
        return NULL;
    }
    return pdevstr;
}


static PyObject *lookupnet(PyObject *self, PyObject *args){
    char ebuf[PCAP_ERRBUF_SIZE];
    char *devstr;
    bpf_u_int32 netp, maskp;

    if (!PyArg_ParseTuple(args, "s", &devstr))
        return NULL;

    if (pcap_lookupnet(devstr, &netp, &maskp, ebuf) < 0){
        PyErr_SetString(PcapError, ebuf);
        return NULL;
    }
    return Py_BuildValue("(l, l)", netp, maskp);
}

/*
 * A wrapper for pcap_freecode that obeys the interface expected for the
 * destructor of a PyCObject.
 */
void freecode_wrapper(void *code){
	pcap_freecode((struct bpf_program*)code); 
	return; 
}

/*
 * Returns a Python object containing a pointer to a malloced bpf_program. The
 * user must explicitly free this using freebpf after use.
 */
static PyObject *compile(PyObject *self, PyObject *args){
    PyObject *pcap_ptr;
    char *bpfstr;
    pcap_t *pptr;
    int optimise;
    bpf_u_int32 netmask;
    struct bpf_program *bptr;

    if (!PyArg_ParseTuple(args, "Osii", &pcap_ptr, &bpfstr, &optimise, &netmask))
        return NULL;

    bptr = malloc(sizeof(struct bpf_program));
    if (!bptr){
        PyErr_SetString(PyExc_MemoryError, "Can't allocate memory for BPF program.");
        return NULL;
    }
    pptr = (pcap_t*)PyCObject_AsVoidPtr(pcap_ptr);

    if (pcap_compile(pptr, bptr, bpfstr, optimise, netmask) < 0){
        PyErr_SetString(PcapError, "Filter program compilation error.");
        return NULL;
    }
    return PyCObject_FromVoidPtr((void*)bptr, freecode_wrapper);
}

static PyObject *setfilter(PyObject *self, PyObject *args){
    PyObject *ptr, *bpf;
    pcap_t *pptr;
    struct bpf_program *pbpf;

    if (!PyArg_ParseTuple(args, "OO", &ptr, &bpf))
        return NULL;

    pptr = (pcap_t*)PyCObject_AsVoidPtr(ptr);
    pbpf = (struct bpf_program*)PyCObject_AsVoidPtr(bpf);

    if (pcap_setfilter(pptr, pbpf) < 0){
        PyErr_SetString(PcapError, pcap_geterr(pptr));
        return NULL;
    }
    Py_INCREF(Py_None);
    return Py_None;
}


static PyObject *snapshot(PyObject *self, PyObject *args){
    PyObject *dptr;
    int snaplen;

    if (!PyArg_ParseTuple(args, "O", &dptr))
        return NULL;

    snaplen = pcap_snapshot((pcap_t*)PyCObject_AsVoidPtr(dptr));
    return Py_BuildValue("i", snaplen);
}

static PyObject *is_swapped(PyObject *self, PyObject *args){
    PyObject *dptr;
    int swapped;

    if (!PyArg_ParseTuple(args, "O", &dptr))
        return NULL;

    swapped = pcap_is_swapped((pcap_t*)PyCObject_AsVoidPtr(dptr));
    return Py_BuildValue("i", swapped);
}

static PyObject *version(PyObject *self, PyObject *args){
    PyObject *dptr;
    int major, minor;

    if (!PyArg_ParseTuple(args, "O", &dptr))
        return NULL;

    major = pcap_major_version((pcap_t*)PyCObject_AsVoidPtr(dptr));
    minor = pcap_minor_version((pcap_t*)PyCObject_AsVoidPtr(dptr));

    return Py_BuildValue("(i, i)", major, minor);
}


static PyObject *stats(PyObject *self, PyObject *args){
    PyObject *ptr;
    struct pcap_stat pstat; 

    if (!PyArg_ParseTuple(args, "O", &ptr))
        return NULL;

    if (pcap_stats((pcap_t*)PyCObject_AsVoidPtr(ptr), &pstat) < 0){
        PyErr_SetString(PcapError, strerror(errno));
        return NULL;
    }
    return Py_BuildValue("{s:i, s:i, s:i}", "ps_recv", (int)pstat.ps_recv, 
                                            "ps_drop", (int)pstat.ps_drop, 
                                            "ps_ifdrop", (int)pstat.ps_ifdrop);
}


static PyObject *pfileno(PyObject *self, PyObject *args){
    PyObject *dptr;
	FILE *f;

    if (!PyArg_ParseTuple(args, "O", &dptr))
        return NULL;

    f = pcap_file((pcap_t*)PyCObject_AsVoidPtr(dptr));
    return Py_BuildValue("i", fileno(f));
}


static PyMethodDef PcapMethods[] = {
    {"open_live",       open_live,      METH_VARARGS,   "Open a device."},
    {"dump_open",       dump_open,      METH_VARARGS,   "Open a dump file."},
    {"open_offline",    open_offline,   METH_VARARGS,   "Open a file for reading."},
    {"close",           closeptr,       METH_VARARGS,   "Close a pointer."},
    {"dump_close",      dump_close,     METH_VARARGS,   "Close a dump file."},
    {"datalink",        datalink,       METH_VARARGS,   "Get the link layer type."},
    {"dispatch",        dispatch,       METH_VARARGS,   "Dispatch."},
    {"loop",            loop,           METH_VARARGS,   "Loop."},
    {"inject",          inject,         METH_VARARGS,   "Inject a packet."},
    {"dump",            dump,           METH_VARARGS,   "Dump."},
    {"lookupdev",       lookupdev,      METH_VARARGS,   "Lookup a device."},
    {"lookupnet",       lookupnet,      METH_VARARGS,   "Lookup the network specifications of a device."},
    {"compile",         compile,        METH_VARARGS,   "Compile a BPF program."},
    {"setfilter",       setfilter,      METH_VARARGS,   "Set a filter."},
    {"snapshot",        snapshot,       METH_VARARGS,   "Return the snapshot length passed to pcap_live."},
    {"is_swapped",      is_swapped,     METH_VARARGS,   "True if the current savefile uses a different byte order than the current system."},
    {"fileno",			pfileno,		METH_VARARGS,   "Returns the file descriptor number of the current file."},
    {"version",         version,        METH_VARARGS,   "Return the major and minor version of the pcap used to write the save file."},
    {"stats",           stats,          METH_VARARGS,   "Get stats for the feed."},
    {NULL, NULL, 0, NULL}        /* Sentinel */
};


void init_pcap(void){
    PyObject *module;
    module = Py_InitModule4("_pcap", PcapMethods, NULL, NULL, PYTHON_API_VERSION);
    PcapError = PyErr_NewException("_pcap.PcapError", NULL, NULL);
    PyModule_AddObject(module, "PcapError", PcapError);

    /* BPF Data Link Types */
    PyModule_AddIntConstant(module, "DLT_NULL",         (long) DLT_NULL);
    PyModule_AddIntConstant(module, "DLT_EN10MB",       (long) DLT_EN10MB);
    PyModule_AddIntConstant(module, "DLT_EN3MB",        (long) DLT_EN3MB);
    PyModule_AddIntConstant(module, "DLT_AX25",         (long) DLT_AX25);
    PyModule_AddIntConstant(module, "DLT_PRONET",       (long) DLT_PRONET);
    PyModule_AddIntConstant(module, "DLT_CHAOS",        (long) DLT_CHAOS);
    PyModule_AddIntConstant(module, "DLT_IEEE802",      (long) DLT_IEEE802);
    PyModule_AddIntConstant(module, "DLT_ARCNET",       (long) DLT_ARCNET);
    PyModule_AddIntConstant(module, "DLT_SLIP",         (long) DLT_SLIP);
    PyModule_AddIntConstant(module, "DLT_PPP",          (long) DLT_PPP);
    PyModule_AddIntConstant(module, "DLT_FDDI",         (long) DLT_FDDI);
    PyModule_AddIntConstant(module, "DLT_ATM_RFC1483",  (long) DLT_ATM_RFC1483);
    PyModule_AddIntConstant(module, "DLT_LOOP",         (long) DLT_LOOP);
    PyModule_AddIntConstant(module, "DLT_ENC",          (long) DLT_ENC);
    PyModule_AddIntConstant(module, "DLT_RAW",          (long) DLT_RAW);
    PyModule_AddIntConstant(module, "DLT_SLIP_BSDOS",   (long) DLT_SLIP_BSDOS);
    PyModule_AddIntConstant(module, "DLT_PPP_BSDOS",    (long) DLT_PPP_BSDOS);
    PyModule_AddIntConstant(module, "DLT_OLD_PFLOG",    (long) DLT_OLD_PFLOG);
    PyModule_AddIntConstant(module, "DLT_PFSYNC",       (long) DLT_PFSYNC);
    PyModule_AddIntConstant(module, "DLT_IEEE802_11",   (long) DLT_IEEE802_11);
    PyModule_AddIntConstant(module, "DLT_PFLOG",        (long) DLT_PFLOG);
}
