/*
	Copyright (c) 2003, Nullcube Pty Ltd
	All rights reserved.

	Redistribution and use in source and binary forms, with or without
	modification, are permitted provided that the following conditions are met:

	*   Redistributions of source code must retain the above copyright notice, this
		list of conditions and the following disclaimer.
	*   Redistributions in binary form must reproduce the above copyright notice,
		this list of conditions and the following disclaimer in the documentation
		and/or other materials provided with the distribution.
	*   Neither the name of Nullcube nor the names of its contributors may be used to
		endorse or promote products derived from this software without specific prior
		written permission.

	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>

#include <Python.h>

PyObject *KQueueError;

static PyObject *pykqueue(PyObject *self, PyObject *args){
    int k;
    if (!PyArg_ParseTuple(args, ""))
        return NULL;

    k = kqueue();
    if (k < 0){
        PyErr_SetFromErrno(KQueueError);
        return NULL;
    }
    return PyInt_FromLong((long int) k);
}

static PyObject *pykevent(PyObject *self, PyObject *args){
    PyObject *pchangelist, *ptimespec, *tmp, *ret, *udata;
    int nevents, kq;
    long seconds, nanoseconds;
    struct timespec timeout, *tptr;
    struct kevent *changelist, *eventlist;
    int i, csize, esize;

    if (!PyArg_ParseTuple(args, "iOiO", &kq, &pchangelist, &nevents, &ptimespec))
        return NULL;

    if (ptimespec == Py_None){
        tptr = NULL;
    } else {
        if (!PyTuple_Check(ptimespec)){
			PyErr_SetString(KQueueError, "Timeout specification must be a tuple.");
            return NULL;
        }
        tmp = PyTuple_GetItem(ptimespec, 0);
        seconds = (int)PyInt_AsLong(tmp);
        tmp = PyTuple_GetItem(ptimespec, 1);
        nanoseconds = (int)PyInt_AsLong(tmp);
        timeout.tv_sec = seconds;
        timeout.tv_nsec = nanoseconds;
        tptr = &timeout;
    }

    if (pchangelist == Py_None){
        changelist = NULL;
        csize = 0;
    } else {
        csize = PySequence_Size(pchangelist);
        changelist = malloc(sizeof(struct kevent) * csize);
        for (i = 0; i < PySequence_Size(pchangelist); i++){
            PyObject *attr;

            tmp = PySequence_GetItem(pchangelist, i);

            attr = PyObject_GetAttrString(tmp, "ident");
            if (!attr){
                return NULL;
            }
            if (!PyNumber_Check(attr)){
                PyErr_SetString(KQueueError, "ident must be a number.");
                return NULL;
                Py_DECREF(attr);
            }
            changelist[i].ident = PyInt_AsLong(attr);
            Py_DECREF(attr);

            attr = PyObject_GetAttrString(tmp, "_filter");
            if (!attr){
                return NULL;
            }
            if (!PyNumber_Check(attr)){
                PyErr_SetString(KQueueError, "filter must be a number.");
                return NULL;
                Py_DECREF(attr);
            }
            changelist[i].filter = PyInt_AsLong(attr);
            Py_DECREF(attr);

            attr = PyObject_GetAttrString(tmp, "flags");
            if (!attr){
                return NULL;
            }
            if (!PyNumber_Check(attr)){
                PyErr_SetString(KQueueError, "flags must be a number.");
                return NULL;
                Py_DECREF(attr);
            }
            changelist[i].flags = PyInt_AsLong(attr);
            Py_DECREF(attr);

            attr = PyObject_GetAttrString(tmp, "fflags");
            if (!attr){
                return NULL;
            }
            if (!PyNumber_Check(attr)){
                PyErr_SetString(KQueueError, "fflags must be a number.");
                return NULL;
                Py_DECREF(attr);
            }
            changelist[i].fflags = PyInt_AsLong(attr);
            Py_DECREF(attr);

            udata = PyObject_GetAttrString(tmp, "udata");
            if (!udata){
                return NULL;
            }
            Py_INCREF(udata);
            changelist[i].udata = (void*)udata;

            attr = PyObject_GetAttrString(tmp, "data");
            if (!attr){
                return NULL;
            }
            if (!PyNumber_Check(attr)){
                PyErr_SetString(KQueueError, "data must be a number.");
                return NULL;
                Py_DECREF(attr);
            }
            changelist[i].data = PyInt_AsLong(attr);
            Py_DECREF(attr);

            Py_DECREF(tmp);
        }
    }

    if (nevents)
        eventlist = malloc(sizeof(struct kevent) * nevents);
    else
        eventlist = NULL;
    esize = kevent(kq, changelist, csize, eventlist, nevents, tptr);
    free(changelist);
    if (esize == -1){
        PyErr_SetFromErrno(KQueueError);
        free(eventlist);
        return NULL;
    }

    if (esize > 0){
        ret = PyList_New(esize);
        for (i = 0; i < esize; i++){
            tmp = Py_BuildValue(
                        "(iihHOi)",
                        eventlist[i].filter,
                        eventlist[i].ident,
                        eventlist[i].flags,
                        eventlist[i].fflags,
                        (PyObject*)eventlist[i].udata,
                        eventlist[i].data
                    );
            if (PyList_SetItem(ret, i, tmp) < 0){
                Py_DECREF(ret);
                Py_DECREF(tmp);
                free(eventlist);
                return NULL;
            }
        }
        free(eventlist);
        return ret;
    } 
    free(eventlist);
    Py_INCREF(Py_None);
    return Py_None;
}

static PyMethodDef KEventMethods[] = {
	{"kqueue",			pykqueue,		METH_VARARGS,	"Initialise a kqueue."},
	{"kevent",			pykevent,		METH_VARARGS,	"Register or retrieve kevents."},
	{NULL, NULL, 0, NULL}		 /* Sentinel */
};

void init_kqueue(void){
	PyObject *module;
	module = Py_InitModule4("_kqueue", KEventMethods, NULL, NULL, PYTHON_API_VERSION);
	KQueueError = PyErr_NewException("_kqueue.KQueueError", NULL, NULL);
	PyModule_AddObject(module, "KQueueError", KQueueError);

    /* Filters */
    PyModule_AddIntConstant(module, "EVFILT_READ",              (long) EVFILT_READ);       
    PyModule_AddIntConstant(module, "EVFILT_WRITE",             (long) EVFILT_WRITE);       
    /*PyModule_AddIntConstant(module, "EVFILT_AIO",               (long) EVFILT_AIO);*/
    PyModule_AddIntConstant(module, "EVFILT_VNODE",             (long) EVFILT_VNODE);       
    PyModule_AddIntConstant(module, "EVFILT_PROC",              (long) EVFILT_PROC);       
    PyModule_AddIntConstant(module, "EVFILT_SIGNAL",            (long) EVFILT_SIGNAL);       

    /* Flags */
    PyModule_AddIntConstant(module, "EV_ADD",                   (long) EV_ADD);       
    PyModule_AddIntConstant(module, "EV_ENABLE",                (long) EV_ENABLE);       
    PyModule_AddIntConstant(module, "EV_DISABLE",               (long) EV_DISABLE);       
    PyModule_AddIntConstant(module, "EV_DELETE",                (long) EV_DELETE);       
    PyModule_AddIntConstant(module, "EV_ONESHOT",               (long) EV_ONESHOT);       
    PyModule_AddIntConstant(module, "EV_CLEAR",                 (long) EV_CLEAR);       
    PyModule_AddIntConstant(module, "EV_EOF",                   (long) EV_EOF);       
    PyModule_AddIntConstant(module, "EV_ERROR",                 (long) EV_ERROR);       

    /* Notes */
    PyModule_AddIntConstant(module, "NOTE_LOWAT",               (long) NOTE_LOWAT);       
    PyModule_AddIntConstant(module, "NOTE_EOF",                 (long) NOTE_EOF);       
    PyModule_AddIntConstant(module, "NOTE_DELETE",              (long) NOTE_DELETE);       
    PyModule_AddIntConstant(module, "NOTE_WRITE",               (long) NOTE_WRITE);       
    PyModule_AddIntConstant(module, "NOTE_EXTEND",              (long) NOTE_EXTEND);       
    PyModule_AddIntConstant(module, "NOTE_ATTRIB",              (long) NOTE_ATTRIB);       
    PyModule_AddIntConstant(module, "NOTE_LINK",                (long) NOTE_LINK);       
    PyModule_AddIntConstant(module, "NOTE_RENAME",              (long) NOTE_RENAME);       
    PyModule_AddIntConstant(module, "NOTE_REVOKE",              (long) NOTE_REVOKE);       
    PyModule_AddIntConstant(module, "NOTE_TRUNCATE",            (long) NOTE_TRUNCATE);       
    PyModule_AddIntConstant(module, "NOTE_EXIT",                (long) NOTE_EXIT);       
    PyModule_AddIntConstant(module, "NOTE_FORK",                (long) NOTE_FORK);       
    PyModule_AddIntConstant(module, "NOTE_EXEC",                (long) NOTE_EXEC);       
    PyModule_AddIntConstant(module, "NOTE_PCTRLMASK",           (long) NOTE_PCTRLMASK);       
    PyModule_AddIntConstant(module, "NOTE_PDATAMASK",           (long) NOTE_PDATAMASK);       
    PyModule_AddIntConstant(module, "NOTE_TRACK",               (long) NOTE_TRACK);       
    PyModule_AddIntConstant(module, "NOTE_TRACKERR",            (long) NOTE_TRACKERR);       
    PyModule_AddIntConstant(module, "NOTE_CHILD",               (long) NOTE_CHILD);       
}
