/*
 * libtu/output.c
 *
 * Copyright (c) Tuomo Valkonen 1999-2000. 
 * See the included file LICENSE for details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <strings.h>
#include <string.h>

#include <libtu/misc.h>
#include <libtu/output.h>
#include <libtu/util.h>


/* verbose
 */

static bool verbose_mode=FALSE;
static int verbose_indent_lvl=0;
static bool progname_enable=TRUE;

#define INDENTATOR_LENGTH 4

static char indentator[]={' ', ' ', ' ', ' '};


void verbose(const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	
	verbose_v(p, args);
	
	va_end(args);
}
		   

void verbose_v(const char *p, va_list args)
{
	int i;
	
	if(verbose_mode){
		for(i=0; i<verbose_indent_lvl; i++)
			writef(stdout, indentator, INDENTATOR_LENGTH);
		
		vprintf(p, args);
		fflush(stdout);
	}
}


void verbose_enable(bool enable)
{
	verbose_mode=enable;
}


int verbose_indent(int depth)
{
	int old=verbose_indent_lvl;
	
	if(depth>=0)
		verbose_indent_lvl=depth;
	
	return old;
}
		

/* warn
 */

void warn_progname_enable(bool enable)
{
	progname_enable=enable;
}


static void put_prog_name()
{
	const char*progname;
	
	if(!progname_enable)
		return;
	
	progname=prog_execname();
	
	if(progname==NULL)
		return;
	
	fprintf(stderr, "%s: ", (char*)progname);
}


void warn(const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	
	warn_v(p, args);
	
	va_end(args);
}


void warn_v(const char *p, va_list args)
{
	put_prog_name();
	vfprintf(stderr, p, args);
	putc('\n', stderr);
}


void warn_obj(const char *obj, const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	
	warn_obj_v(obj, p, args);
	
	va_end(args);
}


void warn_obj_line(const char *obj, int line, const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	
	warn_obj_line_v(obj, line, p, args);
	
	va_end(args);
}


void warn_obj_v(const char *obj, const char *p, va_list args)
{
	put_prog_name();
	if(obj!=NULL)
		fprintf(stderr,"%s: ", obj);
	vfprintf(stderr, p, args);
	putc('\n', stderr);
}


void warn_obj_line_v(const char *obj, int line, const char *p, va_list args)
{
	put_prog_name();
	if(obj!=NULL){
		if(line>0)
			fprintf(stderr, TR("%s:%d: "), obj, line);
		else		
			fprintf(stderr, "%s: ", obj);
	}else{
		if(line>0)
			fprintf(stderr, TR("%d: "), line);
	}
	vfprintf(stderr, p, args);
	putc('\n', stderr);
}


void warn_err()
{
	put_prog_name();
	fprintf(stderr, "%s\n", strerror(errno));
}


void warn_err_obj(const char *obj)
{
	put_prog_name();
	if(obj!=NULL)
		fprintf(stderr, "%s: %s\n", obj, strerror(errno));
	else
		fprintf(stderr, "%s\n", strerror(errno));

}


void warn_err_obj_line(const char *obj, int line)
{
	put_prog_name();
	if(obj!=NULL){
		if(line>0)
			fprintf(stderr, TR("%s:%d: %s\n"), obj, line, strerror(errno));
		else
			fprintf(stderr, "%s: %s\n", obj, strerror(errno));
	}else{
		if(line>0)
			fprintf(stderr, TR("%d: %s\n"), line, strerror(errno));
		else
			fprintf(stderr, TR("%s\n"), strerror(errno));
	}

}


/* die
 */

void die(const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	
	die_v(p, args);
	
	va_end(args);
}


void die_v(const char *p, va_list args)
{
	warn_v(p, args);
	exit(EXIT_FAILURE);
}


void die_obj(const char *obj, const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	die_obj_v(obj, p, args);
	va_end(args);
}


void die_obj_line(const char *obj, int line, const char *p, ...)
{
	va_list args;
	
	va_start(args, p);
	die_obj_line_v(obj, line, p, args);
	va_end(args);
}


void die_obj_v(const char *obj, const char *p, va_list args)
{
	warn_obj_v(obj, p, args);
	exit(EXIT_FAILURE);
}


void die_obj_line_v(const char *obj, int line, const char *p, va_list args)
{
	warn_obj_line_v(obj, line, p, args);
	exit(EXIT_FAILURE);
}


void die_err()
{
	warn_err();
	exit(EXIT_FAILURE);
}


void die_err_obj(const char *obj)
{
	warn_err_obj(obj);
	exit(EXIT_FAILURE);
}


void die_err_obj_line(const char *obj, int line)
{
	warn_err_obj_line(obj, line);
	exit(EXIT_FAILURE);
}
