
/* LDAP posixAccount handler for Pure-FTPd */
/* (C)opyleft 2001-2002 by Frank DENIS <j@pureftpd.org> */

#include <config.h>
#ifdef WITH_LDAP
# include "ftpd.h"
# include "parser.h"
# include "log_ldap_p.h"
# include "log_ldap.h"
# include "messages.h"
# include "crypto.h"
# ifdef WITH_DMALLOC
#  include <dmalloc.h>
# endif

void pw_ldap_parse(const char * const file)
{
    if (generic_parser(file, ldap_config_keywords) != 0) {
        die(421, LOG_ERR, MSG_CONF_ERR ": " MSG_ILLEGAL_CONFIG_FILE_LDAP ": %s" , file);
    }
    if (ldap_host == NULL) {
        if ((ldap_host = strdup(LDAP_DEFAULT_SERVER)) == NULL) {
            die_mem();
        }
    }
    if (port_s == NULL) {
        port = LDAP_DEFAULT_PORT;
    } else {
        port = atoi(port_s);
        if (port <= 0 || port > 65535) {
            port = LDAP_DEFAULT_PORT;
        }
        free(port_s);
        port_s = NULL;
    }
    if (default_uid_s != NULL) {
        default_uid = (uid_t) strtoul(default_uid_s, NULL, 10);        
        free(default_uid_s);
        default_uid_s = NULL;        
    }
    if (default_gid_s != NULL) {
        default_gid = (gid_t) strtoul(default_gid_s, NULL, 10);
        free(default_gid_s);
        default_gid_s = NULL;
    }        
    if (base == NULL) {
        die(421, LOG_ERR, MSG_LDAP_MISSING_BASE);
    }
    if (root == NULL) {
        pwd = NULL;
    }    
}

void pw_ldap_exit(void)
{
    if (ldap_host != NULL) {
        free((void *) ldap_host);
        ldap_host = NULL;
    }
    if (port_s != NULL) {
        free((void *) port_s);
        port_s = NULL;
    }        
    port = -1;
    if (root != NULL) {
        free((void *) root);
        root = NULL;
    }
    if (pwd != NULL) {
        free((void *) pwd);
        pwd = NULL;
    }
    if (base != NULL) {
        free((void *) base);
        base = NULL;
    }
    if (default_uid_s != NULL) {
        free((void *) default_uid_s);
        default_uid_s = NULL;
    }
    if (default_gid_s != NULL) {
        free((void *) default_gid_s);
        default_gid_s = NULL;
    }
}

static LDAP *pw_ldap_connect(void)
{
    LDAP *ld;

    if (ldap_host == NULL || port < 0) {
        return NULL;
    }
    if ((ld = ldap_init(ldap_host, port)) == NULL) {
        return NULL;
    }
    if (ldap_bind_s(ld, root, pwd, LDAP_AUTH_SIMPLE) != LDAP_SUCCESS) {
        return NULL;
    }
    
    return ld;
}

static LDAPMessage *pw_ldap_uid_search(LDAP * const ld, 
                                       const char *uid,
                                       char *attrs[])
{
    char *alloca_filter;
    size_t uid_size;
    size_t filter_size;
    int rc;    
    LDAPMessage *res;
    
    if (uid == NULL || *uid == 0) {
        return NULL;
    }
    uid_size = strlen(uid);
    if (uid_size > MAX_LDAP_UID_LENGTH) {
        return NULL;
    }
    filter_size = 
        (sizeof  "(&(objectClass=" LDAP_POSIXACCOUNT ")(" LDAP_UID "=" - 1U) +
        uid_size + (sizeof "))" - 1U) + 1U;
    if ((alloca_filter = ALLOCA(filter_size)) == NULL) {
        return NULL;
    }
    if (SNCHECK(snprintf(alloca_filter, filter_size, 
                         "(&(objectClass=" LDAP_POSIXACCOUNT ")(" LDAP_UID "=%s))", 
                         uid), filter_size)) {
        ALLOCA_FREE(alloca_filter);
        return NULL;
    }
    rc = ldap_search_s(ld, base, LDAP_SCOPE_SUBTREE, 
                       alloca_filter, attrs, 0, &res);
    ALLOCA_FREE(filter_size);
    if (rc != LDAP_SUCCESS) {
        return NULL;
    }
    return res;
}

static char *pw_ldap_getvalue(LDAP * const ld,
                              LDAPMessage * const res,
                              const char * const attribute)
{
    char **vals;
    char *ret;
    
    if ((vals = ldap_get_values(ld, res, attribute)) == NULL ||
        vals[0] == NULL) {
        return NULL;
    }    
    ret = strdup(vals[0]);
    ldap_value_free(vals);
    
    return ret;
}

static void pw_ldap_getpwnam_freefields(struct passwd * const p) 
{
    if (p->pw_passwd != NULL) {
        free(p->pw_passwd);
        p->pw_passwd = NULL;
    }
    if (p->pw_dir != NULL) {
        free(p->pw_dir);
        p->pw_dir = NULL;
    }
    if (p->pw_shell != NULL) {
        free(p->pw_shell);
        p->pw_shell = NULL;
    }
}

static int pw_ldap_validate_name(const char *name)
{
    if (name == NULL || *name == 0) {
        return -1;
    }
    do {
        if ((*name >= 'a' && *name <= 'z') ||
            (*name >= 'A' && *name <= 'Z') ||
            (*name >= '0' && *name <= '9') ||
            *name == ' ' || *name == '-' || *name == '@' ||
            *name == '_' || *name == '\'' || *name == '.') {
            /* God bless the Perl 'unless' keyword */
        } else {
            return -1;
        }            
        name++;
    } while (*name != 0);
    
    return 0;
}

static struct passwd *pw_ldap_getpwnam(const char *name)
{
    static struct passwd pwret;
    LDAP *ld;
    LDAPMessage *res;
    char *attrs[] = {                  /* OpenLDAP forgot a 'const' ... */
        LDAP_UIDNUMBER, LDAP_GIDNUMBER, LDAP_HOMEDIRECTORY,
            LDAP_USERPASSWORD, LDAP_LOGINSHELL, NULL
    };
    const char *pw_uid_s = NULL;
    const char *pw_gid_s = NULL;
    const char *pw_passwd_ldap = NULL;

    memset(&pwret, 0, sizeof pwret);
    pwret.pw_name = pwret.pw_passwd = pwret.pw_gecos = pwret.pw_dir =
        pwret.pw_shell = NULL;
    pwret.pw_uid = (uid_t) -1;
    pwret.pw_gid = (gid_t) -1;    
    if (pw_ldap_validate_name(name) != 0) {
        return NULL;
    }
    if ((ld = pw_ldap_connect()) == NULL) {
        return NULL;
    }
    if ((res = pw_ldap_uid_search(ld, name, attrs)) == NULL) {
        goto error;
    }
    pw_ldap_getpwnam_freefields(&pwret);    
    pwret.pw_name = (char *) name;
    if ((pw_passwd_ldap =
         pw_ldap_getvalue(ld, res, LDAP_USERPASSWORD)) == NULL) {
        goto error;
    }
    pwret.pw_passwd = strdup(pw_passwd_ldap);
    if (pwret.pw_passwd == NULL) {
        goto error;
    }
    free((void *) pw_passwd_ldap);
    pw_passwd_ldap = NULL;    
    if ((pw_uid_s = pw_ldap_getvalue(ld, res, LDAP_UIDNUMBER)) == NULL ||
        *pw_uid_s == 0 || 
        (pwret.pw_uid = (uid_t) strtoul(pw_uid_s, NULL, 10)) <= (uid_t) 0) {
        pwret.pw_uid = default_uid;
    } 
    if (pw_uid_s != NULL) {
        free((void *) pw_uid_s);
        pw_uid_s = NULL;
    }
    if ((pw_gid_s = pw_ldap_getvalue(ld, res, LDAP_GIDNUMBER)) == NULL ||
        *pw_gid_s == 0 ||
        (pwret.pw_gid = (gid_t) strtoul(pw_gid_s, NULL, 10)) <= (gid_t) 0) {
        pwret.pw_gid = default_gid;
    } 
    if (pw_gid_s != NULL) {
        free((void *) pw_gid_s);
        pw_gid_s = NULL;
    }
    if ((pwret.pw_dir = 
         pw_ldap_getvalue(ld, res, LDAP_HOMEDIRECTORY)) == NULL ||
        *pwret.pw_dir == 0) {
        goto error;
    }    
    if ((pwret.pw_shell = 
         pw_ldap_getvalue(ld, res, LDAP_LOGINSHELL)) == NULL) {
        pwret.pw_shell = strdup(DEFAULT_SHELL);
    }

    ldap_msgfree(res);
    ldap_unbind(ld);    
    
    return &pwret;
    
    error:    
    if (res != NULL) {
        ldap_msgfree(res);
    }
    ldap_unbind(ld);
    pw_ldap_getpwnam_freefields(&pwret);
    if (pw_uid_s != NULL) {
        free((void *) pw_uid_s);
    }
    if (pw_gid_s != NULL) {
        free((void *) pw_gid_s);
    }    
    if (pw_passwd_ldap != NULL) {
        free((void *) pw_passwd_ldap);
    }    

    return NULL;
}

void pw_ldap_check(AuthResult * const result,
                   const char *account, const char *password,
                   const struct sockaddr_storage * const sa,
                   const struct sockaddr_storage * const peer)
{
    struct passwd *pw;
    const char *spwd;                  /* Stored pwd */
    const char *cpwd = NULL;           /* Computed pwd */
    signed char nocase = 0;            /* Insensitive strcmp */
    
    (void) sa;
    (void) peer;
    result->auth_ok = 0;
    if (account == NULL || *account == 0 || password == NULL ||
        (pw = pw_ldap_getpwnam(account)) == NULL) {
        return;
    }
    result->auth_ok--;                  /* -1 */
    spwd = pw->pw_passwd;
    if (strncasecmp(spwd, PASSWD_LDAP_MD5_PREFIX,
                    sizeof PASSWD_LDAP_MD5_PREFIX - 1U) == 0) {
        spwd += (sizeof PASSWD_LDAP_MD5_PREFIX - 1U);
        if (strlen(spwd) >= 32U) {
            nocase++;
        }
        cpwd = crypto_hash_md5(password, nocase);
    } else if (strncasecmp(spwd, PASSWD_LDAP_SHA_PREFIX,
                           sizeof PASSWD_LDAP_SHA_PREFIX - 1U) == 0) {
        spwd += (sizeof PASSWD_LDAP_SHA_PREFIX - 1U);
        if (strlen(spwd) >= 40U) {
            nocase++;
        }
        cpwd = crypto_hash_sha1(password, nocase);
    } else if (strncasecmp(spwd, PASSWD_LDAP_SSHA_PREFIX,
                           sizeof PASSWD_LDAP_SSHA_PREFIX - 1U) == 0) {
        spwd += (sizeof PASSWD_LDAP_SSHA_PREFIX - 1U);
        cpwd = crypto_hash_ssha1(password, spwd);
    } else if (strncasecmp(spwd, PASSWD_LDAP_SMD5_PREFIX,
                           sizeof PASSWD_LDAP_SMD5_PREFIX - 1U) == 0) {
        spwd += (sizeof PASSWD_LDAP_SMD5_PREFIX - 1U);
        cpwd = crypto_hash_smd5(password, spwd);
    } else if (strncasecmp(spwd, PASSWD_LDAP_CRYPT_PREFIX,
                           sizeof PASSWD_LDAP_CRYPT_PREFIX - 1U) == 0) {
        spwd += (sizeof PASSWD_LDAP_CRYPT_PREFIX - 1U);
        cpwd = (const char *) crypt(password, spwd);
    } else if (*password != 0) {
        cpwd = password;               /* Cleartext */        
    } else {
        return;                      /* Refuse null passwords */
    }
    if (cpwd == NULL) {
        return;
    }
    if (nocase != 0) {        
        if (strcasecmp(cpwd, spwd) != 0) {
            return;
        }
    }    
    if (strcmp(cpwd, spwd) != 0) {
        return;
    }    
    result->uid = pw->pw_uid;
    result->gid = pw->pw_gid;
    if (result->uid == (uid_t) 0 || result->gid == (gid_t) 0) {
        return;
    }
    if ((result->dir = strdup(pw->pw_dir)) == NULL) {
        return;
    }
    result->slow_tilde_expansion = 1;
    result->auth_ok =- result->auth_ok;       /* 1 */
}
#else
extern signed char v6ready;
#endif
