
################################################################################
# Copyright (c) QinetiQ Plc 2003
#
# Licensed under the LGPL. For full license details see the LICENSE file.
################################################################################

"""
Implementation of Point.
"""

class Point:
    """
    Point provides an interface for a single point in 2d space.

    It has accessor functions for X and Y coords, provides a constructor
    that can handle the strings returned from Postgres for the point
    type and support equallity operations with other Point objects.
    """
    
    def __init__(self,s=None):
        """
        Constructor. Optional (arg s) is a string as returned by postgres.
        It is of the form '(x.y)' where x and y are floating point numbers.

        If (arg s) is None or omitted the Point is initalised to (0.0,0.0).
        """
        if s:
            self.fromString(s)
        else:
            self.setX(0.0)
            self.setY(0.0)

    def fromString(self,s):
        """
        Initialise the Point from a string.

        (arg s) should be of the form '(x.y)' where x and y are floating
        point numbers.
        """
        seq = eval(s,{},{})
        self.setX(seq[0])
        self.setY(seq[1])
        
    def setX(self,x):
        """
        Set the X coord of the Point.

        (arg x) is a float.
        """
        self._x = x

    def getX(self):
        """
        Return the X coord of the Point.
        """
        return self._x

    def setY(self,y):
        """
        Set the Y coord of the Point.

        (arg x) is a float.
        """

        self._y = y

    def getY(self):
        """
        Return the X coord of the Point.
        """

        return self._y

    def __str__(self):
        """
        Generate a string representation of the Point that is
        suitable to use in a Postgres query.
        """
        return "'(%f,%f)'" % (self.getX(), self.getY())

    def __repr__(self):
        """
        Generate a represention of the Point as a string
        suitable for 'evaling' as a tuple.
        """
        return "(%f,%f)" % (self.getX(), self.getY())

    def __eq__(self,other):
        """
        Support equality operations.

        A Point is equal to another point is X == other.X and Y = other.Y.
        """
        if (type(self) != type(other)):
            return False

        if self.getX() == other.getX() and \
           self.getY() == other.getY():
            return True
        return False

    def __ne__(self,other):
        return not self.__eq__(other)

# Some helpful factory methods

def pointFromValues(x,y):
    """
    Return a Point object.

    (arg x) is the X coord as a float
    (arg y) is the Y coord as a float
    """
    
    p = Point()
    p.setX(x)
    p.setY(y)
    return p

def pointFromSequence(seq):
    """
    Return a Point object.

    (arg seq) is a sequence of the form (x,y) where x is the X coord as a float
    and y is the Y coord as a float.
    """

    p = Point()
    p.setX(seq[0])
    p.setY(seq[1])
    return p
