
#include "conf.h"

#include <iostream>
#include <fstream>
#include <cassert>
#include <cstring>
#include <cctype>

// POSIX includes
#include <dirent.h>

#include "config.h"
#include "manager_impl.hh"
#include "error_impl.hh"
#include "config_impl.hh"
#include "error_messages.hh"
#include "string_list_impl.hh"
#include "getdata.hh"

#include "preload.h"
#define LT_NON_POSIX_NAMESPACE 1
#ifdef USE_LTDL
//
#include <ltdl.h>
#endif

static void free_lt_handle(PspellManagerLtHandle h) 
{
#ifdef USE_LTDL
  int s;
  s = lt_dlclose((lt_dlhandle)h);
  assert (s == 0);
  s = lt_dlexit();
  assert (s == 0);
#endif
}

PspellCanHaveError * new_pspell_manager_class(PspellConfig * config)
{

  PspellString name = config->retrieve("module");
  unsigned int i; 
  for (i = 0; i != pspell_manager_funs_size; ++i) {
    if (strcmp(name.c_str(), pspell_manager_funs[i].name) == 0) {
      return (*pspell_manager_funs[i].fun)(config, 0);
    }
  }
  
#ifdef USE_LTDL
  int s = lt_dlinit();
  assert(s == 0);
  PspellString libname;
  libname  = LIBDIR "/libpspell_";
  libname += name;
  libname += ".la";
  lt_dlhandle h = lt_dlopen (libname.c_str());
  if (h == 0)
    return (new PspellCanHaveErrorImpl())
      ->set_error(cant_load_module, name.c_str());
  lt_ptr_t fun = lt_dlsym (h, "new_pspell_manager_class");
  assert (fun != 0);
  PspellCanHaveError * m = (*(NewPspellManagerClass)(fun))(config, h);
  assert (m != 0);
  if (m->error_number() != 0)
    free_lt_handle(h);
  return m;
#else
  return (new PspellCanHaveErrorImpl())
    ->set_error(cant_load_module, name.c_str());
#endif
}

int better_match(const char * cur, const char * req, const char * best)
{
  if (strcmp(cur,req) == 0)
    if (strcmp(req,best) == 0)
      return -1;
    else
      return 1;
  else
    if (strcmp(req,best) == 0)
      return 0;
    else
      if (cur[0] == '\0' && req[0] != '\0' && best[0] != '\0')
	return 1;
      else if (cur[0] != '\0' && req[0] != '\0' && best[0] == '\0')
	return 0;
      else
	return -1;
}

PspellCanHaveError * find_word_list(PspellConfig * c) 
{
  PspellConfig * config = c->clone();
  
  ////////////////////////////////////////////////////////////////////
  //
  // Look for a dictionary with a matching language
  // and then the best match based on 
  // 1) if the "spelling" matches
  // then 
  // 2) if the "jargen" matches
  // then if they both matched it will look for the word list based on
  // the module search order
  //

  PspellStringListImpl search_path;
  config->retrieve_list("word-list-path",     search_path);

  PspellStringListImpl module_search_order;
  config->retrieve_list("module-search-order", module_search_order);

  PspellString         best_file;
  struct Match {
    PspellString spelling;
    PspellString jargon;
    PspellString module;
  };
  Match best;

  PspellString requested_spelling = config->retrieve("spelling");
  
  PspellString requested_jargon   = config->retrieve("jargon");

  const char * p;
  PspellString prefix;
  for (p = config->retrieve("language-tag"); *p; ++p)
    prefix += tolower(*p);
  // ignore encoding code
  if ((p = strchr(prefix.c_str(), '.')) != 0) {
    *((char *)p) = '\0';
    config->replace("language-tag", prefix.c_str());
    prefix.clear();
    prefix = config->retrieve("language-tag");

   }
  if ((p = strchr(prefix.c_str(), '_')) != 0) {
    if (requested_spelling.empty()) {
      const char * temp_str = config->retrieve("pspell-data-dir");
      if (temp_str == 0)
	temp_str = config->retrieve("data-dir");
      assert(temp_str != 0);
      PspellString f = temp_str;
      f += "/region-to-spelling.map";
      STD ifstream in0(f.c_str());
      if (!in0) {
	delete config;
	return (new PspellCanHaveErrorImpl())
	  ->set_error(cant_read_file, f.c_str());
      }
      PspellGetLineFromStream in(&in0);
      PspellString key,data;
      while (true) {
	if (!getdata_pair(in,key,data)) break;
	if (strcmp(key.c_str(), prefix.c_str()) == 0) {
	  config->replace("spelling", data.c_str());
	  requested_spelling = config->retrieve("spelling");
	}
      }
    }
    *((char *)p) = '\0';
    config->replace("language-tag", prefix.c_str());
    prefix.clear();
    prefix = config->retrieve("language-tag");
  }
  prefix += '-';

  PspellStringListEmulation els = search_path.elements_obj();
  const char * dir_name;

  PspellString temp;
  
  while ( (dir_name = els.next()) != 0 ) {
    DIR * d = opendir(dir_name);
    if (d == 0) continue;

    struct dirent * entry;
    while ( (entry = readdir(d)) != 0) {
      
      // check if it starts with prefix
      if (strncmp(entry->d_name, prefix.c_str(), prefix.size()) != 0)
	continue;
      // check if it ends in ".pwli"
      p = strchr(entry->d_name, '.');
      if (p == 0 || strcmp(p + 1, "pwli") != 0)
	continue;

      //
      // we got a match
      // so get the rest of the nessary info from the file name
      //
      Match cur;
      p = entry->d_name + prefix.size();
      const char * e = strpbrk(p, "-.");
      if (*e == '-') {
	cur.spelling.assign(p, e-p);
	p = e + 1;
	e = strpbrk(p, "-.");
      }
      if (*e == '-') {
	cur.jargon.assign(p, e-p);
	p = e + 1;
	e = strpbrk(p, "-.");
      }
      assert(*e == '.');
      cur.module.assign(p, e-p);

      bool module_better = true;
      PspellStringListEmulation es = module_search_order.elements_obj();
      const char * m;
      while ( (m = es.next()) != 0) {
	if (strcmp(m,cur.module.c_str()) == 0)  {
	  break;
	} else if (strcmp(m, best.module.c_str()) == 0) {
	  module_better = false;
	}
      }

      if (m == 0)
	continue; // requested module not available

      //
      // check to see if we got a better match than the current
      // best_match if any
      //
      int best_match = -1;
      if (best_file.empty())      	
	best_match = true;
      
      if (best_match == -1)
	best_match = better_match(cur.spelling.c_str(), 
				  requested_spelling.c_str(), 
				  best.spelling.c_str());
      if (best_match == -1)
	best_match = better_match(cur.jargon.c_str(), 
				  requested_jargon.c_str(), 
				  best.jargon.c_str());
      
      if (best_match == -1)
	if (module_better)
	  best_match = true;
	else
	  best_match = false;
      
      //
      //
      //
      if (best_match) {
	best_file  = dir_name;
	best_file += '/';
	best_file +=entry->d_name;
	best      = cur;
      }
    }
    closedir(d);
  }
  //
  // set config to best match
  //
  if (best_file.size() != 0) {
    STD ifstream F(best_file.c_str());
    if (!F)
      return (new PspellCanHaveErrorImpl())
	->set_error(cant_read_file, best_file.c_str());
    PspellGetLineFromStream GL(&F);
    PspellString main_wl,flags;
    getdata_pair(GL, main_wl, flags);
    if (!F) 
      return (new PspellCanHaveErrorImpl())
	->set_error(bad_file_format,  best_file.c_str(), "");
    config->replace("master", main_wl.c_str());
    config->replace("master-flags", flags.c_str());
    config->replace("module", best.module.c_str());
    config->replace("spelling", best.spelling.c_str());
    config->replace("jargon", best.jargon.c_str());
  } else {
    PspellString temp = 
      "I'm sorry I can't find any suitable word lists for "
      "the language-tag \"";
    temp += config->retrieve("language-tag");
    temp += "\".";
    delete config;
    return (new PspellCanHaveErrorImpl())
      ->set_error(PERROR_NO_WORDLIST_FOR_LANG, temp.c_str());
  }
  
  return config;
}

PspellCanHaveError * new_pspell_manager(PspellConfig * c) 
{
  PspellCanHaveError * possible_err = find_word_list(c);
  if (possible_err->error_number() != 0)
    return possible_err;
  PspellConfig * config = (PspellConfig *)(possible_err);
  possible_err = new_pspell_manager_class(config);
  delete config;
  return possible_err;
}

void delete_pspell_manager(PspellManager * m) 
{
  PspellManagerLtHandle h = ((PspellManagerImplBase *)(m))->lt_handle();
  delete m;
  if (h != 0) free_lt_handle(h);
}




