
#include <iostream>
#include <cstring>
#include <pspell/pspell.h>

static void print_word_list(PspellManager * manager, 
			    const PspellWordList *wl) 
{
  if (wl == 0) {
    cout << "Error: " << manager->error_message() << endl;
  } else {
    PspellStringEmulation * els = wl->elements();
    const char * word;
    while ( (word = els->next()) != 0) {
      cout << word << endl;
    }
  }
}

static void check_for_error(PspellManager * manager)
{
  if (manager->error_number() != 0)
    cout << "Error: " << manager->error_message() << endl;
}

int main(int argc, const char *argv[]) {

  if (argc < 2) {
    cout << "Usage: " << argv[0] 
	 << " <language> [<spelling>|- [<jargon>|- [<encoding>]]]\n";
    return 1;
  }

  PspellConfig * config = new_pspell_config();

  config->replace("language-tag", argv[1]);

  if (argc >= 3 && argv[2][0] != '-')
    config->replace("spelling", argv[2]);

  if (argc >= 4 && argv[3][0] != '-')
    config->replace("jargon", argv[3]);

  if (argc >= 5 && argv[4][0] != '-')
    config->replace("encoding", argv[4]);
  
  PspellCanHaveError * ret = new_pspell_manager(config);

  delete_pspell_config(config);

  if (ret->error_number() != 0) {
    cout << ret->error_message() << endl;
    delete_pspell_can_have_error(ret);
    return 1;
  }
  PspellManager * manager = (PspellManager *) ret;

  cout << "Using: ";
  cout << manager->config().retrieve("language-tag");
  cout << "-";
  cout << manager->config().retrieve("spelling");
  cout << "-";
  cout << manager->config().retrieve("jargon");
  cout << "-";
  cout << manager->config().retrieve("module");
  cout << "\n\n";
   
  cout << "Type \"h\" for help.\n" << endl;
  
  int have;
  char word[81];
  while (cin.getline(word, 80)) {

    // remove trailing spaces
    char * word_end = strchr(word, '\0') - 1;
    while (word_end != word && *word_end == ' ') 
      --word_end;
    ++word_end;
    *word_end = '\0';

    cout << "\n";
    switch (word[0]) {
    case '\0':
      break;
    case 'h':
      cout << 
	"Usage: \n"
	"  h(elp)      help\n"
	"  c <word>    check if a word is the correct spelling\n"
	"  s <word>    print out a list of suggestions for a word\n"
	"  a <word>    add a word to the personal word list\n"
	"  i <word>    ignore a word for the rest of the session\n"
	"  p           dumps the personal word list\n"
	"  P           dumps the session word list\n"
	"  m           dumps the master  word list\n"
	"  S           saves all word lists\n"
	"  C           clear the cuurent sesstion word list\n"
	"  x           quite\n";
      break;
    case 'p':
      print_word_list(manager, manager->personal_word_list());
      break;
    case 'P':
      print_word_list(manager, manager->session_word_list());
      break;
    case 'm':
      print_word_list(manager, manager->master_word_list());
      break;
    case 'S':
      manager->save_all_word_lists();
      check_for_error(manager);
      break;
    case 'C': 
      manager->clear_session();
      check_for_error(manager);
      break;
    case 'x':
      goto END;
    case 'c':
      if (strlen(word) < 3) {
	cout << "Usage: " << word[0] << "  <word>\n";
      } else {
	have = manager->check(word + 2);
	if (have == 1) 
	  cout << "correct" << endl;
	else if (have == 0)
	  cout << "incorrect" << endl;
	else
	  cout << "Error: " << manager->error_message() << endl;
      }
      break;
    case 's':
      if (strlen(word) < 3) 
	cout << "Usage: " << word[0] << "  <word>\n";
      else 
	print_word_list(manager, manager->suggest(word + 2));
      break;
    case 'a':
      if (strlen(word) < 3) {
	cout << "Usage: " << word[0] << "  <word>\n";
      } else {
	manager->add_to_personal(word + 2);
	check_for_error(manager);
      }
      break;
    case 'i':
      if (strlen(word) < 3) {
	cout << "Usage: " << word[0] << "  <word>\n";
      } else {
	manager->add_to_session(word + 2);
	check_for_error(manager);
      }
      break;
    default:
      cout << "Unknown Command: " << word << endl;
    }
    cout << "\n";
  }
 END:
  delete_pspell_manager(manager);
  return 0;
}
