#include "sockets.h"

sockets::sockets() {
}

/* Generic function to make a connection to a given server/port. Service is
   the port name/number, type is either SOCK_STREAM or SOCK_DGRAM, and
   netaddress is the host name to connect to. Returns 1 if successful and 0
   if anything went wrong. */
int
sockets::makeConnection(char *netaddress, char *service, int type) {
  /* First convert service from a string, to a number... */
  int port = -1;
  struct in_addr *addr;
  int connected;
  struct sockaddr_in address;

  if (type == SOCK_STREAM) 
    port = aToPort(service, "tcp");
  if (type == SOCK_DGRAM)
    port = aToPort(service, "udp");
  if (port == -1) {
    ptrDisplay->showError("Invalid socket type");
    return(0);
  }
  
  addr = aToAddr(netaddress);
  if (addr == NULL) {
    ptrDisplay->showError("Invalid network address");
    return(0);
  }
  
  memset((char *) &address, 0, sizeof(address));
  address.sin_family = PF_INET;
  address.sin_port = (port);
  address.sin_addr.s_addr = addr->s_addr;
  
  sockfd = socket(PF_INET, type, 0);
  
  ptrDisplay->showText("Connecting to ");
  ptrDisplay->showText(inet_ntoa(*addr));
  ptrDisplay->showText(" on port ");
  ptrDisplay->showText(htons(port));
  ptrDisplay->showText("...\n");
  
  /* Just connect if TCP */
  if (type == SOCK_STREAM) {
    connected = connect(sockfd, (struct sockaddr *) &address, sizeof(address));
    if (connected < 0) {
      //perror("Connect");
      return 0;
    }
    return(1);
  }

  /* Bind to address if UDP */
  if (type == SOCK_DGRAM) {
    if (bind(sockfd, (struct sockaddr *) &address, sizeof(address)) < 0) {
      //perror("bind");
      return(0);
    }
    return(1);
  }
  
  return 0;
}


/* Take a service name, and a service type, and return a port number. If the
   service name is not found, try it as a decimal number. The number
   returned is byte ordered for the network. */
int
sockets::aToPort(char *service, char *proto) {
  int port;
  long int lport;
  struct servent *serv;
  char *errpos;
  
  /* First try to read it from /etc/services */
  serv = getservbyname(service, proto);
  if (serv != NULL)
    port = serv->s_port;
  else { /* Not in services, maybe a number? */
    lport = strtol(service, &errpos, 0);
    if ((*errpos != '\0') || (lport < 1) || (lport > 65535))
      return -1; /* Invalid port address */
    port = htons(lport);
  }
  return port;
}


/* Read from a socket until a linefeed character is received. It fills the
   buffer "str" up to the maximum size "count". Note that if a single line
   exceeds the length of count, the extra data will be read and discarded! */
int
sockets::sockGets(char *buffer, size_t count) {
  
  // Exit if no data is available within TIMEOUT seconds
  if (!sockPoll(TIMEOUT)) {
    //cerr << "NO_DATA";
    return(0);
  }
  
  ssize_t bytesRead;
  unsigned int totalCount = 0;
  char lastRead = 0;
  
  //cerr << "(S)";
  while (lastRead != 10) {
    bytesRead = read(sockfd, &lastRead, 1);

    // Error on read(2)
    if (bytesRead == -1) {
      //cerr << "sockgets(): " << strerror(errno);
      return(-1);
    }
    
    //cerr << "-";
    // If EOF and first try
    if ((!lastRead) && (totalCount == 0)) {
      usleep(1000); // Safenet from "runaway" bug
      return(-2);
    }
    
    //cerr << "[" << (int) lastRead << "]";
    // Break if EOF found
    if (!lastRead)
      break;
    
    // Something has been read, increase pointer
    totalCount++;
    
    // Add to buffer if no overflow and no CR/LF
    if ((totalCount < count) && (lastRead != 10) && (lastRead != 13))
      *buffer++ = lastRead;
  }
  //cerr << "(E)\n";
  
  if (count > 0)
    *buffer = '\0';
  return(totalCount);
}


/* Poll to see if there is anything to read on the socket. If
   no data is available return 0, otherwise return 1. */
int
sockets::sockPoll(int wait) {
  fd_set r;
  int rt;
  struct timeval tv;
  
  while (true) {
    FD_ZERO(&r);
    FD_SET(sockfd, &r);
    
    // Check for data, timeout in TIMEOUT secs if no data is available
    // Minimal delay is 0ms
    tv.tv_sec = wait;
    tv.tv_usec = 0;
    rt = select(sockfd+1, &r, NULL, NULL, &tv);
    
    //cerr << "A";
    // ROCK IT... we have something
    if (rt > 0)
      return(1);
    
    //cerr << "B";    
    // No FDs ready, or timeout
    if (rt == 0)
      return(0);
    
    //cerr << "C";
    // Something bad happened, should do more
    if (rt == -1) {
      //cerr << "sockPoll(): " << strerror(errno);
      return(-1);
    }
  }
}


/* Write a character string out to a socket. It will return -1 if the
   connection is closed while it is trying to write. */
int
sockets::sockPuts(const char *str) {
  //cerr << "{" << str << "}";
  return sockWrite(str, strlen(str));
}


/* This is just like the write(2) system call, except that it will
   make sure that all data is transmitted. */
int
sockets::sockWrite(const char* vptr, size_t n) {
  size_t nleft;
  ssize_t nwritten;
  const char* ptr;
  
  ptr = vptr;
  nleft = n;
  while (nleft > 0) {
    if ((nwritten = write(sockfd, ptr, nleft)) <= 0)
      return(nwritten);
    
    nleft -= nwritten;
    ptr += nwritten;
  }
  
  return(n);
}


/* Empty socket by reading until nothing is left to be fetched. This is a really
   bad solution but until I know how this can be done more efficiently this is
   the way to go. */
void
sockets::readToEOF() {
  char buf[2] = "";  // char + terminating '\0'
  
  // First poll with timeout = 0, shorter delay if data is N/A forever
  while (sockPoll(0)) {
    if (sockGets(buf, 1) > 0)
      continue;
    else
      return;
  }
}


/* Converts ascii text to in_addr struct.  NULL is returned if the address
   can not be found. */
struct in_addr *aToAddr(char *address) {
  struct hostent *host;
  static struct in_addr saddr;

  /* First try it as aaa.bbb.ccc.ddd. */    /* LINUX */
  /*
  inet_aton(address, &saddr);
  if (saddr.s_addr != 0) {
    return &saddr;
  }
  */
  
  /* First try it as aaa.bbb.ccc.ddd. */    /* SOLARIS (obsoleted on LINUX) */
  saddr.s_addr = inet_addr(address);
  if (saddr.s_addr != INADDR_NONE) {
    return &saddr;
  }

  host = gethostbyname(address);
  if (host != NULL) {
    return (struct in_addr *) *host->h_addr_list;
  }
  return NULL;
}

sockets::~sockets() {
  close(sockfd);
}
