/*
** pork_chat.c
** Copyright (C) 2003 Ryan McCabe <ryan@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
*/

#include <config.h>

#include <unistd.h>
#include <stdlib.h>
#include <ncurses.h>
#include <stdio.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>

#include <pork.h>
#include <pork_util.h>
#include <pork_list.h>
#include <pork_buddy.h>
#include <pork_events.h>
#include <pork_acct.h>
#include <pork_set.h>
#include <pork_proto.h>
#include <pork_format.h>
#include <pork_screen.h>
#include <pork_screen_cmd.h>
#include <pork_imwindow.h>
#include <pork_color.h>
#include <pork_chat.h>

static void chat_destroy_user_list_cb(void *param, void *data) {
	struct pork_acct *acct = (struct pork_acct *) param;
	struct chat_user *chat_user = (struct chat_user *) data;

	if (acct->proto->chat_user_free != NULL)
		acct->proto->chat_user_free(acct, chat_user);

	free(chat_user->nname);
	free(chat_user->name);
	free(chat_user);
}

struct chatroom *chat_new(	struct pork_acct *acct,
							char *chat_title,
							char *chat_title_full,
							struct imwindow *win)
{
	struct chatroom *chat;

	chat = xcalloc(1, sizeof(*chat));
	chat->title = xstrdup(chat_title);
	chat->title_quoted = acct->proto->filter_text(chat_title);
	chat->title_full = xstrdup(chat_title_full);
	chat->title_full_quoted = acct->proto->filter_text(chat_title_full);
	chat->win = win;
	win->data = chat;

	acct->chat_list = dlist_add_head(acct->chat_list, chat);

	return (chat);
}

int chat_send_msg(	struct pork_acct *acct,
					struct chatroom *chat,
					char *msg)
{
	if (acct->proto->chat_send == NULL || msg == NULL)
		return (-1);

	if (!event_generate(acct->events, EVENT_SEND_CHAT_MSG,
		chat->title, msg, acct->refnum))
	{
		if (acct->proto->chat_send(acct, chat, msg) == -1)
			return (-1);

		imwindow_send_msg(chat->win, acct, chat->title_quoted,
			msg, OPT_FORMAT_CHAT_SEND);
	}

	return (0);
}

int chat_recv_msg(	struct pork_acct *acct,
					struct chatroom *chat,
					char *user,
					char *msg)
{
	if (!event_generate(acct->events, EVENT_RECV_CHAT_MSG,
		chat->title, user, msg, acct->refnum))
	{
		if (!chat_user_is_ignored(acct, chat, user))
			imwindow_recv_msg(chat->win, acct, user, msg, OPT_FORMAT_CHAT_RECV);
	}

	return (0);
}

int chat_recv_notice(	struct pork_acct *acct,
						struct chatroom *chat,
						char *user,
						char *msg)
{
	if (!event_generate(acct->events, EVENT_RECV_CHAT_NOTICE,
		chat->title, user, msg, acct->refnum))
	{
		if (!chat_user_is_ignored(acct, chat, user)) {
			imwindow_recv_msg(chat->win, acct, user,
				msg, OPT_FORMAT_CHAT_RECV_NOTICE);
		}
	}

	return (0);
}

int chat_send_notice(struct pork_acct *acct, struct chatroom *chat, char *msg) {
	if (acct->proto->chat_send_notice == NULL)
		return (-1);

	if (!event_generate(acct->events, EVENT_RECV_CHAT_NOTICE,
		chat->title, msg, acct->refnum))
	{
		if (acct->proto->chat_send_notice(acct, chat, msg) == -1)
			return (-1);

		imwindow_send_msg(chat->win, acct, chat->title,
			msg, OPT_FORMAT_CHAT_SEND_NOTICE);
	}

	return (0);
}

int chat_ignore(struct pork_acct *acct, char *chat_name, char *user) {
	struct chat_user *chat_user;
	struct chatroom *chat;
	char buf[512];

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	chat_user = chat_find_user(acct, chat, user);
	if (chat_user == NULL) {
		screen_err_msg("%s is not a member of %s", user, chat->title_quoted);
		return (-1);
	}

	chat_user->ignore = 1;

	if (acct->proto->chat_ignore != NULL) {
		if (acct->proto->chat_ignore(acct, chat, user) == -1)
			return (-1);
	}

	fill_format_str(OPT_FORMAT_CHAT_IGNORE, buf, sizeof(buf),
		acct->username, chat->title_quoted, user);
	screen_win_msg(cur_window(), 1, 1, 1, "%s", buf);

	return (0);
}

int chat_unignore(struct pork_acct *acct, char *chat_name, char *user) {
	struct chat_user *chat_user;
	struct chatroom *chat;
	char buf[512];

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	chat_user = chat_find_user(acct, chat, user);
	if (chat_user == NULL) {
		screen_err_msg("%s is not a member of %s", user, chat->title_quoted);
		return (-1);
	}

	chat_user->ignore = 0;

	if (acct->proto->chat_unignore != NULL) {
		if (acct->proto->chat_unignore(acct, chat, user) == -1)
			return (-1);
	}

	fill_format_str(OPT_FORMAT_CHAT_UNIGNORE, buf, sizeof(buf),
		acct->username, chat->title_quoted, user);
	screen_win_msg(cur_window(), 1, 1, 1, "%s", buf);

	return (0);
}

int chat_user_is_ignored(	struct pork_acct *acct,
							struct chatroom *chat,
							char *user)
{
	struct chat_user *chat_user;

	chat_user = chat_find_user(acct, chat, user);
	if (chat_user == NULL)
		return (0);

	return (chat_user->ignore);
}

int chat_users(struct pork_acct *acct, char *chat_name) {
	struct chatroom *chat;

	if (acct->proto->chat_users == NULL)
		return (-1);

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	return (acct->proto->chat_users(acct, chat));
}

int chat_who(struct pork_acct *acct, char *chat_name) {
	struct chatroom *chat;

	if (acct->proto->chat_who == NULL)
		return (-1);

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	return (acct->proto->chat_who(acct, chat));
}

int chat_join(struct pork_acct *acct, char *args) {
	struct imwindow *imwindow = NULL;
	char buf[512];
	int ret = 0;

	if (acct->proto->chat_join == NULL || acct->proto->chat_name == NULL)
		return (-1);

	if (args == NULL) {
		imwindow = cur_window();

		if (imwindow->type == TYPE_CHAT)
			args = imwindow->target;
		else
			return (-1);
	}

	if (acct->proto->chat_name(args, buf, sizeof(buf)) == -1) {
		screen_err_msg("Invalid chat name: %s", args);
		return (-1);
	}

	imwindow = imwindow_find_chat_target(acct, buf);
	if (imwindow == NULL) {
		imwindow = screen_new_chat_window(acct, buf);
		if (imwindow == NULL) {
			screen_err_msg("Unable to create a new window for %s", buf);
			return (-1);
		}
	}

	if (imwindow->data == NULL) {
		if (!event_generate(acct->events, EVENT_SEND_CHAT_JOIN, buf,
			acct->refnum))
		{
			ret = acct->proto->chat_join(acct, buf);
		}
	}

	screen_goto_window(imwindow->refnum);
	return (ret);
}

int chat_leave(struct pork_acct *acct, char *chat_name, int close_window) {
	struct chatroom *chat;
	struct imwindow *win;

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	if (event_generate(acct->events, EVENT_SEND_CHAT_LEAVE,
			chat->title, acct->refnum))
	{
		return (0);
	}

	if (acct->proto->chat_leave != NULL) {
		if (acct->proto->chat_leave(acct, chat) == -1) {
			screen_err_msg("Error leaving chat room %s for %s",
				chat->title_quoted, acct->username);
			return (-1);
		}
	}

	win = chat->win;
	win->data = NULL;
	chat->win = NULL;

	if (close_window)
		screen_close_window(win);

	return (chat_free(acct, chat, 0));
}

int chat_leave_all(struct pork_acct *acct) {
	dlist_t *cur;

	cur = acct->chat_list;
	while (cur != NULL) {
		struct chatroom *chat = cur->data;
		dlist_t *next = cur->next;

		chat_leave(acct, chat->title, 0);
		cur = next;
	}

	return (0);
}

int chat_user_kicked(	struct pork_acct *acct,
						struct chatroom *chat,
						char *kicked,
						char *kicker,
						char *reason)
{
	char buf[512];
	int silent = 0;

	if (event_generate(acct->events, EVENT_RECV_CHAT_KICK,
			chat->title, kicked, kicker, reason, acct->refnum))
	{
		silent = 1;
	}

	chat_user_left(acct, chat, kicked, 1);

	if (silent == 1)
		return (0);

	reason = acct->proto->filter_text(reason);

	fill_format_str(OPT_FORMAT_CHAT_KICK, buf, sizeof(buf),
		kicker, chat->title_quoted, kicked, reason);

	free(reason);
	screen_win_msg(chat->win, 1, 1, 1, "%s", buf);
	return (0);
}

int chat_forced_leave(struct pork_acct *acct, char *chat_name, char *person, char *reason) {
	struct chatroom *chat;

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	chat_user_kicked(acct, chat, acct->username, person, reason);
	return (chat_free(acct, chat, 1));
}

int chat_invite(struct pork_acct *acct, char *chat_name, char *user, char *msg)
{
	struct chatroom *chat;

	if (acct->proto->chat_invite == NULL)
		return (-1);

	chat = chat_find(acct, chat_name);
	if (chat == NULL)
		return (-1);

	if (!event_generate(acct->events, EVENT_SEND_CHAT_INVITE, chat->title,
		user, msg, acct->refnum))
	{
		char buf[512];

		if (acct->proto->chat_invite(acct, chat, user, msg) == -1)
			return (-1);

		msg = acct->proto->filter_text(msg);

		fill_format_str(OPT_FORMAT_CHAT_INVITE, buf, sizeof(buf),
			acct->username, chat->title_quoted, user, msg);

		free(msg);
		screen_win_msg(cur_window(), 1, 1, 1, "%s", buf);
	}

	return (0);
}

struct chatroom *chat_find(struct pork_acct *acct, char *chat_name) {
	if (acct->proto->chat_find == NULL)
		return (NULL);

	return (acct->proto->chat_find(acct, chat_name));
}

void chat_list(struct pork_acct *acct) {
	dlist_t *cur;

	cur = acct->chat_list;
	if (cur == NULL) {
		screen_output("%s is not a member of any chat rooms", acct->username);
		return;
	}

	screen_output("%s is joined to the following chat rooms", acct->username);
	do {
		struct chatroom *chat = cur->data;

		screen_output("  %s in window refnum %u",
			chat->title_quoted, chat->win->refnum);

		cur = cur->next;
	} while (cur != NULL);
}

int chat_free(struct pork_acct *acct, struct chatroom *chat, int silent) {
	dlist_t *cur;

	cur = dlist_find(acct->chat_list, chat, NULL);
	if (cur == NULL)
		return (-1);

	acct->chat_list = dlist_remove(acct->chat_list, cur);

	if (chat->win != NULL) {
		if (!silent) {
			char buf[512];

			fill_format_str(OPT_FORMAT_CHAT_INFO, buf, sizeof(buf),
				acct->username, opt_get_str(OPT_TEXT_CHAT_LEAVE),
				chat->title_quoted);

			screen_win_msg(chat->win, 1, 1, 1, "%s", buf);
		}

		chat->win->data = NULL;
		chat->win = NULL;
	}

	if (acct->proto->chat_free != NULL)
		acct->proto->chat_free(acct, chat->data);

	dlist_destroy(chat->user_list, acct, chat_destroy_user_list_cb);

	free(chat->title);
	free(chat->title_quoted);
	free(chat->title_full);
	free(chat->title_full_quoted);
	free(chat->topic);
	free(chat->mode);
	free(chat);

	return (0);
}

int chat_got_invite(struct pork_acct *acct,
					char *chat_name,
					char *user,
					char *message)
{
	if (!event_generate(acct->events, EVENT_RECV_CHAT_INVITE,
			chat_name, user, message, acct->refnum))
	{
		char buf[512];

		chat_name = acct->proto->filter_text(chat_name);

		if (message == NULL)
			message = "";

		message = acct->proto->filter_text(message);

		fill_format_str(OPT_FORMAT_CHAT_INVITE, buf, sizeof(buf),
			user, chat_name, acct->username, message);

		free(message);
		free(chat_name);
		screen_win_msg(cur_window(), 1, 1, 1, "%s", buf);
	}

	return (0);
}

int chat_created(struct pork_acct *acct, struct chatroom *chat) {
	char buf[512];

	fill_format_str(OPT_FORMAT_CHAT_CREATE, buf, sizeof(buf),
		acct->username, chat->title_quoted,
		acct->username, chat->title_full_quoted);

	screen_win_msg(chat->win, 1, 1, 1, "%s", buf);
	return (0);
}

struct chat_user *chat_user_joined(	struct pork_acct *acct,
									struct chatroom *chat,
									char *user,
									int silent)
{
	char buf[512];
	struct chat_user *chat_user;

	if (chat_find_user(acct, chat, user) != NULL)
		return (NULL);

	chat->num_users++;

	acct->proto->normalize(buf, user, sizeof(buf));

	chat_user = xcalloc(1, sizeof(*chat_user));
	chat_user->name = xstrdup(user);
	chat_user->nname = xstrdup(buf);

	chat->user_list = dlist_add_head(chat->user_list, chat_user);

	if (event_generate(acct->events, EVENT_RECV_CHAT_JOIN,
		chat->title, user, acct->refnum))
	{
		return (chat_user);
	}

	if (!silent) {
		fill_format_str(OPT_FORMAT_CHAT_INFO, buf, sizeof(buf),
			user, opt_get_str(OPT_TEXT_CHAT_JOIN), chat->title_quoted);

		screen_win_msg(chat->win, 1, 1, 1, "%s", buf);
	}

	return (chat_user);
}

static dlist_t *chat_find_user_node(struct pork_acct *acct,
									struct chatroom *chat,
									char *user)
{
	dlist_t *cur;

	cur = chat->user_list;
	while (cur != NULL) {
		struct chat_user *chat_user = cur->data;

		if (!acct->proto->user_compare(user, chat_user->nname))
			break;

		cur = cur->next;
	}

	return (cur);
}

inline struct chat_user *chat_find_user(struct pork_acct *acct,
										struct chatroom *chat,
										char *user)
{
	dlist_t *cur = chat_find_user_node(acct, chat, user);

	if (cur == NULL)
		return (NULL);

	return (cur->data);
}

int chat_user_left(	struct pork_acct *acct,
					struct chatroom *chat,
					char *user,
					int silent)
{
	dlist_t *node;
	struct chat_user *chat_user;

	node = chat_find_user_node(acct, chat, user);
	if (node != NULL) {
		chat->num_users--;

		chat_user = node->data;
		chat->user_list = dlist_remove(chat->user_list, node);
		chat_destroy_user_list_cb(acct, chat_user);
	}

	if (event_generate(acct->events, EVENT_RECV_CHAT_LEAVE,
			chat->title, user, acct->refnum))
	{
		return (1);
	}

	if (!silent) {
		char buf[512];

		fill_format_str(OPT_FORMAT_CHAT_INFO, buf, sizeof(buf),
			user, opt_get_str(OPT_TEXT_CHAT_LEAVE), chat->title_quoted);

		screen_win_msg(chat->win, 1, 1, 1, "%s", buf);
	}

	return (0);
}

int chat_set_topic(struct pork_acct *acct, struct chatroom *chat, char *topic) {
	int ret = 0;

	if (acct->proto->chat_set_topic == NULL)
		return (-1);

	if (!event_generate(acct->events, EVENT_SEND_CHAT_TOPIC,
		chat->title, topic, acct->refnum))
	{
		ret = acct->proto->chat_set_topic(acct, chat, topic);
	}

	return (ret);
}

int chat_got_topic(	struct pork_acct *acct,
					struct chatroom *chat,
					char *set_by,
					char *topic)
{
	if (!event_generate(acct->events, EVENT_RECV_CHAT_TOPIC,
		chat->title, set_by, topic, acct->refnum))
	{
		char buf[512];

		topic = acct->proto->filter_text(topic);

		fill_format_str(OPT_FORMAT_CHAT_TOPIC, buf, sizeof(buf),
			set_by, chat->title_quoted, topic, acct);

		free(topic);
		screen_win_msg(cur_window(), 1, 1, 1, "%s", buf);
	}

	return (0);
}

int chat_kick(	struct pork_acct *acct,
				struct chatroom *chat,
				char *user,
				char *reason)
{
	if (acct->proto->chat_kick == NULL)
		return (-1);

	if (event_generate(acct->events, EVENT_RECV_CHAT_KICK,
			chat->title, user, reason, acct->refnum))
	{
		return (0);
	}

	return (acct->proto->chat_kick(acct, chat, user, reason));
}
