#include <config.h>

#include <unistd.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>

#include <pork.h>
#include <pork_util.h>
#include <pork_list.h>
#include <pork_misc.h>
#include <pork_queue.h>
#include <pork_color.h>
#include <pork_inet.h>
#include <pork_acct.h>
#include <pork_proto.h>
#include <pork_imwindow.h>
#include <pork_screen.h>
#include <pork_screen_cmd.h>
#include <pork_chat.h>
#include <pork_msg.h>
#include <pork_irc.h>

struct irc_input *irc_tokenize(char *buf) {
	struct irc_input *in = xcalloc(1, sizeof(*in));
	size_t i = 0;
	size_t len = array_elem(in->tokens);
	char *p;
	int cmd_offset = 0;
	int numeric;

	in->orig = xstrdup(buf);

	if (*buf == ':') {
		cmd_offset = 1;
		buf++;
	}

	p = strchr(buf, ':');
	if (p != NULL) {
		*p++ = '\0';
		in->args = p;
	}

	while (i < len && (p = strsep(&buf, " ")) != NULL && *p != '\0')
		in->tokens[i++] = p;

	if (i < 2) {
		free(in);
		return (NULL);
	}

	in->cmd = in->tokens[cmd_offset];
	in->num_tokens = i;

	numeric = strtol(in->cmd, &p, 10);
	if (*p == '\0')
		in->numeric = numeric;
	else
		in->numeric = -1;

	return (in);
}

static int irc_callback_compare(void *l, void *r) {
	char *str = l;
	struct callback_handler *cb = r;

	return (strcasecmp(str, cb->str));
}

static void irc_callback_cleanup(void *p __notused, void *data) {
	struct callback_handler *cb = data;

	free(cb->str);
	free(cb);
}

int irc_callback_init(irc_session_t *session) {
	int ret;

	ret = hash_init(&session->callbacks, 5,
			irc_callback_compare, irc_callback_cleanup);

	return (ret);
}

int irc_callback_clear(irc_session_t *session) {
	hash_destroy(&session->callbacks);
	return (0);
}

int irc_callback_add(	irc_session_t *session,
						char *cmd,
						int (*handler)(struct pork_acct *, struct irc_input *))
{
	struct callback_handler *cb;

	cb = xcalloc(1, sizeof(*cb));
	cb->str = xstrdup(cmd);
	cb->handler = handler;

	u_int32_t hash = string_hash(cmd, session->callbacks.order);
	hash_add(&session->callbacks, cb, hash);
	return (0);
}

static int irc_handler_ctcp_version(struct pork_acct *acct,
									struct irc_input *in)
{
	char buf[256];

	snprintf(buf, sizeof(buf), "\x01VERSION %s %s - http://dev.ojnk.net\x01",
		PACKAGE_NAME, PACKAGE_VERSION);

	irc_send_notice(acct->data, in->tokens[0], buf);
	return (0);
}

static int irc_handler_ctcp_ping(	struct pork_acct *acct,
									struct irc_input *in)
{
	struct timeval tv;
	char buf[256];

	gettimeofday(&tv, NULL);
	snprintf(buf, sizeof(buf), "\x01PING %ld %ld\x01",
		tv.tv_sec, tv.tv_usec);

	irc_send_notice(acct->data, in->tokens[0], buf);
	return (0);
}

#if 0
static int irc_handler_ctcp_ping_reply(	struct pork_acct *acct,
										struct irc_input *in)
{
	struct timeval tv;
	char buf[256];
	int ping_sec;
	int ping_usec = 0;
	char *tok;
	char *p = in->args;

	gettimeofday(&tv, NULL);

	tok = strsep(&p, " .");
	if (tok == NULL) {
		/* XXX - invalid ping */
		return (-1);
	}

	if (str_to_int(tok, &ping_sec) != 0) {
		return (-1);
	} 

	if (p != NULL && *p != '\0') {
		if (str_to_int(p, &ping_usec) != 0) {
			return (-1);
		}
	}

	snprintf(buf, sizeof(buf), "\x01PING %f seconds\x01",
		(float) (tv.tv_sec - ping_sec) + ((tv.tv_usec - ping_usec) / 1000000.0));

	return (1);
}
#endif

static int irc_handler_print_arg(struct pork_acct *acct, struct irc_input *in) {
	char *str;

	if (in->args == NULL)
		return (-1);

	if (!strncasecmp(in->args, "End of /", 8))
		return (0);

	str = irc_text_filter(in->args);
	screen_output("%s", str);

	free(str);
	return (0);
}

int irc_callback_run(irc_session_t *session, struct irc_input *in) {
	dlist_t *node;
	u_int32_t hash = string_hash(in->cmd, session->callbacks.order);
	struct callback_handler *cb;

	node = hash_find(&session->callbacks, in->cmd, hash);
	if (node == NULL || node->data == NULL) {
		irc_handler_print_arg(session->data, in);
		return (0);
	}

	cb = node->data;

	return (cb->handler(session->data, in));
}

static ssize_t irc_read_data(int sock, char *buf, size_t len) {
	size_t i;

	if (buf == NULL)
		return (-1);

	for (i = 1 ; i < len ; i++) {
		char c;
		size_t ret;

		top:
			ret = read(sock, &c, 1);
			if (ret == 1) {
				if (c == '\r')
					goto top;

				if (c == '\n') {
					i--;
					break;
				}

				*buf++ = c;
			} else if (ret == 0) {
				if (i == 1)
					return (0);
				else
					break;
			} else {
				if (errno == EINTR)
					goto top;

				return (-1);
			}
	}

	*buf = '\0';
	return (i);
}

int irc_input_dispatch(irc_session_t *session) {
	char buf[8192];
	int ret;
	struct irc_input *in;
	struct pork_acct *acct = session->data;

	ret = irc_read_data(session->sock, buf, sizeof(buf));
	if (ret < 1)
		return (-1);

	in = irc_tokenize(buf);
	if (in == NULL)
		return (-1);

	if (!event_generate(acct->events, EVENT_RECV_RAW,
		in->cmd, in->orig, acct->refnum))
    {
		ret = irc_callback_run(session, in);
	}

	free(in->orig);
	free(in);
	return (ret);
}

static int irc_handler_001(struct pork_acct *acct, struct irc_input *in) {
	acct->connected = 1;

	if (in->num_tokens >= 3) {
		free(acct->username);
		acct->username = xstrdup(in->tokens[2]);
	}

	return (irc_handler_print_arg(acct, in));
}

static int irc_handler_print_num(struct pork_acct *acct, struct irc_input *in) {
	char *str;

	if (in->args == NULL || in->num_tokens < 4)
		return (-1);

	str = irc_text_filter(in->args);
	screen_output("There are %s %s", in->tokens[3], str);

	free(str);
	return (0);
}

static int irc_handler_print_tok(struct pork_acct *acct, struct irc_input *in) {
	u_int32_t i;
	char buf[2048];
	size_t len = sizeof(buf);
	u_int32_t off = 0;

	if (in->num_tokens < 4)
		return (-1);

	for (i = 3 ; i < in->num_tokens ; i++) {
		int ret;

		ret = snprintf(&buf[off], len, "%s ", in->tokens[i]);
		if (ret < 0 || (size_t) ret >= len)
			return (-1);
		len -= ret;
		off += ret;
	}

	if (off > 0) {
		char *str;

		buf[off - 1] = '\0';

		str = irc_text_filter(buf);
		screen_output("%s", str);
		free(str);
	}

	return (0);
}

static int irc_handler_352(struct pork_acct *acct, struct irc_input *in) {
	char *info;
	char buf[2048];

	if (in->num_tokens < 9 || in->args == NULL)
		return (-1);

	info = strchr(in->args, ' ');
	if (info != NULL)
		info++;
	else
		info = in->args;

	info = irc_text_filter(info);

	snprintf(buf, sizeof(buf), "%s\t%-9s %-3s %s@%s (%s)",
		in->tokens[3], in->tokens[7], in->tokens[8],
		in->tokens[4], in->tokens[5], info);
	
	free(info);
	screen_win_msg(cur_window(), 0, 0, 0, "%s", buf);
	return (0);
}

/* channel names */
static int irc_handler_353(struct pork_acct *acct, struct irc_input *in) {
	struct chatroom *chat;
	char *p = in->args;
	char *tok;
	char buf[2048];
	size_t offset = 0;
	size_t len = sizeof(buf);
	int ret;
	int add = 1;
	char *chat_name;

	if (in->num_tokens < 5 || in->args == NULL)
		return (-1);

	chat = chat_find(acct, in->tokens[4]);
	if (chat == NULL)
		add = 0;

	chat_name = irc_text_filter(in->tokens[4]);
	ret = snprintf(buf, sizeof(buf), "Users on %%G%s%%W:%%x ", chat_name);
	free(chat_name);

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);
	offset += ret;
	len -= ret;

	tok = strsep(&p, " ");
	while (tok != NULL && tok[0] != '\0') {
		if (!isalpha(tok[0])) {
			ret = snprintf(&buf[offset], len, "%c", tok[0]);
			if (ret < 0 || (size_t) ret >= len)
				return (-1);
			offset += ret;
			len -= ret;
			tok++;
		}

		if (add) {
			struct chat_user *user;
		
			user = chat_user_joined(acct, chat, tok, 1);
			if (user == NULL)
				add = 0;
		}

		ret = snprintf(&buf[offset], len, "%s ", tok);
		if (ret < 0 || (size_t) ret >= len)
			return (-1);
		offset += ret;
		len -= ret;
		tok = strsep(&p, " ");
	}

	if (offset > 0) {
		buf[offset - 1] = '\0';
		screen_output("%s", buf);
	}

	return (0);
}

static int irc_handler_join(struct pork_acct *acct, struct irc_input *in) {
	char *p;
	struct chatroom *chat;

	p = strchr(in->tokens[0], '!');
	if (p != NULL)
		*p++ = '\0';

	if (!acct->proto->user_compare(acct->username, in->tokens[0])) {
		struct imwindow *win;

		win = imwindow_find_chat_target(acct, in->args);
		if (win == NULL)
			return (-1);

		chat = chat_new(acct, in->args, in->args, win);
		if (chat == NULL)
			return (-1);
	} else {
		chat = chat_find(acct, in->args);
		if (chat == NULL)
			return (-1);

		chat_user_joined(acct, chat, in->tokens[0], 0);
	}

	return (0);
}

static int irc_handler_privmsg(struct pork_acct *acct, struct irc_input *in) {
	char *p;

	if (in->num_tokens < 3 || in->args == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p != NULL)
		*p++ = '\0';

	/* ^A */
	if (in->args[0] == 0x01) {
		p = strrchr(&in->args[1], 0x01);
		if (p == NULL)
			goto no_ctcp;
		*p++ = '\0';

		in->cmd = &in->args[1];
		p = strchr(in->cmd, ' ');
		if (p != NULL)
			*p++ = '\0';
		in->args = p;

		if (irc_callback_run(acct->data, in) == -1) {
			/* unknown ctcp */;
		}

		return (0);
	}

no_ctcp:
	if (!acct->proto->user_compare(acct->username, in->tokens[2]))
		pork_msg_recv(acct, in->tokens[0], in->args, 0);
	else {
		struct chatroom *chat;

		chat = chat_find(acct, in->tokens[2]);
		if (chat == NULL)
			return (-1);

		chat_recv_msg(acct, chat, in->tokens[0], in->args);
	}

	return (0);
}

static int irc_handler_ping(struct pork_acct *acct, struct irc_input *in) {
	if (in->num_tokens < 3)
		return (-1);

	return (irc_send_pong(acct->data, in->tokens[2]));
}

static int irc_handler_311(struct pork_acct *acct, struct irc_input *in) {
	struct imwindow *win = cur_window();
	char *info;

	if (in->num_tokens < 6)
		return (-1);

	screen_win_msg(win, 0, 0, 1,
		"%%D-%%m-%%M-%%W%s%%M-%%D(%%c%s%%W@%%c%s%%D)%%M-%%m-%%D-",
		in->tokens[3], in->tokens[4], in->tokens[5]);

	if (in->args == NULL)
		info = xstrdup("");
	else
		info = irc_text_filter(in->args);

	screen_win_msg(win, 0, 0, 1, "%%D-%%Ci%%crcname%%W:%%x %s", info);
	free(info);
	return (0);
}

static int irc_handler_319(struct pork_acct *acct, struct irc_input *in) {
	char *chans;

	if (in->args == NULL)
		return (-1);

	chans = irc_text_filter(in->args);

	screen_win_msg(cur_window(), 0, 0, 1, "%%Cc%%channels%%W:%%x %s", chans);

	free(chans);
	return (0);
}

static int irc_handler_312(struct pork_acct *acct, struct irc_input *in) {
	char *info;

	if (in->num_tokens < 5)
		return (-1);

	if (in->args == NULL)
		info = xstrdup("");
	else
		info = irc_text_filter(in->args);

	screen_win_msg(cur_window(), 0, 0, 1,
		"%%D-%%m-%%Cs%%cerver%%W:%%x %s %%D(%%x%s%%D)%%x",
		in->tokens[4], info);

	free(info);
	return (0);
}

static int irc_handler_317(struct pork_acct *acct, struct irc_input *in) {
	char timebuf[128];
	u_int32_t idle_time;
	u_int32_t temp;
	struct imwindow *win = cur_window();
	time_t signon;
	char *p;

	if (in->num_tokens < 6)
		return (-1);

	if (str_to_uint(in->tokens[4], &idle_time) != 0)
		return (-1);

	time_to_str_full(idle_time, timebuf, sizeof(timebuf));

	screen_win_msg(win, 0, 0, 1,
		"%%D-%%m-%%M--%%Ci%%cdle%%W:%%x %s", timebuf);

	if (str_to_uint(in->tokens[5], &temp) != 0)
		return (-1);

	signon = (time_t) temp;

	snprintf(timebuf, sizeof(timebuf), "%s", asctime(localtime(&signon)));
	p = strchr(timebuf, '\n');
	if (p != NULL)
		*p = '\0';
	
	screen_win_msg(win, 0, 0, 1,
		"%%D-%%m-%%Cs%%cignon%%W:%%x %s", timebuf);
	return (0);
}

int irc_handler_301(struct pork_acct *acct, struct irc_input *in) {
	char *msg;

	if (in->args == NULL)
		msg = xstrdup("");
	else
		msg = irc_text_filter(in->args);

	screen_win_msg(cur_window(), 0, 0, 1,
		"%%D-%%m-%%M--%%Ca%%cway%%W:%%x %s", msg);
	free(msg);
	return (0);
}

int irc_handler_313(struct pork_acct *acct, struct irc_input *in) {
	char *msg;

	if (in->num_tokens < 4)
		return (-1);

	if (in->args == NULL)
		msg = xstrdup("is an IRC operator");
	else
		msg = irc_text_filter(in->args);

	screen_win_msg(cur_window(), 0, 0, 1,
		"%%Co%%cperator%%W:%%x %s %s", in->tokens[3], msg);

	free(msg);
	return (0);
}

int irc_handler_314(struct pork_acct *acct, struct irc_input *in) {
	char *info;
	struct imwindow *win = cur_window();

	if (in->num_tokens < 6)
		return (-1);

	if (in->args == NULL)
		info = xstrdup("");
	else
		info = irc_text_filter(in->args);

	screen_win_msg(win, 0, 0, 1,
		"%%D-%%b-%%B-%%W%s%%B-%%D(%%c%s%%W@%%c%s%%D)%%B-%%D(%%Cw%%cho%%Cw%%cas%%D)%%b-%%D-",
		in->tokens[3], in->tokens[4], in->tokens[5]);

	screen_win_msg(win, 0, 0, 1, "%%D-%%Ci%%crcname%%W:%%x %s", info);

	free(info);
	return (0);
}

int irc_handler_nick(struct pork_acct *acct, struct irc_input *in) {
	char *p;
	dlist_t *cur = acct->chat_list;

	if (in->args == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p != NULL)
		*p = '\0';

	if (!acct->proto->user_compare(acct->username, in->tokens[0])) {
		free(acct->username);
		acct->username = xstrdup(in->args);
	}

	while (cur != NULL) {
		struct chatroom *chat = cur->data;
		dlist_t *cur_user = chat->user_list;

		while (cur_user != NULL) {
			struct chat_user *user = cur_user->data;

			if (!strcasecmp(in->tokens[0], user->name)) {
				free(user->name);
				free(user->nname);

				user->name = xstrdup(in->args);
				user->nname = xstrdup(in->args);

				screen_win_msg(chat->win, 1, 1, 1, 
					"%%YNick change%%W:%%x %s is now known as %s",
					in->tokens[0], in->args);
			}

			cur_user = cur_user->next;
		}

		cur = cur->next;
	}

	return (0);
}

int irc_handler_332(struct pork_acct *acct, struct irc_input *in) {
	struct chatroom *chat;
	struct imwindow *win;
	char *topic;
	char *chan;

	if (in->num_tokens < 4 || in->args == NULL)
		return (-1);

	chat = chat_find(acct, in->tokens[3]);
	if (chat != NULL)
		win = chat->win;
	else
		win = cur_window();

	chan = irc_text_filter(in->tokens[3]);
	topic = irc_text_filter(in->args);

	screen_win_msg(win, 0, 1, 1,
		"Topic for %%c%s%%W:%%x %s", chan, topic);

	free(chan);
	free(topic);
	return (0);
}

int irc_handler_333(struct pork_acct *acct, struct irc_input *in) {
	struct chatroom *chat;
	struct imwindow *win;
	time_t time_set;
	char buf[64];
	char *chan;

	if (in->num_tokens < 6)
		return (-1);

	if (str_to_int(in->tokens[5], (int *) &time_set) == -1)
		return (-1);

	if (date_to_str(time_set, buf, sizeof(buf)) == -1)
		return (-1);

	chat = chat_find(acct, in->tokens[3]);
	if (chat != NULL)
		win = chat->win;
	else
		win = cur_window();

	chan = irc_text_filter(in->tokens[3]);

	screen_win_msg(win, 0, 1, 1,
		"Topic for %%c%s%%x set by %%c%s%%x on %s",
		chan, in->tokens[4], buf);

	free(chan);
	return (0);
}

int irc_handler_005(struct pork_acct *acct, struct irc_input *in) {
	u_int32_t i;
	irc_session_t *irc = acct->data;

	irc_handler_print_tok(acct, in);

	for (i = 0 ; i < in->num_tokens ; i++) {
		if (!strncasecmp(in->tokens[i], "NICKLEN=", 8))
			str_to_int(&in->tokens[i][8], &irc->nick_len);
		else if (!strncasecmp(in->tokens[i], "CHANTYPES=", 10))
			irc->chantypes = xstrdup(&in->tokens[i][10]);
		else if (!strncasecmp(in->tokens[i], "TOPICLEN=", 9))
			str_to_int(&in->tokens[i][9], &irc->topic_len);
		else if (!strncasecmp(in->tokens[i], "KICKLEN=", 8))
			str_to_int(&in->tokens[i][8], &irc->kick_len);
		else if (!strncasecmp(in->tokens[i], "CHANMODES=", 10))
			irc->chanmodes = xstrdup(&in->tokens[i][10]);
	}

	return (0);
}

int irc_handler_notice(struct pork_acct *acct, struct irc_input *in) {
	char *p;
	int ret = 0;

	if (in->args == NULL)
		return (-1);

	if (!strcasecmp(in->tokens[0], "NOTICE")) {
		in->args += 4;
		return (irc_handler_print_arg(acct, in));
	}

	if (in->num_tokens < 3)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p == NULL)
		return (irc_handler_print_arg(acct, in));
	*p++ = '\0';

	/* CTCP reply */
	if (in->args[0] == 0x01) {
		/* XXX d this */; 
	}

	if (!strcasecmp(acct->username, in->tokens[2]))
		ret = pork_notice_recv(acct, in->tokens[0], in->args);
	else {
		struct chatroom *chat;

		chat = chat_find(acct, in->tokens[2]);
		if (chat == NULL)
			return (-1);

		ret = chat_recv_notice(acct, chat, in->tokens[0], in->args);
	}

	return (ret);
}

int irc_handler_mode(struct pork_acct *acct, struct irc_input *in) {
	char c;
	int op;

	if (in->num_tokens < 3 || in->args == NULL)
		return (-1);

	c = in->args[0];
	if (c == '+')
		op = MODE_PLUS;
	else if (c == '-')
		op = MODE_MINUS;
	else
		return (-1);

	if (!strcasecmp(in->tokens[2], acct->username)) {
		char *p = &in->args[1];
		char new_mode[24];
		char mode_str[24];
		u_int32_t i = 0;
		int changes = 0;
		irc_session_t *irc = acct->data;

		while ((c = *p++) != '\0') {
			if (c == '+')
				op = MODE_PLUS;
			else if (c == '-')
				op = MODE_MINUS;
			else {
				if (op == MODE_PLUS) {
					if (i >= sizeof(new_mode))
						return (-1);

					changes++;
					new_mode[i++] = c;
				} else {
					char *q;

					q = strchr(irc->umode, c);
					if (q != NULL) {
						*q = '#';
						changes++;
					}
				}
			}
		}

		if (changes > 0) {
			new_mode[i] = '\0';
			p = irc->umode;
			i = 0;

			while ((c = *p++) != '\0') {
				if (c != '#')
					mode_str[i++] = c;
			}

			xstrncpy(irc->umode, mode_str, sizeof(irc->umode));
			xstrncat(irc->umode, new_mode, sizeof(irc->umode));

			screen_win_msg(cur_window(), 1, 0, 1,
				"%%YMode %%G%s%%W:%%x %s", acct->username, in->args);
		}
	} else {
		struct chatroom *chat;
		char *args;

		chat = chat_find(acct, in->tokens[2]);
		if (chat == NULL)
			return (-1);

		args = strchr(in->args, ' ');
		if (args != NULL)
			*args++ = '\0';
	}

	return (0);
}

int irc_handler_quit(struct pork_acct *acct, struct irc_input *in) {
	char *msg;
	char *p;
	dlist_t *cur = acct->chat_list;

	if (in->num_tokens < 2)
		return (-1);

	if (in->args != NULL)
		msg = irc_text_filter(in->args);
	else
		msg = xstrdup("Leaving");

	p = strchr(in->tokens[0], '!');
	if (p == NULL)
		return (-1);
	*p++ = '\0';

	while (cur != NULL) {
		struct chatroom *chat = cur->data;
		dlist_t *cur_user = chat->user_list;

		while (cur_user != NULL) {
			dlist_t *next = cur_user->next;
			struct chat_user *user = cur_user->data;

			if (!strcasecmp(in->tokens[0], user->name)) {
				chat_user_left(acct, chat, user->name, 1);

				screen_win_msg(chat->win, 1, 1, 1,
					"%%W%s %%D(%%c%s%%D)%%x has quit: %s",
					in->tokens[0], p, msg);
			}

			cur_user = next;
		}

		cur = cur->next;
	}

	free(msg);
	return (0);	
}

int irc_handler_part(struct pork_acct *acct, struct irc_input *in) {
	char *p;
	struct chatroom *chat;
	int ret = 0;

	if (in->num_tokens < 3)
		return (-1);

	chat = chat_find(acct, in->tokens[2]);
	if (chat == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p == NULL)
		return (-1);
	*p++ = '\0';

	if (strcasecmp(acct->username, in->tokens[0]))
		ret = chat_user_left(acct, chat, in->tokens[0], 0);

	return (ret);
}

int irc_handler_kick(struct pork_acct *acct, struct irc_input *in) {
	struct chatroom *chat;
	char *msg;
	char *p;
	int ret;

	if (in->num_tokens < 4)
		return (-1);

	chat = chat_find(acct, in->tokens[2]);
	if (chat == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p == NULL)
		return (-1);
	*p++ = '\0';

	if (in->args == NULL)
		msg = "No reason given";
	else
		msg = in->args;

	if (!strcasecmp(acct->username, in->tokens[3]))
		ret = chat_forced_leave(acct, chat->title, in->tokens[0], msg);
	else
		ret = chat_user_kicked(acct, chat, in->tokens[3], in->tokens[0], msg);

	return (ret);
}

int irc_handler_topic(struct pork_acct *acct, struct irc_input *in) {
	char *p;
	struct chatroom *chat;

	if (in->num_tokens < 3)
		return (-1);

	chat = chat_find(acct, in->tokens[2]);
	if (chat == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p != NULL)
		*p++ = '\0';

	return (chat_got_topic(acct, chat, in->tokens[0], in->args));
}

int irc_handler_invite(struct pork_acct *acct, struct irc_input *in) {
	char *p;

	if (in->num_tokens < 3 || in->args == NULL)
		return (-1);

	p = strchr(in->tokens[0], '!');
	if (p != NULL)
		*p++ = '\0';

	return (chat_got_invite(acct, in->args, in->tokens[0], NULL));
}

int irc_handler_err_msg(struct pork_acct *acct, struct irc_input *in) {
	char *msg;
	char *str;

	if (in->num_tokens < 4)
		return (-1);

	str = irc_text_filter(in->tokens[3]);

	if (in->args == NULL)
		msg = xstrdup("Unknown error");
	else
		msg = irc_text_filter(in->args);

	screen_err_msg("%s: %s", str, msg);

	free(str);
	free(msg);
	return (0);
}

int irc_callback_add_defaults(irc_session_t *session) {

#if 0
	/* Print everything after the : */
	irc_callback_add(session, "372", irc_handler_print_arg);
	irc_callback_add(session, "255", irc_handler_print_arg);
	irc_callback_add(session, "265", irc_handler_print_arg);
	irc_callback_add(session, "266", irc_handler_print_arg);
	irc_callback_add(session, "250", irc_handler_print_arg);
	irc_callback_add(session, "375", irc_handler_print_arg);
	irc_callback_add(session, "002", irc_handler_print_arg);
	irc_callback_add(session, "003", irc_handler_print_arg);
	irc_callback_add(session, "251", irc_handler_print_arg);
#endif

	irc_callback_add(session, "004", irc_handler_print_tok);
	irc_callback_add(session, "252", irc_handler_print_num);
	irc_callback_add(session, "254", irc_handler_print_num);

	irc_callback_add(session, "005", irc_handler_005);
	irc_callback_add(session, "353", irc_handler_353);
	irc_callback_add(session, "352", irc_handler_352);
	irc_callback_add(session, "001", irc_handler_001);

	irc_callback_add(session, "301", irc_handler_301);
	irc_callback_add(session, "311", irc_handler_311);
	irc_callback_add(session, "319", irc_handler_319);
	irc_callback_add(session, "312", irc_handler_312);
	irc_callback_add(session, "317", irc_handler_317);
	irc_callback_add(session, "313", irc_handler_313);
	irc_callback_add(session, "332", irc_handler_332);
	irc_callback_add(session, "333", irc_handler_333);
	irc_callback_add(session, "314", irc_handler_314);

	irc_callback_add(session, "263", irc_handler_err_msg);
	irc_callback_add(session, "401", irc_handler_err_msg);
	irc_callback_add(session, "402", irc_handler_err_msg);
	irc_callback_add(session, "403", irc_handler_err_msg);
	irc_callback_add(session, "404", irc_handler_err_msg);
	irc_callback_add(session, "405", irc_handler_err_msg);
	irc_callback_add(session, "406", irc_handler_err_msg);
	irc_callback_add(session, "407", irc_handler_err_msg);
	irc_callback_add(session, "408", irc_handler_err_msg);
	irc_callback_add(session, "413", irc_handler_err_msg);
	irc_callback_add(session, "414", irc_handler_err_msg);
	irc_callback_add(session, "415", irc_handler_err_msg);
	irc_callback_add(session, "421", irc_handler_err_msg);
	irc_callback_add(session, "423", irc_handler_err_msg);
	irc_callback_add(session, "432", irc_handler_err_msg);
	irc_callback_add(session, "433", irc_handler_err_msg);
	irc_callback_add(session, "437", irc_handler_err_msg);
	irc_callback_add(session, "442", irc_handler_err_msg);
	irc_callback_add(session, "444", irc_handler_err_msg);
	irc_callback_add(session, "461", irc_handler_err_msg);
	irc_callback_add(session, "467", irc_handler_err_msg);
	irc_callback_add(session, "471", irc_handler_err_msg);
	irc_callback_add(session, "472", irc_handler_err_msg);
	irc_callback_add(session, "473", irc_handler_err_msg);
	irc_callback_add(session, "474", irc_handler_err_msg);
	irc_callback_add(session, "475", irc_handler_err_msg);
	irc_callback_add(session, "476", irc_handler_err_msg);
	irc_callback_add(session, "477", irc_handler_err_msg);
	irc_callback_add(session, "482", irc_handler_err_msg);

	irc_callback_add(session, "JOIN", irc_handler_join);
	irc_callback_add(session, "PRIVMSG", irc_handler_privmsg);
	irc_callback_add(session, "NOTICE", irc_handler_notice);
	irc_callback_add(session, "MODE", irc_handler_mode);
	irc_callback_add(session, "PING", irc_handler_ping);
	irc_callback_add(session, "NICK", irc_handler_nick);
	irc_callback_add(session, "QUIT", irc_handler_quit);
	irc_callback_add(session, "PART", irc_handler_part);
	irc_callback_add(session, "KICK", irc_handler_kick);
	irc_callback_add(session, "TOPIC", irc_handler_topic);
	irc_callback_add(session, "INVITE", irc_handler_invite);

	irc_callback_add(session, "VERSION", irc_handler_ctcp_version);
	irc_callback_add(session, "PING", irc_handler_ctcp_ping);

#if 0
	irc_callback_add(session, "USERINFO", irc_handler_userinfo);
	irc_callback_add(session, "CLIENTINFO", irc_handler_clientinfo);
	irc_callback_add(session, "TIME", irc_handler_time);
	irc_callback_add(session, "ECHO", irc_handler_echo);
	irc_callback_add(session, "ACTION", irc_handler_action);
#endif

	return (0);
}
