/*
	in.pop3gwd, a POP3 proxy
	Copyright (C) 1997 Andrea Borgia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* io.c: line read and write functions */
/* ----------------------------------- */

/* Code adapted from "UNIX Network programming" by W. Richard Stevens */

#include "pop3-gw.h"



/*
 * Read a line from a descriptor, one byte at a time, looking for the POP3 
 * terminator.  The terminator is *not* stored in the buffer, but a null is 
 * added at the end of the line.
 * The number of characters up to, but not including, the null (the same as 
 * strlen(3)) is returned if the operation was successful; return value is 
 * negative in case of error.
 */

int readline(int filedes, char *line, int maxlen, int maxwait, const char termin[]) {
	fd_set	master, copy;
	int	count = 0;
	int 	readchars, result;
	struct timeval	deadline;	
	char	ch;
	int 	cr_flag = FALSE;	/* TRUE if CR has been found; waiting for LF */
	int	term_flag = FALSE;	/* TRUE if either CRLF has been found or EOF */


	FD_ZERO(&master);
	FD_SET(filedes, &master);
	count = 0;
	
	while (++count < maxlen && term_flag == FALSE) {
	
		bcopy(&master, &copy, sizeof(fd_set));	/* select() trashes copy */
		deadline.tv_sec = maxwait;
		deadline.tv_usec = 0;

		if (select(filedes+1, &copy, (fd_set *)NULL, (fd_set *)NULL, &deadline) <= 0) {
#ifdef DEBUG
			syslog(LOG_PRIO, "Network timeout signal after %d seconds while reading data", maxwait);
#endif
			term_flag = TRUE;
			result = BAD;		/* there was an error */
		}
		else {				
			if ((readchars = read(filedes, &ch, 1)) == 1) {
				if (ch == termin[0]) {
					count--; 	/* do not count terminators */
					cr_flag = TRUE;
					result = count; /* ??? */
				}
				else
					if (ch == termin[1] && cr_flag == TRUE) {
						count--; /* do not count terminators */
						term_flag = TRUE;
						result = count;
					}
					else {
						*(line++) = ch;
						cr_flag = FALSE;
					}
			}
			else
				if (readchars == 0) {
					term_flag = TRUE;
					if (count == 1)
						result = 0;	/* EOF, no data read */
				}
				else
					result = readchars; 	/* same error code as read(2) */
		}
	}
	
	*line = 0;
	return(result);
}



/*
 * Write "n" bytes to a descriptor.
 * Use in place of write() when filedes is a stream socket.
 * return value is less than zero in case of error
 */

int writen(int filedes, char *line, int nbytes, int maxwait) {
	int	nleft, nwritten;
	fd_set  master, copy;
	struct timeval	deadline;

	
	FD_ZERO(&master);
	FD_SET(filedes, &master);
	nleft = nbytes;
	
	do {
		bcopy(&master, &copy, sizeof(fd_set));  /* select() trashes copy */
		deadline.tv_sec = maxwait;
		deadline.tv_usec = 0;

		if (select(filedes+1, (fd_set *)NULL, &copy, (fd_set *)NULL, &deadline) <= 0) {
#ifdef DEBUG
			syslog(LOG_PRIO, "Network timeout signal after %d seconds while writing data", maxwait);
#endif
			nwritten = BAD;	/* caller will get an error */
		}
		else {
			nwritten = write(filedes, line, nleft);
			nleft = nleft - nwritten;
			line = line + nwritten;
		}
	} while (nleft > 0 && nwritten > 0);
	
	if (nwritten == BAD)
		return(nwritten);
	else
		return(nbytes - nleft);
}
