/*
 * 	PluSHS - Pluf Simple Hostname Scanner
 * 	Copyright (C) 2001, 2002 pluf <pluf@wanadoo.es> 
 * 
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * plushs.c
 * 
 */
#include "config.h"

#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <ctype.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>

#if defined(HAVE_STRINGS_H) && (defined(SOLARIS) || defined(AIX))
#include <strings.h>
#else
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "plushs.h"

extern FILE *pfildes;

/*
 * Misc variables 
 */
char IPstring[IPRANGE_SIZE+1];	/* single ip address */
const char *progname;		/* programm name */

/*
 * statistics variables 
 */
float resolv_count = 0;         /* total IP address resolved */
float timeout_count = 0;        /* total timeouts */
float search_count = 0;         /* total successful searchs */
float aliases_count = 0;        /* total aliasses found */

struct loop {
	unsigned char a;
	unsigned char b;
	unsigned char c;
	unsigned char d;
};

/*
 * Searchs concordances betwen s1 (pattern) and s2 (hostname)
 */
int
search_pattern(s1, s2)
	char *s1, *s2;
{
	register int i, j;
	int x, y, count, yes;

        i = j = x = y = count = 0;
	
	/* if pattern is longer than hostname
	 * jump to next pattern
	 */
	if (strlen(s1) > strlen(s2))
		return (0);

	x = ((strlen(s2) / strlen(s1)) -1);
	y = strlen(s2) % strlen(s1);

	for (i = 0; i < (strlen(s1) * x + 1 + y); i++, s2++) {
		yes = 1;
		for (j = 0; j < strlen(s1); j++)
			if (*(s1 + j) != *(s2 + j))
				yes = 0;
		if (yes)
			++count;
	}
	return (count);
}


/*
 * Extract patterns from file, then call
 * search_pattern() function 
 */
int
extract_pattern(name, file)
	char *name;
	FILE *file;
{
	register int i;
	char pat[MAX_PATTERN];

	(void)bzero((char*)&pat, sizeof(pat));
	(void)fseek(file, 0L, SEEK_SET);

	while ((fgets(pat, sizeof(pat), file)) != NULL) {
		/* set '\0' character to end string */
		for (i = 0; i < MAX_PATTERN; i++)
			if (pat[i] == '\n')
				pat[i] = '\0';
		if (search_pattern(pat, name))
			return (0);
	}
	return (-1);
}

/*
 * Check the correct address ip or ip ranges format
 */
int
check_addr(fin)
        signed short *fin;
{
        unsigned char i, j;

        for (i = 0, j = 1; i < 8; i++, j += 2) {
                /* [2] check for size error */
                if (j <= 7)
                        if ((*(fin + j) != -1) && (*((fin + j) - 1) > *(fin + j)))
                        error(7);
                /* [3] check for "x equal to y" error */
                if (j <= 7)
                        if ((*(fin + j) != -1) && (*((fin + j) - 1) == *(fin + j)))
                                error(7);
                /* [4] check for range error */
                if (*(fin + i) < 0 || *(fin + i) > 255) {
                        if (*(fin + i) == -1)
                                continue;
                                error(7);
                }
        }

        return (0);
}

/*
 * Convert a string to an int vector, that, 
 * contain ip ranges.
 */
int
extract_addr(signed short *fin, char *address, FILE *list)
{
	char *p, *pa, buf[32];
        unsigned char i = 0, j = 0, 
			ot = 0, num = 0;

	(void)bzero((char*)&IPstring, sizeof(IPstring));
	(void)strncpy(IPstring, address, IPRANGE_SIZE);

	if (pv.lflag) {
		if (fgets(address, IPRANGE_SIZE, list) == NULL)
			return (-1);

		/* check non data */
		if (*address == '\n' || *address == '\0')
			return (-1);

		(void)bzero((char*)&IPstring, sizeof(IPstring));		
		(void)snprintf(IPstring, IPRANGE_SIZE, "%s", address);
		/* delete '\n' char */
		IPstring[strlen(IPstring) - 1] = '\0';
	}

	bzero((char*)buf, 32);
        pa = buf;

	INIT(fin);

        while (*(address + i) != '\0') {
		CHECK_NON_DIGIT(*(address+i));
		if (*(address + i) == '.') {
			buf[(j * 8) + (ot + 1)] = '\0';
			++j; ot = 0;
		}
		else
			buf[(j * 8) + (ot++)] = *(address + i);
		++i;
	}
	if (j < 3 || j > 3)
		error(7);

        p = (char*)malloc(4);
        
	for (j = 0; j < 4; j++) {
		i = ot = 0;
		while (buf[(j * 8) + i] != '\0') {
		/* si el caracter no es - entonces copiar al buffer p */
		/* ot controla los indices del buffer (4) */
			if (buf[(j * 8) + i] != '-')
				*(p + (ot++)) = buf[(j * 8) + i];
			else {
		/* cuando se encuentra - se inserta final-cadena en buffer
		 * y se aade a fin, luego se establece el indice a 0
		 */
				*(p+(ot+1)) = '\0';
				fin[num] = atoi(p);
				++num;
				bzero((char*)p, sizeof(p));
				ot = 0;
			}
		++i;
		}
		*(p + (ot + 1)) = '\0';
		fin[num] = atoi(p);
		bzero((char*)p, sizeof(p));
		if (num == 1 || num == 3 || num == 5)
			num += 1;
		else
			num += 2;
	}
		
	check_addr(fin);
	return (0);
}

/* 
 * Increment timeout count variable
 */
void
sigalrm_handling()
{
	timeout_count += 1;
	pv.t = 1;

	signal(SIGALRM, (void (*)())sigalrm_handling);
	return;
}

void
sigint_handling()
{
	printf("\n---- Stop ----\n");
	exit(0);
	return;
}

/*
 * check for single ip
 */
int
check_ipsingle(fin)
	signed short *fin;
{
	if ((*(fin+1) == -1) && (*(fin+3) == -1) && 
	     (*(fin+5) == -1) && (*(fin+7) == -1))
		return (0);
	else
		return (-1);
}

/*
 * Get total ip addresses to scan
 */
int
get_total_ips(fin)
	signed short fin[8];
{
	int all = 0, yes = 0;
	register int i, j;

	if (fin[1] == -1 && fin[3] == -1 && 
	     fin[5] == -1 && fin[7] == -1)
		return (0);

	for (i = 7; i >= 1; i -= 2)
		if (fin[i] != -1) {
			yes = 1;
			for (j = 7; j >= i; j -= 2)
				if (fin[j] != -1 && j != i ) {
					all *= ((fin[i] + 1) - fin[i-1]);
					yes = 0;
					break;
				}
			if (yes)
				all += ((fin[i] + 1) - fin[i-1]);
		}
	return (all);
}                

/*
 * Show statistics information about network scan
 */
int
display_statistics(fin, timeout, wait, file)
	signed short fin[8];
	int timeout, wait;
 	FILE *file;
{
        float total = 0;

	if ((total = get_total_ips(fin)) == 0)
		return (0);

(void)fprintf(DESC,"\n\n========| Network Statistics |====================\n\n");
(void)fprintf(DESC,"Ip range to scan      %s\n\n", IPstring);
(void)fprintf(DESC,"  Successfull:   [%5.1f%%]\n", ((resolv_count * 100)
										 / total));
(void)fprintf(DESC,"Unsuccessfull:   [%5.1f%%]\n", (((total - (resolv_count
							 + timeout_count)) * 100) / total));
(void)fprintf(DESC,"     Timeouts:   [%5.1f%%]\n\n", ((timeout_count * 100)
										 / total)); 
(void)fprintf(DESC,"=-----------------------------------------------=\n\n");
(void)fprintf(DESC,"  Total ips to check:  %6.0f\n", total);
(void)fprintf(DESC,"  Successfull checks:  %6.0f\n", resolv_count);
(void)fprintf(DESC,"Unsuccessfull checks:  %6.0f\n", (total - (resolv_count +
									   timeout_count)));
(void)fprintf(DESC,"            Timeouts:  %6.0f\n", timeout_count);
(void)fprintf(DESC,"       Aliases found:  %6.0f\n", aliases_count);
(void)fprintf(DESC," Successfull searchs:  %6.0f\n\n", search_count);
(void)fprintf(DESC,"=-----------------------------------------------=\n\n");
(void)fprintf(DESC,"     String format:  %s\n", vars.format);
(void)fprintf(DESC,"    Timeout set to:  %d seconds\n", timeout);
(void)fprintf(DESC,"Wait second set to:  %d seconds\n\n", wait);

	return (0);
}


int
main(argc, argv)
	int argc;
	char *argv[];
{
	struct hostent *host, *gethostbyaddr();
	struct loop singflag, count;
	char oval[4], *file = NULL, *pattern = NULL, **ptr;
	FILE *fildes=NULL, *sfile=NULL, *iplistfile=NULL, *plist=NULL;
	int opts, single = 0;
	signed short final[8];
	char address[IPRANGE_SIZE+1];

	if (argc == 1) {
		puts(TITLE);
		printf("Try 'plushs -h' for more information.\n");
		exit(0);
	}
		
	progname = argv[0];

	while ((opts = getopt(argc, argv, "cho:s:al:Dt:w:x:vmf:TP:")) != EOF) {
		switch (opts) {
		case 'c':
			pv.cflag = 1;
			break;
		case 'o':
			if (strlen(optarg) > MAX_FILE)
				error(1);
			file = optarg;	/* log file descriptor */
			pv.oflag = 1;
			break;
		case 's':
			if (strlen(optarg) > MAX_PATTERN)
				error(2);
			pattern = optarg; /* single searchs log file descriptor */
			pv.sflag = 1;
			break;
		case 'a':
			pv.aflag = 1;
			break;
		case 'l':
			if (strlen(optarg) > MAX_FILE)
				error(3);
			/* ip addresses file */ 
			if ((iplistfile = fopen(optarg, "r")) == NULL)
				error(13);
			pv.lflag = 1;
			break;
		case 'D':
			pv.dflag = 1;
			break;
		case 't':
			pv.tflag = 1;
			pv.timeout = atoi(optarg);
			break;
		case 'T':
			pv.Tflag = 1;
			sethostent(1);
			break;
		case 'P':
			pv.Pflag = 1;
			if (strlen(optarg) > MAX_PLUGIN_LIST)
				error(8);

			if (strncmp(optarg, "list", 5) == 0) {
				show_plugins_list();
				exit(0);
			}
			if (activate_selected_plugins(optarg) != 0) {
				printf("format incorrect: 1,2,3,etc\n");
				exit(-1);
			}
			break;
		case 'w':
			pv.wflag = 1;
			pv.wait = atoi(optarg);
			break;
		case 'x':
			if (strlen(optarg) > MAX_FILE)
				error(4);
			/* patterns list file */
			if ((plist = fopen(optarg, "r")) == NULL)
				error(9);
			pv.xflag = 1;
			break;
		case 'v':
			pv.verbose = 1;
			putchar('\n');
			puts(TITLE);
			putchar('\n');
			break;
		case 'm':
			pv.mflag = 1;
			break;
		case 'f':
			if (strlen(optarg) > FORMAT_SIZE)
				error(5);
			strncpy(vars.format, optarg, FORMAT_SIZE - 1);
			check_format_string();
			pv.fflag = 1;
			break;
		case 'h':
		default:
			usage(0);
			break;
		}
	}

	if (argc == 1 || ((pv.dflag && !pv.oflag) && !pv.mflag) ||
 		 (pv.sflag && pv.xflag) || (pv.oflag && pv.mflag)) 
		usage(0);

	/* if lflag,  don't pass IP address by command line */
	if (!pv.lflag) {
		argc -= optind;
       		argv += optind;

		if (argc != 1)
			usage(0);

		/* check address size */
		if (strlen(*argv) > IPRANGE_SIZE)
			error(6);

		sscanf(*argv, "%s", address);
	}

	extract_addr(final, address, iplistfile);

	/* open log file && plugins log file */
	if (pv.oflag) {
		if ((fildes = fopen(file, "a")) == NULL)
			error(10);

		if ((pfildes = fopen(PLOGFILE, "a")) == NULL)
			error(11);

	}

	if (pv.sflag || pv.xflag)
		if ((sfile = fopen(LOG_FILE, "a")) == 0)
			error(12);

	/* daemonize */
	if (pv.dflag) {
		switch (fork()) {
		case 0:
			break;
		case -1:
			perror("fork()");
			exit(-1);
			break;
		default:
			exit(-1);
			break;
		}
		close(STDIN_FILENO);
		close(STDOUT_FILENO);
		close(STDERR_FILENO);
	}
	if (!pv.tflag || !pv.timeout)
		pv.timeout = DEFAULT_TIMEOUT;
	if (!pv.wflag)
		pv.wait = DEFAULT_WAIT;

	signal(SIGALRM, (void (*)())sigalrm_handling);
	signal(SIGINT, (void (*)())sigint_handling);

	do {		
		if (pv.mflag)
			if ((fildes = fopen(IPstring, "a")) == NULL)        
				error(10);

		/* check for ip single */
		single = check_ipsingle(final);

		resolv_count = 0;
		timeout_count = 0;
		search_count = 0;

		if (final[1] == -1)
			++singflag.a;
		if (final[3] == -1)
			++singflag.b;
		if (final[5] == -1)
			++singflag.c;
		if (final[7] == -1)
			++singflag.d;	

		if (pv.lflag)
			fprintf(SFLAGS, "> %s \n", IPstring);


		count.a = final[0];
		if (singflag.a)
			goto A_Sing;	

	/* 1: A class [ AAA.xxx.xxx.xxx ] */
	while (final[0] <= final[1]) {
A_Sing:
		if (single)
			fprintf(SFLAGS, SA, final[0], ((final[1] !=-1)? final[1]: 0));

		count.b = final[2];
		if (singflag.b)    
	                goto B_Sing;

		/* 2: B class [ xxx.BBB.xxx.xxx ] */
		while (final[2] <= final[3]) {
B_Sing:				
			if (single)
				fprintf(SFLAGS, SB, final[2], ((final[3]!=-1)? final[3]: 0));	
			count.c = final[4];
		        if (singflag.c) 
               	        	goto C_Sing;

			/* 3: C class [ xxx.xxx.CCC.xxx ] */
			while (final[4] <= final[5]) {
C_Sing:			
				if (single)
					fprintf(SFLAGS, SC, final[4], ((final[5]!=-1)? final[5]: 0));
				count.d = final[6];
				if (singflag.d)
					goto D_Sing;
			
		/* 4: end */
		while (final[6] <= final[7]) {
D_Sing:					
			oval[0] = final[0];
			oval[1] = final[2];
			oval[2] = final[4];
			oval[3] = final[6];

			/* sleep 'wait' seconds */
			if (pv.wflag) 
				sleep(pv.wait);
			/* set timeout alarm signal */
			host = NULL;		
			alarm(pv.timeout);
			host = gethostbyaddr(oval, 4, AF_INET);
			alarm(0);

			/* check for break timeout limit */
			if (host != NULL && pv.t == 0) {
				/* (1) print host name */
				if (!single) { 
					(void)fprintf(SFLAGS, "%d.%d.%d.%d ==> %s", final[0], 
							final[2], final[4], final[6], host->h_name);
				}
				else
					(void)fprintf(SFLAGS, SD, final[0], final[2], 
							final[4], final[6], host->h_name);

				/* (2) search single pattern */
				if (pv.sflag)
					if (search_pattern(pattern, host->h_name)) {
						fprintf(sfile, "%d.%d.%d.%d ==> %s\n", final[0], final[2], 
									final[4], final[6], host->h_name);
						(void)fflush(sfile);
						search_count += 1;

						if (pv.fflag) {
							alarm(0);
							executing(host->h_name, fildes, final);
						}
						else if (pv.verbose)
							(void)fprintf(SFLAGS, FOUND);
					}

				 /* (3) searchs multiples patterns */				
				 if (pv.xflag)
					if (!extract_pattern(host->h_name, plist)) {
						fprintf(sfile, "%d.%d.%d.%d ==> %s\n", final[0], final[2],
									final[4], final[6], host->h_name);
						fflush(sfile);
						search_count += 1;
						if (pv.fflag) {
							alarm(0);
							executing(host->h_name, fildes, final);
						}
						else
							if (pv.verbose)
								fprintf(SFLAGS, FOUND);
					}

				/* (4) show alias */
				if (pv.aflag) {
					ptr = host->h_aliases;
					while (*ptr != NULL) {
						if (single)
							fprintf(SFLAGS, SALIAS, *ptr);
						else
							fprintf(SFLAGS, SALIAS, *ptr);
						++ptr;
						aliases_count += 1;		
					}
				}
				if (pv.oflag || pv.mflag)
					(void)fflush(fildes);

				resolv_count += 1;
			}
			if (pv.verbose) {
				if (pv.t == 1) {
					if (single)
						fprintf(SFLAGS, SD, final[0], final[2], 
								final[4], final[6], TIMEOUT);
					else	
						fprintf(SFLAGS, "%d.%d.%d.%d = %s\n", final[0], final[2], 
							final[4], final[6], TIMEOUT);
					pv.t = 0;
				}
				else if (h_errno == HOST_NOT_FOUND) {
					if (!single)
						(void)fprintf(SFLAGS, "%d.%d.%d.%d ==> .", final[0], 
								final[2], final[4], final[6]);
					else
						fprintf(SFLAGS, SD, final[0], final[2], final[4], 
									final[6], ".");
					h_errno = 0;
				}
			}
			
			if (pv.oflag || pv.mflag)
				(void)fflush(fildes);

			load_plugins(address);

			if (!single)
				putchar('\n');
			final[6] += 1;
				}
				final[6] = count.d;
				final[4] += 1;
			}
			final[4] = count.c;
			final[2] += 1;
		}
		final[2] = count.b;
		final[0] += 1;

		}
		final[0] = count.a;		
			
		display_statistics(final, pv.timeout, pv.wait, fildes);
		
		if (pv.oflag || pv.mflag)
			fflush(fildes);
		if (pv.mflag)
			fclose(fildes);
		
	} while (pv.lflag? extract_addr(final, address, iplistfile) == 0: 0);
	
	if (pv.Tflag)
		endhostent();

	/* close all descriptors files */
	if (pv.oflag)
		fclose(fildes);
	if (pv.sflag || pv.xflag)
		fclose(sfile);
	if (pv.lflag)
		fclose(iplistfile);
	if (pv.xflag)
		fclose(plist);

	return (0);
}
