/* Copyright (C) 1997, 1998, 1999 Michael Wiedmann

   This file is part of pi-address.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Additionally, you are granted permission to assume, for the purposes
   of distributing this program in object code or executable form
   under Section 3 of the GNU Public License, that the QT library 
   is normally distributed with the major components of the 
   operating system on which the executable or object code runs.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
   USA.

   Written by Michael Wiedmann <mw@miwie.in-berlin.de>

   $Id: pilot.cpp,v 1.15 2000/04/24 16:11:52 mw Exp $
*/


#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include "pilot.h"

#include "pilot.moc"

/*! 
  \class Pilot pilot.h
  \brief The Pilot class is used to communicate with the Pilot.

  This class is used to communicate with the Palm Pilot. It provides 
  member functions to connect, disconnect, install a database,
  retrieve a database, etc. to/from a Pilot.

  General usage:
  <UL>
  <LI>Create a Pilot object
  <LI>Connect to the Pilot using \c connect(...)
  <LI>Perform any operations
  <LI>Disconnect from Pilot using \c disconnect()
  <LI>Destroy Pilot object
  </UL> 

  You need a correct installed copy of <B>pilot-link</B> to use this 
  wrapper class.
*/  

//! Default constructor.

Pilot::Pilot()
  : QObject()
{
  ;
}

/*!
  Overloaded constructor. Initializes private members and creates a 
  QProgressDialog.

  Expects the following arguments:
  \arg \c parent pointer to parent widget.
  \arg \c optional timeout value in seconds (Default: 30 sec.)
*/

Pilot::Pilot(QWidget *parent, int iTime)
  : QObject()
{
  sd       = 0;
  iTimeout = iTime;

  memset((char *)&userInfo, 0x00, sizeof(struct PilotUser));
  memset((char *)&sysInfo,  0x00, sizeof(struct SysInfo));  
  memset((char *)&cardInfo, 0x00, sizeof(struct CardInfo));

  pilotProgress = new QProgressDialog("             Connecting                 \n\n",
                                      "Cancel", 2*iTimeout, parent,
                                      "progress", TRUE);
}

//! Destructor. Destroys used resources.
Pilot::~Pilot()

{
 if (pilotProgress)
   {
     pilotProgress->setProgress(pilotProgress->totalSteps());
     delete pilotProgress;
   }
}


/*! 
  This member function connects to a Pilot:

  \arg \c qsDeviceName reference to a QString object. 
  The contents of this QString has to be the device to be used.
*/

Pilot::ConnectResult Pilot::connect(QString &qsDeviceName, bool bCheckDevice)
{
  struct pi_sockaddr addr;
  int iRet;

  fd_set fd_r, fd_r_in;
  struct timeval tv;

  // do we have to test the device for access?
  if (bCheckDevice)
    {
      // workaround to prevent access to non-existent device
      QFile qfDevice(qsDeviceName.data());
      if (qfDevice.open(IO_ReadWrite))
        qfDevice.close();
      else
        return (DeviceError);
    }
  
  if (sd != 0)
    return (AlreadyConnected);
  
  if (!(sd = pi_socket(PI_AF_SLP, PI_SOCK_STREAM, PI_PF_PADP)))
    return (SocketError);

  addr.pi_family = PI_AF_SLP;
  strcpy(addr.pi_device, (const char *)qsDeviceName);
  
  iRet = pi_bind(sd, (struct sockaddr *)&addr, sizeof(addr));
  if (iRet == -1)
    return (BindError);

  iRet = pi_listen(sd, 1);
  if (iRet == -1)
    return (ListenError);

  qsLabel.sprintf("   Waiting for connection on %s...   ",
                  (const char *)qsDeviceName);
  qsLabel.append("\nPress HotSync-Button now!\n");

  pilotProgress->setLabelText(qsLabel.data());
  pilotProgress->setProgress(0);
  int i=0;

  sz = pilotProgress->sizeHint();
  pilotProgress->resize(sz);
  
  FD_ZERO(&fd_r);
  FD_SET(sd, &fd_r);

  while (1)
    {
      if (pilotProgress->wasCancelled())
        return (Cancelled);
  
      if (i == 2*iTimeout)
	return (TimeoutError);

      pilotProgress->setProgress(i++);
      pilotProgress->resize(sz);

      fd_r_in    = fd_r;
      tv.tv_sec  = 0;
      tv.tv_usec = 500000;

      if (select(sd+1, &fd_r_in, 0, 0, &tv) >= 0)
	{
	  if (FD_ISSET(sd, &fd_r_in))
	    {
              sd = pi_accept(sd, 0, 0);
              if (sd == -1)
	        return (AcceptError);
              else
		{
                  pilotProgress->setProgress(i++);
                  pilotProgress->resize(sz);
                  dlp_ReadStorageInfo(sd, 0, &cardInfo);
                  pilotProgress->setProgress(i++);
                  pilotProgress->resize(sz);
		  dlp_ReadUserInfo(sd, &userInfo);

                  break;
		}
	    }
	}
      else
	return (SelectError);
    }
  
  pilotProgress->setProgress(i++);
  pilotProgress->resize(sz);

  return (Success);
}


//! Disconnects from a Pilot.

void Pilot::disconnect()
{
  if (sd == 0)
    return;
  
  pi_close(sd);
  sd = 0;
}

/*! 
  Get the user name from a Pilot.

  \arg \c qsName reference to a QString object. On successful return this 
  QString holds the queried user name.

  Returns TRUE on success, FALSE otherwise.
*/

bool Pilot::getUserName(QString &qsName)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (userInfo.username[0] == 0x00)
    if (dlp_ReadUserInfo(sd, &userInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);
  bReturn = TRUE;
  qsName.sprintf("%s", userInfo.username);
  
  return bReturn;
}


/*! 
  Get the user ID from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/
bool Pilot::getUserID(unsigned long &userID)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  if (userInfo.username[0] == 0x00)
    if (dlp_ReadUserInfo(sd, &userInfo) < 0)
      return bReturn;

  bReturn = TRUE;
  userID = userInfo.userID;
  
  return bReturn;
}

/*! 
  Get the ROM version from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/
bool Pilot::getRomVersion(unsigned long &version)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (sysInfo.romVersion == 0l)
    if (dlp_ReadSysInfo(sd, &sysInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  version = sysInfo.romVersion;
  
  return bReturn;
}

/*! 
  Get the name from a Pilot.

  Expects a reference to a QString object, which holds on successful return
  the name (return value : TRUE).

  Returns TRUE on success, FALSE otherwise.

  Note: The meaning of \e name is not clear!
*/
   
bool Pilot::getName(QString &qsName)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (sysInfo.name[0] == 0x00)
    if (dlp_ReadSysInfo(sd, &sysInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);
  bReturn = TRUE;
  qsName.sprintf("%s", sysInfo.name);
  
  return bReturn;
}


bool Pilot::getLocale(unsigned long &locale)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (sysInfo.locale == 0l)
    if (dlp_ReadSysInfo(sd, &sysInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
 locale  = sysInfo.locale;
  
  return bReturn;
}

/*! 
  Get the ROM size from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/
bool Pilot::getRomSize(unsigned long &romSize)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (cardInfo.romSize == 0l)
    if (dlp_ReadStorageInfo(sd, 0, &cardInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  romSize = cardInfo.romSize;
  
  return bReturn;
}


/*! 
  Get the RAM size from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/
bool Pilot::getRamSize(unsigned long &ramSize)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (cardInfo.ramSize == 0l)
    if (dlp_ReadStorageInfo(sd, 0, &cardInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  ramSize = cardInfo.ramSize;
  
  return bReturn;
}

/*! 
  Get the amount of free RAM from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/

bool Pilot::getRamFree(unsigned long &ramFree)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (cardInfo.ramSize == 0l)
    if (dlp_ReadStorageInfo(sd, 0, &cardInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  ramFree = cardInfo.ramFree;
  
  return bReturn;
}


/*! 
  Get the name of the card from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/

bool Pilot::getCardName(QString &qsName)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (cardInfo.ramSize == 0l)
    if (dlp_ReadStorageInfo(sd, 0, &cardInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  qsName.sprintf("%s", cardInfo.name);
  
  return bReturn;
}

/*! 
  Get the name of the manufacturer from a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/

bool Pilot::getCardManufacturer(QString &qsName)
{
  bool bReturn = FALSE;

  if (sd == 0)
    return bReturn;

  if (cardInfo.ramSize == 0l)
    if (dlp_ReadStorageInfo(sd, 0, &cardInfo) < 0)
      return bReturn;

  pilotProgress->setProgress(pilotProgress->progress()+1);
  pilotProgress->resize(sz);

  bReturn = TRUE;
  qsName.sprintf("%s", cardInfo.manufacturer);
  
  return bReturn;
}


/*!
  Installs a database to a Pilot.

  Expects the name of the database and a full qualified filename.

  Returns TRUE on success, FALSE otherwise.
*/

Pilot::InstallResult Pilot::installDB(QString qsDBName, QString qsFName)
{
  struct pi_file *pi_f;
  struct DBInfo dbInfo;
  int iFlags;
  int db;
  int version;
  void *buffer;
  int l, j;
  bool bErr=FALSE;
  
  version = pi_version(sd);
  
  if (dlp_OpenConduit(sd)<0)
    return (InstConduitError);

  pi_f = pi_file_open(qsFName.data());
  if (pi_f == 0)
    return (InstOpenError);

  pi_file_get_info(pi_f, &dbInfo);
  iFlags = dbInfo.flags;
  
  dlp_DeleteDB(sd, 0, dbInfo.name);

  if (dlp_CreateDB(sd, dbInfo.creator, dbInfo.type, 0, 
                   iFlags, dbInfo.version, dbInfo.name, &db) < 0)
    {
      pi_file_close(pi_f);
      return (InstCreateError);
    }
  
  pi_file_get_app_info(pi_f, &buffer, &l);

  if (l>0)
    dlp_WriteAppBlock(sd, db, buffer, l);

  int nentries;
  pi_file_get_entries(pi_f, &nentries);

  pilotProgress->setTotalSteps(nentries);
  pilotProgress->setProgress(0);

  for (j=0; j<nentries; j++)
    {
      unsigned long id;
      int size;
      int attr;
      int category;

      qsLabel.sprintf("         Writing record no. %3d of %3d            \n\n",
                      j+1, nentries);
  
      pilotProgress->setLabelText(qsLabel.data());
      pilotProgress->setProgress(j+1);

      if (pi_file_read_record(pi_f, j, &buffer, &size, &attr, 
                              &category, &id) < 0)
	{
	  bErr=TRUE;
	  break;
	}

      if ((attr & (dlpRecAttrArchived|dlpRecAttrDeleted)) && 
          (version  < 0x0101))
        continue;
      if (dlp_WriteRecord(sd, db, attr, id, category, buffer, size, 0)<0)
	{
	  bErr=TRUE;
	  break;
	}
    }

  if (bErr)
    dlp_AddSyncLogEntry(sd, "Error writing AddressDB");
  else
    dlp_AddSyncLogEntry(sd, "AddressDB written");
  pi_file_close(pi_f);
  pilotProgress->setProgress(pilotProgress->totalSteps());
  
  return (dlp_CloseDB(sd, db) ? InstCloseDBError : InstallSuccess);
}


/*!
  Retrieve a database from a Pilot.

  Expects the name fo the database and a full qualified filename.

  Returns TRUE on success, FALSE otherwise.
*/

Pilot::RetrieveResult Pilot::retrieveDB(QString qsDBName, QString qsFName)
{
  struct DBInfo info;
  struct pi_file *pi_f;
  QString qsTmp;

  int db, l, j;
  unsigned char buffer[0xffff];

  if (sd == 0)
    return (NoSocketError);
  
  if (dlp_OpenConduit(sd)<0)
    return (ConduitError);

  if (dlp_FindDBInfo(sd, 0, 0, qsDBName.data(), 0, 0, &info) < 0)
    return (FindDBError);
  
  info.flags &= 0xff;

  pi_f = pi_file_create(qsFName.data(), &info);
  if (pi_f == 0)
    return (FileCreateError);

  if (dlp_OpenDB(sd, 0, dlpOpenRead|dlpOpenSecret, info.name, &db) < 0)
    return (OpenDBError);
  
  l = dlp_ReadAppBlock(sd, db, 0, buffer, 0xffff);
  if (l>0)
    pi_file_set_app_info(pi_f, buffer, l);
  
  if (dlp_ReadOpenDBInfo(sd, db, &l) < 0)
    return (ReadDBInfoError);

  pilotProgress->setTotalSteps(l);
  pilotProgress->reset();
  pilotProgress->setProgress(0);
  
  for(j=0; j<l; j++)
    {
      unsigned long id;
      int size, attr, category;

      if ((dlp_ReadRecordByIndex(sd, db, j, buffer, &id, &size, &attr, &category) < 0))
	{
	  dlp_CloseDB(sd, db);
	  return (ReadRecordError);
	}

      if (attr & (dlpRecAttrArchived|dlpRecAttrDeleted))
        continue;

      qsLabel.sprintf("         Fetching record no. %3d of %3d            \n\n", j+1, l);
  
      pilotProgress->setLabelText(qsLabel.data());
      pilotProgress->setProgress(j);
      
      if (pi_file_append_record(pi_f, buffer, size, attr, category, id) < 0)
	{
          dlp_CloseDB(sd, db);
	  return(FileAppendError);
	}
    }

  dlp_AddSyncLogEntry(sd, "AddressDB fetched");
  pi_file_close(pi_f);
  pilotProgress->setProgress(pilotProgress->totalSteps());

  return ((dlp_CloseDB(sd, db) == 0) ? RetrieveSuccess : CloseDBError);
}


/*! 
  Purges all marked records on all databases on a Pilot.

  Returns TRUE on success, FALSE otherwise.
*/

void Pilot::purgeDB()
{
  struct DBInfo dbInfo;
  int i, h;
  
  if (sd == 0)
    return;

  if (dlp_OpenConduit(sd) < 0)
    return;

  qsLabel.sprintf("        Purging DBs on Pilot       \n\n");
  pilotProgress->setLabelText(qsLabel.data());
  pilotProgress->resize(sz);

  i=0;
  for(;;)
    {
      if (pilotProgress->wasCancelled())
	break;

      if (dlp_ReadDBList(sd, 0, 0x80, i, &dbInfo) < 0)
	break;
      i = dbInfo.index + 1;
      if (dbInfo.flags & 1)
	continue;

      pilotProgress->setProgress(pilotProgress->progress()+1);

      h = 0;
      if ((dlp_OpenDB(sd, 0, 0x40|0x80, dbInfo.name, &h) >= 0) &&
	  (dlp_CleanUpDatabase(sd, h) >= 0)                    &&
          (dlp_ResetSyncFlags(sd, h) >= 0) )
	{
          pilotProgress->setProgress(pilotProgress->progress()+1);
	}
      else
	{
          qsLabel.sprintf("     Purging %s : failed  \n\n", dbInfo.name);
          pilotProgress->setLabelText(qsLabel.data());
          pilotProgress->resize(sz);
          pilotProgress->setProgress(pilotProgress->progress()+1);
	}

      if (h != 0)
        dlp_CloseDB(sd, h);
    }

  voidSyncFlags();

  dlp_AddSyncLogEntry(sd, "Databases purged");
}

void Pilot::voidSyncFlags()
{
  if (sd == 0)
    return;
  if (dlp_ReadUserInfo(sd, &userInfo) >=0)
    {
      userInfo.lastSyncPC = 0x00000000;
      userInfo.lastSyncDate = userInfo.successfulSyncDate = time(0);
      dlp_WriteUserInfo(sd, &userInfo);
    }
}
