/* Copyright (C) 1997, 1998, 1999, 2000 Michael Wiedmann

   This file is part of pi-address.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   Additionally, you are granted permission to assume, for the purposes
   of distributing this program in object code or executable form
   under Section 3 of the GNU Public License, that the QT library 
   is normally distributed with the major components of the 
   operating system on which the executable or object code runs.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
   USA.

   Written by Michael Wiedmann <mw@miwie.in-berlin.de>
*/


#include <ctype.h>
#include <pwd.h>
#include <stdlib.h>
#include <unistd.h>

#include "common.h"

#include "pia.h"
#include "pi-dlp.h"
#include "addrdlg.h"
#include "notedlg.h"
#include "pilot.h"
#include "rpdbfile.h"
#include "wpdbfile.h"
#include "rcsvfile.h"
#include "cateditdlg.h"

#include <qapp.h>
#include <qdatetm.h>
#include <qframe.h>
#include <qkeycode.h>
#include <qfiledlg.h>
#include <qmsgbox.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qprogdlg.h>
#include <qregexp.h>
#include <qtooltip.h>
#include <qtstream.h>

#include "icons.xpm"

#include "pia.moc"

#include "iso8859_5.tab"
#include "koi8_r.tab"

extern "C" {
// TODO: forward declarations (because of a name conflict of h-files I
//       cannot easily include the proper h-file).
extern int pack_Address(Address *a, unsigned char *record, int line);
}


PiAddress::PiAddress( QWidget *parent, const char *name )
  : QWidget( parent, name ),
    toolBar( this, 0),
    statusBar(20, this, 0)
{
    QString qsTmp;

    QPixmap p1(openXpm);
    QPixmap p2(newXpm );
    QPixmap p3(saveXpm);

    rMouseMenu = new QPopupMenu();
    idEdit   = rMouseMenu->insertItem("Edit",   this, SLOT(editAddr()));
    idNew    = rMouseMenu->insertItem("New",    this, SLOT(newAddr()));
    idClone  = rMouseMenu->insertItem("Clone",  this, SLOT(cloneAddr()));
    rMouseMenu->insertSeparator();
    idDelete = rMouseMenu->insertItem("Delete", this, SLOT(deleteAddr()));
    idDelCat = rMouseMenu->insertItem("Delete Addresses", this, SLOT(deleteAddresses()));
    rMouseMenu->insertSeparator();
    idExec1  = rMouseMenu->insertItem("User def. appl. #1", this, SLOT(execUser1()));

    fileMenu = new QPopupMenu();
    fileMenu->insertItem( p1, "&Open...",  this, SLOT(open()), CTRL+Key_O );
    idReread = fileMenu->insertItem("&Reread database", this, 
				    SLOT(rereadFile()), CTRL+Key_R);
    fileMenu->insertItem( p2, "&New", this, SLOT(news()), CTRL+Key_N );
    idSave   = fileMenu->insertItem( p3, "&Save", this, SLOT(save()), CTRL+Key_S );
    idSaveAs = fileMenu->insertItem( "&Save As...", this, SLOT(save_as()), CTRL+Key_A );
    fileMenu->insertItem( "&Close", this, SLOT(closeDoc()), CTRL+Key_W );
    fileMenu->insertSeparator();

    QPopupMenu *exportMenu = new QPopupMenu;
    idExport = exportMenu->insertItem("Export as CSV...", this, SLOT(exportCSV()));
    idExportLDIF = exportMenu->insertItem("Export as LDIF...", this, SLOT(exportLDIF()));
    idExportXML  = exportMenu->insertItem("Export as XML...", this, SLOT(exportXML()));
    fileMenu->insertItem("Export", exportMenu);

    QPopupMenu *importMenu = new QPopupMenu;
    importMenu->insertItem("Import CSV...", this, SLOT(importCSV()));
    importMenu->insertItem("Import PilotMgr CSV", this, SLOT(importPMgr()));
#ifdef IMPORT_XML
    importMenu->insertItem("Import XML...", this, SLOT(importXML()));
#endif
    fileMenu->insertItem("Import", importMenu);

    fileMenu->insertSeparator();
    printID = fileMenu->insertItem( "&Print", this, SLOT(printer()), CTRL+Key_P  );
    fileMenu->insertSeparator();
    fileMenu->insertItem( "&Reread config file",  this, SLOT(rereadConfig()));
    
    fileMenu->insertSeparator();
    fileMenu->insertItem( "E&xit",  this, SLOT(quit()), CTRL+Key_Q );

    fileMenu->setItemEnabled(idSave,   FALSE);
    fileMenu->setItemEnabled(idReread, FALSE);
    fileMenu->setItemEnabled(idSaveAs, FALSE);
    fileMenu->setItemEnabled(printID,  FALSE);
    fileMenu->setItemEnabled(idExport, FALSE);
    fileMenu->setItemEnabled(idExportLDIF, FALSE);
    fileMenu->setItemEnabled(idExportXML,  FALSE);

    recordMenu = new QPopupMenu;
    idRecordEdit  = recordMenu->insertItem("Edit...", this,
					   SLOT(editAddr()));
    idRecordNew   = recordMenu->insertItem("New...", this,
					   SLOT(newAddr()));
    idRecordClone = recordMenu->insertItem("Clone...", this,
					   SLOT(cloneAddr()));
    recordMenu->insertSeparator();
    idRecordDelete = recordMenu->insertItem("Delete...", this,
					    SLOT(deleteAddr()));
    idRecordDeleteAddresses = recordMenu->insertItem("Delete addresses...", 
						     this, 
						     SLOT(deleteAddresses()));
    recordMenu->insertSeparator();
    idRecordExec1 = recordMenu->insertItem("User def. appl. #1", 
					   this, 
					   SLOT(execUser1()));
    
    pilotMenu  = new QPopupMenu;
    pilotRead  = pilotMenu->insertItem("&Read...",  this, SLOT(readPilot()));
    pilotWrite = pilotMenu->insertItem("&Write...", this, SLOT(writePilot()));
    pilotMenu->insertSeparator();
    pilotPurge = pilotMenu->insertItem("&Purge DBs...", this, SLOT(purgePilot()));

    pilotList = pilotMenu->insertItem("&List Info...", this, SLOT(infoPilot()));

    pilotMenu->setItemEnabled(pilotWrite, FALSE);

    QPopupMenu *infoMenu = new QPopupMenu();
    infoMenu->insertItem("DB &Info", this, SLOT(info()));

    helpMenu = new QPopupMenu;
    idFAQ = helpMenu->insertItem("&F A Q", this, SLOT(helpFAQ()) );
    helpMenu->insertSeparator();
    helpMenu->insertItem("&About", this, SLOT(about()), CTRL+Key_H );
    helpMenu->insertItem("&Copyright", this, SLOT(copyright()), CTRL+Key_C);
    helpMenu->insertItem("About &Qt", this, SLOT(aboutQt()) );

    menu = new QMenuBar(this);
    menu->insertItem("&File",    fileMenu);
    menu->insertItem("&Records", recordMenu);
    menu->insertItem("&Pilot",   pilotMenu);
    menu->insertItem("&Info",    infoMenu); 
    
    menu->insertSeparator();
    menu->insertItem( "&Help", helpMenu );
    menu->setSeparator( QMenuBar::InWindowsStyle );

    int xPos, yPos;
    
    int hMenu = menu->height();
   
    populateToolbar();
    toolBar.setGeometry(0, hMenu, width(), hMenu);
    int hToolBar = toolBar.height();
   
    xPos = lMargin;
    yPos = hMenu + hToolBar + tMargin;
    
    lCat = new QLabel(this);
    lCat->setText("Category");
    lCat->setMinimumSize(lCat->sizeHint());
    lCat->setGeometry(lMargin, hMenu + hToolBar + tMargin,
                      lCat->sizeHint().width(), lCat->sizeHint().height());
    
    cat = new QComboBox(this);
    cat->setGeometry(lMargin+lCat->x()+6+lCat->width(), yPos-6, 100, 20);
    cat->insertItem("Unfiled");
    cat->insertItem("Company");
    cat->insertItem("Private");
    cat->insertItem("QuickList");
    cat->insertItem("All");
    cat->setCurrentItem(cat->count()-1);

    cat->setGeometry(lMargin+lCat->x()+6+lCat->width(), yPos-6, 100, 20);
    cat->adjustSize();
    QToolTip::add(cat, "Change category");
    connect(cat, SIGNAL(activated(int)),
            this,SLOT(displayAddresses()));

    pbCatEdit = new QPushButton("Ed&it Cat.", this);
    pbCatEdit->setMinimumSize(pbCatEdit->sizeHint());
    connect(pbCatEdit,  SIGNAL(clicked()), this, SLOT(editCat()));

    QToolTip::add(pbCatEdit, "Edit categories");
    pbCatEdit->move(cat->x()+lMargin+cat->width()+8+lMargin, yPos-6);
    // pbCatEdit->setEnabled(FALSE);
    
    lbAddr = new ListBox(this);
    lbAddr->setFont(QFont("fixed", 12));

    yPos += cat->height() + 6;
    
    lbAddr->setGeometry(lMargin, yPos, 300, 250);
    QToolTip::add(lbAddr, "List of all records");

    connect(lbAddr, SIGNAL(rMouseButtonPressed()), 
            this,   SLOT(rMouseActivated()));

    yPos += tMargin+lbAddr->height();
    
    lSearch = new QLabel(this);
    lSearch->setText("Search:");
    lSearch->setGeometry(lMargin, yPos,
                         lSearch->sizeHint().width(), 
                         lSearch->sizeHint().height());

    leSearch = new QLineEdit(this);
    leSearch->setGeometry(lMargin+lSearch->width()+6, yPos-4,
                          220, leSearch->sizeHint().height());

#ifdef _DEBUG_
    debug("DBG (CTOR) Qt-Version -> %s", qVersion());
#endif
    if (strcmp(qVersion(), "2.") < 0)
      connect (leSearch, SIGNAL(textChanged(const char *)), 
               this, SLOT(searchRecord(const char *)));
    else
      connect (leSearch, SIGNAL(textChanged(const QString &)), 
               this, SLOT(searchRecord(const QString &)));

    connect (leSearch, SIGNAL(returnPressed()),
             this, SLOT(editAddr()));
    QToolTip::add(leSearch, "Search for record");

    pbEdit = new QPushButton("&Edit...", this);
    pbEdit->setMinimumSize(pbEdit->sizeHint());
    QSize sz2 = pbEdit->sizeHint();
    connect(pbEdit, SIGNAL(clicked()), this, SLOT(editAddr()));
    QToolTip::add(pbEdit, "Edit selected record");
    
    pbNew  = new QPushButton("&New...", this);
    pbNew->setMinimumSize(pbNew->sizeHint());
    QSize sz3 = pbNew->sizeHint();
    connect(pbNew, SIGNAL(clicked()), this, SLOT(newAddr()));
    QToolTip::add(pbNew, "Add new record");

    pbClone = new QPushButton("&Clone...", this);
    pbClone->setMinimumSize(pbClone->sizeHint());
    QSize sz4 = pbClone->sizeHint();
    connect(pbClone, SIGNAL(clicked()), this, SLOT(cloneAddr()));
    QToolTip::add(pbClone, "Clone selected record");

    pbDelete = new QPushButton("&Delete...", this);
    pbDelete->setMinimumSize(pbDelete->sizeHint());
    QSize sz5 = pbDelete->sizeHint();
    connect(pbDelete, SIGNAL(clicked()), this, SLOT(deleteAddr()));
    QToolTip::add(pbDelete, "Delete selected record");

    pbNote = new QPushButton("&Show Note", this);
    pbNote->setMinimumSize(pbNote->sizeHint());
    connect(pbNote, SIGNAL(clicked()), this, SLOT(showNote()));
    QToolTip::add(pbNote, "Show attached Note");
    
    int h2, h3, h4, h5, w2, w3, w4, w5, w, h;
    h2 = sz2.height();
    h3 = sz3.height();
    h4 = sz4.height();
    h5 = sz5.height();
    w2 = sz2.width();
    w3 = sz3.width();
    w4 = sz4.width();
    w5 = sz5.width();
    
    h = (h2 > h3) ? h2 : h3;
    h = (h4 > h ) ? h4 : h;
    h = (h5 > h ) ? h5 : h;
    w = (w2 > w3) ? w2 : w3;
    w = (w4 > w ) ? w4 : w;
    w = (w5 > w ) ? w5 : w;
    
    QSize sz(w, h);
    pbEdit   ->resize(sz);
    pbNew    ->resize(sz);
    pbClone  ->resize(sz);
    pbDelete ->resize(sz);
    pbNote   ->resize(sz);
    pbCatEdit->resize(sz);
    
    yPos += leSearch->height() + tMargin;
    
    pbEdit  ->move(lMargin,                                        yPos);
    pbNew   ->move(lMargin+w+bttnMargin,                           yPos);
    pbClone ->move(lMargin+w+bttnMargin+w+bttnMargin,              yPos);
    pbDelete->move(lMargin+w+bttnMargin+w+bttnMargin+w+bttnMargin, yPos);

    lAddr = new QLabel(this);
    lAddr->setGeometry(lbAddr->x()+lbAddr->width()+lMargin, lbAddr->y(),
                       300, 
                       lbAddr->height()+2*tMargin);

    QToolTip::add(lAddr, "Display of selected record");

    connect(lbAddr, SIGNAL(highlighted(int)),
            this, SLOT(enableButtons(int)));
    connect(lbAddr, SIGNAL(highlighted(int)),
            this, SLOT(displayRecord(int)));
    connect(lbAddr, SIGNAL(selected(int)),
            this, SLOT(editAddr(int)));

    yPos += h + statusBar.height() + bMargin;
    xPos  = lMargin+lbAddr->width()+lMargin+lAddr->width()+lMargin;
    toolBar.resize(xPos, toolBar.height());
    setFixedSize(xPos, yPos);
    setMinimumSize(xPos, yPos);
    setMaximumSize(xPos, yPos);

    pbNote->move(xPos-lMargin-pbNote->width(), pbDelete->y());
    pbNote->setEnabled(FALSE);

    qsFname = "";
    QString s("- untitled -"); 
    qsTmp   = s.leftJustify(36, ' ');
    QString t("File: ");
    t += qsTmp;
    statusMsg   = statusBar.InsertTextRegion(1, "Ready");
    statusDevice= statusBar.InsertTextRegion(0, "Dev: /dev/pilot   ");

    statusFname = statusBar.InsertTextRegion(0, (const char *)t);
    statusCount = statusBar.InsertTextRegion(0, "Records: 000");
   
    statusBar.setGeometry(0, height()-20, width(), 20);

    bModified = FALSE;
    disableButtons();

    leSearch->setFocus();

    AddrList = 0;
    pAppInfo = new AddressAppInfo;
    initAppInfo();

    creaTime = modTime = backupTime = 0;

    pPrinter = 0;

    initDefValues();

    loadSystemConfigFile();
    loadUserConfigFile();
    loadDefaultDB();
}

PiAddress::~PiAddress()
{
  if (AddrList != 0)
  {
    delete AddrList;
    AddrList = 0;
  }

  if (pPrinter)
  {
    delete pPrinter;
    pPrinter = 0;
  }
}

void PiAddress::insertSorted(AddressRecord *newRec)
{
 AddressRecord *a;
 int i, iRet=1;
 bool bInserted = FALSE;
 const char *newKey, *currentKey;
 bool newFamily = FALSE, currentFamily=FALSE;

   for (a=AddrList->first(), i=0; a!=0; a=AddrList->next(), i++)
   {
      // Insert the records sorted by Family Name then  First Name.
      // If no Family Name exists then insert the record by First Name
      // If no First Name or Family Name exists insert by Company.
      // Similarly, you want to check the new record against the current
      // record in the same order. 
      // By existence we also check to make sure the field contains at
      // least 1 character.
      if (a->getFN() && strlen(a->getFN()))
	 {
	    currentKey = a->getFN();
	    currentFamily = TRUE;
	 }
      else if (a->getGN() && strlen(a->getGN()))
	 currentKey = a->getGN();
      else if (a->getCompany() && strlen(a->getCompany()))
	currentKey = a->getCompany();
      
      if (newRec->getFN() && strlen(newRec->getFN()))
	 {
	    newKey = newRec->getFN();
	    newFamily = TRUE;
	 }
      else if (newRec->getGN() && strlen(newRec->getGN()))
	newKey = newRec->getGN();
      else if (newRec->getCompany() && strlen(newRec->getCompany()))
	newKey = newRec->getCompany();
      
     if (newKey && currentKey) {
	iRet = stricmp(newKey, currentKey);
	// The keys matched.  Check to see if we compared last names.
	// If so do a secondary sort by first name.
	if (!iRet && newFamily && currentFamily && a->getFN() 
	    && newRec->getFN())
	   iRet = stricmp(newRec->getFN(), a->getFN());
     }
     
     
     if (iRet < 0)
	{
	   AddrList->insert(i, newRec);
	   bInserted = TRUE;
	   break;
	}
   }
   if (!bInserted)
      AddrList->append(newRec);
   
}

void PiAddress::showEditDlg(AddressRecord *addrRec)
{
  bool bCancel=FALSE;

  // create and display dialog
  AddressDlg addrDlg(this, 0, addrRec, pAppInfo, &bCancel, qsEditFont.data());

  // template list already initialized?
  if (AddrList == 0)
    {
      AddrList = new QList<AddressRecord>;
      AddrList->setAutoDelete(TRUE);
    }

  // Dialog finished 
  if (!bCancel)
    {
      // is there a "Family Name" OR a "Given Name" OR a "Company"
      if (strlen(addrRec->getFN())      || 
          strlen(addrRec->getCompany()) ||
          strlen(addrRec->getGN()))
	{
         insertSorted(addrRec);
         addrRec->putModified(TRUE);
         bModified = TRUE;
         if (AddrList->count() == 1)
           creaTime = time(0);
         modTime = time(0);
         fileMenu->setItemEnabled(idSaveAs,     TRUE);
         fileMenu->setItemEnabled(idExport,     TRUE);
         fileMenu->setItemEnabled(idExportLDIF, TRUE);
         fileMenu->setItemEnabled(idExportXML,  TRUE);
         toolBar.EnableItem(idToolbarExport,    TRUE);
         fileMenu->setItemEnabled(printID,      TRUE);
         toolBar.EnableItem(idToolbarPrint,     TRUE);
         statusMessageReady();
        }
      else
	{
	  delete addrRec;
	}
      
      // update display
      lAddr->repaint();
      displayAddresses();
      updateStatusCount();
    }
}

void PiAddress::cloneAddr()
{
  Address addr;
  int idx = getIndex(lbAddr->currentItem());
  AddressRecord *a = AddrList->at(idx);

  if (!a)
    return;

  // create a copy of the selected record
  a->cloneAddress(&addr);

  // create new Address Record  
  AddressRecord *addrRec = new AddressRecord(&addr, 0, TRUE);

  idx=cat->currentItem();
  addrRec->setCat((idx == cat->count()-1) ? 0 : idx);
  addrRec->putPrivate(a->getPrivate());

  // show the edit dialog
  showEditDlg(addrRec);
}

void PiAddress::newAddr()
{
  // create new - empty - Address Record  
  AddressRecord *addrRec = new AddressRecord();

  int idx=cat->currentItem();
  addrRec->setCat((idx == cat->count()-1) ? 0 : idx);

  // show the edit dialog
  showEditDlg(addrRec);
}

void PiAddress::rMouseActivated()
{
 // get saved mouse position
 QPoint  pos = lbAddr->getMousePos();
 QPoint  p(pos.x() + lbAddr->x() + qApp->mainWidget()->x(), 
           pos.y() + lbAddr->y() + qApp->mainWidget()->y());

 bool bFlag;
 if (AddrList == 0 || AddrList->count() == 0 || lbAddr->count() == 0)
   bFlag = FALSE;
 else
   bFlag = TRUE;

 // update items in PopupMenu
 rMouseMenu->setItemEnabled(idEdit,   bFlag);
 rMouseMenu->setItemEnabled(idDelete, bFlag);
 rMouseMenu->setItemEnabled(idDelCat, bFlag);
 rMouseMenu->setItemEnabled(idClone,  bFlag);
 rMouseMenu->setItemEnabled(idExec1,  bFlag);
 if (qsUserProg1.isEmpty())
   rMouseMenu->setItemEnabled(idExec1,  FALSE);

 if (bFlag && (cat->currentItem() == 0 || 
               cat->currentItem() >= cat->count()-1))
   {
     rMouseMenu->setItemEnabled(idDelCat, FALSE);
   }
 
 // popup (action via signal/slot)
 rMouseMenu->popup(p);
}

void PiAddress::newCat(int idx, char *s)
{
  int last = cat->count()-1;

  // remember last category name
  QString qs = cat->text(last);

  // change category name in combobox
  cat->changeItem(s, last);

  // and insert category "All" again at end
  cat->insertItem(qs.data());

  // append new category name in AppInfo
  memset(pAppInfo->category.name[last], 0x00, 16);
  strncpy(pAppInfo->category.name[last], s, 16);
  
  cat->setCurrentItem(cat->count()-1);
  lbAddr->setCurrentItem(0);  
  displayRecord(0);

  bModified = TRUE;
  statusMessageReady();
}

void PiAddress::deleteCat(int idx)
{
  if (AddrList == 0 || AddrList->count() == 0)
    return;

  // 1: scan all records and set category to 0 if category = idx
  // 2: correct all categories above "idx"

  AddressRecord *a;
  int i;

  for(a=AddrList->first(); a!=0; a=AddrList->next())
    {
      i = a->getCat();

      if (i == idx)
        a->setCat(0);

      if (i > idx)
	a->setCat(i-1);
    }

  // delete item from category combobox
  cat->removeItem(idx);
  cat->setCurrentItem(0);

  // FIXME: correct AppInfo!
  //        - clear names of categories in AppInfo
  //        - fill names of categories in AppInfo from cat combobox
  //          (all but the last one!)

  memset(&(pAppInfo->category.name[0]), 0x00, 16*16);
  for (i=0; i<cat->count()-1; i++)
    strcpy(pAppInfo->category.name[i], cat->text(i));

  cat->setCurrentItem(cat->count()-1);
  lbAddr->setCurrentItem(0);  
  displayRecord(0);

  // mark DB modified
  bModified = TRUE;
  statusMessageReady();
}

void PiAddress::changeCat(int idx, char *s)
{
  // change name of category in pAppInfo  ...
  memset(pAppInfo->category.name[idx], 0x00, 16);
  strncpy(pAppInfo->category.name[idx], s, 16);
  
  // ... and combobox
  cat->changeItem(s, idx);

  cat->setCurrentItem(cat->count()-1);
  lbAddr->setCurrentItem(0);  
  displayRecord(0);
  
  // mark DB modified
  bModified = TRUE;
  statusMessageReady();
}

void PiAddress::editCat()
{
 CatEditDlg catEditDlg(this, "EditCatDlg", &(pAppInfo->category), bConfirmOps);

 connect(&catEditDlg, SIGNAL(newCat(int, char *)), this, SLOT(newCat(int, char*)));
 connect(&catEditDlg, SIGNAL(deleteCat(int)), this, SLOT(deleteCat(int)));
 connect(&catEditDlg, SIGNAL(changeCat(int, char *)), this, SLOT(changeCat(int , char *)));

 catEditDlg.exec();
}

void PiAddress::showNote()
{
  AddressRecord *a = AddrList->at(getIndex(lbAddr->currentItem()));

  NoteDlg noteDlg(this, "NoteDlg", a->getNote());
  noteDlg.exec();
}


void PiAddress::editAddr()
{
  editAddr(lbAddr->currentItem());
}


void PiAddress::editAddr(int i)
{
  if (i == -1)
    return;

  statusMessageReady();

  int k = getIndex(i);

  // get Address Record at index
  AddressRecord *a = AddrList->at(k);

  bool bCancel=FALSE;
  // create and display dialog with Address Dialog
  AddressDlg addrDlg(this, 0, a, pAppInfo, &bCancel, qsEditFont.data());

  if (!bCancel)
    {
      // record modification time
      modTime = time(0);

      // update display
      displayAddresses();
      displayRecord(i);
      lAddr->repaint();
      updateStatusCount();
      leSearch->setText("");
      lbAddr->setFocus();
      lbAddr->setCurrentItem(i);
      lbAddr->centerCurrentItem();
      leSearch->setFocus();
      bModified = TRUE;
      statusMessageReady();
    }
}


void PiAddress::displayAddresses()
{
  int i, j;
  AddressRecord *a;
  QString s, t, tmp;
  QString msg, t0, t1, t2, t3, t4, t5;
  
  lbAddr->clear();
  if (AddrList == 0 || AddrList->count() == 0)
    return;

  QFontMetrics fm(lbAddr->font());
    
    for(a=AddrList->first(), i=0; a!=0; a=AddrList->next(), i++)
    {
      if (a && 
          ((cat->currentItem() == cat->count()-1) || 
           (cat->currentItem() == a->getCat()))
          )
	{
          j = a->getShowPhone();
	  
	  int posCR;
          // ist there a "Family Name"
          if (strlen(a->getFN()))
	    {
              QString qsFN = a->getFN();
	      posCR = qsFN.find(0x0a);
	      if (posCR != -1)
	        qsFN.truncate(posCR);
	      
              if (a->getGN())
		{
		  QString qsGN = a->getGN();
		  posCR = qsGN.find(0x0a);
		  if (posCR != -1)
		    qsGN.truncate(posCR);
                  t.sprintf("%s, %s", qsFN.data(), qsGN.data());
		}
              else
                t.sprintf("%s", qsFN.data());
	    }
          else if (strlen(a->getGN()))
	    {
	      QString qsGN1 = a->getGN();
	      posCR = qsGN1.find(0x0a);
	      if (posCR != -1)
		qsGN1.truncate(posCR);
              t.sprintf("%s", qsGN1.data());
	    }
	  else
	    {
	      QString qsCompany = a->getCompany();
	      posCR = qsCompany.find(0x0a);
	      if (posCR != -1)
		qsCompany.truncate(posCR);
              t.sprintf("%s", qsCompany.data());
	    }

          t0 = "";
	  
          t1 = a->getPhone1();
	  t2 = a->getPhone2();
	  t3 = a->getPhone3();
	  t4 = a->getPhone4();
	  t5 = a->getPhone5();

          switch (j)
	    {
	    case 0:
              if (t1.isEmpty())
		break;
	      posCR = t1.find(0x0a);
	      if (posCR != -1)
	        t1.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t1, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx0()][0]);
	      break;
	    case 1:
              if (t2.isEmpty())
                 break;
	      posCR = t2.find(0x0a);
	      if (posCR != -1)
	        t2.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t2, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx1()][0]);
	      break;
	    case 2:
              if (t3.isEmpty())
                break;
	      posCR = t3.find(0x0a);
	      if (posCR != -1)
	        t3.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t3, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx2()][0]);
	      break;
	    case 3:
              if (t4.isEmpty())
		break;
	      posCR = t4.find(0x0a);
	      if (posCR != -1)
	        t4.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t4, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx3()][0]);
	      break;
	    case 4:
              if (t5.isEmpty())
		break;
	      posCR = t5.find(0x0a);
	      if (posCR != -1)
	        t5.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t5, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx4()][0]);
	      break;
	    default:
              if (t1.isEmpty())
		break;
	      posCR = t1.find(0x0a);
	      if (posCR != -1)
	        t1.truncate(posCR);
              t0.sprintf("%s %c", (const char *)t1, 
                         pAppInfo->phoneLabels[a->getPhoneLabelIdx0()][0]);
	      break;
	    }

          int wPhone = fm.width((const char *)t0);
          int wName  = lbAddr->width() - wPhone - 20;

          if (wName < 100)
	    {
              wName = 100;
              wPhone= lbAddr->width() - wName - 10;
	    }

          tmp = t;
          while (fm.width((const char *)tmp) < wName)
	    {
             tmp += " ";
             t = tmp;
	    }

          tmp = t;
          while (fm.width((const char *)tmp) > wName)
	    {
             tmp = t.left(t.length()-1);
             t = tmp;
	    }

	  s.sprintf("%s %s", t.data(), t0.data());
	  lbAddr->insertItem((const char *)s);
	}
    }
}

// Display a single record in the right window
void PiAddress::displayRecord(int i)
{
  QString qsTmp, qsFN, qsGN, qsTitle, qsCompany;
  QString qsPhone1, qsPhone2, qsPhone3, qsPhone4, qsPhone5;
  QString qsAddress, qsZIP, qsTown, qsState, qsCountry, qsNote;
  QString qsCustom1, qsCustom2, qsCustom3, qsCustom4;
  
  QPainter p(lAddr);
  QFont fBig  (qsRecordFont.data(), 14, QFont::Bold);
  QFont fSmall(qsRecordFont.data(), 12);

  AddressRecord *a = AddrList->at(getIndex(i));

  if (a)
    {
      qsFN      = a->getFN();
      qsGN      = a->getGN();
      qsTitle   = a->getTitle();
      qsCompany = a->getCompany();
      qsPhone1  = a->getPhone1();
      qsPhone2  = a->getPhone2();
      qsPhone3  = a->getPhone3();
      qsPhone4  = a->getPhone4();
      qsPhone5  = a->getPhone5();
    
      qsAddress = a->getAddress();
      qsZIP     = a->getZIP();
      qsTown    = a->getTown();
      qsState   = a->getState();
      qsCountry = a->getCountry();

      qsNote    = a->getNote();
      
      qsCustom1 = a->getDefined1();
      qsCustom2 = a->getDefined2();
      qsCustom3 = a->getDefined3();
      qsCustom4 = a->getDefined4();
      
      p.eraseRect(lAddr->rect());
      
      int x1 = 10, x2 = 100, y = 20;

      if (qsFN.length() > 0)
	{
          qsTmp = qsGN.data();
          if  (!qsGN.isEmpty())
            qsTmp.append(" ");
          qsTmp.append(qsFN.data());
	}
      else if (qsGN.length() > 0)
          qsTmp = qsGN.data();
      else
	  qsTmp = qsCompany.data();
      
      p.setFont(fBig); 
      p.drawText(x1, y, (const char *)qsTmp);
      y += 28;

      p.setFont(fSmall); 
      if (!qsPhone1.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->phoneLabels[a->getPhoneLabelIdx0()]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsPhone1);
          y += 18;
	}

      if (!qsPhone2.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->phoneLabels[a->getPhoneLabelIdx1()]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsPhone2);
          y += 18;
	}

      if (!qsPhone3.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->phoneLabels[a->getPhoneLabelIdx2()]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsPhone3);
          y += 18;
	}

      if (!qsPhone4.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->phoneLabels[a->getPhoneLabelIdx3()]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsPhone4);
          y += 18;
	}
      
      if (!qsPhone5.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->phoneLabels[a->getPhoneLabelIdx4()]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsPhone5);
          y += 18;
	}
      
      if (!qsTitle.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[13]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsTitle);
          y += 18;
	}
      
      if (!qsCompany.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[2]);
          p.drawText(x1, y, (const char *)qsTmp);
          p.drawText(x2, y, (const char *)qsCompany);
          y += 18;
	}
      
      y += 10;

      if (!qsAddress.isEmpty())
	{
          int pos0=0, pos1;
	  pos1 = qsAddress.find(0x0a, 0);
	  if (pos1 != -1)
	    {
              while ((pos1 = qsAddress.find(0x0a, pos0)) != -1)
		{
		  qsTmp = qsAddress.mid(pos0, pos1-pos0);
	          p.drawText(x1, y, (const char *)qsTmp);
	          y += 14;
                  pos0 = pos1+1;
		}
	      qsTmp = qsAddress.mid(pos0, qsNote.length());
	      p.drawText(x1, y, (const char *)qsTmp);
	    }
          else
            p.drawText(x1, y, (const char *)qsAddress);

          y += 18;
	}

      // german database
      if (stricmp(cat->text(0), "Nicht abgelegt") == 0)
	{
          qsTmp = qsZIP.data();
          qsTmp.append("  ");
          qsTmp.append(qsTown.data());
          p.drawText(x1, y, (const char *)qsTmp);

          if (!qsZIP.isEmpty() || !qsTown.isEmpty())
            y += 18;
      
          if (!qsState.isEmpty())
	  {
            p.drawText(x1, y, (const char *)qsState);
            y += 18;
	  }
	}
      // american database
      else
	{
          qsTmp = qsTown.data();
	  if (!qsState.isEmpty())
	    qsTmp.append(", ");
          qsTmp.append(qsState.data());
	  qsTmp.append(" ");
          qsTmp.append(qsZIP.data());
	  p.drawText(x1, y, (const char *)qsTmp);

          if (!qsState.isEmpty() || !qsZIP.isEmpty() || !qsTown.isEmpty())
	    y += 18;
	}

      if (!qsCountry.isEmpty())
	{
          p.drawText(x1, y, (const char *)qsCountry);
          y += 18;
	}

      if (!qsCustom1.isEmpty() ||
          !qsCustom2.isEmpty() ||
          !qsCustom3.isEmpty() ||
          !qsCustom4.isEmpty())
        y += 10;

      if (!qsCustom1.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[14]);
	  p.drawText(x1, y, (const char *)qsTmp);
	  p.drawText(x2, y, (const char *)qsCustom1);
          y += 18;
	}
      if (!qsCustom2.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[15]);
	  p.drawText(x1, y, (const char *)qsTmp);
	  p.drawText(x2, y, (const char *)qsCustom2);
          y += 18;
	}
      if (!qsCustom3.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[16]);
	  p.drawText(x1, y, (const char *)qsTmp);
	  p.drawText(x2, y, (const char *)qsCustom3);
          y += 18;
	}
      if (!qsCustom4.isEmpty())
	{
	  qsTmp.sprintf("%s:", pAppInfo->labels[17]);
	  p.drawText(x1, y, (const char *)qsTmp);
	  p.drawText(x2, y, (const char *)qsCustom4);
          y += 18;
	}

      if (!qsNote.isEmpty())
	{
          pbNote->setEnabled(TRUE);
	}
      else
	{
          pbNote->setEnabled(FALSE);
	}
    }
}

int PiAddress::getIndex(int i)
{
 int iCat, k, ct;

 // index of selected category
 iCat = cat->currentItem();

 // if category equals "All" we are done (All ist the last category!)
 if (iCat == cat->count()-1)
   return (i);

 AddressRecord *a;
 ct = -1;
 for (a=AddrList->first(), k=0; a!=0; a=AddrList->next(), k++)
   {
    if (a->getCat() == iCat)
      ct++;

    if (ct == i)
      break;
   }
 return k;
}

void PiAddress::enableButtons(int index)
{
  index = index;

  pbEdit   ->setEnabled(TRUE);
  pbClone  ->setEnabled(TRUE);
  pbDelete ->setEnabled(TRUE);
  pbCatEdit->setEnabled(TRUE);

  recordMenu->setItemEnabled(idRecordEdit,            TRUE);
  recordMenu->setItemEnabled(idRecordClone,           TRUE);
  recordMenu->setItemEnabled(idRecordDelete,          TRUE);
  recordMenu->setItemEnabled(idRecordDeleteAddresses, TRUE);
  if (!qsUserProg1.isEmpty())
    recordMenu->setItemEnabled(idRecordExec1,          TRUE);
  else
    recordMenu->setItemEnabled(idRecordExec1,          FALSE);

  if (cat->currentItem() == 0 ||
      cat->currentItem() >= cat->count()-1)
  {
    recordMenu->setItemEnabled(idRecordDeleteAddresses, FALSE);
  }
}

void PiAddress::disableButtons()
{
  pbEdit   ->setEnabled(FALSE);
  pbClone  ->setEnabled(FALSE);
  pbDelete ->setEnabled(FALSE);
  pbNote   ->setEnabled(FALSE);
  pbCatEdit->setEnabled(FALSE);

  recordMenu->setItemEnabled(idRecordEdit,            FALSE);
  recordMenu->setItemEnabled(idRecordClone,           FALSE);
  recordMenu->setItemEnabled(idRecordDelete,          FALSE);
  recordMenu->setItemEnabled(idRecordDeleteAddresses, FALSE);
  recordMenu->setItemEnabled(idRecordExec1,           FALSE);
}

void PiAddress::execUser1()
{
  if (qsUserProg1.isEmpty())
    return;

  int i   = lbAddr->currentItem();
  int idx = getIndex(i);
  AddressRecord *a = AddrList->at(idx);

  if (!a)
    return;

  int j=0;
  QString qsUser = qsUserProg1;
  QString qsTmp;

  // expand %XY sequences
  while ((j = qsUser.find('%', j)) > 0)
    {
      // switch over next character
      char ch;
      ch = *(const char *)qsUser.mid(j+1, 1);
      switch ((int)ch)
	{
	case 'l':
	  qsTmp="";
	  qsTmp = a->getFN();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'f':
	  qsTmp="";
	  qsTmp = a->getGN();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'i':
	  qsTmp="";
	  qsTmp = a->getTitle();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'o':
	  qsTmp="";
	  qsTmp = a->getCompany();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'a':
	  qsTmp="";
	  qsTmp = a->getAddress();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'z':
	  qsTmp="";
	  qsTmp = a->getZIP();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 't':
	  qsTmp="";
	  qsTmp = a->getTown();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 's':
	  qsTmp="";
	  qsTmp = a->getState();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	case 'n':
	  qsTmp="";
	  qsTmp = a->getCountry();
	  qsUser.remove(j, 2);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;
	  
	case '%':
	  qsUser.remove(j, 1);
	  j++;
	  break;
	  
	case 'c':
	  qsTmp="";
	  qsTmp = a->getDefined(atoi(qsUser.mid(j+2,1)));
	  qsUser.remove(j, 3);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;
	  
	case 'p':
	  qsTmp="";
	  qsTmp = a->getPhone(atoi(qsUser.mid(j+2,1)));
	  qsUser.remove(j, 3);
	  if (!qsTmp.isEmpty())
	    qsUser.insert(j, qsTmp.data());
	  else
	    qsUser.insert(j, " ");
	  break;

	default:
	  // invalid: remove and advance to next char
	  qsUser.remove(j, 1);
	  j++;
	  break;
	}
    }

  // execute user application
  if (system((const char *)qsUser))
    {
      QString s;
      s.sprintf("Couldn't start the user defined application!\n\n"
       	        "Please correct the problem in the configuration file \n"
       	        "(/etc/pi-addressrc or ~/.pi-addressrc) and try again.\n");
      messageBox(s);
    }
}

// delete all addresses of the selected category
void PiAddress::deleteAddresses()
{
  int iCurrent=cat->currentItem();
  
  if (iCurrent == 0 || iCurrent >= cat->count()-1)
    return;

  // ask user first!
  if (bConfirmOps && QMessageBox::warning(this, "Delete Address Records",
                           "Do you really want to delete the records in this category?",
                           "Yes", "No", "Cancel", 1) != 0)
    return;
  
  AddressRecord *a;
  int i;
  for (i=0; i<AddrList->count(); i++)
    {
      a=AddrList->at(i);
      if (!a || (a->getCat() != cat->currentItem()))
	continue;
      
      AddrList->remove(a);
      // NOTE: QList->remove() moves the next list item backward!
      i--;
    }

  cat->setCurrentItem(cat->count()-1);
  displayAddresses();
  if (lbAddr->count())
    {
      lbAddr->setCurrentItem(0);
      lbAddr->centerCurrentItem();
    }
  bModified = TRUE;
  modTime = time(0);
  statusMessageReady();
  updateStatusCount();
}

// delete a single Address record
void PiAddress::deleteAddr()
{
  QString msg;
  int i   = lbAddr->currentItem();
  int idx = getIndex(i);

  AddressRecord *a = AddrList->at(idx);

  if (!a)
    return;

  int b=0;
  if (bConfirmOps)
    {
      // ask the user
      b=QMessageBox::warning(this, "Delete Address Record",
                            "Do you really want to delete the record ?",
                            "Yes", "No", "Cancel", 1);
    }

  if (b==0 && lbAddr->currentItem() >= 0)
    {
      AddrList->remove(idx);
      displayAddresses();
      if (lbAddr->count())
	{
          lbAddr->setCurrentItem((i-1)<0 ? 0 : (i-1));
          lbAddr->centerCurrentItem();
	}
      bModified = TRUE;
      modTime = time(0);
      statusMessageReady();
    }

  updateStatusCount();
}

void PiAddress::searchRecord(const QString &t)
{
  searchRecord((const char *)t);
}

// search for an entry, select it and scroll the listbox
void PiAddress::searchRecord(const char *t)
{
  for (int i=0; i<(int)lbAddr->count(); i++)
    {
      if (!strncasecmp(t, lbAddr->text(i), strlen(t)))
	{
          lbAddr->setFocus();
          lbAddr->setCurrentItem(i);
          lbAddr->centerCurrentItem();
          leSearch->setFocus();
	  return;
	}
    }
}

void PiAddress::writeFile()
{
  if (AddrList == 0 || AddrList->count() == 0)
    return;

  int  iBufSize = 2048;
  char *ptrBuffer = (char *)malloc(iBufSize);
  memset(ptrBuffer, 0x00, iBufSize);

  QString qsTmp;
  statusMessageReady();
    
  // initialize DBInfo
  memset(&dbInfo, 0x00, sizeof(dbInfo));
  dbInfo.type    = pi_mktag('D','A','T','A');
  dbInfo.creator = pi_mktag('a','d','d','r');
  dbInfo.createDate = creaTime;
  dbInfo.modifyDate = modTime;
  dbInfo.backupDate = backupTime = time(0);
  strcpy(&dbInfo.name[0], "AddressDB");
  
  bool bCancel = FALSE;
  WritePDBFile *pdbFile = new WritePDBFile((const char *)qsFname, 
                                           &dbInfo, &bCancel);

  // create buffer for AppInfo and init. from current data
  int l = fillAppInfo(ptrBuffer);

  // call pi_file_set_app_info()
  pdbFile->setAppInfo(ptrBuffer, l);

  // append records
  int i;
  AddressRecord *addrRec;
  Address a;
  long lUid=0l;
  
  int iRecSize, iMaxRecSize=0;
  int n=(int)AddrList->count();

  // determine the largest record
  for (addrRec  = AddrList->first(), i=0;
       addrRec != 0;
       addrRec  = AddrList->next(),  i++)
    {
      fillAddress(addrRec, &a);
      iRecSize = pack_Address(&a, (unsigned char *)0, 0);
      if (iRecSize > iMaxRecSize)
        iMaxRecSize = iRecSize;
    }
#ifdef _DEBUG_
    debug("Max. record size: %d", iMaxRecSize);
#endif

  // reallocate buffer (multiple of 1kB)
  iBufSize = ((iMaxRecSize / 1024) + 1) << 10;
  ptrBuffer = (char *)realloc(ptrBuffer, iBufSize);

#ifdef _DEBUG_
    debug("Buffer reallocated: %d", iBufSize);
#endif

  for (addrRec  = AddrList->first(), i=0; 
       addrRec != 0; 
       addrRec  = AddrList->next(),  i++)
    {
      fillAddress(addrRec, &a);
      memset(ptrBuffer, 0x00, iBufSize);

      l = pack_Address(&a, (unsigned char *)ptrBuffer, iBufSize);
      if (l > 0)
	{
          int attr;

          // is this a secret record?
          attr  = (addrRec->getPrivate()) ? dlpRecAttrSecret : 0;

          // was this record modified?
          attr |= (addrRec->getModified()) ? dlpRecAttrDirty : 0;

          qsTmp.sprintf("Writing record no. %03d of %03d", i+1, n);
          statusBar.SetText(statusMsg, qsTmp);

          // append this record 
          if (pdbFile->appendRecord(ptrBuffer, l, attr, addrRec->getCat(), lUid++) < 0)
	    {
              debug("Error writing record no. %d. Aborting.", i+1);
	      break;
	    }
	}
      else
	{
	  // should not happen!
          debug("Skipping record no. %d [too big (%d byte)].",
                i+1, pack_Address(&a, 0, 0));
	}
    }
  
  if (pdbFile->closeFile() < 0)
    {
      dbInfo.backupDate = backupTime = 0;
      fileMenu->setItemEnabled(idSave,   FALSE);
      fileMenu->setItemEnabled(idReread, FALSE);
      toolBar.EnableItem(idToolbarSave,  FALSE);
      QMessageBox::warning(this, "File Write Error",
	                   "Could not write database file.\n");
    }
  else
    {
      bModified = FALSE;
      fileMenu->setItemEnabled(idSave,      TRUE);
      fileMenu->setItemEnabled(idReread,    TRUE);
      toolBar.EnableItem(idToolbarSave,     TRUE);
      if (!bDisablePilotFunctions) 
	{
	  pilotMenu->setItemEnabled(pilotWrite, TRUE);
	  toolBar.EnableItem(idToolbarWrite,    TRUE);
	}
    }

  delete(ptrBuffer);
  
  delete pdbFile;
  fileMenu->setItemEnabled(idSaveAs, TRUE);
  startTimer4StatusMsg();
}

void PiAddress::fillAddress(AddressRecord *addrRec, Address *a)
{
  a->entry[0]  = (char *)addrRec->getFN();
  a->entry[1]  = (char *)addrRec->getGN();
  a->entry[2]  = (char *)addrRec->getCompany();
  a->entry[3]  = (char *)addrRec->getPhone1();
  a->entry[4]  = (char *)addrRec->getPhone2();
  a->entry[5]  = (char *)addrRec->getPhone3();
  a->entry[6]  = (char *)addrRec->getPhone4();
  a->entry[7]  = (char *)addrRec->getPhone5();
  a->entry[8]  = (char *)addrRec->getAddress();
  a->entry[9]  = (char *)addrRec->getTown();
  a->entry[10] = (char *)addrRec->getState();
  a->entry[11] = (char *)addrRec->getZIP();
  a->entry[12] = (char *)addrRec->getCountry();
  a->entry[13] = (char *)addrRec->getTitle();
  a->entry[14] = (char *)addrRec->getDefined1();
  a->entry[15] = (char *)addrRec->getDefined2();
  a->entry[16] = (char *)addrRec->getDefined3();
  a->entry[17] = (char *)addrRec->getDefined4();
  a->entry[18] = (char *)addrRec->getNote();

  a->showPhone = addrRec->getShowPhone();

  a->phoneLabel[0] = addrRec->getPhoneLabelIdx0();
  a->phoneLabel[1] = addrRec->getPhoneLabelIdx1();
  a->phoneLabel[2] = addrRec->getPhoneLabelIdx2();
  a->phoneLabel[3] = addrRec->getPhoneLabelIdx3();
  a->phoneLabel[4] = addrRec->getPhoneLabelIdx4();
}

int PiAddress::fillAppInfo(char *p)
{
  char *p1, *ptr = p;
  int i;
  
  *ptr++ = 0x00;
  *ptr++ = 0x0F;

  p1 = ptr;

  // the last category ("All") ist not stored in the DB-file!
  for (i=0; i<cat->count()-1; i++, ptr += 16)
    strcpy(ptr, cat->text(i));
  ptr = p1 + 16*16;

  for (i=0; i<16; i++)
    *ptr++ = i;
  
  *ptr++ = 0x0F;
  *ptr++ = 0x00;
  *ptr++ = 0x00;
  *ptr++ = 0x00;

  *ptr++ = 0x00;
  *ptr++ = 0x3F;
  *ptr++ = 0xFF;
  *ptr++ = 0xFF;
  
  p1 = ptr;
  for(i=0; i<22; i++)
    memcpy(ptr, &pAppInfo->labels[0][0], 22*16);
  ptr = p1 + 22*16;
  
  // Country and gap fill
  *ptr++ = 0x08;
  *ptr++ = 0x00;
  *ptr++ = 0x00;
  *ptr++ = 0x00;
  
  return (ptr - p);
}

int PiAddress::initAppInfo()
{
  int i;
  
  memset(pAppInfo, 0x00, sizeof(AddressAppInfo));

  strcpy(pAppInfo->labels[0],  "Last Name");
  strcpy(pAppInfo->labels[1],  "First Name");
  strcpy(pAppInfo->labels[2],  "Company");
  strcpy(pAppInfo->labels[3],  "Work");
  strcpy(pAppInfo->labels[4],  "Home");
  strcpy(pAppInfo->labels[5],  "Fax");
  strcpy(pAppInfo->labels[6],  "Other");
  strcpy(pAppInfo->labels[7],  "E-mail");
  strcpy(pAppInfo->labels[8],  "Address");
  strcpy(pAppInfo->labels[9],  "City");
  strcpy(pAppInfo->labels[10], "State");
  strcpy(pAppInfo->labels[11], "ZIP-Code");
  strcpy(pAppInfo->labels[12], "Country");
  strcpy(pAppInfo->labels[13], "Title");
  strcpy(pAppInfo->labels[14], "Defined 1");
  strcpy(pAppInfo->labels[15], "Defined 2");
  strcpy(pAppInfo->labels[16], "Defined 3");
  strcpy(pAppInfo->labels[17], "Defined 4");
  strcpy(pAppInfo->labels[18], "Note");
  strcpy(pAppInfo->labels[19], "Main");
  strcpy(pAppInfo->labels[20], "Pager");
  strcpy(pAppInfo->labels[21], "Mobile");
  
  for(i=3; i<8; i++)
    strcpy(pAppInfo->phoneLabels[i-3], pAppInfo->labels[i]);
  for(i=19; i<22; i++)
    strcpy(pAppInfo->phoneLabels[i-19+5], pAppInfo->labels[i]);

  strcpy(pAppInfo->category.name[0], "Unfiled");
  strcpy(pAppInfo->category.name[1], "Company");
  strcpy(pAppInfo->category.name[2], "Private");
  strcpy(pAppInfo->category.name[3], "Quicklist");

  return 0;
}

char *PiAddress::getDir()
{
 char *p;
 struct passwd *pwd;
 p = getenv("PILOT");
 if (!p) {
   pwd = getpwuid(getuid());
   p = pwd->pw_dir;
 }
#ifdef _DEBUG_
 debug("getDir() : %s", p);
#endif
 return p;
}

void PiAddress::loadConfigFile(QString *qsFile)
{
  QString qsLine, qsTmp;
  int pos;
  
  QFile rcFile(qsFile->data());
  if (!rcFile.exists())
    return;

  // open file for reading
  if (!rcFile.open(IO_ReadOnly))
    return;
  
  QTextStream t(&rcFile);
  
  while (!t.eof())
    {
      qsLine = "";
      
      qsLine = t.readLine();

      // skip over comment lines and empty lines      
      if (*(const char *)qsLine == '#' || qsLine.length() == 0)
	continue;

#ifdef _DEBUG_
	debug ("DBG (loadConfigFile): %s", (const char *)qsLine);
#endif
      
      // look for "Debug"
      pos = qsLine.find("Debug", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      iDebug = qsTmp.toInt();
	    }
	}

      // look for "DisablePilotFunctions"
      pos = qsLine.find("DisablePilotFunctions", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      bDisablePilotFunctions = qsTmp.toInt() > 0 ? TRUE : FALSE;
	      if (bDisablePilotFunctions)
		statusBar.SetText(statusDevice, "Dev: disabled");
	    }
	}

      // look for "CheckDevice"
      pos = qsLine.find("CheckDevice", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      bCheckDevice = qsTmp.toInt() > 0 ? TRUE : FALSE;
	    }
	}
      
      // look for "PrintingFormat"
      pos = qsLine.find("PrintingFormat", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
              qsPrintingFormat = qsLine.mid(pos+1, qsLine.length()-pos-1);
	}

      // look for "PrintingPointSize"
      pos = qsLine.find("PrintingPointSize", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      iPrintingPointSize = qsTmp.toInt();
	    }
	}

      // look for "PrintingFontName"
      pos = qsLine.find("PrintingFontName", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);

              if ( (qsLine.find("Helvetica", pos, FALSE) == -1) &&
                   (qsLine.find("Courier",   pos, FALSE) == -1) &&
                   (qsLine.find("Times",     pos, FALSE) == -1) )
		continue;

              qsPrintingFontName = qsTmp.stripWhiteSpace();
	    }
	}

      // look for "PrintTo"
      pos = qsLine.find("PrintTo", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);

              if ( (qsLine.find("Printer", pos, FALSE) == -1) &&
                   (qsLine.find("File",    pos, FALSE) == -1) )
		continue;

              qsPrintTo = qsTmp.stripWhiteSpace();
	    }
	}

      // look for "PrintFileName"
      pos = qsLine.find("PrintFileName", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);

              qsPrintFileName = qsTmp.stripWhiteSpace();
	    }
	}

      // look for "HelpApplication""
      pos = qsLine.find("HelpApplication", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsHelpApplication = qsTmp.stripWhiteSpace();
              if (qsHelpApplication.isEmpty())
                helpMenu->setItemEnabled(idFAQ, FALSE);
              else
                helpMenu->setItemEnabled(idFAQ, TRUE);
	    }
	}

      // look for "DefaultDB"
      pos = qsLine.find("DefaultDB", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      qsDefaultDB = qsTmp.stripWhiteSpace();
	      QString s(qsDefaultDB.data());
	      pos = s.find("~/");
	      if (pos != -1) {
		struct passwd *pwd = getpwuid(getuid());
		s.replace(0, 1, pwd->pw_dir);
	      }
              QFile f(s.data());
              if (f.exists())
                readFile(f);
	    }
	}

      // look for "DeviceName"
      pos = qsLine.find("DeviceName", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsDeviceName = qsTmp.stripWhiteSpace();
              qsTmp  = "Dev: ";
              qsTmp += (const char *)qsDeviceName;
              statusBar.SetText(statusDevice, qsTmp);
            }
        }

      // look for "ListFont"
      pos = qsLine.find("ListFont", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsListFont = qsTmp.stripWhiteSpace();
              lbAddr->setFont(QFont(qsListFont.data(), 12));
            }
        }

      // look for "RecordFont"
      pos = qsLine.find("RecordFont", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsRecordFont = qsTmp.stripWhiteSpace();
            }
        }

      // look for "EditFont"
      pos = qsLine.find("EditFont", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsEditFont = qsTmp.stripWhiteSpace();
            }
        }

      // look for "SrcCodePage"
      pos = qsLine.find("SrcCodePage", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsSrcCodePage = qsTmp.stripWhiteSpace();
            }
        }

      // look for "DstCodePage"
      pos = qsLine.find("DstCodePage", 0, FALSE);
      if (pos != -1)
        {
          // found, look for "="
          pos = qsLine.find("=", pos);
          if (pos != -1)
            {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsDstCodePage = qsTmp.stripWhiteSpace();
            }
        }

      // look for "PilotManagerAppInfo"
      pos = qsLine.find("PilotManagerAppInfo", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      qsPMgrAppInfo = qsTmp.stripWhiteSpace();
	    }
	}

      // LDIF configuration variables (which phone label to use)
      pos = qsLine.find("LDIFtelephonenumber", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFtelephonenumber = qsTmp.toInt();
	    }
	}

      pos = qsLine.find("LDIFhomephone", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFhomephone = qsTmp.toInt();
	    }
	}

      pos = qsLine.find("LDIFfacsimiletelephonenumber", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFfacsimiletelephonenumber = qsTmp.toInt();
	    }
	}

      pos = qsLine.find("LDIFmail", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFmail = qsTmp.toInt();
	    }
	}

      pos = qsLine.find("LDIFpagerphone", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFpagerphone = qsTmp.toInt();
	    }
	}

      pos = qsLine.find("LDIFcellphone", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      iLDIFcellphone = qsTmp.toInt();
	    }
	}

      // User Program variables
      pos = qsLine.find("UserApplication1", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
              qsUserProg1 = qsTmp.stripWhiteSpace();
	    }
	}
      pos = qsLine.find("ApplicationName1", 0, FALSE);
      if (pos != -1)
	{
	  // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
	      qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
	      recordMenu->changeItem(qsTmp.data(), idRecordExec1);
	      rMouseMenu->changeItem(qsTmp.data(), idExec1);
	    }
	}

      // confirm operations?
      pos = qsLine.find("ConfirmOperations", 0, FALSE);
      if (pos != -1)
	{
          // found, look for "="
	  pos = qsLine.find("=", pos);
	  if (pos != -1)
	    {
              qsTmp = qsLine.mid(pos+1, qsLine.length()-pos-1);
   	      bConfirmOps = qsTmp.toInt() > 0 ? TRUE : FALSE;
	    }
	}
    }

  rcFile.close();

  evaluateConfValues();
}


void PiAddress::evaluateConfValues()
{
  if (bDisablePilotFunctions)
    {
#ifdef _DEBUG_
      debug("DBG (evaluateConfValues): bDisableConfValues -> TRUE");
#endif
      pilotMenu->setItemEnabled(pilotWrite, FALSE);
      pilotMenu->setItemEnabled(pilotRead,  FALSE);
      pilotMenu->setItemEnabled(pilotPurge, FALSE);
      pilotMenu->setItemEnabled(pilotList,  FALSE);
      toolBar.EnableItem(idToolbarWrite,    FALSE);
      toolBar.EnableItem(idToolbarRead,     FALSE);
    }
  else
    {
      pilotMenu->setItemEnabled(pilotPurge, TRUE);
      pilotMenu->setItemEnabled(pilotList,  TRUE);
    }
}


void PiAddress::loadSystemConfigFile()
{
  QString qsFile("/etc/pi-addressrc");
  loadConfigFile(&qsFile);
}

void PiAddress::loadUserConfigFile()
{
  // look for user home dir
  char *p = getpwuid(getuid())->pw_dir;

  // cannot find home dir
  if (!p)
    return;

  QString qsFile(p);
  qsFile.append("/.pi-addressrc");

  loadConfigFile(&qsFile);
}

void PiAddress::initDefValues()
{
  // Default values
  qsPrintingFormat   = "%l %f %p1 %p2 %p3 %c1";
  qsPrintingFontName = "Helvetica";
  iPrintingPointSize = 8;
  qsPrintTo = "Printer";
  qsPrintFileName = "pi-address.ps";
    
  qsHelpApplication = "xterm -e lynx /usr/doc/pi-address/html/pia-faq.html";
  iDebug = 0;
  qsDefaultDB = "";
  qsDeviceName = "/dev/pilot";

  qsListFont   = "fixed";
  qsRecordFont = "Helvetica";
  qsEditFont   = "fixed";

  qsSrcCodePage = "ISO8859-1";
  qsDstCodePage = "ISO8859-1";

  qsPMgrAppInfo = "";
  
  bDisablePilotFunctions = FALSE;

  bCheckDevice = TRUE;

  // which phone label to use for LDIF variables (0 = empty)
  iLDIFtelephonenumber          = 1;
  iLDIFhomephone                = 2;
  iLDIFfacsimiletelephonenumber = 3;
  iLDIFmail                     = 5;
  iLDIFpagerphone               = 0;
  iLDIFcellphone                = 0;

  qsUserProg1 = "";

  bConfirmOps = TRUE;
}


void PiAddress::rereadConfig()
{
  initDefValues();
  loadUserConfigFile();
  if (lbAddr->count())
    {
      int i = lbAddr->currentItem();
      displayAddresses();
      displayRecord(lbAddr->currentItem());
      lbAddr->setCurrentItem(i);
    }
  
  statusBar.SetText(statusMsg, "Reread configuration file");
  startTimer4StatusMsg();
}

void PiAddress::loadDefaultDB()
{
  int i, pos;
  char *p;
  
  if (qApp->argc() < 2)
    return;
 
  for (i=1; i<qApp->argc(); i++)
    {
      QString s(qApp->argv()[i]);
      pos = s.find("~/");
      if (pos != -1) 
	{
	  p = getpwuid(getuid())->pw_dir;

	  if (p)
	    s.replace(0, 2, p);
	}
      
      QFile f(s.data());

      if (f.exists())
	{
          readFile(f);
	  break;
	}
    }
}


void PiAddress::importPMgr()
{
    if (bModified && confirmNoSave() != 0)
      return;

    QString csvName(getpwuid(getuid())->pw_dir);
    csvName += "/.csvAddr";

    // get a file name
    QString qsTmp(QFileDialog::getOpenFileName(csvName, 0, this));
    if (qsTmp.isNull())
      return;

    QFile f((const char *)qsTmp);
    if (!f.exists())
      {
        QMessageBox::warning(this, "File Open Error",
			     "Could not open the file for reading.\n");
	return;
      }

    // actually read the file
    readFile(f, TRUE);
}


void PiAddress::rereadFile()
{
  QFile f((const char *)qsFname);
  readFile(f);
}


void PiAddress::readFile(QFile &qfFile, bool bPMgr)
{
    int i, rc;
    QString qsTmp;
    int aiCat[16];
    char *acTmp[16];

    QFileInfo fi(qfFile);
  
    if (AddrList && AddrList->count())
      {
	delete AddrList;
	AddrList = 0;
      }

    memset(pAppInfo, 0x00, sizeof(AddressAppInfo));

    AddrList = new QList<AddressRecord>;
    AddrList->setAutoDelete(TRUE);

    ReadPDBFile *pdbFile = 0;
    ReadCSVFile *csvFile = 0;
    
    bool bCancel=FALSE;
    if (!bPMgr)
      pdbFile = new ReadPDBFile(qfFile, &bCancel);
    else
      csvFile = new ReadCSVFile(qfFile, &bCancel, &qsPMgrAppInfo);
    
    if (bCancel == FALSE)
      {
	if (!bPMgr)
          rc = pdbFile->getAppInfo(pAppInfo);
	else
	  rc = csvFile->getAppInfo(pAppInfo);
	
        // display correct categories
        cat->clear();
	memset(acTmp, 0x00, sizeof(acTmp));

	// collect category names in temp. array
        for (i=0; i<16; i++)
          {
	      acTmp[i] = pAppInfo->category.name[i];
	      aiCat[i] = i;
          }

	// reorder category entries in case there are 'empty'
	// categories in between!
	for (i=0; i<16; i++)
	  {
	    // valid entry?
	    if (strlen(acTmp[i]))
	      // yes, continue
	      continue;

	    // search next entry
	    for (int j=i+1; j<16; j++)
	      {
		if (strlen(acTmp[j]) == 0)
		  continue;

		// swap entries
		acTmp[i] = acTmp[j];
		acTmp[j] = 0;
		// change category index
		aiCat[j] = i;

		break;
	      }
	  }

	// copy all 'non-empty' categories to combobox
	i=0;
	while (strlen(acTmp[i]))
	  cat->insertItem(acTmp[i++]);
	
        // FIXME: other countries?
        if (stricmp(cat->text(0), "Nicht abgelegt") == 0)
          cat->insertItem("Alle");
        else
          cat->insertItem("All");
        cat->setCurrentItem(cat->count()-1);

        cat->adjustSize();

        // get count of entries in DB
        int iEntries = 0;
	if (!bPMgr)
          iEntries = pdbFile->getNEntries();
	else
          iEntries = csvFile->getNEntries();

	// read all entries
	int iAttrs, iCat;
	
        for (i=0; i<iEntries; i++)
	  {
            Address a;
            memset(&a, 0x00, sizeof(a));
	    
            qsTmp.sprintf("Reading record no. %03d", i+1);
	    statusBar.SetText(statusMsg, qsTmp);

            qsTmp.sprintf("Records: %03d", i+1);
            statusBar.SetText(statusCount, qsTmp);
	    
	    if (!bPMgr)
	      rc = pdbFile->readRecord(i, &a, &iAttrs, &iCat);
	    else
	      rc = csvFile->readRecord(i, &a, &iAttrs, &iCat);;
	    
            if (rc > 0)
	      {
                AddressRecord *addrRec = new AddressRecord();
                bool bPrivate = (iAttrs & dlpRecAttrSecret) ? TRUE : FALSE;

		// CAUTION: category index has to be converted in case 
		//          category names have been reordered because
		//          there have been 'empty' category names!
		fillAddrRecord(addrRec, &a, aiCat[iCat], bPrivate);

                // is there a "Family Name" OR a "Company" Or a "First Name"
                // AND  is the record NOT deleted?
                if ( (addrRec->getFN() || addrRec->getCompany()
		      || addrRec->getGN()) && 
                     !(iAttrs & dlpRecAttrDeleted))
                    insertSorted(addrRec);
		else
                  delete addrRec;
	      }
	  }

        qsTmp.sprintf("File: %s", (const char *)fi.filePath());
        statusBar.SetText(statusFname, qsTmp);
	if (!bPMgr)
	  {
            qsFname = qfFile.name();
            fileMenu->setItemEnabled(idSave,   TRUE);
            fileMenu->setItemEnabled(idReread, TRUE);
	  }
        toolBar.EnableItem(idToolbarSave,      TRUE);
        fileMenu->setItemEnabled(idSaveAs,     TRUE);
        fileMenu->setItemEnabled(idExport,     TRUE);
        fileMenu->setItemEnabled(idExportLDIF, TRUE);
        fileMenu->setItemEnabled(idExportXML,  TRUE);
        toolBar.EnableItem(idToolbarExport,    TRUE);
        fileMenu->setItemEnabled(printID,      TRUE);
        toolBar.EnableItem(idToolbarPrint,     TRUE);
        if (!bDisablePilotFunctions)
	  {
	    pilotMenu->setItemEnabled(pilotWrite,  TRUE);
            toolBar.EnableItem(idToolbarWrite,  TRUE);
	  }
      }

    displayAddresses();
    if (lbAddr->count())
      {
        displayRecord(0);
        lAddr->repaint();
        updateStatusCount();
        lbAddr->setCurrentItem(0);
        lbAddr->centerCurrentItem();
        leSearch->setFocus();
      }

    if (!bPMgr)
      {
        creaTime   = pdbFile->getCreateDate();
        if (creaTime < 0)
          creaTime = 0;
        modTime    = pdbFile->getModifyDate();
        if (modTime < 0)
          modTime = 0;
        backupTime = pdbFile->getBackupDate();
        if (backupTime < 0)
          backupTime = 0;
        qsTmp.sprintf("Last mod: %s", ctime(&modTime));
        // ctime returns a buffer containing a newline!
        qsTmp.truncate(qsTmp.length()-1);
        bModified = FALSE;
        statusBar.SetText(statusMsg, qsTmp);
      }
    else
      {
	creaTime = modTime = backupTime = 0;
	bModified = TRUE;
      }
    
    startTimer4StatusMsg();

    if (bPMgr)
      delete csvFile;
    else
      delete pdbFile;
}


void PiAddress::open()
{   
    if (bModified && confirmNoSave() != 0)
      return;

    // check for env. variable
    QString qsTmp1 = "./";
    QFileInfo fiDir(getDir());
    // is it a directory?
    if (fiDir.isDir())
      qsTmp1 = getDir();

    // get a file name
    QString qsTmp(QFileDialog::getOpenFileName(qsTmp1, "*.pdb", this));
    if (qsTmp.isNull())
      return;

    QFile f((const char *)qsTmp);
    if (!f.exists())
      {
	// report error
        QMessageBox::warning(this, "File Open Error",
			     "Could not open the file for reading.\n");
	return;
      }

    // actually read the file
    readFile(f);
}


void PiAddress::transformCodePage(Address *a)
{
  if (stricmp(qsSrcCodePage, qsDstCodePage) == 0)
    return;

  // look for valid translation table  
  if ( (stricmp(qsSrcCodePage, "cp1251") == 0) &&
       (stricmp(qsDstCodePage, "iso8859-5") == 0)
     )
    {
      transformAddress(a, &c_iso8859_5[0]);
    }

  if ( (stricmp(qsSrcCodePage, "cp1251") == 0) &&
       (stricmp(qsDstCodePage, "koi8-r") == 0)
     )
    {
      transformAddress(a, &c_koi8_r[0]);
    }
}


void PiAddress::transformAddress(Address *a, unsigned short int *table)
{
  int i, j;
  int ch;
  char *p;
  
  for (j=0; j<19; j++)
    if (a->entry[j])
      for (i=0; i<strlen(a->entry[j]); i++)
        {
          p  = a->entry[j] + i;
	  ch = (char)(*(table+(int)(*p)));
	  *p = ch;
        }
}


void PiAddress::fillAddrRecord(AddressRecord *addrRec, Address *a, 
                               int iCat, bool bPrivate)
{
  transformCodePage(a);
  
  if (a->entry[0])  
    addrRec->putFN(a->entry[0]);
  if (a->entry[1])  
    addrRec->putGN(a->entry[1]);
  if (a->entry[2])  
    addrRec->putCompany(a->entry[2]);
  if (a->entry[3])  
    addrRec->putPhone1(a->entry[3]);
  if (a->entry[4])  
    addrRec->putPhone2(a->entry[4]);
  if (a->entry[5])  
    addrRec->putPhone3(a->entry[5]);
  if (a->entry[6])  
    addrRec->putPhone4(a->entry[6]);
  if (a->entry[7])  
    addrRec->putPhone5(a->entry[7]);
  if (a->entry[8])  
    addrRec->putAddress(a->entry[8]);
  if (a->entry[9])  
    addrRec->putTown(a->entry[9]);
  if (a->entry[10])  
    addrRec->putState(a->entry[10]);
  if (a->entry[11])  
    addrRec->putZIP(a->entry[11]);
  if (a->entry[12])  
    addrRec->putCountry(a->entry[12]);
  if (a->entry[13])  
    addrRec->putTitle(a->entry[13]);
  if (a->entry[14])  
    addrRec->putDefined1(a->entry[14]);
  if (a->entry[15])  
    addrRec->putDefined2(a->entry[15]);
  if (a->entry[16])  
    addrRec->putDefined3(a->entry[16]);
  if (a->entry[17])  
    addrRec->putDefined4(a->entry[17]);
  if (a->entry[18])  
    addrRec->putNote(a->entry[18]);
  addrRec->setCat(iCat);
  addrRec->putShowPhone(a->showPhone);
  addrRec->putPrivate(bPrivate);
  addrRec->putPhoneLabelIdx0(a->phoneLabel[0]);
  addrRec->putPhoneLabelIdx1(a->phoneLabel[1]);
  addrRec->putPhoneLabelIdx2(a->phoneLabel[2]);
  addrRec->putPhoneLabelIdx3(a->phoneLabel[3]);
  addrRec->putPhoneLabelIdx4(a->phoneLabel[4]);
}


int PiAddress::confirmNoSave()
{
  int b=0;
  if (bConfirmOps)
    b=QMessageBox::warning(this, "Modified Records",
                           "There is unsaved modified data.\n"
                           "Do you want to continue?\n",
                           "Yes", "No", "Cancel", 1);
  return b;
}


// create a new - empty - address DB
void PiAddress::news()
{
  if (bModified && confirmNoSave() != 0)
   return;

  // clear the ListBox
  lbAddr->clear();
  if (AddrList != 0)
    {
     delete AddrList;
     AddrList = 0;
    }

  initAppInfo();
  initCat();
  
  disableButtons();
  fileMenu->setItemEnabled(printID,  FALSE);
  toolBar.EnableItem(idToolbarPrint, FALSE);
  updateStatusCount();
  bModified = FALSE;

  statusMessageReady();
  statusBar.SetText(statusFname, "File: - untitled - ");
  qsFname = "";
  fileMenu->setItemEnabled(idSave,       FALSE);
  fileMenu->setItemEnabled(idReread,     FALSE);
  toolBar.EnableItem(idToolbarSave,      FALSE);
  fileMenu->setItemEnabled(idSaveAs,     FALSE);
  fileMenu->setItemEnabled(idExport,     FALSE);
  fileMenu->setItemEnabled(idExportLDIF, FALSE);
  fileMenu->setItemEnabled(idExportXML,  FALSE);
  toolBar.EnableItem(idToolbarExport,    FALSE);
  pilotMenu->setItemEnabled(pilotWrite,  FALSE);
  toolBar.EnableItem(idToolbarWrite,     FALSE);
  creaTime = time(0);
  modTime  = creaTime;
  backupTime = 0; 
}


void PiAddress::save()
{
  writeFile();
}


bool PiAddress::fileOverwrite()
{
  int b=0;
  if (bConfirmOps)
    b=QMessageBox::warning(this, "File Exists",
                                 "The file already exists.\n"
                                 "Do you want to overwrite it?\n",
                                 "Yes", "No", "Cancel", 1);
  return ((b==0) ? TRUE : FALSE);
}


void PiAddress::save_as()
{
  QString f(QFileDialog::getSaveFileName(getDir(), "*.pdb", this));

  if (!f.isEmpty())
    {
      if (f.find(".pdb", f.length()-4, FALSE) == -1)
        f.append(".pdb");
	
      // got a valid filename
      QFileInfo fi(f);
      if (fi.exists() && !fileOverwrite())
         return;

      qsFname = fi.filePath();
      writeFile();

      QString qsTmp;
      qsTmp.sprintf("File: %s", (const char *)qsFname);
      statusBar.SetText(statusFname, qsTmp);
    }
}

// close the current Address DB
void PiAddress::closeDoc()
{
  if (bModified && confirmNoSave() != 0)
     return;

  initAppInfo();
  initCat();
  
  lbAddr->clear();
  delete AddrList;
  AddrList = 0;
  disableButtons();
  bModified = FALSE;
  updateStatusCount();
	 
  statusMessageReady();
  statusBar.SetText(statusFname, "File: - untitled -");
  qsFname = "";
  fileMenu->setItemEnabled(idSave,       FALSE);
  fileMenu->setItemEnabled(idReread,     FALSE);
  toolBar.EnableItem(idToolbarSave,      FALSE);
  fileMenu->setItemEnabled(idSaveAs,     FALSE);
  fileMenu->setItemEnabled(idExport,     FALSE);
  fileMenu->setItemEnabled(idExportLDIF, FALSE);
  fileMenu->setItemEnabled(idExportXML,  FALSE);
  toolBar.EnableItem(idToolbarExport,    FALSE);
  pilotMenu->setItemEnabled(pilotWrite,  FALSE);
  toolBar.EnableItem(idToolbarWrite,     FALSE);

  creaTime = modTime = backupTime = 0;
}

// export the current Address DB as LDIF-file (Netscape)
void PiAddress::exportLDIF()
{
    QString msg;
    int i, iCat = cat->currentItem();
    AddressRecord *a;
    
    if (AddrList == 0 || AddrList->count() == 0)
      return;

    // Note: Netscape imports LDIF files with 8-bit data only
    //       if the filename extension is '.4ld'!
    QString f(QFileDialog::getSaveFileName(0, "*.4ld", this));
    if (!f.isEmpty())
      {
        QFile outFile(f);
        if (outFile.exists() && !fileOverwrite())
	  return;

	if (outFile.open(IO_WriteOnly))
	  {
            QTextStream t(&outFile);
	    QString qsTmp;

            for(a=AddrList->first(), i=0; a!=0; a=AddrList->next(), i++)
	      {
		char aOut[1024];
		size_t aOutLen;
		
                // write only records which are selected through category
                if ( (a->getCat() != iCat) && (iCat < cat->count()-1) )
                  continue;

		qsTmp.sprintf("cn=%s %s%s%s",
                              (a->getGN())     ? a->getGN() : "", 
                              (a->getFN())     ? a->getFN() : "", 
			      (a->getPhone(iLDIFmail)) ? ",mail="   : "",
			      (a->getPhone(iLDIFmail)) ? a->getPhone(iLDIFmail) : "");
		encode_base64(qsTmp.data(), qsTmp.length(), 
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("dn:: %s", aOut);
                t << qsTmp.data();

		qsTmp.sprintf("changetype: add\n");
		t << qsTmp.data();

		qsTmp.sprintf("%s %s", 
			      (a->getGN()) ? a->getGN() : "", 
			      (a->getFN()) ? a->getFN() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("cn:: %s", aOut);
		t << qsTmp.data();

		if (iLDIFmail > 0)
		  {
		    qsTmp.sprintf("%s", (a->getPhone(iLDIFmail)) ? 
                                         a->getPhone(iLDIFmail)  : 
                                         "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("mail:: %s", aOut);
	            t << qsTmp.data();
		  }

		qsTmp.sprintf("%s", (a->getGN()) ? a->getGN() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("givenname:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getFN()) ? a->getFN() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("sn:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getNote()) ? a->getNote() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("description:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getCompany()) ? a->getCompany() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("o:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getAddress()) ? a->getAddress() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("streetaddress:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getZIP()) ? a->getZIP() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("postalcode:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getCountry()) ? a->getCountry() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("countryname:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getTown()) ? a->getTown() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("locality:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getState()) ? a->getState() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("st:: %s", aOut);
	        t << qsTmp.data();

		qsTmp.sprintf("%s", (a->getTitle()) ? a->getTitle() : "");
		encode_base64(qsTmp.data(), qsTmp.length(),
			      aOut, sizeof(aOut), &aOutLen);
		qsTmp.sprintf("title:: %s", aOut);
	        t << qsTmp.data();

		if (iLDIFtelephonenumber > 0)
		  {
		    qsTmp.sprintf("%s", (a->getPhone(iLDIFtelephonenumber)) ? 
                                         a->getPhone(iLDIFtelephonenumber)  : 
                                         "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("telephonenumber:: %s", aOut);
	            t << qsTmp.data();
		  }

		if (iLDIFfacsimiletelephonenumber > 0)
		  {
		    qsTmp.sprintf("%s", 
                               (a->getPhone(iLDIFfacsimiletelephonenumber)) ? 
                                a->getPhone(iLDIFfacsimiletelephonenumber)  : 
                                "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("facsimiletelephonenumber:: %s", aOut);
	            t << qsTmp.data();
		  }

		if (iLDIFhomephone > 0)
		  {
		    qsTmp.sprintf("%s", (a->getPhone(iLDIFhomephone)) ? 
                                         a->getPhone(iLDIFhomephone)  : 
                                         "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("homephone:: %s", aOut);
	            t << qsTmp.data();
		  }

		if (iLDIFpagerphone > 0)
		  {
		    qsTmp.sprintf("%s", (a->getPhone(iLDIFpagerphone)) ? 
                                         a->getPhone(iLDIFpagerphone)  : 
                                         "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("pagerphone:: %s", aOut);
	            t << qsTmp.data();
		  }

		if (iLDIFcellphone > 0)
		  {
		    qsTmp.sprintf("%s", (a->getPhone(iLDIFcellphone)) ? 
                                         a->getPhone(iLDIFcellphone)  : 
                                         "");
		    encode_base64(qsTmp.data(), qsTmp.length(),
			          aOut, sizeof(aOut), &aOutLen);
		    qsTmp.sprintf("cellphone:: %s", aOut);
	            t << qsTmp.data();
		  }

		qsTmp.sprintf("objectclass: top\n");
		t << qsTmp.data();
		qsTmp.sprintf("objectclass: person\n\n");
		t << qsTmp.data();

                msg.sprintf("Exporting record no. %3d of %3d", 
                            i+1, AddrList->count());

		statusBar.SetText(statusMsg, msg);
	      }
	    
	    outFile.close();
          }
        else
          QMessageBox::warning(this, "File Open Error",
	       		       "Could not open the file for writing.\n");
      }

      startTimer4StatusMsg();
}


// export the current Address DB as CSV-file
void PiAddress::exportCSV()
{
    QString msg;
    int i, iCat = cat->currentItem();
    AddressRecord *a;
    
    if (AddrList == 0 || AddrList->count() == 0)
      return;
    
    QString f(QFileDialog::getSaveFileName(0, "*.csv", this));
    if (!f.isEmpty())
      {
        QFile outFile(f);
        if (outFile.exists() && !fileOverwrite())
	  {
	    return;
	  }
	if (outFile.open(IO_WriteOnly))
	    {
                QTextStream t(&outFile);

		QString *s;
                s = new QString();
		
                for(a=AddrList->first(), i=0; a!=0; a=AddrList->next(), i++)
		{
                  // write only records which are selected through category
                  if ( (a->getCat() != iCat) && (iCat < cat->count()-1) )
                    continue;

                  s->sprintf("\"%s\",\"%s\",\"%s\",\"%s\","
                             "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
                             "\"%s\",\"%s\",\"%s\",\"%s\",\"%s\","
                             "\"%s\",\"%s\",\"%s\",\"%s\",",
                             (a->getFN())       ? a->getFN()       : "", 
                             (a->getGN())       ? a->getGN()       : "", 
                             (a->getTitle())    ? a->getTitle()    : "", 
                             (a->getCompany())  ? a->getCompany()  : "", 
                             (a->getPhone1())   ? a->getPhone1()   : "", 
                             (a->getPhone2())   ? a->getPhone2()   : "",
                             (a->getPhone3())   ? a->getPhone3()   : "", 
                             (a->getPhone4())   ? a->getPhone4()   : "", 
                             (a->getPhone5())   ? a->getPhone5()   : "",
                             (a->getAddress())  ? a->getAddress()  : "", 
                             (a->getTown())     ? a->getTown()     : "", 
                             (a->getState())    ? a->getState()    : "",
                             (a->getZIP())      ? a->getZIP()      : "", 
                             (a->getCountry())  ? a->getCountry()  : "", 
                             (a->getDefined1()) ? a->getDefined1() : "",
                             (a->getDefined2()) ? a->getDefined2() : "", 
                             (a->getDefined3()) ? a->getDefined3() : "",
                             (a->getDefined4()) ? a->getDefined4() : "");
		  t << (const char *)*s;

                  s->sprintf("\"%s\",\"%s\"\n",
                             (a->getNote())     ? a->getNote()     : "", 
                             (a->getPrivate())  ? "1"              : "0");

		  t << (const char *)*s;

                  msg.sprintf("Exporting record no. %3d of %3d", 
                              i+1, AddrList->count());

		  statusBar.SetText(statusMsg, msg);
		}
	      delete s;
	      outFile.close();
	    }
        else
          QMessageBox::warning(this, "File Open Error",
	       		       "Could not open the file for writing.\n");
      }
      startTimer4StatusMsg();
}

// export the current Address DB as XML-file
void PiAddress::exportXML()
{
    QString msg;
    int j, i, iCat = cat->currentItem();
    AddressRecord *a;
    
    if (AddrList == 0 || AddrList->count() == 0)
      return;
    
    // build default filename
    QString defName;
    if (qsFname.length()) {
      defName = qsFname.mid(qsFname.findRev('/')+1, 
			    qsFname.findRev('.')-qsFname.findRev('/')-1);
      defName += ".xml";
    }
    else
      defName = "";
    
    QString f(QFileDialog::getSaveFileName(defName.data(), "*.xml", this));
    if (!f.isEmpty())
      {
        QFile outFile(f);
        if (outFile.exists() && !fileOverwrite())
	  {
	    return;
	  }
	if (outFile.open(IO_WriteOnly))
	    {
                QTextStream t(&outFile);
		struct tm *pgmt;
		
		QString *s;
                s = new QString();

		// write "header"
		t << "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>" << endl;
		t << "<!DOCTYPE pilot-addresses SYSTEM \"pilot-addresses.dtd\" >" << endl << endl;
		t << "<pilot-addresses>" << endl << endl;
		
		// write "dbinfo"
		t << "  <dbinfo>" << endl;
		t << "    <dbtitle>Palm Pilot Address DB (XML export)</dbtitle>" << endl;
		s->sprintf("    <dbcreator version=\"%s\">pi-address</dbcreator>", VERSION);
		t << (const char *)*s << endl;
		
		t << "    <!-- iso-8601: YYYY-MM-DDThh:mm:ssZ                -->" << endl;
		t << "    <!-- rfc-822:  WDay, DD Mon YYYY hh::mm::ss +XX00  -->" << endl;
		t << "    <!-- free:     free format                         -->" << endl;
		t << "    <!-- date/time this file was written               -->" << endl;
		time_t t1 = time(0);
		pgmt = gmtime(&t1);
		s->sprintf("    <dbfiledate format=\"iso-8601\">%4d-%02d-%02dT%02d:%02d:%02dZ</dbfiledate>", 
			   pgmt->tm_year+1900, pgmt->tm_mon+1, pgmt->tm_mday, pgmt->tm_hour,
			   pgmt->tm_min, pgmt->tm_sec);
		t << (const char *)*s << endl;
		t << "    <!-- date/time the DB was created                  -->" << endl;
		pgmt = gmtime(&creaTime);
		s->sprintf("    <dbcreatedate format=\"iso-8601\">%4d-%02d-%02dT%02d:%02d:%02dZ</dbcreatedate>",
			   pgmt->tm_year+1900, pgmt->tm_mon+1, pgmt->tm_mday, pgmt->tm_hour,
			  pgmt->tm_min, pgmt->tm_sec);
		t << (const char *)*s << endl;
		t << "    <!-- date/time the DB was last modified            -->" << endl;
		pgmt = gmtime(&modTime);
		s->sprintf("    <dbmoddate format=\"iso-8601\">%4d-%02d-%02dT%02d:%02d:%02dZ</dbmoddate>", pgmt->tm_year+1900, pgmt->tm_mon+1, pgmt->tm_mday, pgmt->tm_hour,
			  pgmt->tm_min, pgmt->tm_sec);
		t << (const char *)*s << endl;
		
		t << "  </dbinfo>" << endl << endl;

		// write "appinfo"
		t << "  <!-- \"appinfo\" is optional and contains the -->" << endl;
		t << "  <!-- names of all categories and fieldnames -->" << endl;
		t << "  <appinfo>" << endl;
		t << "    <!-- up to 16 categories -->" << endl;
		t << "    <!--   (15 chars each)   -->" << endl;
		t << "    <categories>" << endl;
		for(i=0; i<cat->count()-1; i++) {
		  s->sprintf("      <category no=\"%d\">%s</category>", 
			     i+1, cat->text(i));
		  t << (const char *)*s << endl;
		}
		t << "    </categories>" << endl;
		t << "    <!-- all field names -->" << endl;
		t << "    <fieldnames>" << endl;
		s->sprintf("      <fieldlastname>%s</fieldlastname>",
			   pAppInfo->labels[0]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldfirstname>%s</fieldfirstname>",
			   pAppInfo->labels[1]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldtitle>%s</fieldtitle>",
			   pAppInfo->labels[13]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldcompany>%s</fieldcompany>",
			   pAppInfo->labels[2]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldphone1>%s</fieldphone1>",
			   pAppInfo->labels[3]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldphone2>%s</fieldphone2>",
			   pAppInfo->labels[4]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldphone3>%s</fieldphone3>",
			   pAppInfo->labels[5]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldphone4>%s</fieldphone4>",
			   pAppInfo->labels[6]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldphone5>%s</fieldphone5>",
			   pAppInfo->labels[7]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldaddress>%s</fieldaddress>",
			   pAppInfo->labels[8]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldcity>%s</fieldcity>",
			   pAppInfo->labels[9]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldstate>%s</fieldstate>",
			   pAppInfo->labels[10]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldzip>%s</fieldzip>",
			   pAppInfo->labels[11]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldcountry>%s</fieldcountry>",
			   pAppInfo->labels[12]);
		t << (const char *)*s << endl;
		s->sprintf("      <fielddefined1>%s</fielddefined1>",
			   pAppInfo->labels[14]);
		t << (const char *)*s << endl;
		s->sprintf("      <fielddefined2>%s</fielddefined2>",
			   pAppInfo->labels[15]);
		t << (const char *)*s << endl;
		s->sprintf("      <fielddefined3>%s</fielddefined3>",
			   pAppInfo->labels[16]);
		t << (const char *)*s << endl;
		s->sprintf("      <fielddefined4>%s</fielddefined4>",
			   pAppInfo->labels[17]);
		t << (const char *)*s << endl;
		s->sprintf("      <fieldnote>%s</fieldnote>",
			   pAppInfo->labels[18]);
		t << (const char *)*s << endl;

		t << "    </fieldnames>" << endl;

		t << "    <phonenames>" << endl;
		for(i=0; i<8; i++) {
		  s->sprintf("      <phonename%d>%s</phonename%d>",
			     i+1, pAppInfo->phoneLabels[i], i+1);
		  t << (const char *)*s << endl;
		}
		t << "    </phonenames>" << endl;
		
		t << "  </appinfo>" << endl << endl;

#ifdef IMPORT_XML
		t << "  <!-- If you want to import an XML file to your Address DB -->" << endl;
		t << "  <!-- make sure the file conforms to the following format: -->" << endl;
		t << "  <!-- 1: all records must be enclosed with <addrrecords>   -->" << endl;
		t << "  <!--    and </addrrecords> tags                           -->" << endl;
		t << "  <!-- 2: every record must be enclosed with <record> and   -->" << endl;
	        t << "  <!--    </record> tags                                    -->" << endl;
		t << "  <!-- 3: every field must be enclosed with its own tag     -->" << endl;
		t << "  <!--    (see the FAQ for a list of valid tags or read     -->" << endl;
		t << "  <!--    carefully the DTD)                                -->" << endl;
		t << "  <!-- 4: the following characters must be quoted           -->" << endl;
		t << "  <!--    with their entities: FIXME                        -->" << endl;
#endif		
		t << "  <addressrecords>" << endl;
		
		QString qsDest, qsPrivate;
                for(a=AddrList->first(),i=0,j=0; a!=0; a=AddrList->next(), i++)
		{
                  // write only records which are selected via category
                  if ( (a->getCat() != iCat) && (iCat < cat->count()-1) )
                    continue;

		  j++;
		  if (a->getPrivate())
		    qsPrivate = "yes";
		  else
		    qsPrivate = "no";
		  s->sprintf("    <record no=\"%d\" category=\"%d\" showphone=\"%d\" private=\"%s\" >",
			     j, a->getCat()+1, a->getShowPhone()+1, qsPrivate.data());
		  t << (const char *)*s << endl;

		  if (a->getFN() && strlen(a->getFN())) {
		    encodeEntities(a->getFN(), qsDest);
		    s->sprintf("      <lastname>%s</lastname>", qsDest.data());
		  }
		  else
		    s->sprintf("      <lastname></lastname>");
		  t << (const char *)*s << endl;
		  if (a->getGN() && strlen(a->getGN())) {
		    encodeEntities(a->getGN(), qsDest);
		    s->sprintf("      <firstname>%s</firstname>", 
			       qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getTitle() && strlen(a->getTitle())) {
		    encodeEntities(a->getTitle(), qsDest);
		    s->sprintf("      <title>%s</title>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getCompany() && strlen(a->getCompany())) {
		    encodeEntities(a->getCompany(), qsDest);
		    s->sprintf("      <company>%s</company>", qsDest.data());
		    t << (const char *)*s << endl;
		  }

		  // FIXME: add phonename attribute for each phone element !
		  if (a->getPhone1() && strlen(a->getPhone1())) {
		    encodeEntities(a->getPhone1(), qsDest);
		    s->sprintf("      <phone1 phonename=\"%d\">%s</phone1>", 
			       a->getPhoneLabelIdx(0)+1, qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getPhone2() && strlen(a->getPhone2())) {
		    encodeEntities(a->getPhone2(), qsDest);
		    s->sprintf("      <phone2 phonename=\"%d\">%s</phone2>", 
			       a->getPhoneLabelIdx(1)+1, qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getPhone3() && strlen(a->getPhone3())) {
		    encodeEntities(a->getPhone3(), qsDest);
		    s->sprintf("      <phone3 phonename=\"%d\">%s</phone3>", 
			       a->getPhoneLabelIdx(2)+1, qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getPhone4() && strlen(a->getPhone4())) {
		    encodeEntities(a->getPhone4(), qsDest);
		    s->sprintf("      <phone4 phonename=\"%d\">%s</phone4>", 
			       a->getPhoneLabelIdx(3)+1, qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getPhone5() && strlen(a->getPhone5())) {
		    encodeEntities(a->getPhone5(), qsDest);
		    s->sprintf("      <phone5 phonename=\"%d\">%s</phone5>", 
			       a->getPhoneLabelIdx(4)+1, qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getAddress() && strlen(a->getAddress())) {
		    encodeEntities(a->getAddress(), qsDest);
		    s->sprintf("      <address>%s</address>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getTown() && strlen(a->getTown())) {
		    encodeEntities(a->getTown(), qsDest);
		    s->sprintf("      <city>%s</city>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getState() && strlen(a->getState())) {
		    encodeEntities(a->getState(), qsDest);
		    s->sprintf("      <state>%s</state>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getZIP() && strlen(a->getZIP())) {
		    encodeEntities(a->getZIP(), qsDest);
		    s->sprintf("      <zip>%s</zip>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getCountry() && strlen(a->getCountry())) {
		    encodeEntities(a->getCountry(), qsDest);
		    s->sprintf("      <country>%s</country>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getDefined(1) && strlen(a->getDefined(1))) {
		    encodeEntities(a->getDefined(1), qsDest);
		    s->sprintf("      <defined1>%s</defined1>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getDefined(2) && strlen(a->getDefined(2))) {
		    encodeEntities(a->getDefined(2), qsDest);
		    s->sprintf("      <defined2>%s</defined2>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getDefined(3) && strlen(a->getDefined(3))) {
		    encodeEntities(a->getDefined(3), qsDest);
		    s->sprintf("      <defined3>%s</defined3>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  if (a->getDefined(4) && strlen(a->getDefined(4))) {
		    encodeEntities(a->getDefined(4), qsDest);
		    s->sprintf("      <defined4>%s</defined4>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
		  // FIXME
#if 0
		  if (a->getNote() && strlen(a->getNote())) {
		    encodeEntities(a->getNote(), qsDest);
		    s->sprintf("      <note>%s</note>", qsDest.data());
		    t << (const char *)*s << endl;
		  }
#endif		  
		  t << "    </record>" << endl;

                  msg.sprintf("Exporting record no. %3d of %3d", 
                              i+1, AddrList->count());

		  statusBar.SetText(statusMsg, msg);
		}
	        delete s;

		t << "  </addressrecords>" << endl << endl;
   	        t << "</pilot-addresses>" << endl;

	        outFile.close();
	    }
        else
          QMessageBox::warning(this, "File Open Error",
	       		       "Could not open the file for writing.\n");
      }
      startTimer4StatusMsg();
}


void PiAddress::encodeEntities(const char *str, QString &qsDest)
{
  QString s(str);

  s.replace(QRegExp("&"),  "&amp;");
  s.replace(QRegExp("<"),  "&lt;");
  s.replace(QRegExp(">"),  "&gt;");
  s.replace(QRegExp("\""), "&quot;");
  s.replace(QRegExp("'"),  "&apos;");

  qsDest = s;
  
  return;
}


// import CSF-file into Address DB
void PiAddress::importCSV()
{
  QString msg;

  if (AddrList == 0)
  {
   AddrList = new QList<AddressRecord>;
   AddrList->setAutoDelete(TRUE);
  }

  QString f(QFileDialog::getOpenFileName(0, "*.csv", this));
  if (!f.isEmpty())
  {
   QFile inFile(f);
   if (inFile.exists())
   {
    if (inFile.open(IO_ReadOnly))
    {
     AddressRecord *a;

     int n=0;
     int i, ch;
     QString tmp[20], tmp1;      


     // read all lines
     while (!inFile.atEnd())
     {
       ch = inFile.getch();
       if (ch == -1)
	   break;

       if (ch == 0x0a)
	 continue;
       
       inFile.ungetch(ch);
       
       n++; 
       msg.sprintf("Reading record %03d", n);
       statusBar.SetText(statusMsg, msg);

       a = new AddressRecord();

       // initialize temp. QString's
       for (i=0; i<20; i++)
	 tmp[i] = "";

       // 20 fields for one record      
       for (i=0; i<20; i++)
	{
          // look for starting '"'
          while ((ch = inFile.getch()) != '"')
	    ;

          tmp1 = "";
	  
          // read until next '"' and insert read char to tmp QString
	  // FIXME: should check for max. length?
	  while ((ch = inFile.getch()) != (int)'"')
	    tmp1.insert(tmp1.length(), (char)ch);

          // save in corresponding QString
          tmp[i]=tmp1;
	}

       a->putFN((const char *)tmp[0]);
       a->putGN((const char *)tmp[1]);
       a->putTitle((const char *)tmp[2]);
       a->putCompany((const char *)tmp[3]);
       a->putPhone1((const char *)tmp[4]);
       a->putPhone2((const char *)tmp[5]);
       a->putPhone3((const char *)tmp[6]);
       a->putPhone4((const char *)tmp[7]);
       a->putPhone5((const char *)tmp[8]);
       a->putAddress((const char *)tmp[9]);
       a->putZIP((const char *)tmp[12]);
       a->putTown((const char *)tmp[10]);
       a->putState((const char *)tmp[11]);
       a->putCountry((const char *)tmp[13]);
       a->putDefined1((const char *)tmp[14]);
       a->putDefined2((const char *)tmp[15]);
       a->putDefined3((const char *)tmp[16]);
       a->putDefined4((const char *)tmp[17]);
       a->putNote((const char *)tmp[18]);
       a->putPrivate((tmp[19].find('0', 0) == -1) ? TRUE : FALSE);

       a->putPhoneLabelIdx0(0);
       a->putPhoneLabelIdx1(1);
       a->putPhoneLabelIdx2(2);
       a->putPhoneLabelIdx3(3);
       a->putPhoneLabelIdx4(4);

       int i;
       i = cat->currentItem();
       if (i>=cat->count()-1)
	 i = 0;
       a->setCat(i);

       for (i=0; i<5; i++)
         {
           if (strlen(tmp[i+4].data()))
	     {
               a->putShowPhone(i);
	       break;
	     }
         }

       insertSorted(a);
     }
     startTimer4StatusMsg();

     bModified = TRUE;
     QFileInfo fi(inFile);
     QString fn = fi.filePath();
    }
    else
    {
     msg.sprintf("File \"%s\" could not be opened for reading.\n", 
                 (const char*)f);
     QMessageBox::information(this, "File open error", (const char *)msg,
                              "Ok", 0, 0, 1);
    }
   }
   else
   {
    msg.sprintf("File \"%s\" could not be found.\n", (const char*)f);
    QMessageBox::information(this, "File not found", (const char *)msg,
                             "Ok", 0, 0, 1);
   } 
  }
  displayAddresses();
  updateStatusCount();
  if (lbAddr->count())
    {
      lbAddr->setFocus();
      lbAddr->setCurrentItem(0);
    }
  leSearch->setFocus();
}

// import XML-file into Address DB
void PiAddress::importXML()
{
  QString msg;

  if (AddrList == 0)
  {
   AddrList = new QList<AddressRecord>;
   AddrList->setAutoDelete(TRUE);
  }

  QString f(QFileDialog::getOpenFileName(0, "*.xml", this));
  if (!f.isEmpty())
  {
   QFile inFile(f);
   if (inFile.exists())
   {
    if (inFile.open(IO_ReadOnly))
    {
      QTextStream t(&inFile);
      QString s;

      int i, j, n=0;
      bool bStartOfRecords=FALSE;
      AddressRecord *a;

      // read all lines
      while (!t.eof())
      {
        s = t.readLine();

	// ignore comment lines
	if (s.find(QRegExp("^\\s*<!--\\.*$")) != -1)
	  continue;

	// look for start of records
	if (!bStartOfRecords) {
	  if (s.find(QRegExp("<addrrecords>",FALSE)) != -1) {
	    bStartOfRecords = TRUE;
	  }
	  continue;
	}

	// look for end of records
	if (s.find(QRegExp("</addrrecords>",FALSE)) != -1) {
	  break;
	}
	
	// look for start of record
	if (s.find(QRegExp("<record",FALSE)) != -1) {
	  n++; 
	  msg.sprintf("Reading record %03d", n);
	  statusBar.SetText(statusMsg, msg);

	  a = new AddressRecord();

	  while (!t.eof()) {
	    QString qsValue;
	    
	    s = t.readLine();

	    i = s.find(QRegExp("<lastname",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</lastname>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putFN(qsValue.data());
	    }

	    i = s.find(QRegExp("<firstname",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</firstname>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putGN(qsValue.data());
	    }

	    i = s.find(QRegExp("<title",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</title>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putTitle(qsValue.data());
	    }

	    i = s.find(QRegExp("<company",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</company>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putCompany(qsValue.data());
	    }

	    i = s.find(QRegExp("<phone1",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</phone1>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putPhone1(qsValue.data());
	    }

	    i = s.find(QRegExp("<phone2",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</phone2>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putPhone2(qsValue.data());
	    }

	    i = s.find(QRegExp("<phone3",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</phone3>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putPhone3(qsValue.data());
	    }

	    i = s.find(QRegExp("<phone4",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</phone4>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putPhone4(qsValue.data());
	    }

	    i = s.find(QRegExp("<phone5",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</phone5>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putPhone5(qsValue.data());
	    }
	    
	    i = s.find(QRegExp("<address",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</address>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putAddress(qsValue.data());
	    }

	    i = s.find(QRegExp("<city",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</city>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putTown(qsValue.data());
	    }

	    i = s.find(QRegExp("<state",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</state>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putState(qsValue.data());
	    }

	    i = s.find(QRegExp("<zip",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</zip>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putZIP(qsValue.data());
	    }

	    i = s.find(QRegExp("<country",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</country>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putCountry(qsValue.data());
	    }

	    i = s.find(QRegExp("<defined1",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</defined1>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putDefined1(qsValue.data());
	    }

	    i = s.find(QRegExp("<defined2",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</defined2>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putDefined2(qsValue.data());
	    }

	    i = s.find(QRegExp("<defined3",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</defined3>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putDefined3(qsValue.data());
	    }

	    i = s.find(QRegExp("<defined4",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</defined4>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putDefined4(qsValue.data());
	    }

	    // FIXME: note-field!
#if 0
	    i = s.find(QRegExp("<note",FALSE));
	    if (i != -1) {
	      i = s.find('>');
	      j = s.find("</note>",0,FALSE);
	      qsValue = s.mid(i+1, j-i-1);
	      a->putNote(qsValue.data());
	    }
#endif
	    // look for end of record
	    if (s.find(QRegExp("</record>",FALSE)) != -1) {

	      a->putPhoneLabelIdx0(0);
	      a->putPhoneLabelIdx1(1);
	      a->putPhoneLabelIdx2(2);
	      a->putPhoneLabelIdx3(3);
	      a->putPhoneLabelIdx4(4);

	      i = cat->currentItem();
	      if (i>=cat->count()-1)
		i = 0;
	      a->setCat(i);

	      // insert record in list
	      insertSorted(a);
	      break;
	    }
	  }
	}
	
#if 0
       a->putPrivate((tmp[19].find('0', 0) == -1) ? TRUE : FALSE);

       int i;
       for (i=0; i<5; i++)
         {
           if (strlen(tmp[i+4].data()))
	     {
               a->putShowPhone(i);
	       break;
	     }
         }

       insertSorted(a);
#endif
     }
     startTimer4StatusMsg();

     bModified = TRUE;
     QFileInfo fi(inFile);
     QString fn = fi.filePath();
    }
    else
    {
     msg.sprintf("File \"%s\" could not be opened for reading.\n", 
                 (const char*)f);
     QMessageBox::information(this, "File open error", (const char *)msg,
                              "Ok", 0, 0, 1);
    }
   }
   else
   {
    msg.sprintf("File \"%s\" could not be found.\n", (const char*)f);
    QMessageBox::information(this, "File not found", (const char *)msg,
                             "Ok", 0, 0, 1);
   } 
  }
  displayAddresses();
  updateStatusCount();
  if (lbAddr->count())
    {
      lbAddr->setFocus();
      lbAddr->setCurrentItem(0);
    }
  leSearch->setFocus();
}


void PiAddress::initCat()
{
  int i;
  
  cat->clear();
  for (i=0; pAppInfo->category.name[i][0]; i++)
    cat->insertItem(pAppInfo->category.name[i]);

  cat->insertItem("All");
  cat->setCurrentItem(cat->count()-1);
}

void PiAddress::info()
{
  QString s, t1, t2, t3;

  t1 = (creaTime)   ? ctime(&creaTime)   : "-\n";
  t2 = (modTime)    ? ctime(&modTime)    : "-\n";
  t3 = (backupTime) ? ctime(&backupTime) : "-\n";

  s.sprintf("Creation time:     %sModification time: %sBackup time:       %s", 
            (const char *)t1, (const char *)t2, (const char *)t3);

  QMessageBox::about(this, "Pilot Address Application: DB Info", 
                     (const char *)s);
}

void PiAddress::messageBox(QString &s)
{
  QString t;
  t.sprintf("Pilot Address Application %s", VERSION);
  QMessageBox::about(this, (const char *)t, (const char *)s);
}

void PiAddress::helpFAQ()
{
  if (!qsHelpApplication.isEmpty())
    if (system((const char *)qsHelpApplication))
      {
	QString t;
	t.sprintf("Couldn't start the HelpApplication!\n\n"
		  "Please correct the problem in the configuration file \n"
		  "(/etc/pi-addressrc or ~/.pi-addressrc) and try again.\n");
	messageBox(t);
      }
}

void PiAddress::copyright()
{
  QString s;
  s.sprintf("Pilot Address Application %s", VERSION);
  QMessageBox::about(this, (const char *)s,
   "This program is free software; you can redistribute it and/or modify\n"
   "it under the terms of the GNU General Public License as published by\n"
   "the Free Software Foundation; either version 2, or (at your option)\n"
   "any later version.\n"
   "\n"
   "Additionally, you are granted permission to assume, for the purposes\n"
   "of distributing this program in object code or executable form\n"
   "under Section 3 of the GNU Public License, that the QT library\n"
   "is normally distributed with the major components of the\n"
   "operating system on which the executable or object code runs.\n"
   "\n"
   "This program is distributed in the hope that it will be useful,\n"
   "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
   "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
   "GNU General Public License for more details.\n"
   "\n"
   "You should have received a copy of the GNU General Public License\n"
   "along with this program; if not, write to the Free Software\n"
   "Foundation, Inc., 59 Temple Place - Suite 330, Boston,\n"
   "MA 02111-1307, USA.\n"
   "\n"
   "Written by Michael Wiedmann <mw@miwie.in-berlin.de>\n");
}


void PiAddress::about()
{
  QString s;
  s.sprintf("Pilot Address Application %s", VERSION);
  
  QMessageBox::about(this, (const char *)s,
               "This program lets you load/edit/save the\n"
	       "Address DB of a 3Com Palm Pilot.\n"
               "\n"
               "New in 0.3.4:\n"
	       "* Export records in XML format (experimental)\n"
	       "* Bugfix: tried to read systemwide config file\n"
               "  '/etc/pi-address.conf' instead of '/etc/pi-addressrc'\n"
               "* Bugfix: junk characters in category names after\n"
               "  importing PilotManager data files\n"
               "\n"
               "THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY\n"
               "EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT\n"
               "LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY\n"
               "AND FITNESS FOR A PARTICULAR PURPOSE.\n"
               "\n"
               "Copyright 1997-2000 Michael Wiedmann <mw@miwie.in-berlin.de>");
}


void PiAddress::statusMessageReady()
{
  QString qsTmp;
  qsTmp.sprintf("Ready %s", (bModified) ? "(Modified)" : "");
  statusBar.SetText(statusMsg, (const char *)qsTmp);
}

void PiAddress::aboutQt()
{
  QMessageBox::aboutQt( this, "Pilot Address Application" );
}


void PiAddress::readPilot()
{
  QString qsTmp, qsTmpFile;

  if (bModified && confirmNoSave() != 0)
     return;

  bModified = FALSE;
  statusMessageReady();

  // clear the ListBox
  lbAddr->clear();
  if (AddrList != 0)
    {
     delete AddrList;
     AddrList = 0;
    }

  qsTmpFile = "./";
  QFileInfo fiDir(getDir());
  if (fiDir.isDir())
    qsTmpFile = getDir();

  qsTmpFile = tempnam((const char *)qsTmpFile, "pia-");
  qsTmpFile.append(".pdb");

  statusBar.SetText(statusFname, "File: - untitled - ");
  qsFname = "";
  fileMenu->setItemEnabled(idSave,       FALSE);
  fileMenu->setItemEnabled(idReread,     FALSE);
  toolBar.EnableItem(idToolbarSave,      FALSE);
  fileMenu->setItemEnabled(idSaveAs,     FALSE);
  fileMenu->setItemEnabled(idExport,     FALSE);
  fileMenu->setItemEnabled(idExportLDIF, FALSE);
  fileMenu->setItemEnabled(idExportXML,  FALSE);
  toolBar.EnableItem(idToolbarExport,    FALSE);
  fileMenu->setItemEnabled(printID,      FALSE);
  toolBar.EnableItem(idToolbarPrint,     FALSE);
  pilotMenu->setItemEnabled(pilotWrite,  FALSE);
  toolBar.EnableItem(idToolbarWrite,     FALSE);

  updateStatusCount();
  creaTime = modTime = backupTime = 0;

  Pilot::ConnectResult Ret;
  Pilot *pilot = new Pilot(this);

  Ret = pilot->connect(qsDeviceName, bCheckDevice);
  if (Ret == Pilot::Success)
    {
      QString qsUser;
      if (pilot->getUserName(qsUser))
	{
          qsTmp.sprintf("User: %s", (const char *)qsUser);

	    if (pilot->retrieveDB("AddressDB", (const char *)qsTmpFile) == 0)
	    {
	      // success, now just read the temp. file as usual
              QFile f(qsTmpFile.data());
	      readFile(f);

              statusBar.SetText(statusFname, "File: - untitled - ");
              qsFname = "";

              fileMenu->setItemEnabled(idSaveAs,     TRUE);
              fileMenu->setItemEnabled(idSave,       FALSE);
              fileMenu->setItemEnabled(idReread,     FALSE);
              toolBar.EnableItem(idToolbarSave,      FALSE);
              fileMenu->setItemEnabled(idExport,     TRUE);
              fileMenu->setItemEnabled(idExportLDIF, TRUE);
              fileMenu->setItemEnabled(idExportXML,  TRUE);
              toolBar.EnableItem(idToolbarExport,    TRUE);
              fileMenu->setItemEnabled(printID,      TRUE);
              toolBar.EnableItem(idToolbarPrint,     TRUE);
              pilotMenu->setItemEnabled(pilotWrite,  FALSE);
              toolBar.EnableItem(idToolbarWrite,     FALSE);
	      bModified = TRUE;
              updateStatusCount();
	    }
          else
	    {
  	      // failure
              qsTmp.sprintf("Error fetching AddressDB");
	    }
          // delete temp. file
          if (QFile::exists((const char *)qsTmpFile))
	    unlink((const char *)qsTmpFile);
	}
      else
	qsTmp.sprintf("Error reading Username");
    }
  else
    qsTmp.sprintf("Connecting to %s failed", qsDeviceName.data());

  statusBar.SetText(statusMsg, qsTmp);
  startTimer4StatusMsg();

  pilot->disconnect();
  delete pilot;
}

void PiAddress::writePilot()
{
  QString qsTmp;

  if (bModified && confirmNoSave() != 0)
     return;

  if (qsFname.length() == 0)
    return;

  Pilot::ConnectResult Ret;
  Pilot *pilot = new Pilot(this);

  Ret = pilot->connect(qsDeviceName, bCheckDevice);
  if (Ret == Pilot::Success)
    {
      QString qsUser;
      if (pilot->getUserName(qsUser))
	{
          qsTmp.sprintf("User: %s", (const char *)qsUser);
	  if (pilot->installDB("AddressDB", (const char *)qsFname) == 0)
	    // success
            updateStatusCount();
          else
  	    // failure
            qsTmp.sprintf("Error installing AddressDB");
	}
      else
	qsTmp.sprintf("Error reading Username");
    }
  else
    qsTmp.sprintf("Connecting to %s failed", qsDeviceName.data());

  statusBar.SetText(statusMsg, qsTmp);
  startTimer4StatusMsg();

  pilot->disconnect();
  delete pilot;
}


void PiAddress::infoPilot()
{
  QString qsTmp, qsAbout;
  bool bRet=FALSE;
  QString qsUser;
  QString qsCardName;
  QString qsCardManufacturer;
  unsigned long ulUserID;
  unsigned long ulRomVersion;
  unsigned long ulRomSize;
  unsigned long ulRamSize;
  unsigned long ulRamFree;
  unsigned long ulLocale;

  Pilot *pilot = new Pilot(this);

  if (pilot->connect(qsDeviceName, bCheckDevice) == 0)
    {
      bRet = pilot->getUserName(qsUser);
      bRet = pilot->getUserID(ulUserID);
      bRet = pilot->getRomVersion(ulRomVersion);
      bRet = pilot->getRomSize(ulRomSize);
      bRet = pilot->getRamSize(ulRamSize);
      bRet = pilot->getRamFree(ulRamFree);
      bRet = pilot->getCardName(qsCardName);
      bRet = pilot->getCardManufacturer(qsCardManufacturer);
      bRet = pilot->getLocale(ulLocale);
      
      qsTmp.sprintf("Got info from Pilot");
    }
  else  
    qsTmp.sprintf("Connecting to %s failed", qsDeviceName.data());

  pilot->disconnect();
  delete pilot;

  if (bRet)
    {
      qsAbout.sprintf("User Name:     %s\nUser ID:       %7ld\nROM Size:      %7ld\nROM Version:   %lx\nRAM Size:      %7ld\nRAM Free:      %7ld\nCard Name:     %s\nManufacturer:  %s\nLocalization:  0x%8.8lX", 
	              qsUser.data(), ulUserID, ulRomSize, ulRomVersion, 
                      ulRamSize, ulRamFree, qsCardName.data(), 
                      qsCardManufacturer.data(), ulLocale);

      QMessageBox::about(this, "Pilot Address Application: Pilot Info", 
                         qsAbout.data());
    }
  else
    {
      qsAbout.sprintf("Failed to get info from Pilot");
      QMessageBox::about(this, "Pilot Address Application: Pilot Info", 
                         qsAbout.data());
    }

  statusBar.SetText(statusMsg, qsTmp);
  startTimer4StatusMsg();
}


void PiAddress::purgePilot()
{
  QString qsTmp;

  Pilot *pilot = new Pilot(this);

  if (pilot->connect(qsDeviceName, bCheckDevice) == 0)
    {
      pilot->purgeDB();
      qsTmp.sprintf("Purge successful");
    }
  else  
    qsTmp.sprintf("Connecting to %s failed", qsDeviceName.data());

  statusBar.SetText(statusMsg, qsTmp);
  startTimer4StatusMsg();
  
  pilot->disconnect();
  delete pilot;
}

enum {
  h_space =   4
};

void PiAddress::printer()
{
  if (pPrinter == 0)
    pPrinter = new QPrinter();

  bool bPrint2Printer = TRUE;
  if (qsPrintTo.find("File", 0, FALSE) >= 0)
    {
      bPrint2Printer = FALSE;
      pPrinter->setOutputToFile(TRUE);
      
      const char *p = pPrinter->outputFileName();
      if (p == 0 || *p == 0)
	{
          // set generic filename
          pPrinter->setOutputFileName((const char *)qsPrintFileName);
	}
    }

  if (pPrinter->setup(0) == FALSE)
    return;

  if (pPrinter->outputToFile())
    {
      bPrint2Printer = FALSE;
      const char *p = pPrinter->outputFileName();
      if (p == 0 || *p == 0)
        pPrinter->setOutputFileName((const char *)qsPrintFileName);
    } 

  // should not happen
  if (AddrList == 0 || AddrList->count() == 0)
    return;

  int maxW, maxH, i, x, y;

  int marginLeft  = 42;
  int marginTop   = 42;
  int marginBtm   = 42;

  if (pPrinter == 0)
    pPrinter = new QPrinter();

  pPrinter->setCreator("pi-address");


  pPrinter->setDocName("pi-address");
  
  QString qsTmp;
  qsTmp.sprintf("Printing to %s...", 
                (bPrint2Printer) ? pPrinter->printerName() 
                                 : pPrinter->outputFileName());
  statusBar.SetText(statusMsg, (const char *)qsTmp);

  QPrinter::PageSize pageSize = pPrinter->pageSize();
  QPrinter::Orientation orientation = pPrinter->orientation();

  if (orientation == QPrinter::Portrait)
    {
      switch (pageSize)
	{
	  case QPrinter::A4:
            maxW = 595;
            maxH = 842;
            break;
          case QPrinter::B5:
            maxW = 516;
            maxH = 730;
            break;
	  case QPrinter::Letter:
            maxW = 612;
            maxH = 792;
            break;
	  case QPrinter::Legal:
            maxW = 612;
            maxH = 1008;
            break;
	  case QPrinter::Executive:
            maxW = 540;
            maxH = 720;
            break;
	  default:
            maxW = 595; 
            maxH = 842;
            break;
	}
    } 
  else
    {
      switch (pageSize)
	{
	  case QPrinter::A4:
            maxW = 842;
            maxH = 595;
            break;
          case QPrinter::B5:
            maxW = 730; 
            maxH = 516;
            break;
	  case QPrinter::Letter:
            maxW = 792; 
            maxH = 612;
            break;
	  case QPrinter::Legal:
            maxW = 1008; 
            maxH = 612;
            break;
	  case QPrinter::Executive:
            maxW = 720; 
            maxH = 540;
            break;
	  default:
            maxW = 842; 
            maxH = 595;
            break;
	}
    }

  int iCat = cat->currentItem();
  int n    = (int)AddrList->count();
  AddressRecord *a;
  
  QProgressDialog *printProgress = new QProgressDialog("Printing records...", 
                                                       "Cancel", n, this,
                                                       "progress", TRUE);
  printProgress->setProgress(0);

  QPainter p;
  p.begin(pPrinter);

  QFont ft ((const char *)qsPrintingFontName, iPrintingPointSize);
  QFont ftb((const char *)qsPrintingFontName, iPrintingPointSize, QFont::Bold);

  QFontInfo fi(ft);

  initPrintingFields();
  
  x = marginLeft;
  y = marginTop;

  p.setFont(ft);  

  int iCurrent = lbAddr->currentItem();

  char last = 0;

  for(a=AddrList->first(), i=0; a!=0; a=AddrList->next(), i++)
    {
      printProgress->setProgress(i+1);
      if (printProgress->wasCancelled())
        break;

      // new page if necassery
      if (y > maxH - marginBtm)
	{
  	  pPrinter->newPage();
          x = marginLeft;
          y = marginTop;
          p.setFont(ft);  
	}

      // only process records which meet chosen category
      if ( (a->getCat() != iCat) && (iCat < cat->count()-1) )
	continue;

      qsTmp.sprintf("Printing record no. %d...", i+1);
      statusBar.SetText(statusMsg, (const char *)qsTmp);

      const char *ptr = a->getFN();
      char c;
      if (ptr != 0 && *ptr != 0)
         c = tolower(*ptr);
      else
         c = 0;
      if ( c != last )
	{
          p.drawText(x, y, " ");
	  y += (int)(fi.pointSize() * 0.5);
          p.setFont(ftb);
          QString s1;
          s1.sprintf("%c", toupper(c));
	  p.drawText(x, y, s1);
	  y += (int)(fi.pointSize());
	  last = c;
          p.setFont(ft);
	}

      // actually print the record      
      printAddress(a, &p, x, y);

      y += (int)(fi.pointSize() * 1.5);
    }

  printProgress->setProgress(n);
  delete printProgress;

  p.end();

  lbAddr->setFocus();
  if (iCurrent >= 0)
    {
      lbAddr->setCurrentItem(iCurrent);
      lbAddr->centerCurrentItem();
    }
  
  startTimer4StatusMsg();
}

void PiAddress::initPrintingFields()
{
  int i, j, width;
  char ch;
  QString qsTmp;
  QRegExp r("^[0-9]");

  memset(&aiPrintingFields[0], -1, sizeof(aiPrintingFields));
  
  for(i=0, j=0; i<(int)qsPrintingFormat.length(); )
    {
      // search for next '%'
      i = qsPrintingFormat.find('%', i);
      // FIXME:
      if (i == -1 || j == 39)
	break;
  
      i++;
      qsTmp = qsPrintingFormat.mid(i, qsPrintingFormat.length()-i-1);

      // look for specified width
      if (qsTmp.contains(r) > 0)
	{
          width = atoi((const char *)qsTmp);
          while (qsTmp.contains(r) > 0)
	    {
	      qsTmp = qsPrintingFormat.mid(++i, 
                                           qsPrintingFormat.length()-i-1);
	    }
	}
      else
        width = 20;

      ch = *(const char *)qsPrintingFormat.mid(i, 1);
      
      switch (tolower(ch))
        {
	    // Last name
	    case 'l':
	      aiPrintingFields[j++] = 0;
              aiPrintingFields[j++] = width;
	      break;

	    // First name	      
	    case 'f':
	      aiPrintingFields[j++] = 1;
              aiPrintingFields[j++] = width;
	      break;

	    // Title
  	    case 'i':
	      aiPrintingFields[j++] = 2;
              aiPrintingFields[j++] = width;
	      break;

	    // Company
	    case 'o':
	      aiPrintingFields[j++] = 3;
              aiPrintingFields[j++] = width;
	      break;
	  
	    // handle phone 1 - 5
	    case 'p':
              ch = *(const char *)qsPrintingFormat.mid(++i, 1);
              if (ch >= 0x31 && ch <= 0x35)
		{
		  aiPrintingFields[j++] = ch - 0x31 + 4;
                  aiPrintingFields[j++] = width;
		}
	      break;

            // Address
	    case 'a':
	      aiPrintingFields[j++] = 9;
              aiPrintingFields[j++] = width;
	      break;

	    // ZIP
	    case 'z':
	      aiPrintingFields[j++] = 10;
              aiPrintingFields[j++] = width;
	      break;

	    // Town
	    case 't':
	      aiPrintingFields[j++] = 11;
              aiPrintingFields[j++] = width;
	      break;

	    // State
	    case 's':
	      aiPrintingFields[j++] = 12;
              aiPrintingFields[j++] = width;
	      break;

	    // Country
	    case 'n':
	      aiPrintingFields[j++] = 13;
              aiPrintingFields[j++] = width;
	      break;
	    
	    // handle customer field 1 - 4
	    case 'c':  
              ch = *(const char *)qsPrintingFormat.mid(++i, 1);
              if (ch >= 0x31 && ch <= 0x34)
		{
		  aiPrintingFields[j++] = ch - 0x31 + 14;
                  aiPrintingFields[j++] = width;
		}
	      break;
	}
    }
}


void PiAddress::printAddress(AddressRecord *a, QPainter *p, int x, int y)
{
  int xx = x;
  int yy = y;
  int i, w;
  
  QString qsTmp;
  QFontMetrics fm = p->fontMetrics();
  int wChar = fm.width("8");

  if (a && p)
    {
      for(i=0; aiPrintingFields[i] != -1; i+=2)
	{
         switch (aiPrintingFields[i])
	   {
	   case 0:
	     qsTmp = a->getFN();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;
	     
	   case 1:
	     qsTmp = a->getGN();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 2:
	     qsTmp = a->getTitle();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 3:
	     qsTmp = a->getCompany();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 4:
	     qsTmp = a->getPhone1();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 5:
	     qsTmp = a->getPhone2();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 6:
	     qsTmp = a->getPhone3();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 7:
	     qsTmp = a->getPhone4();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 8:
	     qsTmp = a->getPhone5();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 9:
	     qsTmp = a->getAddress();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 10:
	     qsTmp = a->getZIP();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 11:
	     qsTmp = a->getTown();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 12:
	     qsTmp = a->getState();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 13:
	     qsTmp = a->getCountry();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 14:
	     qsTmp = a->getDefined1();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 15:
	     qsTmp = a->getDefined2();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 16:
	     qsTmp = a->getDefined3();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;

	   case 17:
	     qsTmp = a->getDefined4();
             w = wChar * aiPrintingFields[i+1];
             while (fm.width((const char *)qsTmp) > w)
	         qsTmp = qsTmp.left(qsTmp.length()-1);
             p->drawText(xx, yy, (const char *)qsTmp);
             xx += w + h_space;
	     break;
	   }
	}
    }
}


void PiAddress::startTimer4StatusMsg()
{
  QTimer *timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()),
          this,  SLOT(statusMessageReady()));

  timer->start(midTime, TRUE);
}


void PiAddress::paintEvent(QPaintEvent *)
{
  if (lbAddr->count() && lbAddr->currentItem() != -1)
    displayRecord(lbAddr->currentItem());
}


void PiAddress::quit()
{
  if (bModified && confirmNoSave() != 0)
    return;

  qApp->quit();
}


void PiAddress::updateStatusCount()
{
  int ct=0;

  if (AddrList != 0)
    ct = AddrList->count();
  
  QString s;
  s.sprintf("Records: %03d", ct);
  statusBar.SetText(statusCount, (const char *)s);
  if (ct == 0)
    {
       disableButtons();
       lAddr->repaint();
       fileMenu->setItemEnabled(printID,  FALSE);
       toolBar.EnableItem(idToolbarPrint, FALSE);
    }
  else
    enableButtons(0);

  int i = lbAddr->currentItem();
  if (i != -1)
    displayRecord(i);
}

void PiAddress::populateToolbar()
{
  QPixmap pOpen(openXpm);
  QPixmap pNew (newXpm);
  QPixmap pSave(saveXpm);
  QPixmap pImport(importXpm);
  QPixmap pExport(exportXpm);
  QPixmap pPrint(printXpm);
  QPixmap pReadPilot(readXpm);
  QPixmap pWritePilot(writeXpm);

  toolBar.Insert(pOpen, this, SLOT(open()), "Open file");
  toolBar.Insert(pNew,  this, SLOT(news()), "New file");
  idToolbarSave  = toolBar.Insert(pSave, this, SLOT(save()), "Save file");
  toolBar.EnableItem(idToolbarSave, FALSE);

  toolBar.InsertSeparator();
  idToolbarPrint = toolBar.Insert(pPrint, this, SLOT(printer()), 
                   "Print records");
  toolBar.EnableItem(idToolbarPrint, FALSE);

  toolBar.InsertSeparator();
  idToolbarRead  = toolBar.Insert(pReadPilot,  this, SLOT(readPilot()),  
                   "Read from Pilot");
  idToolbarWrite = toolBar.Insert(pWritePilot, this, SLOT(writePilot()), 
                   "Write to Pilot");
  toolBar.EnableItem(idToolbarWrite, FALSE);

  toolBar.InsertSeparator();
  toolBar.Insert(pImport, this, SLOT(importCSV()), "Import as CSV");
  idToolbarExport = toolBar.Insert(pExport, this, SLOT(exportCSV()), 
                    "Export as CSV");
  toolBar.EnableItem(idToolbarExport, FALSE);
  toolBar.InsertSeparator();
}


int main( int argc, char ** argv )
{
    int i;
    for (i=0; i<argc; i++)
    {
      if ((!strcmp(argv[i], "-h")) || (!strcmp(argv[i], "--help")))
      {
        printf("pi-address [-h | -v | database-file]\n");
	printf("   -h | --help     shows this message and exits\n");
	printf("   -v | --version  shows version and exits\n");
	printf("   file            database-file to load on startup\n");
        exit(0);
      }
      if ((!strcmp(argv[i], "-v")) || (!strcmp(argv[i], "--version")))
      {
        printf("pi-address %s\n", VERSION);
        exit(0);
      }
    }
   
    QApplication a( argc, argv );
    PiAddress m;
    
    a.setMainWidget( &m );
    m.show();
    return a.exec();
}

